/*
 * tangle-timeline.c
 *
 * This file is part of Tangle Toolkit - A graphical actor library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-timeline.h"

/**
 * SECTION:tangle-timeline
 * @Short_description: A #ClutterTimeline with frame rate setting
 * @Title: TangleTimeline
 */

G_DEFINE_TYPE(TangleTimeline, tangle_timeline, CLUTTER_TYPE_TIMELINE);

enum {
	PROP_0,
	PROP_FPS
};

struct _TangleTimelinePrivate {
	guint fps;
	
	guint last_frame;
	guint next_frame;
};

static void on_new_frame(ClutterTimeline* clutter_timeline, gint msecs, gpointer user_data);

static guint default_fps = 25;
static guint new_frame_signal_id;

ClutterTimeline* tangle_timeline_new(guint msecs) {

	return CLUTTER_TIMELINE(g_object_new(TANGLE_TYPE_TIMELINE, "duration", msecs, NULL));
}

ClutterTimeline* tangle_timeline_new_with_fps(guint msecs, guint fps) {

	return CLUTTER_TIMELINE(g_object_new(TANGLE_TYPE_TIMELINE, "duration", msecs, "fps", fps, NULL));
}

guint tangle_timeline_get_fps(TangleTimeline* timeline) {
	g_return_val_if_fail(TANGLE_IS_TIMELINE(timeline), 0);

	return timeline->priv->fps;
}

void tangle_timeline_set_fps(TangleTimeline* timeline, guint fps) {
	g_return_if_fail(TANGLE_IS_TIMELINE(timeline));

	if (timeline->priv->fps != fps) {
		timeline->priv->fps = fps;
		timeline->priv->next_frame = 0;
		g_object_notify(G_OBJECT(timeline), "fps");
	}
}

guint tangle_timeline_get_default_fps(void) {

	return default_fps;
}

void tangle_timeline_set_default_fps(guint fps) {
	default_fps = fps;
}

static void tangle_timeline_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleTimeline* timeline;
	
	timeline = TANGLE_TIMELINE(object);

	switch (prop_id) {
		case PROP_FPS:
			tangle_timeline_set_fps(timeline, g_value_get_uint(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_timeline_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleTimeline* timeline;

	timeline = TANGLE_TIMELINE(object);

        switch (prop_id) {
		case PROP_FPS:
			g_value_set_uint(value, timeline->priv->fps);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_timeline_class_init(TangleTimelineClass* timeline_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(timeline_class);

	gobject_class->set_property = tangle_timeline_set_property;
	gobject_class->get_property = tangle_timeline_get_property;
	
	/**
	 * TangleTimeline:fps:
	 *
	 * Frames per second.
	 *
	 * The value is always limited by the default frame rate of the Clutter framework.
	 * The actual frame rate is the fraction of the normal #ClutterTimeline frame rate.
	 * For example, if the default frame rate is 60 fps (for example, in the case of
	 * 60 Hz vertical refresh), and the requested frame rate is 20 fps, a #TangleTimeline
	 * emits only every third ::new-frame signal and drop others.
	 */
	g_object_class_install_property(gobject_class, PROP_FPS,
	                                g_param_spec_uint("fps",
	                                "FPS",
	                                "Frames per second",
	                                0, G_MAXUINT, 25,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	
	new_frame_signal_id = g_signal_lookup("new-frame", TANGLE_TYPE_TIMELINE);
	g_assert(new_frame_signal_id != 0);

	g_type_class_add_private (gobject_class, sizeof (TangleTimelinePrivate));
}

static void tangle_timeline_init(TangleTimeline* timeline) {
	guint signal_id;

	timeline->priv = G_TYPE_INSTANCE_GET_PRIVATE(timeline, TANGLE_TYPE_TIMELINE, TangleTimelinePrivate);

	timeline->priv->fps = default_fps;

	g_signal_connect(timeline, "new-frame", G_CALLBACK(on_new_frame), NULL);
}

static void on_new_frame(ClutterTimeline* clutter_timeline, gint msecs, gpointer user_data) {
	TangleTimeline* timeline;
	guint frame_duration;

	timeline = TANGLE_TIMELINE(clutter_timeline);

	if (timeline->priv->fps != 0) {
		if (clutter_timeline_get_direction(clutter_timeline) == CLUTTER_TIMELINE_FORWARD) {
			if (timeline->priv->last_frame < msecs &&
			    timeline->priv->next_frame > msecs) {
				g_signal_stop_emission(timeline, new_frame_signal_id, 0);
			} else {
				frame_duration = 1000 / timeline->priv->fps;
				timeline->priv->next_frame = ((msecs / frame_duration) + 1) * frame_duration;
			}
		} else {
			if (timeline->priv->last_frame > msecs &&
			    timeline->priv->next_frame < msecs) {
				g_signal_stop_emission(timeline, new_frame_signal_id, 0);
			} else {
				frame_duration = 1000 / timeline->priv->fps;
				timeline->priv->next_frame = (msecs / frame_duration) * frame_duration;
			}
		}
		timeline->priv->last_frame = msecs;
	}
}
