/*
 * tangle-set-property-action.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-set-property-action.h"
#include "tangle-misc.h"
#include <string.h>

/**
 * SECTION:tangle-set-property-action
 * @Short_description: An action that sets a property
 * @Title: TangleSetPropertyAction
 */

G_DEFINE_TYPE(TangleSetPropertyAction, tangle_set_property_action, TANGLE_TYPE_ACTION);

enum {
	PROP_0,
	PROP_OBJECT,
	PROP_NAME,
	PROP_VALUE_STRING
};

struct _TangleSetPropertyActionPrivate {
	GObject* object;
	gchar* name;
	GValue value;
	gchar* value_string;
};

static gboolean set_value(TangleSetPropertyAction* action, const GValue* value);

TangleAction* tangle_set_property_action_new() {

	return TANGLE_ACTION(g_object_new(TANGLE_TYPE_SET_PROPERTY_ACTION, NULL));
}

GObject* tangle_set_property_action_get_object(TangleSetPropertyAction* action) {

	return action->priv->object;
}

void tangle_set_property_action_set_object(TangleSetPropertyAction* action, GObject* object) {
	if (action->priv->object != object) {
		if (action->priv->object) {
			g_object_unref(action->priv->object);
		}
		action->priv->object = object;
		g_object_ref(object);
		g_object_notify(G_OBJECT(action), "object");
	}
}

const gchar*tangle_set_property_action_get_name(TangleSetPropertyAction* action) {

	return action->priv->name;
}

void tangle_set_property_action_set_name(TangleSetPropertyAction* action, const gchar* name) {
	if (!action->priv->name || strcmp(action->priv->name, name)) {
		g_free(action->priv->name);
		action->priv->name = g_strdup(name);
		g_object_notify(G_OBJECT(action), name);
	}
}

void tangle_set_property_action_get_value(TangleSetPropertyAction* action, GValue* value) {
	*value = action->priv->value;
}

void tangle_set_property_action_set_value(TangleSetPropertyAction* action, const GValue* value) {	
	if (set_value(action, value)) {
		g_free(action->priv->value_string);
		action->priv->value_string = NULL;
		g_object_notify(G_OBJECT(action), "value-string");
	}
}

const gchar* tangle_set_property_action_get_value_string(TangleSetPropertyAction* action) {
	if (!action->priv->value_string) {
		action->priv->value_string = g_strdup_value_contents(&action->priv->value);
	}
	
	return action->priv->value_string;
}

void tangle_set_property_action_set_value_string(TangleSetPropertyAction* action, const gchar* value_string) {	
	GValue value;

	/* TODO: parse string */

	if (set_value(action, &value)) {
		g_free(action->priv->value_string);
		action->priv->value_string = g_strdup(value_string);
		g_object_notify(G_OBJECT(action), "value-string");	
	}
}

static void tangle_set_property_action_execute(TangleAction* tangle_action, GObject* source, const gchar* trigger, TangleProperties* properties) {
	TangleSetPropertyAction* action;
	
	action = TANGLE_SET_PROPERTY_ACTION(tangle_action);
	g_return_if_fail(action->priv->object != NULL || source != NULL);
	
	/* TODO: This is just very initial version... */
	g_object_set_property((action->priv->object ? action->priv->object : source), action->priv->name, &action->priv->value);
}

static void tangle_set_property_action_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleSetPropertyAction* action;
	
	action = TANGLE_SET_PROPERTY_ACTION(object);

	switch (prop_id) {
		case PROP_OBJECT:
			tangle_set_property_action_set_object(action, g_value_get_object(value));
			break;
		case PROP_NAME:
			tangle_set_property_action_set_name(action, g_value_get_string(value));
			break;
		case PROP_VALUE_STRING:
			tangle_set_property_action_set_value_string(action, g_value_get_string(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_set_property_action_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleSetPropertyAction* action;

	action = TANGLE_SET_PROPERTY_ACTION(object);

        switch (prop_id) {
		case PROP_OBJECT:
			g_value_set_object(value, action->priv->object);
			break;
		case PROP_NAME:
			g_value_set_string(value, action->priv->name);
			break;
		case PROP_VALUE_STRING:
			g_value_set_string(value, tangle_set_property_action_get_value_string(action));
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_set_property_action_finalize(GObject* object) {
	TangleSetPropertyAction* action;
	
	action = TANGLE_SET_PROPERTY_ACTION(object);
	g_free(action->priv->name);
	g_value_unset(&action->priv->value);
	
	G_OBJECT_CLASS(tangle_set_property_action_parent_class)->finalize(object);
}

static void tangle_set_property_action_dispose(GObject* object) {
	TangleSetPropertyAction* action;
	
	action = TANGLE_SET_PROPERTY_ACTION(object);
	TANGLE_UNREF_AND_NULLIFY_OBJECT(action->priv->object);
	
	G_OBJECT_CLASS(tangle_set_property_action_parent_class)->dispose(object);
}

static void tangle_set_property_action_class_init(TangleSetPropertyActionClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);
	TangleActionClass* action_class = TANGLE_ACTION_CLASS(klass);

	gobject_class->finalize = tangle_set_property_action_finalize;
	gobject_class->dispose = tangle_set_property_action_dispose;
	gobject_class->set_property = tangle_set_property_action_set_property;
	gobject_class->get_property = tangle_set_property_action_get_property;

	action_class->execute = tangle_set_property_action_execute;

	/**
	 * TangleSetPropertyAction:object:
	 */
	g_object_class_install_property(gobject_class, PROP_OBJECT,
	                                g_param_spec_object("object",
	                                                    "Object",
	                                                    "The object of which property is set",
	                                                    G_TYPE_OBJECT,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleSetPropertyAction:name:
	 */
	g_object_class_install_property(gobject_class, PROP_NAME,
	                                g_param_spec_string("name",
	                                                    "Name",
	                                                    "The name of the property that is set",
	                                                    NULL,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleSetPropertyAction:value-string:
	 */
	g_object_class_install_property(gobject_class, PROP_VALUE_STRING,
	                                g_param_spec_string("value-string",
	                                                    "Value string",
	                                                    "The value of the property as a string",
	                                                    NULL,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_type_class_add_private (gobject_class, sizeof (TangleSetPropertyActionPrivate));
}

static void tangle_set_property_action_init(TangleSetPropertyAction* action) {
	action->priv = G_TYPE_INSTANCE_GET_PRIVATE(action, TANGLE_TYPE_SET_PROPERTY_ACTION, TangleSetPropertyActionPrivate);
}

static gboolean set_value(TangleSetPropertyAction* action, const GValue* value) {
//	if (tangle_value_compare(&action->priv->value, value)) {
		g_value_unset(&action->priv->value);
		g_value_init(&action->priv->value, G_VALUE_TYPE(value));
		g_value_copy(value, &action->priv->value);
		g_object_notify(G_OBJECT(action), "value");
//	}
	
	return TRUE;
}
