/*
 * tangle-scroll-trick.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-scroll-trick.h"

G_DEFINE_TYPE(TangleScrollTrick, tangle_scroll_trick, TANGLE_TYPE_TRICK);

enum {
	PROP_0,
	PROP_AXIS,
	PROP_OFFSET_X,
	PROP_OFFSET_Y,
	PROP_MAX_OFFSET_X,
	PROP_MAX_OFFSET_Y,
	PROP_PAGE_LENGTH_X,
	PROP_PAGE_LENGTH_Y,
	PROP_REPEAT_AXIS,
};

struct _TangleScrollTrickPrivate {
	TangleScrollAxis axis;
	gfloat offset_x;
	gfloat offset_y;
	gfloat max_offset_x;
	gfloat max_offset_y;
	gfloat page_length_x;
	gfloat page_length_y;
	TangleScrollAxis repeat_axis;
	
	guint fix_repeat_x : 1;
	guint fix_repeat_y : 1;
};

static const GEnumValue scroll_axis_values[] = {
	{ TANGLE_SCROLL_NO_AXIS, "TANGLE_SCROLL_NO_AXIS", "no-axis" },
	{ TANGLE_SCROLL_X_AXIS, "TANGLE_SCROLL_X_AXIS", "x-axis" },
	{ TANGLE_SCROLL_Y_AXIS, "TANGLE_SCROLL_Y_AXIS", "y-axis" },
	{ TANGLE_SCROLL_XY_AXIS, "TANGLE_SCROLL_XY_AXIS", "xy-axis" },
	{ 0, NULL, NULL }
};

GType tangle_scroll_axis_get_type(void) {
	static GType type = 0;
	
	if (!type) {
		type = g_enum_register_static("TangleScrollAxis", scroll_axis_values);
	}
	
	return type;
}

TangleTrick* tangle_scroll_trick_new() {

	return TANGLE_TRICK(g_object_new(TANGLE_TYPE_SCROLL_TRICK, NULL));
}

TangleScrollAxis tangle_scroll_trick_get_axis(TangleScrollTrick* scroll_trick) {

	return scroll_trick->priv->axis;
}

void tangle_scroll_trick_set_axis(TangleScrollTrick* scroll_trick, TangleScrollAxis axis) {
	if (scroll_trick->priv->axis != axis) {
		scroll_trick->priv->axis = axis;
		g_object_notify(G_OBJECT(scroll_trick), "axis");
		tangle_layout_changed(tangle_trick_get_layout(TANGLE_TRICK(scroll_trick)));
	}
}

void tangle_scroll_trick_get_offsets(TangleScrollTrick* scroll_trick, gfloat* offset_x_return, gfloat* offset_y_return) {
	if (offset_x_return) {
		*offset_x_return = scroll_trick->priv->offset_x;
	}
	if (offset_y_return) {
		*offset_y_return = scroll_trick->priv->offset_y;
	}
}

void tangle_scroll_trick_set_offsets(TangleScrollTrick* scroll_trick, gfloat offset_x, gfloat offset_y) {
	gboolean changed = FALSE;
	TangleWidget* widget;
	
	if (scroll_trick->priv->offset_x != offset_x) {
		scroll_trick->priv->offset_x = offset_x;
		g_object_notify(G_OBJECT(scroll_trick), "offset-x");
		changed = TRUE;
	}
	if (scroll_trick->priv->offset_y != offset_y) {
		scroll_trick->priv->offset_y = offset_y;
		g_object_notify(G_OBJECT(scroll_trick), "offset-y");
		changed = TRUE;
	}
	if (changed && (widget = tangle_layout_get_widget(tangle_trick_get_layout(TANGLE_TRICK(scroll_trick))))) {
		clutter_actor_queue_relayout(CLUTTER_ACTOR(widget));
	}
}

void tangle_scroll_trick_get_max_offsets(TangleScrollTrick* scroll_trick, gfloat* max_offset_x_return, gfloat* max_offset_y_return) {
	if (max_offset_x_return) {
		*max_offset_x_return = scroll_trick->priv->max_offset_x;
	}
	if (max_offset_y_return) {
		*max_offset_y_return = scroll_trick->priv->max_offset_y;
	}
}

void tangle_scroll_trick_get_page_lengths(TangleScrollTrick* scroll_trick, gfloat* page_length_x_return, gfloat* page_length_y_return) {
	if (page_length_x_return) {
		*page_length_x_return = scroll_trick->priv->page_length_x;
	}
	if (page_length_y_return) {
		*page_length_y_return = scroll_trick->priv->page_length_y;
	}
}

TangleScrollAxis tangle_scroll_trick_get_repeat_axis(TangleScrollTrick* scroll_trick) {

	return scroll_trick->priv->repeat_axis;
}

void tangle_scroll_trick_set_repeat_axis(TangleScrollTrick* scroll_trick, TangleScrollAxis repeat_axis) {
	if (scroll_trick->priv->repeat_axis != repeat_axis && repeat_axis >= TANGLE_SCROLL_NO_AXIS && repeat_axis <= TANGLE_SCROLL_XY_AXIS) {
		scroll_trick->priv->repeat_axis = repeat_axis;
		g_object_notify(G_OBJECT(scroll_trick), "repeat-axis");	
	}
}

void tangle_scroll_trick_clamp_page_boundaries(TangleScrollTrick* scroll_trick, TangleClamp* clamp) {
	TangleClampDirection direction;
	gfloat original_value;
	gfloat page_size;
	
	g_return_if_fail(TANGLE_IS_SCROLL_TRICK(scroll_trick));
	
	direction = tangle_clamp_get_direction(clamp);
	original_value = tangle_clamp_get_original_value(clamp);
	
	if (direction == TANGLE_CLAMP_HORIZONTAL) {
		if (original_value < 0.0) {
			tangle_clamp_set_clamped_value(clamp, 0.0);
		} else if (original_value > scroll_trick->priv->max_offset_x) {
			tangle_clamp_set_clamped_value(clamp, scroll_trick->priv->max_offset_x);
		} else {
			tangle_clamp_set_clamped_value(clamp, ((guint)((original_value + scroll_trick->priv->page_length_x / 2) / scroll_trick->priv->page_length_x)) * scroll_trick->priv->page_length_x);
		}
	} else if (direction == TANGLE_CLAMP_VERTICAL) {
		if (original_value < 0.0) {
			tangle_clamp_set_clamped_value(clamp, 0.0);
		} else if (original_value > scroll_trick->priv->max_offset_y) {
			tangle_clamp_set_clamped_value(clamp, scroll_trick->priv->max_offset_y);
		} else {
			tangle_clamp_set_clamped_value(clamp, ((guint)((original_value + scroll_trick->priv->page_length_y / 2) / scroll_trick->priv->page_length_y)) * scroll_trick->priv->page_length_y);
		}
	}
}

static void tangle_scroll_trick_manipulate_allocation(TangleTrick* trick, TangleWidget* widget, ClutterActorBox* box, ClutterAllocationFlags flags) {
	TangleScrollTrick* scroll_trick;
	gfloat w, h, page_length_x, page_length_y, max_offset_x, max_offset_y;
		
	scroll_trick = TANGLE_SCROLL_TRICK(trick);
	
	if (clutter_actor_get_request_mode(CLUTTER_ACTOR(widget)) == CLUTTER_REQUEST_HEIGHT_FOR_WIDTH) {
		tangle_layout_get_preferred_width(tangle_trick_get_layout(trick), -1, NULL, &w);
		tangle_layout_get_preferred_height(tangle_trick_get_layout(trick), w, NULL, &h);
		
	} else {
		tangle_layout_get_preferred_height(tangle_trick_get_layout(trick), -1, NULL, &h);
		tangle_layout_get_preferred_width(tangle_trick_get_layout(trick), h, NULL, &w);	
	}

	if (scroll_trick->priv->axis & TANGLE_SCROLL_X_AXIS) {
		page_length_x = box->x2 - box->x1;
		if (w < page_length_x) {
			w = page_length_x;
			scroll_trick->priv->fix_repeat_x = TRUE;
		} else {
			scroll_trick->priv->fix_repeat_x = FALSE;	
		}
		if (scroll_trick->priv->page_length_x != page_length_x) {
			scroll_trick->priv->page_length_x = page_length_x;
			g_object_notify(G_OBJECT(scroll_trick), "page-length-x");

		}
		max_offset_x = w - page_length_x;
		if (scroll_trick->priv->max_offset_x != max_offset_x) {
			scroll_trick->priv->max_offset_x = max_offset_x;
			g_object_notify(G_OBJECT(scroll_trick), "max-offset-x");
		}
		box->x2 = box->x1 + w;
	}

	if (scroll_trick->priv->axis & TANGLE_SCROLL_Y_AXIS) {
		page_length_y = box->y2 - box->y1;
		if (h < page_length_y) {
			h = page_length_y;
			scroll_trick->priv->fix_repeat_y = TRUE;
		} else {
			scroll_trick->priv->fix_repeat_y = FALSE;	
		}
		if (scroll_trick->priv->page_length_y != h) {
			scroll_trick->priv->page_length_y = page_length_y;
			g_object_notify(G_OBJECT(scroll_trick), "page-length-y");
		}
		max_offset_y = h - (box->y2 - box->y1);
		if (max_offset_y < 0) {
			max_offset_y = 0;
		}
		if (scroll_trick->priv->max_offset_y != max_offset_y) {
			scroll_trick->priv->max_offset_y = max_offset_y;
			g_object_notify(G_OBJECT(scroll_trick), "max-offset-y");
		}
		box->y2 = box->y1 + h;
	}
}

static void tangle_scroll_trick_manipulate_child_allocation(TangleTrick* trick, TangleWidget* widget, ClutterActor* actor, ClutterActorBox* box, ClutterAllocationFlags flags) {
	TangleScrollTrick* scroll_trick;
	gfloat width, height;
	guint n;
	
	scroll_trick = TANGLE_SCROLL_TRICK(trick);
	
	if (scroll_trick->priv->axis & TANGLE_SCROLL_X_AXIS) {
		box->x1 -= scroll_trick->priv->offset_x;
		box->x2 -= scroll_trick->priv->offset_x;

		width = scroll_trick->priv->max_offset_x + scroll_trick->priv->page_length_x;
		if ((scroll_trick->priv->repeat_axis & TANGLE_SCROLL_X_AXIS) && box->x1 > scroll_trick->priv->page_length_x) {
			n = (box->x1 - scroll_trick->priv->page_length_x) / width + 1;
			if (scroll_trick->priv->fix_repeat_x) {
				n = ((n + 1) / 2) * 2;
			}
			box->x1 -= n * width;
			box->x2 -= n * width;
		} else if ((scroll_trick->priv->repeat_axis & TANGLE_SCROLL_X_AXIS) && box->x2 < 0.0) {
			n = -box->x2 / width + 1;
			if (scroll_trick->priv->fix_repeat_x) {
				n = ((n + 1) / 2) * 2;
			}
			box->x1 += n * width;
			box->x2 += n * width;
		}
	}
	
	if (scroll_trick->priv->axis & TANGLE_SCROLL_Y_AXIS) {
		box->y1 -= scroll_trick->priv->offset_y;
		box->y2 -= scroll_trick->priv->offset_y;

		height = scroll_trick->priv->max_offset_y + scroll_trick->priv->page_length_y;
		if ((scroll_trick->priv->repeat_axis & TANGLE_SCROLL_Y_AXIS) && box->y1 > scroll_trick->priv->page_length_y) {
			n = (box->y1 - scroll_trick->priv->page_length_y) / height + 1;
			if (scroll_trick->priv->fix_repeat_y) {
				n = ((n + 1) / 2) * 2;
			}
			box->y1 -= n * height;
			box->y2 -= n * height;
		} else if ((scroll_trick->priv->repeat_axis & TANGLE_SCROLL_Y_AXIS) && box->y2 < 0.0) {
			n = -box->y2 / height + 1;
			if (scroll_trick->priv->fix_repeat_y) {
				n = ((n + 1) / 2) * 2;
			}
			box->y1 += n * height;
			box->y2 += n * height;
		}
	}
}

static void tangle_scroll_trick_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleScrollTrick* scroll_trick;
	
	scroll_trick = TANGLE_SCROLL_TRICK(object);

	switch (prop_id) {
		case PROP_AXIS:
			tangle_scroll_trick_set_axis(scroll_trick, g_value_get_enum(value));
			break;
		case PROP_OFFSET_X:
			tangle_scroll_trick_set_offsets(scroll_trick, g_value_get_float(value), scroll_trick->priv->offset_y);
			break;
		case PROP_OFFSET_Y:
			tangle_scroll_trick_set_offsets(scroll_trick, scroll_trick->priv->offset_x, g_value_get_float(value));
			break;
		case PROP_REPEAT_AXIS:
			tangle_scroll_trick_set_repeat_axis(scroll_trick, g_value_get_enum(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_scroll_trick_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleScrollTrick* scroll_trick;

	scroll_trick = TANGLE_SCROLL_TRICK(object);

        switch (prop_id) {
		case PROP_AXIS:
			g_value_set_enum(value, scroll_trick->priv->axis);
			break;
		case PROP_OFFSET_X:
			g_value_set_float(value, scroll_trick->priv->offset_x);
			break;
		case PROP_OFFSET_Y:
			g_value_set_float(value, scroll_trick->priv->offset_y);
			break;
		case PROP_MAX_OFFSET_X:
			g_value_set_float(value, scroll_trick->priv->max_offset_x);
			break;
		case PROP_MAX_OFFSET_Y:
			g_value_set_float(value, scroll_trick->priv->max_offset_y);
			break;
		case PROP_PAGE_LENGTH_X:
			g_value_set_float(value, scroll_trick->priv->page_length_x);
			break;
		case PROP_PAGE_LENGTH_Y:
			g_value_set_float(value, scroll_trick->priv->page_length_y);
			break;
		case PROP_REPEAT_AXIS:
			g_value_set_enum(value, scroll_trick->priv->repeat_axis);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_scroll_trick_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_scroll_trick_parent_class)->finalize(object);
}

static void tangle_scroll_trick_dispose(GObject* object) {
	G_OBJECT_CLASS(tangle_scroll_trick_parent_class)->dispose(object);
}

static void tangle_scroll_trick_class_init(TangleScrollTrickClass* scroll_trick_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(scroll_trick_class);
	TangleTrickClass* trick_class = TANGLE_TRICK_CLASS(scroll_trick_class);

	gobject_class->finalize = tangle_scroll_trick_finalize;
	gobject_class->dispose = tangle_scroll_trick_dispose;
	gobject_class->set_property = tangle_scroll_trick_set_property;
	gobject_class->get_property = tangle_scroll_trick_get_property;
	
	trick_class->manipulate_allocation = tangle_scroll_trick_manipulate_allocation;
	trick_class->manipulate_child_allocation = tangle_scroll_trick_manipulate_child_allocation;	

	/**
	 * TangleScrollTrick:axis:
	 *
	 * The axis of which is scrolled.
	 */
	g_object_class_install_property(gobject_class, PROP_AXIS,
	                                g_param_spec_enum("axis",
	                                "Axis",
	                                "The axis of which is scrolled",
	                                TANGLE_TYPE_SCROLL_AXIS, TANGLE_SCROLL_XY_AXIS,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollTrick:offset-x:
	 *
	 * The value of which every child actor is moved backwards along the horizontal axis.
	 */
	g_object_class_install_property(gobject_class, PROP_OFFSET_X,
	                                g_param_spec_float("offset-x",
	                                "Offset X",
	                                "The value of which every child actor is moved backwards along the horizontal axis",
	                                -G_MAXFLOAT, G_MAXFLOAT, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollTrick:offset-y:
	 *
	 * The value of which every child actor is moved backwards along the vertical axis.
	 */
	g_object_class_install_property(gobject_class, PROP_OFFSET_Y,
	                                g_param_spec_float("offset-y",
	                                "Offset Y",
	                                "The value of which every child actor is moved backwards along the vertical axis",
	                                -G_MAXFLOAT, G_MAXFLOAT, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollTrick:max-offset-x:
	 *
	 * The maximum value of offset-x when all child actors are within scrolling boundaries.
	 */
	g_object_class_install_property(gobject_class, PROP_MAX_OFFSET_X,
	                                g_param_spec_float("max-offset-x",
	                                "Max Offset X",
	                                "The maximum value of offset-x when all child actors are within scrolling boundaries",
	                                0, G_MAXFLOAT, 0,
	                                G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollTrick:max-offset-y:
	 *
	 * The maximum value of offset-y when all child actors are within scrolling boundaries.
	 */
	g_object_class_install_property(gobject_class, PROP_MAX_OFFSET_Y,
	                                g_param_spec_float("max-offset-y",
	                                "Max Offset Y",
	                                "The maximum value of offset-y when all child actors are within scrolling boundaries",
	                                0, G_MAXFLOAT, 0,
	                                G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollTrick:page-length-x:
	 *
	 * The width of the visible area of the associated #TangleWidget (via #TangleLayout)
	 */
	g_object_class_install_property(gobject_class, PROP_PAGE_LENGTH_X,
	                                g_param_spec_float("page-length-x",
	                                "Page Length X",
	                                "The width of the visible area of rhe associated widget (via layout)",
	                                0, G_MAXFLOAT, 0,
	                                G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollTrick:page-length-y:
	 *
	 * The height of the visible area of the associated #TangleWidget (via #TangleLayout)
	 */
	g_object_class_install_property(gobject_class, PROP_PAGE_LENGTH_Y,
	                                g_param_spec_float("page-length-y",
	                                "Page Length Y",
	                                "The height of the visible area of rhe associated widget (via layout)",
	                                0, G_MAXFLOAT, 0,
	                                G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollTrick:repeat-axis:
	 *
	 * Specifies whether the scrolled content should be repeated horizontally and/or vertically.
	 */
	g_object_class_install_property(gobject_class, PROP_REPEAT_AXIS,
	                                g_param_spec_enum("repeat-axis",
	                                "Repeat axis",
	                                "Specifies whether the scrolled content should be repeated horizontally and/or vertically",
	                                TANGLE_TYPE_SCROLL_AXIS,
					TANGLE_SCROLL_NO_AXIS,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof(TangleScrollTrickPrivate));
}

static void tangle_scroll_trick_init(TangleScrollTrick* scroll_trick) {
	scroll_trick->priv = G_TYPE_INSTANCE_GET_PRIVATE(scroll_trick, TANGLE_TYPE_SCROLL_TRICK, TangleScrollTrickPrivate);
	scroll_trick->priv->axis = TANGLE_SCROLL_XY_AXIS;
}

