/*
 * tangle-stretch-action.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-stretch-action.h"
#include "tangle-misc.h"

G_DEFINE_TYPE(TangleStretchAction, tangle_stretch_action, CLUTTER_TYPE_DRAG_ACTION);

enum {
	PROP_0,
	PROP_STRETCHED_ACTOR,
	PROP_MIN_WIDTH,
	PROP_MIN_HEIGHT,
	PROP_MAX_WIDTH,
	PROP_MAX_HEIGHT
};

struct _TangleStretchActionPrivate {
	ClutterActor* stretched_actor;
	gfloat min_width;
	gfloat min_height;
	gfloat max_width;
	gfloat max_height;
	
	gfloat original_width;
	gfloat original_height;
};

ClutterAction* tangle_stretch_action_new(ClutterActor* stretched_actor) {

	return CLUTTER_ACTION(g_object_new(TANGLE_TYPE_STRETCH_ACTION, "stretched-actor", stretched_actor, NULL));
}

ClutterActor* tangle_stretch_action_get_stretched_actor(TangleStretchAction* stretch_action) {

	return stretch_action->priv->stretched_actor;
}

void tangle_stretch_action_set_stretched_actor(TangleStretchAction* stretch_action, ClutterActor* stretched_actor) {
	if (stretch_action->priv->stretched_actor != stretched_actor) {
		if (stretch_action->priv->stretched_actor) {
			g_object_unref(stretch_action->priv->stretched_actor);
		}
		
		stretch_action->priv->stretched_actor = stretched_actor;

		if (stretch_action->priv->stretched_actor) {
			g_object_ref(stretch_action->priv->stretched_actor);
		}
		
		g_object_notify(G_OBJECT(stretch_action), "stretched-actor");
	}
}

void tangle_stretch_action_get_min_size(TangleStretchAction* stretch_action, gfloat* min_width_return, gfloat* min_height_return) {
	if (min_width_return) {
		*min_width_return = stretch_action->priv->min_width;
	}
	if (min_height_return) {
		*min_height_return = stretch_action->priv->min_height;
	}
}

void tangle_stretch_action_set_min_size(TangleStretchAction* stretch_action, gfloat min_width, gfloat min_height) {
	if (stretch_action->priv->min_width = min_width) {
		stretch_action->priv->min_width = min_width;
		g_object_notify(G_OBJECT(stretch_action), "min-width");
	}
	if (stretch_action->priv->min_height = min_height) {
		stretch_action->priv->min_height = min_height;
		g_object_notify(G_OBJECT(stretch_action), "min-height");
	}
}

void tangle_stretch_action_get_max_size(TangleStretchAction* stretch_action, gfloat* max_width_return, gfloat* max_height_return) {
	if (max_width_return) {
		*max_width_return = stretch_action->priv->max_width;
	}
	if (max_height_return) {
		*max_height_return = stretch_action->priv->max_height;
	}
}

void tangle_stretch_action_set_max_size(TangleStretchAction* stretch_action, gfloat max_width, gfloat max_height) {
	if (stretch_action->priv->max_width = max_width) {
		stretch_action->priv->max_width = max_width;
		g_object_notify(G_OBJECT(stretch_action), "max-width");
	}
	if (stretch_action->priv->max_height = max_height) {
		stretch_action->priv->max_height = max_height;
		g_object_notify(G_OBJECT(stretch_action), "max-height");
	}
}

static void tangle_stretch_action_drag_begin(ClutterDragAction* drag_action, ClutterActor* actor, gfloat event_x, gfloat event_y, ClutterModifierType modifiers) {
	TangleStretchAction* stretch_action;
	ClutterActor* handle;
	
	stretch_action = TANGLE_STRETCH_ACTION(drag_action);

	clutter_actor_get_size(stretch_action->priv->stretched_actor, &stretch_action->priv->original_width, &stretch_action->priv->original_height);

	handle = clutter_rectangle_new();
	clutter_container_add_actor(CLUTTER_CONTAINER(clutter_actor_get_stage(actor)), handle);
	clutter_drag_action_set_drag_handle(drag_action, handle);

	if (CLUTTER_DRAG_ACTION_CLASS(tangle_stretch_action_parent_class)->drag_begin) {
		CLUTTER_DRAG_ACTION_CLASS(tangle_stretch_action_parent_class)->drag_begin(drag_action, actor, event_x, event_y, modifiers);
	}
}

static void tangle_stretch_action_drag_motion(ClutterDragAction* drag_action, ClutterActor* actor, gfloat delta_x, gfloat delta_y) {
	TangleStretchAction* stretch_action;
	gfloat press_x, press_y, motion_x, motion_y;
	gfloat width;
	gfloat height;
	
	stretch_action = TANGLE_STRETCH_ACTION(drag_action);

	clutter_drag_action_get_press_coords(drag_action, &press_x, &press_y);
	clutter_drag_action_get_motion_coords(drag_action, &motion_x, &motion_y);

	width = stretch_action->priv->original_width + motion_x - press_x;
	if (width >= stretch_action->priv->min_width && width <= stretch_action->priv->max_width) {
		clutter_actor_set_width(stretch_action->priv->stretched_actor, width);
	}

	height = stretch_action->priv->original_height + motion_y - press_y;
	if (height >= stretch_action->priv->min_height && height <= stretch_action->priv->max_height) {
		clutter_actor_set_height(stretch_action->priv->stretched_actor, height);
	}
	
	if (CLUTTER_DRAG_ACTION_CLASS(tangle_stretch_action_parent_class)->drag_motion) {
		CLUTTER_DRAG_ACTION_CLASS(tangle_stretch_action_parent_class)->drag_motion(drag_action, actor, delta_x, delta_y);
	}
}

static void tangle_stretch_action_drag_end(ClutterDragAction* drag_action, ClutterActor* actor, gfloat event_x, gfloat event_y, ClutterModifierType modifiers) {
	clutter_actor_destroy(clutter_drag_action_get_drag_handle(drag_action));
}

static void tangle_stretch_action_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleStretchAction* stretch_action;
	
	stretch_action = TANGLE_STRETCH_ACTION(object);

	switch (prop_id) {
		case PROP_STRETCHED_ACTOR:
			tangle_stretch_action_set_stretched_actor(stretch_action, CLUTTER_ACTOR(g_value_get_object(value)));
			break;
		case PROP_MIN_WIDTH:
			tangle_stretch_action_set_min_size(stretch_action, g_value_get_float(value), stretch_action->priv->min_height);
			break;
		case PROP_MIN_HEIGHT:
			tangle_stretch_action_set_min_size(stretch_action, stretch_action->priv->min_width, g_value_get_float(value));
			break;
		case PROP_MAX_WIDTH:
			tangle_stretch_action_set_max_size(stretch_action, g_value_get_float(value), stretch_action->priv->max_height);
			break;
		case PROP_MAX_HEIGHT:
			tangle_stretch_action_set_max_size(stretch_action, stretch_action->priv->max_width, g_value_get_float(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_stretch_action_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleStretchAction* stretch_action;

	stretch_action = TANGLE_STRETCH_ACTION(object);

        switch (prop_id) {
		case PROP_STRETCHED_ACTOR:
			g_value_set_object(value, stretch_action->priv->stretched_actor);
			break;
		case PROP_MIN_WIDTH:
			g_value_set_float(value, stretch_action->priv->min_width);
			break;
		case PROP_MIN_HEIGHT:
			g_value_set_float(value, stretch_action->priv->min_height);
			break;
		case PROP_MAX_WIDTH:
			g_value_set_float(value, stretch_action->priv->max_width);
			break;
		case PROP_MAX_HEIGHT:
			g_value_set_float(value, stretch_action->priv->max_height);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_stretch_action_dispose(GObject* object) {
        TangleStretchAction* stretch_action;

	stretch_action = TANGLE_STRETCH_ACTION(object);
	
	TANGLE_UNREF_AND_NULLIFY_OBJECT(stretch_action->priv->stretched_actor);

	G_OBJECT_CLASS(tangle_stretch_action_parent_class)->dispose(object);
}

static void tangle_stretch_action_class_init(TangleStretchActionClass* stretch_action_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(stretch_action_class);
	ClutterDragActionClass* drag_action_class = CLUTTER_DRAG_ACTION_CLASS(stretch_action_class);

	gobject_class->dispose = tangle_stretch_action_dispose;
	gobject_class->set_property = tangle_stretch_action_set_property;
	gobject_class->get_property = tangle_stretch_action_get_property;

	drag_action_class->drag_begin = tangle_stretch_action_drag_begin;
	drag_action_class->drag_motion = tangle_stretch_action_drag_motion;
	drag_action_class->drag_end = tangle_stretch_action_drag_end;

	/**
	 * TangleStretchAction:stretched-actor:
	 *
	 * The actor that is being stretched by this #TangleStretchAction.
	 */
	g_object_class_install_property(gobject_class, PROP_STRETCHED_ACTOR,
	                                g_param_spec_object("stretched-actor",
	                                                    "Stretched actor",
	                                                    "The actor that is being stretched by this stretch action",
	                                                    CLUTTER_TYPE_ACTOR,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleStretchAction:min-width:
	 *
	 * The minimum width of stretching.
	 */
	g_object_class_install_property(gobject_class, PROP_MIN_WIDTH,
	                                g_param_spec_float("min-width",
	                                                   "Min width",
	                                                   "The minimum width of stretching",
	                                                   0.0, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleStretchAction:min-height:
	 *
	 * The minimum height of stretching.
	 */
	g_object_class_install_property(gobject_class, PROP_MIN_HEIGHT,
	                                g_param_spec_float("min-height",
	                                                   "Min height",
	                                                   "The minimum height of stretching",
	                                                   0.0, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleStretchAction:max-width:
	 *
	 * The maximum width of stretching.
	 */
	g_object_class_install_property(gobject_class, PROP_MAX_WIDTH,
	                                g_param_spec_float("max-width",
	                                                   "Max width",
	                                                   "The maximum width of stretching",
	                                                   0.0, G_MAXFLOAT, G_MAXFLOAT,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleStretchAction:max-height:
	 *
	 * The maximum height of stretching.
	 */
	g_object_class_install_property(gobject_class, PROP_MAX_HEIGHT,
	                                g_param_spec_float("max-height",
	                                                   "Max height",
	                                                   "The maximum height of stretching",
	                                                   0.0, G_MAXFLOAT, G_MAXFLOAT,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof(TangleStretchActionPrivate));
}

static void tangle_stretch_action_init(TangleStretchAction* stretch_action) {
	stretch_action->priv = G_TYPE_INSTANCE_GET_PRIVATE(stretch_action, TANGLE_TYPE_STRETCH_ACTION, TangleStretchActionPrivate);
	stretch_action->priv->max_width = G_MAXFLOAT;
	stretch_action->priv->max_height = G_MAXFLOAT;
}
