/*
 * tangle-binding.c
 *
 * This file is part of Tangle Toolkit - A graphical actor library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-binding.h"
#include "tangle-private.h"

/* We know that these are available in the Clutter library (from clutter-script-private.h). */
gboolean clutter_script_parse_node (ClutterScript *script, GValue *value, const gchar *name, JsonNode *node, GParamSpec *pspec);


static void clutter_scriptable_iface_init(ClutterScriptableIface* iface);

/**
 * SECTION:tangle-binding
 * @Short_description: A connection between two property values
 * @Title: TangleBinding
 *
 * #TangleBinding connects two properties of one or two objects
 * so that the values are continuously the same.
 *
 * The implementation tracks both properties, and sets the other
 * when either one changes. However, in case of write-only
 * properties, the binding will be unidirectional.
 *
 * In addition to one-to-one relationship, a #TangleBinding supports
 * also a linear equation between the binded properties. The equation
 * is b = m * a + c, where b is the value of property-b, m is multiplier,
 * a is the value of property-a, and c is constant term.
 */

G_DEFINE_TYPE_WITH_CODE(TangleBinding, tangle_binding, TANGLE_TYPE_OBJECT,
                        G_IMPLEMENT_INTERFACE(CLUTTER_TYPE_SCRIPTABLE, clutter_scriptable_iface_init););

#define MAX_NOTIFY_COUNT 10

enum {
	PROP_0,
	PROP_OBJECT_A,
	PROP_PROPERTY_A,
	PROP_OBJECT_B,
	PROP_PROPERTY_B,
	PROP_DIRECTION
};

struct _TangleBindingPrivate {
	GObject* objects[2];
	gchar* properties[2];
	TangleBindingDirection direction;
	GValue multiplier;
	GValue constant_term;
	
	guint notify_counts[2];
	gulong notify_signal_handler_ids[2];
	GValue current_value[2];
};

static const GEnumValue binding_direction_values[] = {
	{ TANGLE_BINDING_NONE, "TANGLE_BINDING_NONE", "none" },
	{ TANGLE_BINDING_A_TO_B, "TANGLE_BINDING_A_TO_B", "a-to-b" },
	{ TANGLE_BINDING_B_TO_A, "TANGLE_BINDING_", "b-to-a" },
	{ TANGLE_BINDING_BIDIRECTIONAL, "TANGLE_BINDING_BIDIRECTIONAL", "bidirectional" },
	{ 0, NULL, NULL }
};

static ClutterScriptableIface* parent_scriptable_iface = NULL;
static GQuark quark_property_bindings = 0;

static gulong connect_notify_signal(GObject* object, GParamSpec* param_spec, GCallback callback, gpointer user_data);
static void disconnect_notify_signals(TangleBinding* binding);
static void on_notify_a(GObject* object, GParamSpec* param_spec, gpointer user_data);
static void on_notify_b(GObject* object, GParamSpec* param_spec, gpointer user_data);
static void on_destroy_notify(gpointer user_data, GObject* where_the_object_was);
static void synchronize(TangleBinding* binding, gint from_index, gint to_index, GParamSpec* param_spec);

GType tangle_binding_direction_get_type(void) {
	static GType type = 0;
	
	if (!type) {
		type = g_enum_register_static("TangleBindingDirection", binding_direction_values);
	}
	
	return type;
}

TangleBinding* tangle_binding_new(GObject* object_a, const gchar* property_a, GObject* object_b, const gchar* property_b) {

	return TANGLE_BINDING(g_object_new(TANGLE_TYPE_BINDING, "object-a", object_a, "property-a", property_a, "object-b", object_b, "property-b", property_b, NULL));
}

GObject* tangle_binding_get_object_a(TangleBinding* binding) {

	return binding->priv->objects[0];
}

const gchar* tangle_binding_get_property_a(TangleBinding* binding) {

	return binding->priv->properties[0];
}

GObject* tangle_binding_get_object_b(TangleBinding* binding) {

	return binding->priv->objects[1];
}

const gchar* tangle_binding_get_property_b(TangleBinding* binding) {

	return binding->priv->properties[1];
}

TangleBindingDirection tangle_binding_get_direction(TangleBinding* binding) {

	return binding->priv->direction;
}

gboolean tangle_binding_get_multiplier(TangleBinding* binding, GValue* value_return) {
	gboolean retvalue = FALSE;
	
	if (G_VALUE_TYPE(&binding->priv->multiplier)) {
		g_value_init(value_return, G_VALUE_TYPE(&binding->priv->multiplier));
		g_value_copy(&binding->priv->multiplier, value_return);
		retvalue = TRUE;
	}

	return retvalue;
}

void tangle_binding_set_multiplier(TangleBinding* binding, const GValue* value) {
	g_return_if_fail(G_VALUE_TYPE(value) == G_VALUE_TYPE(&binding->priv->current_value[0]));
	g_return_if_fail(tangle_value_is_calculation_supported(G_VALUE_TYPE(value)));
	
	if ((!value && G_VALUE_TYPE(&binding->priv->multiplier)) ||
	    !G_VALUE_TYPE(&binding->priv->multiplier) || tangle_value_compare(&binding->priv->multiplier, value)) {
		if (G_VALUE_TYPE(&binding->priv->multiplier)) {
			g_value_unset(&binding->priv->multiplier);
		}
		if (value) {
			g_value_init(&binding->priv->multiplier, G_VALUE_TYPE(value));
			g_value_copy(value, &binding->priv->multiplier);
		}
		synchronize(binding, 0, 1, NULL);
	}
}
	
gboolean tangle_binding_get_constant_term(TangleBinding* binding, GValue* value_return) {
	gboolean retvalue = FALSE;
	
	if (G_VALUE_TYPE(&binding->priv->constant_term)) {
		g_value_init(value_return, G_VALUE_TYPE(&binding->priv->constant_term));
		g_value_copy(&binding->priv->constant_term, value_return);
		retvalue = TRUE;
	}
	
	return retvalue;
}

void tangle_binding_set_constant_term(TangleBinding* binding, const GValue* value) {
	g_return_if_fail(G_VALUE_TYPE(value) == G_VALUE_TYPE(&binding->priv->current_value[0]));
	g_return_if_fail(tangle_value_is_calculation_supported(G_VALUE_TYPE(value)));
	
	if ((!value && G_VALUE_TYPE(&binding->priv->constant_term)) ||
	    !G_VALUE_TYPE(&binding->priv->constant_term) || tangle_value_compare(&binding->priv->constant_term, value)) {
		if (G_VALUE_TYPE(&binding->priv->constant_term)) {
			g_value_unset(&binding->priv->constant_term);
		}
		if (value) {
			g_value_init(&binding->priv->constant_term, G_VALUE_TYPE(value));
			g_value_copy(value, &binding->priv->constant_term);
		}
		synchronize(binding, 0, 1, NULL);
	}
}

GList* tangle_object_get_bindings_readonly(GObject* object, const gchar* property) {
	GList* list = NULL;
	GHashTable* hash_table;
	
	if ((hash_table = (GHashTable*)g_object_get_qdata(object, quark_property_bindings))) {
		list = g_hash_table_lookup(hash_table, property);
	}
	
	return list;
}

static void tangle_binding_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleBinding* binding;
	
	binding = TANGLE_BINDING(object);

	switch (prop_id) {
		case PROP_OBJECT_A:
			binding->priv->objects[0] = g_value_get_object(value);
			break;
		case PROP_PROPERTY_A:
			binding->priv->properties[0] = g_strdup(g_value_get_string(value));
			break;
		case PROP_OBJECT_B:
			binding->priv->objects[1] = g_value_get_object(value);
			break;
		case PROP_PROPERTY_B:
			binding->priv->properties[1] = g_strdup(g_value_get_string(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_binding_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleBinding* binding;

	binding = TANGLE_BINDING(object);

        switch (prop_id) {
		case PROP_OBJECT_A:
			g_value_set_object(value, binding->priv->objects[0]);
			break;
		case PROP_PROPERTY_A:
			g_value_set_string(value, binding->priv->properties[0]);
			break;
		case PROP_OBJECT_B:
			g_value_set_object(value, binding->priv->objects[1]);
			break;
		case PROP_PROPERTY_B:
			g_value_set_string(value, binding->priv->properties[1]);
			break;
		case PROP_DIRECTION:
			g_value_set_enum(value, binding->priv->direction);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_binding_constructed(GObject* object) {
	TangleBinding* binding;
	GParamSpec* param_specs[2];
	gboolean writable[2] = { FALSE, FALSE };
	GType value_type;
	
	binding = TANGLE_BINDING(object);
	
	g_return_if_fail(binding->priv->objects[0]);
	g_return_if_fail(binding->priv->properties[0]);
	g_return_if_fail(binding->priv->objects[1]);
	g_return_if_fail(binding->priv->properties[1]);

	g_object_weak_ref(binding->priv->objects[0], on_destroy_notify, binding);
	g_object_weak_ref(binding->priv->objects[1], on_destroy_notify, binding);

	if (!(param_specs[0] = g_object_class_find_property(G_OBJECT_GET_CLASS(binding->priv->objects[0]), binding->priv->properties[0]))) {
		g_warning("Object has no property called '%s'.", binding->priv->properties[0]);
	}
	if (!(param_specs[1] = g_object_class_find_property(G_OBJECT_GET_CLASS(binding->priv->objects[1]), binding->priv->properties[1]))) {
		g_warning("Object has no property called '%s'.", binding->priv->properties[1]);
	}
	
	if (param_specs[0] && param_specs[1]) {
		if ((param_specs[0]->flags & G_PARAM_WRITABLE) && !(param_specs[0]->flags & G_PARAM_CONSTRUCT_ONLY)) {
			writable[0] = TRUE;
		}
		if ((param_specs[1]->flags & G_PARAM_WRITABLE) && !(param_specs[1]->flags & G_PARAM_CONSTRUCT_ONLY)) {
			writable[1] = TRUE;
		}

		if (param_specs[0] && writable[1] && (binding->priv->notify_signal_handler_ids[0] = connect_notify_signal(binding->priv->objects[0], param_specs[0], G_CALLBACK(on_notify_a), binding))) {
			binding->priv->direction |= TANGLE_BINDING_A_TO_B;
		}
		if (param_specs[0] && writable[0] && (binding->priv->notify_signal_handler_ids[1] = connect_notify_signal(binding->priv->objects[1], param_specs[1], G_CALLBACK(on_notify_b), binding))) {
			binding->priv->direction |= TANGLE_BINDING_B_TO_A;
		}

		if (binding->priv->direction == TANGLE_BINDING_NONE) {
			g_warning("Both properties '%s' and '%s' were write only.", binding->priv->properties[0], binding->priv->properties[1]);
		} else {
			value_type = G_PARAM_SPEC_VALUE_TYPE(g_object_class_find_property(G_OBJECT_GET_CLASS(binding->priv->objects[0]), binding->priv->properties[0]));
			g_value_init(&binding->priv->current_value[0], value_type);
			g_value_init(&binding->priv->current_value[1], value_type);
		}
	}
}

static void tangle_binding_finalize(GObject* object) {
	TangleBinding* binding;
	
	binding = TANGLE_BINDING(object);
	
	disconnect_notify_signals(binding);
	g_free(binding->priv->properties[0]);
	g_free(binding->priv->properties[1]);

	G_OBJECT_CLASS(tangle_binding_parent_class)->finalize(object);
}

static void tangle_binding_dispose(GObject* object) {
	TangleBinding* binding;
	
	binding = TANGLE_BINDING(object);
	
	g_value_unset(&binding->priv->current_value[0]);
	g_value_unset(&binding->priv->current_value[1]);
	if (G_VALUE_TYPE(&binding->priv->multiplier)) {
		g_value_unset(&binding->priv->multiplier);
	}
	if (G_VALUE_TYPE(&binding->priv->constant_term)) {
		g_value_unset(&binding->priv->constant_term);
	}

	G_OBJECT_CLASS(tangle_binding_parent_class)->dispose(object);
}

static void tangle_binding_class_init(TangleBindingClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);

	gobject_class->constructed = tangle_binding_constructed;
	gobject_class->finalize = tangle_binding_finalize;
	gobject_class->dispose = tangle_binding_dispose;
	gobject_class->set_property = tangle_binding_set_property;
	gobject_class->get_property = tangle_binding_get_property;

	quark_property_bindings = g_quark_from_static_string("tangle-property-bindings");

	/**
	 * TangleBinding:object-a:
	 *
	 * The first object of which property is binded.
	 */
	g_object_class_install_property(gobject_class, PROP_OBJECT_A,
	                                g_param_spec_object("object-a",
	                                "Object A",
	                                "The first object of which property is binded",
	                                G_TYPE_OBJECT,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleBinding:property-a:
	 *
	 * The name of the first binded property.
	 */
	g_object_class_install_property(gobject_class, PROP_PROPERTY_A,
	                                g_param_spec_string("property-a",
	                                "Property A",
	                                "The name of the first binded property",
	                                NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleBinding:object-b:
	 *
	 * The second object of which property is binded.
	 */
	g_object_class_install_property(gobject_class, PROP_OBJECT_B,
	                                g_param_spec_object("object-b",
	                                "Object B",
	                                "The second object of which property is binded",
	                                G_TYPE_OBJECT,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleBinding:property-b:
	 *
	 * The name of the second binded property.
	 */
	g_object_class_install_property(gobject_class, PROP_PROPERTY_B,
	                                g_param_spec_string("property-b",
	                                "Property B",
	                                "The name of the second binded property",
	                                NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleBinding:direction
	 *
	 * The direction of the binding.
	 */
	g_object_class_install_property(gobject_class, PROP_DIRECTION,
	                                g_param_spec_enum("direction",
	                                "Direction",
	                                "The direction of the binding",
	                                TANGLE_TYPE_BINDING_DIRECTION,
					TANGLE_BINDING_BIDIRECTIONAL,
	                                G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof(TangleBindingPrivate));
}

static void tangle_binding_init(TangleBinding* binding) {
	binding->priv = G_TYPE_INSTANCE_GET_PRIVATE(binding, TANGLE_TYPE_BINDING, TangleBindingPrivate);
}

static gboolean tangle_binding_parse_custom_node(ClutterScriptable* scriptable, ClutterScript* script, GValue* value, const gchar* name, JsonNode* node) {
	gboolean retvalue;
	TangleBinding* binding;
	
	binding = TANGLE_BINDING(scriptable);
	
	if (!strcmp(name, "multiplier") || !strcmp(name, "constant-term")) {
		retvalue = clutter_script_parse_node(script, value, name, node, NULL);
	} else {
		retvalue = parent_scriptable_iface->parse_custom_node(scriptable, script, value, name, node);
	}
	
	return retvalue;
}

static void tangle_binding_set_custom_property(ClutterScriptable* scriptable, ClutterScript* script, const gchar* name, const GValue* value) {
	TangleBinding* binding;
	GValue v = { 0 };

	binding = TANGLE_BINDING(scriptable);
	
	if (!strcmp(name, "multiplier")) {
		g_value_init(&v, G_VALUE_TYPE(&binding->priv->current_value[0]));
		g_value_transform(value, &v);
		tangle_binding_set_multiplier(binding, &v);
		g_value_unset(&v);
	} else if (!strcmp(name, "constant-term")) {
		g_value_init(&v, G_VALUE_TYPE(&binding->priv->current_value[0]));
		g_value_transform(value, &v);
		tangle_binding_set_constant_term(binding, &v);
		g_value_unset(&v);
	} else {
		parent_scriptable_iface->set_custom_property(scriptable, script, name, value);
	}
}

static void clutter_scriptable_iface_init(ClutterScriptableIface* iface) {
	if (!(parent_scriptable_iface = g_type_interface_peek_parent (iface))) {
		parent_scriptable_iface = g_type_default_interface_peek(CLUTTER_TYPE_SCRIPTABLE);
	}
	
	iface->parse_custom_node = tangle_binding_parse_custom_node;
	iface->set_custom_property = tangle_binding_set_custom_property;
}

static gulong connect_notify_signal(GObject* object, GParamSpec* param_spec, GCallback callback, gpointer user_data) {
	gulong handler_id = 0;
	GHashTable* hash_table;
	gchar* key;
	GList* list;
	gchar* s;
	
	if (!(hash_table = (GHashTable*)g_object_get_qdata(object, quark_property_bindings))) {
		hash_table = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, (GDestroyNotify)g_list_free);
		g_object_set_qdata(object, quark_property_bindings, hash_table);
	}

	if (g_hash_table_lookup_extended(hash_table, param_spec->name, (gpointer*)&key, (gpointer*)&list)) {
		g_hash_table_steal(hash_table, key);
	} else {
		list = NULL;
		key = g_strdup(param_spec->name);
	}
	list = g_list_prepend(list, user_data);
	g_hash_table_insert(hash_table, key, list);

	if (param_spec->flags & G_PARAM_READABLE) {
		s = g_strconcat("notify::", param_spec->name, NULL);
		handler_id = g_signal_connect(object, s, callback, user_data);
		g_free(s);
	}
	
	return handler_id;
}

static void disconnect_notify_signal(GObject* object, const gchar* property, gulong handler_id, gpointer user_data) {
	GHashTable* hash_table;
	gchar* key;
	GList* list;
	
	if (object) {
		if ((hash_table = (GHashTable*)g_object_get_qdata(object, quark_property_bindings)) &&
		    g_hash_table_lookup_extended(hash_table, property, (gpointer*)&key, (gpointer*)&list)) {
			g_hash_table_steal(hash_table, key);
			list = g_list_remove(list, user_data);
			g_hash_table_insert(hash_table, key, list);
		}
		
		if (handler_id) {
			g_signal_handler_disconnect(object, handler_id);
		}
	}
}

static void disconnect_notify_signals(TangleBinding* binding) {
	disconnect_notify_signal(binding->priv->objects[0], binding->priv->properties[0], binding->priv->notify_signal_handler_ids[0], binding);
	disconnect_notify_signal(binding->priv->objects[1], binding->priv->properties[1], binding->priv->notify_signal_handler_ids[1], binding);
			
	binding->priv->direction = TANGLE_BINDING_NONE;
}

static void synchronize(TangleBinding* binding, gint from_index, gint to_index, GParamSpec* param_spec) {
	GValue value = { 0 };

	binding->priv->notify_counts[from_index]++;

	if (binding->priv->notify_counts[from_index] > MAX_NOTIFY_COUNT) {
		g_warning("Possible loop when synchronizing binding property '%s'.", binding->priv->properties[from_index]);
	} else {
		g_value_init(&value, G_VALUE_TYPE(&binding->priv->current_value[0]));
		g_object_get_property(binding->priv->objects[from_index], binding->priv->properties[from_index], &value);

		if (from_index == 0) {
			if (G_VALUE_TYPE(&binding->priv->multiplier)) {
				tangle_value_multiply(&value, &binding->priv->multiplier);
			}
			if (G_VALUE_TYPE(&binding->priv->constant_term)) {
				tangle_value_add(&value, &binding->priv->constant_term);			
			}
		} else {
			if (G_VALUE_TYPE(&binding->priv->constant_term)) {
				tangle_value_substract(&value, &binding->priv->constant_term);			
			}		
			if (G_VALUE_TYPE(&binding->priv->multiplier)) {
				tangle_value_divide(&value, &binding->priv->multiplier);
			}
		}

		if ((param_spec && g_param_values_cmp(param_spec, &value, &binding->priv->current_value[to_index])) ||
		    (!param_spec && tangle_value_compare(&value, &binding->priv->current_value[to_index]))) {
			g_value_copy(&value, &binding->priv->current_value[to_index]);
			g_object_set_property(binding->priv->objects[to_index], binding->priv->properties[to_index], &value);
		}

		g_value_unset(&value);
	}

	binding->priv->notify_counts[from_index]--;
}

static void on_notify_a(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	synchronize(TANGLE_BINDING(user_data), 0, 1, param_spec);
}

static void on_notify_b(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	synchronize(TANGLE_BINDING(user_data), 1, 0, param_spec);
}

static void on_destroy_notify(gpointer user_data, GObject* where_the_object_was) {
	TangleBinding* binding;
	
	binding = TANGLE_BINDING(user_data);
	
	if (binding->priv->objects[0] == where_the_object_was) {
		binding->priv->objects[0] = NULL;
	} else {
		binding->priv->objects[1] = NULL;	
	}

	disconnect_notify_signals(binding);
}
