/*
 * tangle-template.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */
 
#include "tangle-template.h"
#include "tangle-vault.h"
#include "tangle-actor.h"
#include <clutter/clutter.h>
#include <string.h>

/* We know that these are available in the Clutter library (from clutter-script-private.h). */
gboolean clutter_script_parse_node (ClutterScript *script, GValue *value, const gchar *name, JsonNode *node, GParamSpec *pspec);
GType clutter_script_get_type_from_class (const gchar *name);


static void clutter_scriptable_iface_init(ClutterScriptableIface* iface);

/**
 * SECTION:tangle-template
 * @Short_description: A template object to create other objects
 * @Title: TangleTemplate
 */

G_DEFINE_TYPE_WITH_CODE(TangleTemplate, tangle_template, TANGLE_TYPE_PROPERTIES,
                        G_IMPLEMENT_INTERFACE(CLUTTER_TYPE_SCRIPTABLE, clutter_scriptable_iface_init););

enum {
	PROP_0,
	PROP_OBJECT_TYPE,
	PROP_OBJECT_TYPE_NAME
};

struct _TangleTemplatePrivate {
	GType object_type;
	GHashTable* parameters;
	
	ClutterScript* script;
	GList* child_templates;
	
	guint instance_counter;
};

static gboolean is_value_null(gpointer key, gpointer value, gpointer user_data);
static gchar* str_replace(gchar* string, gsize position, gsize length, const gchar* replacement, gchar** replacement_end_return);
static gboolean substitute_parameters(TangleTemplate* template, GValue* value);
static void set_property(const gchar* name, const GValue* value, gpointer user_data);
static void free_value(gpointer data);
static gboolean apply_parameters(TangleTemplate* template, const gchar* name, GValue* value, const gchar* id_prefix);
static gboolean on_signal(gpointer user_data);
static void add_children(TangleTemplate* template, GObject* object);
void set_template_property(const gchar* name, const GValue* value, gpointer user_data);
void add_template_parameter(gpointer key, gpointer value, gpointer user_data);
void add_child_template(gpointer data, gpointer user_data);

static ClutterScriptableIface* parent_scriptable_iface = NULL;

TangleTemplate* tangle_template_new(GType object_type) {

	return TANGLE_TEMPLATE(g_object_new(TANGLE_TYPE_TEMPLATE, "object-type", object_type, NULL));
}

GType tangle_template_get_object_type(TangleTemplate* template) {

	return template->priv->object_type;
}

GObject* tangle_template_create_object(TangleTemplate* template) {
	GObject* object;
	const gchar* template_id;
	gchar* id = NULL;

	if ((template_id = clutter_get_script_id(G_OBJECT(template)))) {
		id = g_strdup_printf("%s$$%d", template_id, template->priv->instance_counter);
	}
	object = tangle_template_create_object_with_id(template, id);
	g_free(id);

	return object;
}

GObject* tangle_template_create_object_with_id(TangleTemplate* template, const gchar* id) {
	GObject* object = NULL;
	gboolean error = FALSE;
	guint n;
	GParameter* parameters;
	GArray* construct_parameters;
	GArray* nonconstruct_parameters;
	GArray* signal_parameters;
	gpointer type_class;
	guint i;
	GParamSpec* param_spec;
	ClutterScriptable* scriptable;
	ClutterScriptableIface* scriptable_iface;
	GParameter* parameter;

	parameters = tangle_properties_get_as_parameters(TANGLE_PROPERTIES(template), &n);
	construct_parameters = g_array_new(FALSE, FALSE, sizeof(GParameter));
	nonconstruct_parameters = g_array_new(FALSE, FALSE, sizeof(GParameter));
	signal_parameters = g_array_new(FALSE, FALSE, sizeof(GParameter));

	type_class = g_type_class_ref(template->priv->object_type);
	for (i = 0; i < n; i++) {
		if (parameters[i].name[0] == '@') {
			g_array_append_val(signal_parameters, parameters[i]);
		} else if (!apply_parameters(template, parameters[i].name, &parameters[i].value, id)) {
			error = TRUE;
			break;
		} else if ((param_spec = g_object_class_find_property(type_class, parameters[i].name)) &&
		           param_spec->flags & G_PARAM_CONSTRUCT_ONLY) {
			g_array_append_val(construct_parameters, parameters[i]);
		} else {
			g_array_append_val(nonconstruct_parameters, parameters[i]);
		}
	}
	g_type_class_unref(type_class);

	if (!error) {
		object = g_object_newv(template->priv->object_type, construct_parameters->len, (GParameter*)construct_parameters->data);
		
		template->priv->instance_counter++;
		if (id) {
			if (CLUTTER_IS_SCRIPTABLE(object)) {
				clutter_scriptable_set_id(CLUTTER_SCRIPTABLE(object), id);
			} else {
				g_object_set_data_full(object, "clutter-script-id", g_strdup(id), g_free);
			}
		}

		/* TODO: This message will be removed soon! */
		g_print("Created an object '%s' from the template '%s'.\n", id, clutter_get_script_id(G_OBJECT(template)));

		if (!template->priv->script ||
		    !CLUTTER_IS_SCRIPTABLE(object) ||
		    !(scriptable = CLUTTER_SCRIPTABLE(object)) ||
		    !(scriptable_iface = CLUTTER_SCRIPTABLE_GET_IFACE(scriptable)) ||
		    !scriptable_iface->set_custom_property) {
		    	scriptable_iface = NULL;
		}
		for (i = 0; i < nonconstruct_parameters->len; i++) {
			parameter = &g_array_index(nonconstruct_parameters, GParameter, i);
			if (scriptable_iface) {
				scriptable_iface->set_custom_property(scriptable, template->priv->script, parameter->name, &parameter->value);
			} else {
				g_object_set_property(object, parameter->name, &parameter->value);
			}
		}
		
		add_children(template, object);
		
		for (i = 0; i < signal_parameters->len; i++) {
			parameter = &g_array_index(signal_parameters, GParameter, i);
			if (!G_VALUE_HOLDS(&parameter->value, G_TYPE_STRING) || !substitute_parameters(template, &parameter->value)) {
				g_warning("Invalid signal property '%s' of an object created from the template '%s', skipped.", parameter->name, clutter_get_script_id(G_OBJECT(template)));
			} else if (template->priv->script) {
				if (!!tangle_signal_connect_dynamically_from_script(object, parameter->name + 1, g_value_get_string(&parameter->value), template->priv->script)) {
					g_warning("Failed to connect the signal '%s' of an object created from the template '%s' into function '%s', skipped.", parameter->name + 1, clutter_get_script_id(G_OBJECT(template)), g_value_get_string(&parameter->value));
				}
			} else if (!tangle_signal_connect_dynamically(object, parameter->name + 1, g_value_get_string(&parameter->value), NULL)) {
				g_warning("Failed to connect the signal '%s' into function '%s', skipped.", parameter->name + 1, g_value_get_string(&parameter->value));
			}		
		}
	}

	for (i = 0; i < n; i++) {
		g_free((gchar*)parameters[i].name);
		g_value_unset(&parameters[i].value);
	}
	g_free(parameters);

	g_array_free(construct_parameters, TRUE);
	g_array_free(nonconstruct_parameters, TRUE);
	g_array_free(signal_parameters, TRUE);

	return object;
}

void tangle_template_apply_properties(TangleTemplate* template, GObject* object) {
	TangleVault* vault;
	
	vault = tangle_vault_new(4, TANGLE_TYPE_TEMPLATE, template, G_TYPE_OBJECT, object, G_TYPE_ULONG, (gulong)0, G_TYPE_UINT, (guint)0);
	tangle_properties_foreach(TANGLE_PROPERTIES(template), set_property, vault);
	tangle_vault_free(vault);

	add_children(template, object);
}

void tangle_template_animate_properties(TangleTemplate* template, ClutterActor* actor, gulong mode, guint duration) {
	TangleVault* vault;
	
	vault = tangle_vault_new(4, TANGLE_TYPE_TEMPLATE, template, G_TYPE_OBJECT, G_OBJECT(actor), G_TYPE_ULONG, mode, G_TYPE_UINT, duration);
	tangle_properties_foreach(TANGLE_PROPERTIES(template), set_property, vault);
	tangle_vault_free(vault);
}
ngle_template_apply_properties_on_signal(TangleTemplate* template, GObject* target_object, GObject* signal_object, const gchar* signal_name, gboolean signal_return) {
	TangleVault* vault;
	
	vault = tangle_vault_new(5, TANGLE_TYPE_TEMPLATE, template, G_TYPE_OBJECT, target_object, G_TYPE_ULONG, (gulong)0, G_TYPE_UINT, (guint)0, G_TYPE_BOOLEAN, signal_return);
	
	return tangle_signal_connect_vault_flags(signal_object, signal_name, G_CALLBACK(on_signal), vault, G_CONNECT_SWAPPED);
}

gulong tangle_template_animate_properties_on_signal(TangleTemplate* template, ClutterActor* actor, gulong mode, guint duration, GObject* signal_object, const gchar* signal_name, gboolean signal_return) {
	TangleVault* vault;
	
	vault = tangle_vault_new(5, TANGLE_TYPE_TEMPLATE, template, G_TYPE_OBJECT, G_OBJECT(actor), G_TYPE_ULONG, mode, G_TYPE_UINT, duration, G_TYPE_BOOLEAN, signal_return);
	
	return tangle_signal_connect_vault_flags(signal_object, signal_name, G_CALLBACK(on_signal), vault, G_CONNECT_SWAPPED);
}

gboolean tangle_template_has_parameters(TangleTemplate* template) {

	return g_hash_table_size(template->priv->parameters) > 0;
}

GList* tangle_template_get_parameter_names(TangleTemplate* template) {

	return g_hash_table_get_keys(template->priv->parameters);
}

gboolean tangle_template_has_parameter(TangleTemplate* template, const gchar* name) {

	return g_hash_table_lookup_extended(template->priv->parameters, name, NULL, NULL);
}

const GValue* tangle_template_get_parameter_value(TangleTemplate* template, const gchar* name) {
	
	return g_hash_table_lookup(template->priv->parameters, name);
}

gboolean tangle_template_set_parameter_value(TangleTemplate* template, const gchar* name, const GValue* value) {
	gboolean retvalue = FALSE;
	GValue* value_copy;

	if (g_hash_table_lookup_extended(template->priv->parameters, name, NULL, NULL)) {
		value_copy = g_value_init(g_new0(GValue, 1), G_VALUE_TYPE(value));
		g_value_copy(value, value_copy);
		g_hash_table_insert(template->priv->parameters, g_strdup(name), value_copy);

		retvalue = TRUE;
	}

	return retvalue;
}

void tangle_template_clear_parameter_values(TangleTemplate* template) {
	GList* list;
	GList* list_item;

	list = g_hash_table_get_keys(template->priv->parameters);
	for (list_item = list; list_item; list_item = list_item->next) {
		g_hash_table_insert(template->priv->parameters, g_strdup(list_item->data), NULL);
	}
	g_list_free(list);
}

gboolean tangle_template_are_parameters_complete(TangleTemplate* template) {
	gboolean found = FALSE;

	g_hash_table_find(template->priv->parameters, is_value_null, &found);

	return !found;
}

static gboolean tangle_template_validate_property(TangleProperties* properties, const gchar* name, const GValue* value) {
	gboolean retvalue = TRUE;
	TangleTemplate* template;
	gpointer type_class;
	GParamSpec* param_spec = NULL;
	const gchar* s;
	const gchar* t;
	gchar* parameter_name;
	
	template = TANGLE_TEMPLATE(properties);
	
	type_class = g_type_class_ref(template->priv->object_type);
	if (name[0] != '@' && !(param_spec = g_object_class_find_property(G_OBJECT_CLASS(type_class), name))) {
		g_warning("The object of type '%s' does not have property named '%s', skipping.", g_type_name(template->priv->object_type), name);
		retvalue = FALSE;
	} else {
		if (G_VALUE_HOLDS(value, G_TYPE_STRING)) {
			s = g_value_get_string(value);
			while (retvalue && (s = strstr(s, "${"))) {
				s += 2;
				if (*s == '{') {
					g_warning("Template parameter '${}' without a name in property '%s', skipping.", name);
					retvalue = FALSE;
				} else if (!(t = strchr(s + 1, '}'))) {
					g_warning("Start of template parameter '${' without end '}' in property '%s', skipping.", name);
					retvalue = FALSE;
				} else {
					if (s - 2 == g_value_get_string(value) && t[1] == 0) {
						param_spec = NULL;
					}
					
					parameter_name = g_strndup(s, t - s);
					if (!g_hash_table_lookup_extended(template->priv->parameters, parameter_name, NULL, NULL)) {
						g_hash_table_insert(template->priv->parameters, parameter_name, NULL);
					} else {
						g_free(parameter_name);
					}
					
					s = t + 1;
				}
			}
		} else if (!param_spec || !g_value_type_compatible(G_VALUE_TYPE(value), G_PARAM_SPEC_VALUE_TYPE(param_spec))) {
			g_warning("Incompatible value type for property named '%s', skipping.", name);
			retvalue = FALSE;
		}
	}
	g_type_class_unref(type_class);
	
	return retvalue;
}

static void tangle_template_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleTemplate* template;
	
	template = TANGLE_TEMPLATE(object);

	switch (prop_id) {
		case PROP_OBJECT_TYPE:
			if (g_value_get_gtype(value) != G_TYPE_NONE) {
				g_assert(!template->priv->object_type);
				template->priv->object_type = g_value_get_gtype(value);
				g_assert(G_TYPE_IS_OBJECT(template->priv->object_type));
			}
			break;
		case PROP_OBJECT_TYPE_NAME:
			if (g_value_get_string(value)) {
				g_assert(!template->priv->object_type);
				template->priv->object_type = g_type_from_name(g_value_get_string(value));
				if (template->priv->object_type == G_TYPE_INVALID) {
					template->priv->object_type = clutter_script_get_type_from_class(g_value_get_string(value));
				}
				g_assert(G_TYPE_IS_OBJECT(template->priv->object_type));
			}
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_template_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleTemplate* template;

	template = TANGLE_TEMPLATE(object);

        switch (prop_id) {
		case PROP_OBJECT_TYPE:
			g_value_set_gtype(value, template->priv->object_type);
			break;
		case PROP_OBJECT_TYPE_NAME:
			g_value_set_string(value, g_type_name(template->priv->object_type));
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_template_class_init(TangleTemplateClass* template_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(template_class);
	TanglePropertiesClass* properties_class = TANGLE_PROPERTIES_CLASS(template_class);

	/* TODO: Free hash table && unref script! */
	gobject_class->set_property = tangle_template_set_property;
	gobject_class->get_property = tangle_template_get_property;
	
	properties_class->validate_property = tangle_template_validate_property;

	/**
	 * TangleTemplate:object-type:
	 */
	g_object_class_install_property(gobject_class, PROP_OBJECT_TYPE,
	                                g_param_spec_gtype("object-type",
	                                "Object type",
	                                "The GType of the object to be created",
	                                G_TYPE_NONE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleTemplate:object-type-name:
	 */
	g_object_class_install_property(gobject_class, PROP_OBJECT_TYPE_NAME,
	                                g_param_spec_string("object-type-name",
	                                "Object type name",
	                                "The name of the GType of the object to be created",
					NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private (gobject_class, sizeof (TangleTemplatePrivate));
}

static void tangle_template_init(TangleTemplate* template) {
	template->priv = G_TYPE_INSTANCE_GET_PRIVATE(template, TANGLE_TYPE_TEMPLATE, TangleTemplatePrivate);
	template->priv->parameters = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, free_value);
}

static gboolean tangle_template_parse_custom_node(ClutterScriptable* scriptable, ClutterScript* script, GValue* value, const gchar* name, JsonNode* node) {
	gboolean retvalue = FALSE;
	TangleTemplate* template;
	gpointer type_class;
	GParamSpec* param_spec;
	const gchar* s;
	gboolean check_subtemplate = FALSE;
	GObject* object;
	TangleTemplate* t;
	GList* name_list;
	GList* name_list_item;
	JsonArray* array;
	gint i;
	
	template = TANGLE_TEMPLATE(scriptable);

	if (!template->priv->script) {
		template->priv->script = script;
		g_object_ref(template->priv->script);
	}

	if (name[0] == '@') {
		retvalue = clutter_script_parse_node(script, value, name, node, NULL);
	} else if (!strcmp(name, "child-templates")) {
		name_list = NULL;

		if (!g_type_is_a(template->priv->object_type, CLUTTER_TYPE_CONTAINER)) {
			g_critical("Object of type '%s' is not ClutterContainer, but the 'child-template' property found, skipped.", g_type_name(template->priv->object_type));
		} else if (!JSON_NODE_HOLDS_ARRAY(node)) {
			g_critical("Expected an array in 'child-templates' node, skipped.");
		} else {
			array = json_node_get_array(node);
			for (i = json_array_get_length(array) - 1; i >= 0; i--) {
				if (!(s = json_array_get_string_element(array, i))) {
					g_critical("A non-string element in 'child-templates' array, skipped.");
				} else {
					name_list = g_list_prepend(name_list, g_strdup(s));
				}
			}
		}
			
		g_value_init(value, G_TYPE_POINTER);
		g_value_set_pointer(value, name_list);

		retvalue = TRUE;
	} else if (strcmp(name, "object-type") && strcmp(name, "object-type-name") && strcmp(name, "template")) {
		type_class = g_type_class_ref(template->priv->object_type);
		if (!(param_spec = g_object_class_find_property(G_OBJECT_CLASS(type_class), name))) {
			g_warning("Object of type '%s' does not have property named '%s', skipping.", g_type_name(template->priv->object_type), name);
		} else {
			if ((s = json_node_get_string(node)) && strlen(s) >= 4 && s[0] == '$' && s[1] == '{' && strchr(s + 2, '}')[1] == 0) {
				/* The template parameter replaces the whole property value and its type. */
				param_spec = NULL;
			} else if (G_IS_PARAM_SPEC_OBJECT(param_spec)) {
				/* Defer object resolution until the template is instantiated. */
				param_spec = NULL;
				check_subtemplate = TRUE;
			}
			retvalue = clutter_script_parse_node(script, value, name, node, param_spec);
		}
		g_type_class_unref(type_class);

		if (!retvalue && parent_scriptable_iface->parse_custom_node) {
			retvalue = parent_scriptable_iface->parse_custom_node(scriptable, script, value, name, node);
		}

		if (check_subtemplate && retvalue && G_VALUE_HOLDS_STRING(value) &&
		    (object = clutter_script_get_object(script, g_value_get_string(value))) && TANGLE_IS_TEMPLATE(object)) {
			t = TANGLE_TEMPLATE(object);
			name_list = tangle_template_get_parameter_names(t);
			for (name_list_item = name_list; name_list_item; name_list_item = name_list_item->next) {
				s = (const gchar*)name_list_item->data;
				if (!g_hash_table_lookup_extended(template->priv->parameters, s, NULL, NULL)) {
					g_hash_table_insert(template->priv->parameters, g_strdup(s), NULL);
				}
			}
			g_list_free(name_list);					

		}
	}
	
	return retvalue;
}

static void tangle_template_set_custom_property(ClutterScriptable* scriptable, ClutterScript* script, const gchar* name, const GValue* value) {
	TangleTemplate* template;
	TangleTemplate* super_template;
	
	template = TANGLE_TEMPLATE(scriptable);

	if (!strcmp(name, "template")) {
		super_template = TANGLE_TEMPLATE(g_value_get_object(value));

		g_return_if_fail(TANGLE_IS_TEMPLATE(super_template));
		g_return_if_fail(template->priv->object_type == super_template->priv->object_type);

		tangle_properties_foreach(TANGLE_PROPERTIES(super_template), set_template_property, template);
		g_hash_table_foreach(super_template->priv->parameters, add_template_parameter, template);
		g_list_foreach(super_template->priv->child_templates, add_child_template, template);

		template->priv->child_templates = g_list_reverse(template->priv->child_templates);	
	} else if (!strcmp(name, "child-templates")) {
		g_return_if_fail(G_VALUE_HOLDS(value, G_TYPE_POINTER));

		template->priv->child_templates = (GList*)g_value_get_pointer(value);
	} else {
		parent_scriptable_iface->set_custom_property(scriptable, script, name, value);
	}
}

static void clutter_scriptable_iface_init(ClutterScriptableIface* iface) {
	if (!(parent_scriptable_iface = parent_scriptable_iface = g_type_interface_peek_parent (iface))) {
		parent_scriptable_iface = g_type_default_interface_peek(CLUTTER_TYPE_SCRIPTABLE);
	}
	
	iface->parse_custom_node = tangle_template_parse_custom_node;
	iface->set_custom_property = tangle_template_set_custom_property;
}

static gboolean is_value_null(gpointer key, gpointer value, gpointer user_data) {
	gboolean* found_pointer;
	
	found_pointer = (gboolean*)user_data;
	if (value ==  NULL) {
		*found_pointer = TRUE;
	}

	return *found_pointer;
}

static gchar* str_replace(gchar* string, gsize position, gsize length, const gchar* replacement, gchar** replacement_end_return) {
	gsize string_length;
	gsize replacement_length;
	gchar* s;
	
	string_length = strlen(string);
	replacement_length = strlen(replacement);
	s = (gchar*)g_malloc(string_length - length + replacement_length + 1);
	g_memmove(s, string, position);
	g_memmove(s + position, replacement, replacement_length);
	g_memmove(s + position + replacement_length, string + position + length, string_length - position - length + 1);

	if (replacement_end_return) {
		*replacement_end_return = s + position + replacement_length;
	}

	g_free(string);

	return s;
}

static gboolean substitute_parameters(TangleTemplate* template, GValue* value) {
	gboolean retvalue = TRUE;
	gchar* substituted_string;
	gboolean substituted;
	gchar* s;
	gchar* n;
	gchar* e;
	GValue* v;
	gchar* u;

	/* Template parameters will be substituted with the property value. */
	substituted_string = s = g_strdup(g_value_get_string(value));
	substituted = FALSE;

	while (s = strstr(s, "${")) {
		s += 2;
		e = strchr(s + 1, '}');
		n = g_strndup(s, e - s);
		if (!g_hash_table_lookup_extended(template->priv->parameters, n, NULL, (gpointer*)&v)) {
			g_critical("Template does not have a parameter named '%s'.", n);
			retvalue = FALSE;
		} else if (!value) {
			g_critical("Template parameter '%s' has no value.", n);
			retvalue = FALSE;
		} else {
			if (G_VALUE_HOLDS_STRING(v)) {
				u = g_value_dup_string(v);
			} else {
				u = g_strdup_value_contents(v);
			}
			substituted_string = str_replace(substituted_string, s - substituted_string - 2, e - s + 3, u, &s);
			substituted = TRUE;
			g_free(u);
		}
		g_free(n);
	}
	if (retvalue && substituted) {
		g_value_set_string(value, substituted_string);
	}

	g_free(substituted_string);
	
	return retvalue;
}

static void set_property(const gchar* name, const GValue* value, gpointer user_data) {
	TangleVault* vault;
	TangleTemplate* template;
	GObject* object;
	gulong mode;
	guint duration;
	GValue v = { 0 };
	
	vault = TANGLE_VAULT(user_data);
	tangle_vault_get(vault, 4, TANGLE_TYPE_TEMPLATE, &template, G_TYPE_OBJECT, &object, G_TYPE_ULONG, &mode, G_TYPE_UINT, &duration);
	
	g_value_init(&v, G_VALUE_TYPE(value));
	g_value_copy(value, &v);
	if (name[0] == '@') {
		if (!G_VALUE_HOLDS(&v, G_TYPE_STRING) || !substitute_parameters(template, &v)) {
			g_warning("Failed to connect signal handler property '%s' of an object (id: %s).", name, clutter_get_script_id(G_OBJECT(object)));
		} else if (template->priv->script) {
			if (!tangle_signal_connect_dynamically_from_script(object, name + 1, g_value_get_string(&v), template->priv->script)) {
				g_warning("Failed to connect the signal handler property '%s' of an object created from the template '%s', skipped.", name + 1, clutter_get_script_id(G_OBJECT(template)));
			}
		} else if (!tangle_signal_connect_dynamically(object, name + 1, g_value_get_string(&v), NULL)) {
			g_warning("Failed to connect the signal handler property '%s' of an object, skipped.", name + 1);
		}
	} else if (apply_parameters(template, name, &v, clutter_get_script_id(object))) {
		if (!mode || !duration || !CLUTTER_IS_ACTOR(object)) {
			g_object_set_property(object, name, &v);
		} else if (TANGLE_IS_ACTOR(object)) {
			tangle_actor_animatev(TANGLE_ACTOR(object), mode, duration, 1, &name, &v);
		} else {
			clutter_actor_animatev(CLUTTER_ACTOR(object), mode, duration, 1, &name, &v);
		}
	}
	g_value_unset(&v);
}

static void free_value(gpointer data) {
	if (data) {
		g_value_unset(data);
		g_free(data);
	}
}

static gboolean apply_parameters(TangleTemplate* template, const gchar* name, GValue* value, const gchar* id_prefix) {
	gboolean retvalue = TRUE;
	gpointer type_class;
	GObject* object;
	TangleTemplate* t;
	const gchar* cs;
	gsize length;
	GValue* v;
	const GValue* cv;
	GParamSpec* param_spec;
	gchar* s;
	GList* name_list;
	GList* name_list_item;
	gchar* id;

	type_class = g_type_class_ref(template->priv->object_type);

	if (G_VALUE_HOLDS(value, G_TYPE_STRING)) {
		if ((param_spec = g_object_class_find_property(G_OBJECT_CLASS(type_class), name)) && G_IS_PARAM_SPEC_OBJECT(param_spec)) {
			/* Handle deferred object resolution. */
			substitute_parameters(template, value);
			if ((object = clutter_script_get_object(template->priv->script, g_value_get_string(value)))) {
				if (TANGLE_IS_TEMPLATE(object) && !G_VALUE_HOLDS(value, TANGLE_TYPE_TEMPLATE)) {
					/* Create objects from templates recursively. */
					t = TANGLE_TEMPLATE(object);
					g_object_ref(t);
					
					name_list = tangle_template_get_parameter_names(t);
					for (name_list_item = name_list; name_list_item; name_list_item = name_list_item->next) {
						cs = (const gchar*)name_list_item->data;
						if (!(cv = tangle_template_get_parameter_value(template, cs))) {
							g_warning("Subtemplate parameter '%s' not found, skipped.", cs);
						} else {
							tangle_template_set_parameter_value(t, cs, cv);
						}
					}
					g_list_free(name_list);
					
					if (id_prefix) {
						id = g_strconcat(id_prefix, "$", name, NULL);
					} else {
						id = NULL;
					}
					object = tangle_template_create_object_with_id(t, id);
					g_free(id);
					g_value_unset(value);
					g_value_init(value, tangle_template_get_object_type(t));
					g_value_set_instance(value, object);
					g_object_unref(t);				
				} else {
					g_value_unset(value);
					g_value_init(value, G_PARAM_SPEC_VALUE_TYPE(param_spec)),
					g_value_set_instance(value, object);
				}
			}
		} else {
			/* Handle template parameters. */
			cs = g_value_get_string(value);
			length = strlen(cs);
			if (length >= 4 && cs[0] == '$' && cs[1] == '{' && strchr(cs + 2, '}')[1] == 0) {
				/* The template parameter replaces the whole property value and its type. */
				s = g_strndup(cs + 2, length - 3);
				if (!g_hash_table_lookup_extended(template->priv->parameters, s, NULL, (gpointer*)&v)) {
					g_critical("Template does not have a parameter sd '%s'.", s);
					retvalue = FALSE;
				} else if (!v) {
					g_critical("Template parameter '%s' has no value.", s);
					retvalue = FALSE;
				} else if (!(param_spec = g_object_class_find_property(G_OBJECT_CLASS(type_class), name)) ||
					  !g_value_type_transformable(G_VALUE_TYPE(v), G_PARAM_SPEC_VALUE_TYPE(param_spec))) {
					g_critical("Incompatible value type of the template parameter '%s' for property named '%s'.", s, name);
					retvalue = FALSE;			
				} else {
					g_value_unset(value);
					g_value_init(value, G_VALUE_TYPE(v));
					g_value_transform(v, value);
				}
				g_free(s);
			} else {
				retvalue = substitute_parameters(template, value);
			}
		}
	}

	g_type_class_unref(type_class);
	
	return retvalue;
}

static gboolean on_signal(gpointer user_data) {
	TangleVault* vault;
	TangleTemplate* template;
	GObject* object;
	gulong mode;
	guint duration;
	gboolean retvalue;
		
	vault = TANGLE_VAULT(user_data);
	tangle_vault_get(vault, 5, TANGLE_TYPE_TEMPLATE, &template, G_TYPE_OBJECT, &object, G_TYPE_ULONG, &mode, G_TYPE_UINT, &duration, G_TYPE_BOOLEAN, &retvalue);
	
	if (!mode || !duration || !CLUTTER_IS_ACTOR(object)) {
		tangle_template_apply_properties(template, object);
	} else {
		tangle_template_animate_properties(template, CLUTTER_ACTOR(object), mode, duration);
	}
	
	return retvalue;
}

static void add_children(TangleTemplate* template, GObject* object) {
	GList* child_template;
	gint i;
	gchar* s;
	GObject* o;
	const gchar* object_id;
	gchar* id;

	for (child_template = template->priv->child_templates, i = 0; child_template; child_template = child_template->next, i++) {
		s = g_strdup((gchar*)child_template->data);
		/* TODO: substitute_parameters() */
		if (!(o = clutter_script_get_object(template->priv->script, s))) {
			g_critical("A template (id: %s) not found when adding a children from 'child-templates' array, skipped.", s);
		} if (!TANGLE_IS_TEMPLATE(o)) {
			g_critical("An object (id: %s) is not TangleTemplate as expected in 'child-templates' array, skipped.", s);
		} else {
			if ((object_id = clutter_get_script_id(object))) {
				id = g_strdup_printf("%s$%d", object_id, i);
			} else {
				id = NULL;
			}
			clutter_container_add_actor(CLUTTER_CONTAINER(object), CLUTTER_ACTOR(tangle_template_create_object_with_id(TANGLE_TEMPLATE(o), id)));
			g_free(id);
		}
		g_free(s);		
	}

}

void set_template_property(const gchar* name, const GValue* value, gpointer user_data) {
	tangle_properties_set_property(TANGLE_PROPERTIES(user_data), name, value);		
}

void add_template_parameter(gpointer key, gpointer value, gpointer user_data) {
	TangleTemplate* template;
	GValue* value_copy;
	
	template = TANGLE_TEMPLATE(user_data);

	value_copy = g_value_init(g_new0(GValue, 1), G_VALUE_TYPE(value));
	g_value_copy(value, value_copy);
	g_hash_table_insert(template->priv->parameters, g_strdup(key), value_copy);
}

void add_child_template(gpointer data, gpointer user_data) {
	TangleTemplate* template;
	
	template = TANGLE_TEMPLATE(user_data);
	
	template->priv->child_templates = g_list_prepend(template->priv->child_templates, g_strdup(data));
}
