/*
 * tangle-stylesheet.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-stylesheet.h"
#include "tangle-compound-style.h"
#include "tangle-stylable.h"
#include "marshalers.h"
#include <gmodule.h>
#include <stdlib.h>

/**
 * SECTION:tangle-stylesheet
 * @Short_description: A collection of style information
 * @Title: TangleStylesheet
 */

G_DEFINE_TYPE(TangleStylesheet, tangle_stylesheet, G_TYPE_OBJECT);

enum {
	PROP_0,
	PROP_STYLES
};

enum {
	STYLE_ADDED,
	STYLE_REMOVED,
	LAST_SIGNAL
};

struct _TangleStylesheetPrivate {
	GList* styles;
};

static guint signals[LAST_SIGNAL] = { 0 };
static TangleStylesheet* default_stylesheet = NULL;

static TangleStylesheet* load_from_file(const gchar* directory_prefix, const gchar* name);

TangleStylesheet* tangle_stylesheet_new() {

	return TANGLE_STYLESHEET(g_object_new(TANGLE_TYPE_STYLESHEET, NULL));
}

TangleStylesheet* tangle_stylesheet_new_from_file(const gchar* name) {
	TangleStylesheet* stylesheet;
	const gchar working_directory[PATH_MAX];
	gchar* home_directory;
	
	g_return_if_fail(name != NULL || name[0] != 0);
	
	if (name[0] == '/') {
		if (!(stylesheet = load_from_file("", name))) {
			g_warning("Could not load stylesheet '%s'.", name);
		}
	} else if (!(getcwd(working_directory, PATH_MAX) && (stylesheet = load_from_file(working_directory, name))) &&
	    !((home_directory = getenv("HOME")) && (stylesheet = load_from_file(home_directory, name))) &&
	    !(stylesheet = load_from_file("/usr/local/share/tangle/stylesheets/", name)) &&
	    !(stylesheet = load_from_file("/usr/share/tangle/stylesheets/", name))) {
		g_warning("Could not load stylesheet '%s'.", name);
	}
	
	return stylesheet;

}

GList* tangle_stylesheet_get_styles(TangleStylesheet* stylesheet) {

	return g_list_copy(stylesheet->priv->styles);
}

void tangle_stylesheet_add_style(TangleStylesheet* stylesheet, TangleStyle* style) {
	if (!g_list_find(stylesheet->priv->styles, style)) {
		g_signal_emit(stylesheet, signals[STYLE_ADDED], 0, style);
		g_object_ref(style);
	}
}

void tangle_stylesheet_remove_style(TangleStylesheet* stylesheet, TangleStyle* style) {
	if (g_list_find(stylesheet->priv->styles, style)) {
		g_signal_emit(stylesheet, signals[STYLE_REMOVED], 0, style);
		g_object_unref(style);
	}
}

void tangle_stylesheet_apply(TangleStylesheet* stylesheet, GObject* object) {
	gboolean handled = FALSE;
	TangleStyle* style;

	if (TANGLE_IS_STYLABLE(object)) {
		handled = tangle_stylable_apply_stylesheet(TANGLE_STYLABLE(object), stylesheet);
	}
	
	if (!handled && (style = tangle_stylesheet_get_style_for_object(stylesheet, object))) {
		tangle_style_apply(style, object);
		g_object_unref(style);
	}
}

void tangle_stylesheet_unapply(TangleStylesheet* stylesheet, GObject* object) {
	gboolean handled = FALSE;
	TangleStyle* style;

	if (TANGLE_IS_STYLABLE(object)) {
		handled = tangle_stylable_unapply_stylesheet(TANGLE_STYLABLE(object), stylesheet);
	}
	
	if (!handled && (style = tangle_stylesheet_get_style_for_object(stylesheet, object))) {
		tangle_style_unapply(style, object);
		g_object_unref(style);
	}
}

TangleStyle* tangle_stylesheet_get_style_for_object(TangleStylesheet* stylesheet, GObject* object) {
	TangleCompoundStyle* compound_style;
	GList* style_in_list;
	TangleStyle* style;
	
	/* TODO: This can be optimized so that compound style is created only when really needed. */
	compound_style = tangle_compound_style_new_with_stylesheet(G_OBJECT_TYPE(object), stylesheet);
	for (style_in_list = stylesheet->priv->styles; style_in_list; style_in_list = style_in_list->next) {
		style = TANGLE_STYLE(style_in_list->data);
		if (tangle_style_is_for_object(style, object)) {
			tangle_compound_style_add_style(compound_style, style);
		}
	}
	
	return TANGLE_STYLE(compound_style);
}

TangleStylesheet* tangle_stylesheet_get_default() {
	
	if (!default_stylesheet) {
		default_stylesheet = tangle_stylesheet_new_from_file("draft");
	}
	
	return default_stylesheet;
}

void tangle_stylesheet_set_default(TangleStylesheet* stylesheet) {
	if (default_stylesheet) {
		g_object_unref(default_stylesheet);
	}
	default_stylesheet = stylesheet;
	g_object_ref(default_stylesheet);
}

static void tangle_stylesheet_style_added(TangleStylesheet* stylesheet, TangleStyle* style) {
	stylesheet->priv->styles = g_list_append(stylesheet->priv->styles, style);
}

static void tangle_stylesheet_style_removed(TangleStylesheet* stylesheet, TangleStyle* style) {
	stylesheet->priv->styles = g_list_remove(stylesheet->priv->styles, style);
}

static void tangle_stylesheet_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleStylesheet* stylesheet;
	
	stylesheet = TANGLE_STYLESHEET(object);

	switch (prop_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_stylesheet_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleStylesheet* stylesheet;

	stylesheet = TANGLE_STYLESHEET(object);

        switch (prop_id) {
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_stylesheet_dispose(GObject* object) {
	TangleStylesheet* stylesheet;
	
	stylesheet = TANGLE_STYLESHEET(object);
	
	if (stylesheet->priv->styles) {
		g_list_foreach(stylesheet->priv->styles, (GFunc)g_object_unref, NULL);
		g_list_free(stylesheet->priv->styles);
		stylesheet->priv->styles = NULL;
	}
	
	G_OBJECT_CLASS(tangle_stylesheet_parent_class)->dispose(object);
}

static void tangle_stylesheet_class_init(TangleStylesheetClass* stylesheet_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(stylesheet_class);

	gobject_class->dispose = tangle_stylesheet_dispose;
	gobject_class->set_property = tangle_stylesheet_set_property;
	gobject_class->get_property = tangle_stylesheet_get_property;

	stylesheet_class->style_added = tangle_stylesheet_style_added;
	stylesheet_class->style_removed = tangle_stylesheet_style_removed;
	
	signals[STYLE_ADDED] = g_signal_new("style-added",
	                                    G_TYPE_FROM_CLASS(gobject_class),
					    G_SIGNAL_RUN_FIRST,
					    G_STRUCT_OFFSET(TangleStylesheetClass, style_added),
					    NULL, NULL,
					    tangle_marshal_VOID__OBJECT,
					    G_TYPE_NONE, 1,
					    TANGLE_TYPE_STYLE);

	signals[STYLE_REMOVED] = g_signal_new("style-removed",
	                                      G_TYPE_FROM_CLASS(gobject_class),
					      G_SIGNAL_RUN_FIRST,
					      G_STRUCT_OFFSET(TangleStylesheetClass, style_removed),
					      NULL, NULL,
					      tangle_marshal_VOID__OBJECT,
					      G_TYPE_NONE, 1,
					      TANGLE_TYPE_STYLE);

	g_type_class_add_private (gobject_class, sizeof (TangleStylesheetPrivate));
}

static void tangle_stylesheet_init(TangleStylesheet* stylesheet) {
	stylesheet->priv = G_TYPE_INSTANCE_GET_PRIVATE(stylesheet, TANGLE_TYPE_STYLESHEET, TangleStylesheetPrivate);
}


static TangleStylesheet* load_from_file(const gchar* directory_prefix, const gchar* name) {
	TangleStylesheet* stylesheet = NULL;
	gchar* path;
	gchar* module_path;
	gchar* script_path;
	GModule* module;
	const gchar working_directory[PATH_MAX];
	ClutterScript* script;
	GError* error = NULL;
	GList* list;
	GList* item;

	path = g_strconcat(directory_prefix, G_DIR_SEPARATOR_S, name, G_DIR_SEPARATOR_S, NULL);	
	module_path = g_strconcat(path, "stylesheet", NULL);
	script_path = g_strconcat(path, "stylesheet.json", NULL);

	if (g_module_supported()) {
		module = g_module_open(module_path, 0);
	}

	if (getcwd(working_directory, PATH_MAX) &&
	    chdir(path) == 0) {
	    	script = clutter_script_new();
		clutter_script_add_search_paths(script, (const gchar**)&path, 1);
		if (!clutter_script_load_from_file(script, script_path, &error)) {
			g_warning("Could not load stylesheet from %s: %s\n", script_path, error->message);
			g_error_free(error);
		} else {
			stylesheet = tangle_stylesheet_new();
			list = clutter_script_list_objects(script);
			for (item = list; item; item = item->next) {
				if (TANGLE_IS_STYLE(item->data)) {
					tangle_stylesheet_add_style(stylesheet, TANGLE_STYLE(item->data));
				}
			}
			g_list_free(list);
		}
		g_object_unref(script);
		
		chdir(working_directory);
	}
	
	if (!stylesheet && module) {
		g_module_close(module);
	} else if (stylesheet && module) {
		/* It would be very hard to know when all style related objects had been unreferenced. */
		g_module_make_resident(module);
	}

	g_free(path);
	g_free(module_path);
	g_free(script_path);

	return stylesheet;
}
