/*
 * tangle-properties.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-properties.h"
#include "marshalers.h"
#include <string.h>
#include <gobject/gvaluecollector.h>
#include <clutter/clutter.h>

/* We know that these are available in the Clutter library (from clutter-script-private.h). */
gboolean clutter_script_parse_node (ClutterScript *script, GValue *value, const gchar *name, JsonNode *node, GParamSpec *pspec);


static void clutter_scriptable_iface_init(ClutterScriptableIface* iface);

/**
 * SECTION:tangle-properties
 * @Short_description: 	A data structure holding named GValues
 * @Title: TangleProperties
 */

G_DEFINE_TYPE_WITH_CODE(TangleProperties, tangle_properties, TANGLE_TYPE_OBJECT,
                        G_IMPLEMENT_INTERFACE(CLUTTER_TYPE_SCRIPTABLE, clutter_scriptable_iface_init););

struct _TanglePropertiesPrivate {
	GHashTable* hash_table;
};

enum {
	CHANGED,
	LAST_SIGNAL
};

static void free_g_value_slice(gpointer p);
static void free_str_slice(gpointer p);

static ClutterScriptableIface* parent_scriptable_iface = NULL;
static guint signals[LAST_SIGNAL];

TangleProperties* tangle_properties_new(void) {

	return TANGLE_PROPERTIES(g_object_new(TANGLE_TYPE_PROPERTIES, NULL));
}

GType tangle_properties_get_property_type(TangleProperties* properties, const gchar* name) {
	GType type = G_TYPE_INVALID;
	GValue* existing_value;
	
	existing_value = g_hash_table_lookup(properties->priv->hash_table, name);
	if (existing_value) {
		type = G_VALUE_TYPE(existing_value);
	}

	return type;
}

gboolean tangle_properties_get_property(TangleProperties* properties, const gchar* name, GValue* value) {
	gboolean found = FALSE;
	GValue* existing_value;
	
	existing_value = g_hash_table_lookup(properties->priv->hash_table, name);
	if (existing_value) {
		g_value_init(value, G_VALUE_TYPE(existing_value));
		g_value_copy(existing_value, value);
		found = TRUE;
	}
	
	return found;
}

gboolean tangle_properties_get(TangleProperties* properties, const gchar* first_name, GType first_type, ...) {
	const gchar* name;
	GType type;
	GValue* value;
	va_list args;
	guint i;
	gchar* error;

	name = first_name;
	type = first_type;
	va_start(args, first_type);
	do {
		value = g_hash_table_lookup(properties->priv->hash_table, name);
		if (!value) {
			return FALSE;
		}
		g_return_val_if_fail(G_VALUE_HOLDS(value, type), FALSE);
		
		error = NULL;
		G_VALUE_LCOPY(value, args, 0, &error);
		if (error) {
			g_warning("%s: %s", G_STRLOC, error);
			g_free(error);
		}
		name = va_arg(args, const gchar*);
		if (name) {
			type = va_arg(args, GType);
		}
	} while (name);

	return TRUE;
}

gboolean tangle_properties_get_boolean(TangleProperties* properties, const gchar* name) {
	gboolean value = FALSE;
	GValue* existing_value;
	
	existing_value = g_hash_table_lookup(properties->priv->hash_table, name);
	if (!existing_value) {
		g_critical("No such property: '%s'.", name);
	} else {
		value = g_value_get_boolean(existing_value);
	}
	
	return value;
}

gint64 tangle_properties_get_int(TangleProperties* properties, const gchar* name) {
	gint64 value = 0;
	GValue* existing_value;
	
	existing_value = g_hash_table_lookup(properties->priv->hash_table, name);
	if (!existing_value) {
		g_critical("No such property: '%s'.", name);
	} else {
		value = g_value_get_int64(existing_value);
	}
	
	return value;
}

double tangle_properties_get_double(TangleProperties* properties, const gchar* name) {
	gdouble value = 0.0;
	GValue* existing_value;
	
	existing_value = g_hash_table_lookup(properties->priv->hash_table, name);
	if (!existing_value) {
		g_critical("No such property: '%s'.", name);
	} else {
		value = g_value_get_double(existing_value);
	}
	
	return value;
}

const gchar* tangle_properties_get_string(TangleProperties* properties, const gchar* name) {
	const gchar* value = NULL;
	GValue* existing_value;
	
	existing_value = g_hash_table_lookup(properties->priv->hash_table, name);
	if (!existing_value) {
		g_critical("No such property: '%s'.", name);
	} else {
		value = g_value_get_string(existing_value);
	}
	
	return value;
}

gchar* tangle_properties_dup_string(TangleProperties* properties, const gchar* name) {
	gchar* value = NULL;
	GValue* existing_value;
	
	existing_value = g_hash_table_lookup(properties->priv->hash_table, name);
	if (!existing_value) {
		g_critical("No such property: '%s'.", name);
	} else {
		value = g_value_dup_string(existing_value);
	}
	
	return value;
}

GParameter* tangle_properties_get_as_parameters(TangleProperties* properties, guint* n_p) {
	GParameter* parameters;
	GHashTableIter iter;
	guint i;
	const gchar* name;
	GValue* value;

	*n_p = g_hash_table_size(properties->priv->hash_table);
	parameters = g_new0(GParameter, *n_p);
	g_hash_table_iter_init(&iter, properties->priv->hash_table);
	for (i = 0; i < *n_p && g_hash_table_iter_next(&iter, (gpointer*)&name, (gpointer*)&value); i++) {
		parameters[i].name = g_strdup(name);
		g_value_init(&parameters[i].value, G_VALUE_TYPE(value));
		g_value_copy(value, &parameters[i].value);
	}
	
	return parameters;
}

void tangle_properties_set_property(TangleProperties* properties, const gchar* name, const GValue* value) {
	gpointer name_slice;
	GValue* value_slice;
	
	g_return_if_fail(TANGLE_IS_PROPERTIES(properties));
	g_return_if_fail(name != NULL);
	g_return_if_fail(value == NULL || G_IS_VALUE(value));
	
	if (value && TANGLE_PROPERTIES_GET_CLASS(properties)->validate_property) {
		g_return_if_fail(TANGLE_PROPERTIES_GET_CLASS(properties)->validate_property(properties, name, value));
	}
	
	if (value) {
		value_slice = g_slice_new0(GValue);
		g_value_init(value_slice, G_VALUE_TYPE(value));
		g_value_copy(value, value_slice);
		name_slice = g_slice_copy(strlen(name) + 1, name);
		g_hash_table_insert(properties->priv->hash_table, name_slice, value_slice);		
	} else {
		g_hash_table_remove(properties->priv->hash_table, name);
	}

	g_signal_emit(properties, signals[CHANGED], 0, name, value);
}

void tangle_properties_set(TangleProperties* properties, const gchar* first_name, ...) {
	va_list properties_list;

	va_start(properties_list, properties);
	tangle_properties_set_valist(properties, properties_list);
	va_end(properties_list);
}

void tangle_properties_set_valist(TangleProperties* properties, va_list properties_list) {
	const gchar* name;
	GType type;
	GValue value = { 0 };
	gchar* error;

	while ((name = va_arg(properties_list, const gchar*))) {
		type = va_arg(properties_list, GType);
		g_value_init(&value, type);
		error = NULL;
		G_VALUE_COLLECT(&value, properties_list, 0, &error);
		if (error) {
			g_warning("%s: %s", G_STRLOC, error);
			g_free(error);
		} else {
			tangle_properties_set_property(properties, name, &value);
		}
		g_value_unset(&value);
	}	
}

void tangle_properties_set_boolean(TangleProperties* properties, const gchar* name, gboolean boolean_value) {
	GValue value = { 0 };
	
	g_value_init(&value, G_TYPE_BOOLEAN);
	g_value_set_boolean(&value, boolean_value);
	tangle_properties_set_property(properties, name, &value);
	g_value_unset(&value);
}

void tangle_properties_set_int(TangleProperties* properties, const gchar* name, gint64 int_value) {
	GValue value = { 0 };
	
	g_value_init(&value, G_TYPE_INT64);
	g_value_set_int64(&value, int_value);
	tangle_properties_set_property(properties, name, &value);
	g_value_unset(&value);
}

void tangle_properties_set_double(TangleProperties* properties, const gchar* name, double double_value) {
	GValue value = { 0 };
	
	g_value_init(&value, G_TYPE_DOUBLE);
	g_value_set_double(&value, double_value);
	tangle_properties_set_property(properties, name, &value);
	g_value_unset(&value);
}

void tangle_properties_set_string(TangleProperties* properties, const gchar* name, const gchar* string_value) {
	GValue value = { 0 };
	
	g_value_init(&value, G_TYPE_STRING);
	g_value_set_string(&value, string_value);
	tangle_properties_set_property(properties, name, &value);
	g_value_unset(&value);
}

void tangle_properties_foreach(TangleProperties* properties, TanglePropertyCallback callback, gpointer user_data) {
	g_hash_table_foreach(properties->priv->hash_table, (GHFunc)callback, user_data);
}

static void tangle_properties_finalize(GObject* object) {
	g_hash_table_unref(TANGLE_PROPERTIES(object)->priv->hash_table);
	
	G_OBJECT_CLASS(tangle_properties_parent_class)->finalize(object);
}

static void tangle_properties_dispose(GObject* object) {
	g_hash_table_remove_all(TANGLE_PROPERTIES(object)->priv->hash_table);
	
	G_OBJECT_CLASS(tangle_properties_parent_class)->dispose(object);
}

static void tangle_properties_class_init(TanglePropertiesClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);

	gobject_class->finalize = tangle_properties_finalize;
	gobject_class->dispose = tangle_properties_dispose;

	signals[CHANGED] = g_signal_new("changed",
					G_TYPE_FROM_CLASS(gobject_class),
					G_SIGNAL_RUN_LAST,
					0,
					NULL, NULL,
					tangle_marshal_VOID__STRING_POINTER,
					G_TYPE_NONE, 2,
					G_TYPE_STRING,
					G_TYPE_VALUE);

	g_type_class_add_private (gobject_class, sizeof (TanglePropertiesPrivate));
}

static void tangle_properties_init(TangleProperties* properties) {
	properties->priv = G_TYPE_INSTANCE_GET_PRIVATE(properties, TANGLE_TYPE_PROPERTIES, TanglePropertiesPrivate);
	properties->priv->hash_table = g_hash_table_new_full(g_str_hash, g_str_equal, free_str_slice, free_g_value_slice);
}

static void tangle_properties_set_custom_property(ClutterScriptable* scriptable, ClutterScript* script, const gchar* name, const GValue* value) {
	if (g_object_class_find_property(G_OBJECT_GET_CLASS(G_OBJECT(scriptable)), name)) {
		if (parent_scriptable_iface->set_custom_property) {
			parent_scriptable_iface->set_custom_property(scriptable, script, name, value);
		} else {
			g_object_set_property(G_OBJECT(scriptable), name, value);
		}
	} else {
		tangle_properties_set_property(TANGLE_PROPERTIES(scriptable), name, value);	
	}
}

static void clutter_scriptable_iface_init(ClutterScriptableIface* iface) {
	if (!(parent_scriptable_iface = parent_scriptable_iface = g_type_interface_peek_parent (iface))) {
		parent_scriptable_iface = g_type_default_interface_peek(CLUTTER_TYPE_SCRIPTABLE);
	}
	
	iface->set_custom_property = tangle_properties_set_custom_property;
}

static void free_g_value_slice(gpointer p) {
	g_value_unset((GValue*)p);
	g_slice_free(GValue, p);
}

static void free_str_slice(gpointer p) {
	g_slice_free1(strlen(p) + 1, p);
}
