/*
 * tangle-view.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2009-2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-view.h"
#include "tangle-misc.h"

/**
 * SECTION:tangle-view
 * @Short_description: A container that is constructed with ClutterScript
 * @Title: TangleView
 */

static void clutter_scriptable_iface_init(ClutterScriptableIface *iface);

G_DEFINE_TYPE(TangleView, tangle_view, TANGLE_TYPE_WIDGET);

enum {
	PROP_0,
	PROP_FILENAME,
	PROP_SCRIPT
};

struct _TangleViewPrivate {
	gchar* filename;
	ClutterScript* script;
	guint script_id;
	guint loaded : 1;
};

static void add_orphans_to_container(TangleView* view);

ClutterActor* tangle_view_new() {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_VIEW, NULL));
}

ClutterActor* tangle_view_new_with_parent(TangleWidget* parent) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_VIEW, "parent", parent, NULL));
}

ClutterActor* tangle_view_new_with_layout(TangleLayout* layout) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_VIEW, "layout", layout, NULL));
}

void tangle_view_set_filename(TangleView* view, const gchar* filename) {
	GError* error = NULL;

	if (g_strcmp0(view->priv->filename, filename)) {
		if (view->priv->script && view->priv->script_id) {
			/* TODO: unparent every children. */
			clutter_script_unmerge_objects(view->priv->script, view->priv->script_id);
			view->priv->script_id = 0;
		}
		g_free(view->priv->filename);
		view->priv->filename = tangle_actor_lookup_filename(TANGLE_ACTOR(view), filename);
		if (!view->priv->filename) {
			g_warning("%s: file '%s' not found.", G_STRLOC, filename);
		}
		view->priv->loaded = FALSE;
		g_object_notify(G_OBJECT(view), "filename");
	}
}

const gchar* tangle_view_get_filename(TangleView* view) {

	return view->priv->filename;
}

void tangle_view_load_now(TangleView* view) {
	GError* error = NULL;

	g_return_if_fail(view->priv->filename != NULL);

	if (!view->priv->loaded) {
		if (!view->priv->script && !(view->priv->script = tangle_actor_get_script(TANGLE_ACTOR(view)))) {
			view->priv->script = clutter_script_new();
		}

		if (!(view->priv->script_id =
		      clutter_script_load_from_file(view->priv->script, view->priv->filename, &error))) {
			g_critical("Cannot load JSON file '%s': %s\n", view->priv->filename, error->message);
			g_error_free(error);
		} else {
			add_orphans_to_container(view);
			clutter_script_connect_signals(view->priv->script, view);
		}

		view->priv->loaded = TRUE;
	}
}

void tangle_view_start_timelines(TangleView* view) {
	/* TODO: Implement! */
}

void tangle_view_stop_timelines(TangleView* view) {
	/* TODO: Implement! */
}

static ClutterAnimation* tangle_view_animate_transition(TangleActor* actor, ClutterActorBox* current_box, ClutterActorBox* new_box) {
	ClutterAnimation* animation = NULL;
	guint duration;
	gulong mode;
	ClutterActor* parent;

	g_object_get(G_OBJECT(actor), "transition-duration", &duration, "transition-mode", &mode, NULL);
	if (duration && mode) {
		if (current_box && new_box) {
			animation = TANGLE_ACTOR_CLASS(tangle_view_parent_class)->animate_transition(actor, current_box, new_box);
		} else if (current_box) {
			animation = tangle_actor_animate(actor, mode, duration,
			                                 "transition-move-x", -(current_box->x2 - current_box->x1) / 4,
							 "transition-move-y", -(current_box->y2 - current_box->y1) / 4,
							 "transition-scale-x", 2.0,
							 "transition-scale-y", 2.0,
							 "opacity", 0,
							 NULL);
		} else if ((parent = clutter_actor_get_parent(CLUTTER_ACTOR(actor))) &&
		           (!TANGLE_IS_ACTOR(parent) || TANGLE_ACTOR_IS_ALLOCATED(parent))) {
			g_object_set(G_OBJECT(actor),
			             "transition-move-x", (new_box->x2 - new_box->x1) / 4,
				     "transition-move-y", (new_box->y2 - new_box->y1) / 4,
				     "transition-scale-x", 0.5,
				     "transition-scale-y", 0.5,
				     "opacity", 0,
				     NULL);
			animation = tangle_actor_animate(actor, mode, duration,
			                                 "transition-move-x", 0.0,
							 "transition-move-y", 0.0,
							 "transition-scale-x", 1.0,
							 "transition-scale-y", 1.0,
							 "opacity", 255,
							 NULL);
		}
	}

	return animation;
}

static void tangle_view_map(ClutterActor* actor) {
	TangleView* view;
	
	view = TANGLE_VIEW(actor);
	
	if (view->priv->filename && !view->priv->loaded) {
		tangle_view_load_now(view);
	}

	CLUTTER_ACTOR_CLASS(tangle_view_parent_class)->map(actor);	
}

static void tangle_view_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleView* view;
	
	view = TANGLE_VIEW(object);

	switch (prop_id) {
		case PROP_FILENAME:
			tangle_view_set_filename(view, g_value_get_string(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_view_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleView* view;

	view = TANGLE_VIEW(object);

        switch (prop_id) {
		case PROP_FILENAME:
			g_value_set_string(value, view->priv->filename);
			break;
		case PROP_SCRIPT:
			g_value_set_object(value, view->priv->script);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_view_finalize(GObject* object) {
	TangleView* view;
	
	view = TANGLE_VIEW(object);
	g_free(view->priv->filename);

	G_OBJECT_CLASS(tangle_view_parent_class)->finalize(object);
}

static void tangle_view_dispose(GObject* object) {
	TangleView* view;
	
	view = TANGLE_VIEW(object);

	G_OBJECT_CLASS(tangle_view_parent_class)->dispose(object);
}

static void tangle_view_class_init(TangleViewClass* view_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(view_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(view_class);
	TangleActorClass* actor_class = TANGLE_ACTOR_CLASS(view_class);

	gobject_class->finalize = tangle_view_finalize;
	gobject_class->dispose = tangle_view_dispose;
	gobject_class->set_property = tangle_view_set_property;
	gobject_class->get_property = tangle_view_get_property;
	
	clutter_actor_class->map = tangle_view_map;

	actor_class->animate_transition = tangle_view_animate_transition;

	/**
	 * TangleView:filename:
	 */
	g_object_class_install_property(gobject_class, PROP_FILENAME,
	                                g_param_spec_string("filename",
	                                "Filename",
	                                "The name of the JSON file containing child objects",
	                                NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleView:script:
	 */
	g_object_class_install_property(gobject_class, PROP_FILENAME,
	                                g_param_spec_object("script",
	                                "Script",
	                                "The ClutterScript that is used to load JSON file",
	                                CLUTTER_TYPE_SCRIPT,
	                                G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private (gobject_class, sizeof (TangleViewPrivate));
}

static void tangle_view_init(TangleView* view) {
	view->priv = G_TYPE_INSTANCE_GET_PRIVATE(view, TANGLE_TYPE_VIEW, TangleViewPrivate);
}

static void add_orphans_to_container(TangleView* view) {
	GList* objects;
	GList* list;
	ClutterActor* actor;
	
	objects = clutter_script_list_objects(view->priv->script);
	for (list = objects; list; list = list->next) {
		if (list->data != (gpointer)view && CLUTTER_IS_ACTOR(list->data) && !TANGLE_IS_VIEW(list->data)) {
			actor = CLUTTER_ACTOR(list->data);
			if (!clutter_actor_get_parent(actor) && !tangle_actor_get_warehouse_parent(actor)) {
				clutter_container_add_actor(CLUTTER_CONTAINER(view), actor);
			}
		}
	}
	g_list_free(objects);
}
