/*
 * tangle-action.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2009-2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-action.h"

/**
 * SECTION:tangle-action
 * @Short_description: 	A base class for functionality that is connected as a property
 * @Title: TangleAction
 */

G_DEFINE_ABSTRACT_TYPE(TangleAction, tangle_action, G_TYPE_OBJECT);

enum {
	PROP_0,
	PROP_NEXT_ACTION
};

struct _TangleActionPrivate {
	TangleAction* next_action;
};

static void remove_action_recursively(TangleAction* action, GObject* object, const char* name);

void tangle_action_execute(TangleAction* action) {
	tangle_action_execute_full(action, NULL, NULL, NULL);
}

void tangle_action_execute_full(TangleAction* action, GObject* source, const gchar* trigger, TangleProperties* properties) {
	GObject* previous_execution_source;
	TangleActionClass* action_class;

	if (source) {
		g_object_ref(source);
	}

	action_class = TANGLE_ACTION_GET_CLASS(action);
	action_class->execute(action, source, trigger, properties);
	if (action->priv->next_action) {
		tangle_action_execute_full(action->priv->next_action, source, trigger, properties);
	}

	if (source) {
		g_object_unref(source);
	}
}

void tangle_action_set_next_action(TangleAction* action, TangleAction* next_action) {
	if (action->priv->next_action) {
		g_object_unref(action->priv->next_action);
	}
	g_object_ref(next_action);
	action->priv->next_action = next_action;
}

TangleAction* tangle_action_get_next_action(TangleAction* action) {

	return action->priv->next_action;
}

void tangle_action_add(TangleAction* action, GObject* object, const char* name) {
	GParamSpec* param_spec;
	TangleAction* existing_action;
	TangleAction* last_action;
	
	param_spec = g_object_class_find_property(G_OBJECT_GET_CLASS(object), name);
	if (!param_spec) {
		g_critical("No such property: %s\n", name);
	} else if (TANGLE_IS_ACTION(G_PARAM_SPEC_VALUE_TYPE(param_spec))) {
		g_critical("Property is not action: %s\n", name);
	} else {
		g_object_get(object, name, &existing_action, NULL);
		if (existing_action) {
			for (last_action = action; action->priv->next_action; action = action->priv->next_action) {
				/* Nothing */
			}
			tangle_action_set_next_action(last_action, existing_action);
		}
		g_object_set(object, name, action, NULL);
	}
}

void tangle_action_remove(TangleAction* action, GObject* object, const char* name) {
	GParamSpec* param_spec;
	
	g_object_ref(object);

	param_spec = g_object_class_find_property(G_OBJECT_GET_CLASS(object), name);
	if (!param_spec) {
		g_critical("No such property: %s\n", name);
	} else if (TANGLE_IS_ACTION(G_PARAM_SPEC_VALUE_TYPE(param_spec))) {
		g_critical("Property is not action: %s\n", name);
	} else {
		g_object_freeze_notify(object);
		remove_action_recursively(action, object, name);
		g_object_thaw_notify(object);
	}
	
	g_object_unref(G_OBJECT(action));
}


static void tangle_action_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleAction* action;
	
	action = TANGLE_ACTION(object);

	switch (prop_id) {
		case PROP_NEXT_ACTION:
			tangle_action_set_next_action(action, TANGLE_ACTION(g_value_get_object(value)));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_action_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleAction* action;

	action = TANGLE_ACTION(object);

        switch (prop_id) {
		PROP_NEXT_ACTION:
			g_value_set_object(value, action->priv->next_action);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_action_dispose(GObject* object) {
	TangleAction* action;
	
	action = TANGLE_ACTION(object);
	if (action->priv->next_action) {
		g_object_unref(action->priv->next_action);
		action->priv->next_action = NULL;
	}

	G_OBJECT_CLASS(tangle_action_parent_class)->dispose(object);
}

static void tangle_action_class_init(TangleActionClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);

	gobject_class->dispose = tangle_action_dispose;
	gobject_class->set_property = tangle_action_set_property;
	gobject_class->get_property = tangle_action_get_property;

	/**
	 * TangleAction:next-action:
	 */
	g_object_class_install_property(gobject_class, PROP_NEXT_ACTION,
	                                g_param_spec_object("next-action",
	                                "Next action",
	                                "The action that is executed after this action in chain",
	                                TANGLE_TYPE_ACTION,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof(TangleActionPrivate));
}

static void tangle_action_init(TangleAction* action) {
	action->priv = G_TYPE_INSTANCE_GET_PRIVATE(action, TANGLE_TYPE_ACTION, TangleActionPrivate);
}

static void remove_action_recursively(TangleAction* action, GObject* object, const char* name) {
	TangleAction* previous_action;
	TangleAction* existing_action;

	g_object_get(object, name, &existing_action, NULL);
	for (previous_action = NULL;
	     existing_action != action && existing_action->priv->next_action;
	     existing_action = existing_action->priv->next_action) {
		previous_action = existing_action;
	}
	if (existing_action == action) {
		if (previous_action) {
			tangle_action_set_next_action(previous_action, action->priv->next_action);
		} else {
			g_object_set(object, name, action->priv->next_action, NULL);
		}
	}

	if (action->priv->next_action) {
		remove_action_recursively(action->priv->next_action, object, name);
	}
}
