/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

/**
 * \file user.h
 */

#ifndef USER_H
#define USER_H

#include "replyerror.h"
#include <QObject>
#include <QDateTime>
#include <QUrl>
#include <QVariantMap>
#include <QNetworkReply>

namespace QtUbuntuOne {

class UserPrivate;

/**
 * \class User
 * \brief Contains the metadata of an UbuntuOne user representation.
 *
 * User contains the metadata of an UbuntuOne user representation.
 * The metadata is read-only.
 */
class User : public QObject
{
    Q_OBJECT

    Q_PROPERTY(QString id
               READ id
               NOTIFY ready)
    Q_PROPERTY(QString name
               READ name
               NOTIFY ready)
    Q_PROPERTY(QString resourcePath
               READ resourcePath
               NOTIFY ready)
    Q_PROPERTY(QString rootNodePath
               READ rootNodePath
               NOTIFY ready)
    Q_PROPERTY(QStringList userNodePaths
               READ userNodePaths
               NOTIFY ready)
    Q_PROPERTY(qint64 maxBytes
               READ maxBytes
               NOTIFY ready)
    Q_PROPERTY(qint64 usedBytes
               READ usedBytes
               NOTIFY ready)
    Q_PROPERTY(ReplyError::Error error
               READ error
               NOTIFY ready)
    Q_PROPERTY(QString errorString
               READ errorString
               NOTIFY ready)

    friend class Files;

public:
    explicit User(QObject *parent = 0);
    explicit User(User *otherUser, QObject *parent = 0);
    ~User();

    /**
     * Loads metadata from another instance of User.
     *
     * \param User*
     */
    Q_INVOKABLE void loadUser(User *otherUser);

    /**
     * Returns the unique id of the User.
     *
     * \return QString
     */
    QString id() const;

    /**
     * Returns the name of the User.
     *
     * \return QString
     */
    QString name() const;

    /**
     * Returns the resource path of the User.
     *
     * \return QString
     */
    QString resourcePath() const;

    /**
     * Returns the root node path of the User.
     *
     * \return QString
     */
    QString rootNodePath() const;

    /**
     * Returns a list of user node paths of the User.
     *
     * \return QStringList
     */
    QStringList userNodePaths() const;

    /**
     * Returns the maximum storage of the User.
     *
     * \return qint64
     */
    qint64 maxBytes() const;

    /**
     * Returns the used storage of the User.
     *
     * \return qint64
     */
    qint64 usedBytes() const;

    /**
     * Returns the error resulting from the request, or NoError if no error was reported.
     *
     * \return ReplyError::Error The request error.
     */
    ReplyError::Error error() const;

    /**
     * Returns a human-readable string detailing the error resulting from the request, or an empty string if no error was reported.
     *
     * \return QString The request error string.
     */
    QString errorString() const;

public slots:
    /**
     * Cancels the request.
     */
    void cancel();

signals:
    /**
     * Emitted when the request is completed.
     *
     * \param user The User object.
     */
    void ready(User *user);

    /**
     * Emitted when the request is cancelled.
     *
     * \param user The User object.
     */
    void cancelled(User *user);

private:
    explicit User(QNetworkReply *reply, QObject *parent = 0);
    explicit User(UserPrivate &d, QObject *parent = 0);

    QScopedPointer<UserPrivate> d_ptr;

    Q_DECLARE_PRIVATE(User)

    Q_PRIVATE_SLOT(d_func(), void _q_onReplyFinished())
};

}

#endif // USER_H
