/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

/**
 * \file token.h
 */

#ifndef TOKEN_H
#define TOKEN_H

#include "qubuntuone_global.h"
#include "replyerror.h"
#include <QObject>

namespace QtUbuntuOne {

class TokenPrivate;

/**
 * \class Token
 * \brief Contains data for an Authentication request.
 *
 * Token contains metadata that results from a request made using the Authentication class.
 * The metadata takes the form of a QVariant. Token also provides convenience methods for
 * accessing the credentials returned by the request.
 */
class QUBUNTUONESHARED_EXPORT Token : public QObject
{
    Q_OBJECT

    Q_PROPERTY(QString tokenName
               READ tokenName
               NOTIFY ready)
    Q_PROPERTY(QString consumerKey
               READ consumerKey
               NOTIFY ready)
    Q_PROPERTY(QString consumerSecret
               READ consumerSecret
               NOTIFY ready)
    Q_PROPERTY(QString tokenKey
               READ tokenKey
               NOTIFY ready)
    Q_PROPERTY(QString tokenSecret
               READ tokenSecret
               NOTIFY ready)
    Q_PROPERTY(ReplyError::Error error
               READ error
               NOTIFY ready)
    Q_PROPERTY(QString errorString
               READ errorString
               NOTIFY ready)

    friend class Authentication;

public:
    ~Token();

    /**
     * Returns the token name.
     *
     * \return QString The token name.
     */
    QString tokenName() const;

    /**
     * Returns the consumer key.
     *
     * \return QString The consumer key.
     */
    QString consumerKey() const;

    /**
     * Returns the consumer secret.
     *
     * \return QString The consumer secret.
     */
    QString consumerSecret() const;

    /**
     * Returns the token key.
     *
     * \return QString The token key.
     */
    QString tokenKey() const;

    /**
     * Returns the token secret.
     *
     * \return QString The token secret.
     */
    QString tokenSecret() const;

    /**
     * Returns the error resulting from the request, or NoError if no error was reported.
     *
     * \return ReplyError::Error The request error.
     */
    ReplyError::Error error() const;

    /**
     * Returns a human-readable string detailing the error resulting from the request, or an empty string if no error was reported.
     *
     * \return QString The request error string.
     */
    QString errorString() const;

public slots:
    /**
     * Cancels the request.
     */
    void cancel();

signals:
    /**
     * Emitted when the request is completed.
     *
     * \param reply The Token object.
     */
    void ready(Token *token);

    /**
     * Emitted when the request is completed.
     *
     * \param reply The Token object.
     */
    void cancelled(Token *token);

private:
    explicit Token(QNetworkReply *reply, QObject *parent = 0);
    explicit Token(TokenPrivate &d, QObject *parent = 0);

    QScopedPointer<TokenPrivate> d_ptr;

    Q_DECLARE_PRIVATE(Token)

    Q_PRIVATE_SLOT(d_func(), void _q_onReplyFinished())
};

}

#endif // TOKEN_H
