/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

/**
 * \file nodelist.h
 */

#ifndef NODELIST_H
#define NODELIST_H

#include "node.h"
#include <QObject>

namespace QtUbuntuOne {

class NodeListPrivate;

/**
 * \class NodeList
 * \brief Contains a list of UbuntuOne nodes.
 *
 * NodeList contains a list of UbuntuOne nodes
 * resulting from a request to the Ubuntu One API.
 */
class NodeList : public QObject
{
    Q_OBJECT

    Q_PROPERTY(int count
               READ count
               NOTIFY ready)
    Q_PROPERTY(QList<Node*> nodes
               READ nodes
               NOTIFY ready)
    Q_PROPERTY(ReplyError::Error error
               READ error
               NOTIFY ready)
    Q_PROPERTY(QString errorString
               READ errorString
               NOTIFY ready)

    friend class Files;

public:
    ~NodeList();

    /**
     * Returns the number of nodes in the list.
     *
     * \return int
     */
    int count() const;

    /**
     * Returns the list of nodes.
     *
     * \return QList<Node*>
     */
    QList<Node*> nodes() const;

    /**
     * Returns the error resulting from the request, or NoError if no error was reported.
     *
     * \return ReplyError::Error The request error.
     */
    ReplyError::Error error() const;

    /**
     * Returns a human-readable string detailing the error resulting from the request, or an empty string if no error was reported.
     *
     * \return QString The request error string.
     */
    QString errorString() const;

public slots:
    /**
     * Cancels the request.
     */
    void cancel();

signals:
    /**
     * Emitted when the request is completed.
     *
     * \param list The NodeList object.
     */
    void ready(NodeList *list);

    /**
     * Emitted when the request is cancelled.
     *
     * \param list The NodeList object.
     */
    void cancelled(NodeList *list);
    
private:
    explicit NodeList(QNetworkReply *reply, QObject *parent = 0);
    explicit NodeList(NodeListPrivate &d, QObject *parent = 0);

    QScopedPointer<NodeListPrivate> d_ptr;

    Q_DECLARE_PRIVATE(NodeList)

    Q_PRIVATE_SLOT(d_func(), void _q_onReplyFinished())
};

}

#endif // NODELIST_H
