/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/


/*****************************************************************************
 * This file demonstrates the general functions of PeerHood C API.
 * The purpose is to introduce the proper approach about how to
 * use the C API.
 *
 * Proper way to start:
 * 1.  Create C_Callback with ph_c_create_callback.
 * 2.  Get an instance of PeerHood with ph_c_get_instance.
 * 3.  Initialize PeerHood with ph_c_init.
 * 4.  Use ph_c_get_devicelist to see list of found devices.
 * 5.  Use ph_c_get_servicelist on found device to get services
 *     and use ph_c_get_localservicelist to get local services.
 * 6.  Use ph_c_connect_remoteservice to connect service on remote device
 *     and ph_c_connect_localservice to connect a local service.
 * 7.  Use ph_c_connection_write to write data and ph_c_connection_read
 *     to read data from connection (returned by functions in 6.).
 * 8.  Disconnect from service with ph_c_connection_disconnect.
 * 9.  Delete PeerHood instance with ph_c_delete_peerhood
 * 10. Delete callback with ph_c_delete_callback
 *
 * For information about other functions provided by C API of PeerHood
 * see the C API specification (url?).
 ****************************************************************************/
 

#include <stdlib.h>
#include <stdio.h>
#include <sys/stat.h>
#include <string.h>

// Required PeerHood libraries
#include <PeerHood.h>
#include <IteratorWrapper.h>

// FALSE and TRUE are defined in PeerHood
// TRUE == 1
// FALSE == 0
int isserver = FALSE;
int connected = FALSE;
MAbstractConnection* conn;
int notified = 0;
int connections = 0;

void callback_notify(short, const char*, void*);
void callback_newconnection(const unsigned short, MAbstractConnection*, int, void*);
void list_local_services(MPeerHood*);
void list_services(MAbstractDevice*, MPeerHood*);
void do_query(MPeerHood*);


int main(int argc, char* argv[])
{
	char input = '\n';
	MPeerHood* peerhood;
	
	// Define function pointers for notify and new connection functions
	void (*notify)(short, const char*,void*) = NULL;
	void (*newconnection)(const unsigned short, MAbstractConnection*, int,void*) = NULL;
	
	// Initialize function pointers to point to appropriate functions
	notify = &callback_notify;
	newconnection = &callback_newconnection;
	
	// Create callback and give function pointers as parameters
	C_Callback* callback = ph_c_create_callback(notify, newconnection, NULL, NULL);
	
	// Get an instance of peerhood with created callback 
	peerhood = ph_c_get_instance(callback);
	
	// Initialize peerhood
	if(ph_c_init(peerhood, argc, argv) == FALSE)
	{
		perror("PeerHood initialization failed. Quitting.");
		return -1;
	}
	else
	{
		printf("\n\n===============================================================\n");
		printf("|      P H    C    C L I E N T   I N S T R U C T I O N S      |\n");
		printf("===============================================================\n");
		printf("| Create new service with name \"c.test\" by entering \"n\".      |\n");
		printf("| Do a query of devices and their services by pressing enter. |\n");
		printf("| Quit with any other key                                     |\n");
		printf("===============================================================\n");
	}
	
	input = getc(stdin);
	
	// With enter list services, n add service if not added
	while(input == '\n' || input == 'n')
	{
		if(input == 'n' && isserver == FALSE)
		{
			// Registering a service
			unsigned short s = ph_c_register_service(peerhood,"c.test","attributes");
			
			// Register with port
			//unsigned short s = ph_c_register_service(peerhood,"c.test","attributes","30000");
			if(s == 0) printf("Cannot register service\n");
			else
			{
				printf("Service on, port = %d.\n",s);
				isserver = TRUE;
			}
		}
		else
		{

			printf("\tLISTING LOCAL SERVICES\n");
			printf("======================================\n");
			// List local services
			list_local_services(peerhood);
			
			printf("\n\n\tQUERYING OTHER DEVICES\n");
			printf("======================================\n");
			// List other devices and their services
			do_query(peerhood);
		}
		input = getc(stdin);
	}
	printf("Quitting.\nNotifications: %d\nConnections:%d\n",notified, connections);
	
	// Unregister service
	if(isserver == TRUE)	ph_c_unregister_service(peerhood,"c.test");
	
	// Check if connection established
	if(connected == TRUE && ph_c_connection_is_connected(conn) == TRUE)
	{
		// Try to disconnect
		if(ph_c_connection_disconnect(conn) == TRUE) printf("Connection disconnected\n");
		else printf("Unable to disconnect\n");
	}
	
	// Delete peerhood instance
	ph_c_delete_peerhood(peerhood);
	
	// delete callback
	ph_c_delete_callback(callback);
	
	
	return 0;
}

// Listing of local services
void list_local_services(MPeerHood* peerhood)
{
	unsigned int c = 0;
	
	// Get services
	TServiceList* list = ph_c_get_localservicelist(peerhood);
	
	printf("Local services ");
	
	// List not empty
	if(ph_c_servicelist_is_empty(list) == FALSE)
	{
		// Get list size
		printf("(%d):\n", ph_c_servicelist_size(list));
		
		// Initialize service iterator
		ServiceIterator* i = ph_c_new_service_iterator(list);
		
		// If not last
		while(ph_c_service_iterator_is_last(i,list) == FALSE)
		{
			// Get service
			CService* service = ph_c_service_iterator_get_service(i);
			
			// Name of the service
			printf("\t\t%d: %s\n",++c, ph_c_service_get_name(service));
			
			// If it is a test service, try to connect, don't connect to local if acting as server
			if(isserver == FALSE && connected == FALSE && strcmp(ph_c_service_get_name(service),"c.test") == 0)
			{
				printf("connecting... ");
				
				// Establish connection to service
				conn = ph_c_connect_localservice(peerhood,service);
				connected = 1;
				printf("ok!\n");

				// Disconnect
				if(ph_c_connection_disconnect(conn) == TRUE) printf("Disconnected\n");
				else printf("Couldn't disconnect\n");
					
			}
			// Next iterator
			ph_c_service_iterator_next_iterator(i);
		}
		
		// Delete iterator
		ph_c_delete_service_iterator(i);
	}
	else printf("\n\t\tNo local services.\n");
	
	// Delete list of services
	ph_c_delete_servicelist(list);
}


// List services on device
void list_services(MAbstractDevice* device, MPeerHood* peerhood)
{
	unsigned int c = 0;
	
	// Get services
	TServiceList* list = ph_c_device_get_servicelist(device);
	
	// List not empty
	if(ph_c_servicelist_is_empty(list) == FALSE)
	{
		// Get list size
		printf("(%d):\n", ph_c_servicelist_size(list));
		
		// Initialize service iterator
		ServiceIterator* i = ph_c_new_service_iterator(list);
		
		// If not last
		while(ph_c_service_iterator_is_last(i,list) == FALSE)
		{
			// Get service
			CService* service = ph_c_service_iterator_get_service(i);
			
			// Print the service name
			printf("\t\t%d: %s\n",++c, ph_c_service_get_name(service));
			
			// Try to connect
			if(strcmp(ph_c_service_get_name(service),"c.test") == 0)
			{
				printf("connecting to service \"%s\" on device \"%s\"... ", ph_c_service_get_name(service), ph_c_device_get_name(device));
				
				MAbstractConnection* conn = ph_c_connect_remoteservice(peerhood, device, ph_c_service_get_name(service));
				
				if(conn != NULL) printf("ok!\n");
				else printf("failure\n");

				// Disconnect
				if(ph_c_connection_disconnect(conn) == TRUE) printf("Disconnected\n");
				else printf("Couldn't disconnect\n");
					
			}
			
			// Next iterator
			ph_c_service_iterator_next_iterator(i);
		}
		
		// Delete iterator
		ph_c_delete_service_iterator(i);
	}
	else printf("\n\t\tNo services.\n");
	
	// Delete list of services
	ph_c_delete_servicelist(list);
}

// Get list of devices and probe for services
void do_query(MPeerHood* peerhood)
{
	// Get surrounding devices
	TDeviceList* list = ph_c_get_devicelist(peerhood);
	
	if(ph_c_devicelist_is_empty(list) == FALSE)
	{
		printf("%d devices in list\n",ph_c_devicelist_size(list));
		
		// Initialize wrapper for iterator for list
		DeviceIterator* i = ph_c_new_device_iterator(list);

		// While not the last iterator
		while(ph_c_device_iterator_is_last(i,list) == FALSE)
		{
			// Get the device
			MAbstractDevice* dev = ph_c_device_iterator_get_device(i);
			
			printf("Device name:%s\n",ph_c_device_get_name(dev));
			printf("\t%s/%s\n",ph_c_device_get_prototype(dev),ph_c_device_get_address(dev));
			
			// Has peerhood?
			if(ph_c_device_has_peerhood(dev) == TRUE)
			{
				printf("\tPeerHood enabled device, services: ");
				list_services(dev,peerhood);
			}
			else printf("\tReqular device.\n");

			printf("\n");
			
			// Take the next iterator
			ph_c_device_iterator_next_iterator(i);
		}
		
		// Delete the iterator
		ph_c_delete_device_iterator(i);
	}
	else printf("List is empty\n");
	
	// Delete the list of devices
	ph_c_delete_devicelist(list);
}

/**
 * Notify callback - called by C_Callback when notify event occurs 
 */
void callback_notify(short aEvent, const char* aAddress,void* aData)
{
	printf("Some event notify (%d) from %s.\n", aEvent,aAddress);
	notified++;
}

/**
 * New connection callback - called by C_Callback when new connection is established
 */
void callback_newconnection(const unsigned short aPort, MAbstractConnection* aConnection, int aConnectionId, void* aData)
{
	printf("New connection notification (%d - %d)\n", aPort, aConnectionId);
	connections++;
	
	if(ph_c_connection_is_connected(aConnection) == TRUE)
	{
		// Get file descriptor and address
		printf("Remote: %d: %s",ph_c_connection_get_fd(aConnection), ph_c_connection_get_remote_address(aConnection));
				
		// Disconnect
		if(ph_c_connection_disconnect(aConnection) == TRUE) printf("Disconnected\n");
		
		// Delete connection object
		ph_c_connection_delete(aConnection);
	}
}


