/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

// Emacs, hail the mighty -*-C++-*-

/**
 * Copyright 2003-2011 LUT. .
 *
 * @name IFSearch.h
 * @memo PeerHood's IFSearch class. This class is used for searching
 * currently available network interfaces.
 *
 * @version 0.2
 * date     22.07.2003
 * change   15.04.2011
 */

#ifndef __IFSEARCH_H__
#define __IFSEARCH_H__

#include <string>
#include <list>
#include <sys/socket.h>
#include <netinet/in.h>

#define IFI_NAME 16
#define IFI_ALIAS 1

#define D_IF_NOT_FOUND 0
#define D_IF_ETHERNET 1
#define D_IF_WIRELESS 2
#define D_IF_OTHER 3

// Struct to hold interface data
// contains structures for unicast and broadcast addresses
struct IFInfo {
  struct sockaddr_in iUcast, iBcast;
};

/**
 * @memo PeerHood's IFSearch class. This class is used for searching
 * currently available network interfaces.
 * @doc This class defines the functionality for searching currently
 * available network interfaces. These interfaces 
 * are queried by using ioctl -call. Every interface is then added to 
 * CFInfo struct which holds information about network interfaces. These
 * structures then form a linked list which can be easily read.
 */

class CIFSearch
{

 public:
  CIFSearch();
  ~CIFSearch();
  struct IFInfo *GetIFInfo(const std::string& aIface);
  
    /**
   * @memo Get all interfaces from /proc/net/dev
   * @doc Get the interfaces from file
   */
  bool RetrieveInterfaces(std::list <std::string> &ifaces);
  
  /** 
   * @memo check what type the interface is
   * @doc D_IF_OTHER is either loopback or logical/virtual (TODO),
          approved are D_IF_ETHERNET and D_IF_WIRELESS
   */
  unsigned int CheckInterfaceType(const std::string &ifname);
  
  /**
   * @memo Select appropriate device mac
   * @doc Select one defined in config, if not found use first with wireless
          extensions, if none found use the first non logical or loopback in list
   */
  unsigned char* RetrieveHWaddr(const std::string &ifname, bool &isphysical);
  
  /**
   * @memo
   * @doc
   */
  bool IsVirtualInterface(unsigned char* mac_addr);
  
  /**
   * @memo
   * @doc
   */
  bool IsInterfaceUp(const std::string &ifname);
  
  /**
   * @memo Select "appropriate" interface from all possibilities
   * @doc Select either the configured interface, if it cannot be found use the
   *      first one with wireless extensions, if none found use first in list
          (ethernet).
   */
  const std::string* SelectInterface(bool &select_up_if);

private:
	bool LoadParameter(std::string& aParameter);
	// ioctl socket
	int iSockIOCTL;
};
#endif //__IFSEARCH_H__
