/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/* 
 * @name C_Callback.cc
 * @memo C_Callback object including C wrappers. Meant to be used with C applications.
 *
 * @version 0.34
 * date     12.11.2009
 * change   17.03.2010
 */
 
#include "C_Callback.h"

/**
 * @memo C_Callback constructor
 * @doc Create new callback. Give pointers to functions responsible dealing with
 * notification and new connection events. The 
 * notification function parameters are short (event id), const char* (address) and
 * void* (user data). The new connection function parameters are const unsigned short
 * (port number), MAbstractConnection* (connection object), int (connection id) and 
 * void* (user data). Given pointers to used data are stored and passed to corresponding
 * callbacks each time callback function is called.
 *
 * @param callback_notify a pointer to function to deal with notification callbacks
 * @param callback_newconnection a pointer to function to deal with new connections
 * @param notifydata Pointer to notify function data, saved to private variable
 * @param newconndata Pointer to new connection function data, saved to private var
 */
C_Callback::C_Callback(void(*callback_notify)(short, const char*,void*),
	void(*callback_newconnection)(const unsigned short, MAbstractConnection*, int, void*),
	void *notifydata,
	void *newconndata)
{
	if(callback_notify > 0)	callback_notify_function = callback_notify;
	else callback_notify_function = NULL;
	
	if(callback_newconnection > 0) callback_newconnection_function = callback_newconnection;
	else callback_newconnection_function = NULL;
	
	iNotifyData = notifydata;
	iNewConnectionData = newconndata;
}

/**
 * @memo Receives PeerHood notifications.
 * @doc Receives notifications from the PeerHood library. Calls the notify function via
 * the function pointer given in construction of C_Callback.
 *
 * @param aEvent The type of the occurred event.
 * @param aAddress The address of the device that the event concerns.
 */
void C_Callback::Notify(TEventType aEvent, const std::string& aAddress)
{
	switch (aEvent) {
    case PH_DEVICE_LOST:
			if(callback_notify_function > 0) callback_notify_function(0, aAddress.c_str(), iNotifyData);
      break;

    case PH_DEVICE_FOUND:
			if(callback_notify_function > 0) callback_notify_function(1, aAddress.c_str(), iNotifyData);
      break;

  	case PH_WEAK_LINK:
			if(callback_notify_function > 0) callback_notify_function(2, aAddress.c_str(), iNotifyData);
    	break;

    default:
			if(callback_notify_function > 0) callback_notify_function(-1, aAddress.c_str(), iNotifyData);
		break;
  }
}

/**
 * @memo Handler for a new connection notification.
 * @doc Called when new connections are established. Forwards the connection information
 * to new connection function via the function pointer given in construction phase.
 *
 * @param aPort port number
 * @param aConnection Connection object
 * @param aConnectionId Id of the connection
 */
void C_Callback::NewConnection(const unsigned short aPort, MAbstractConnection* aConnection, int aConnectionId)
{
	if(callback_newconnection_function > 0) callback_newconnection_function(aPort, aConnection, aConnectionId, iNewConnectionData);
}

/**
 * @memo Create new C_Callback
 * @doc Create new C_Callback object. Function pointers are required to be given as
 * parameters for dealing with notification and new connection callbacks. The 
 * notification function parameters are short (event id), const char* (address) and
 * void* (user data). The new connection function parameters are const unsigned short
 * (port number), MAbstractConnection* (connection object), int (connection id) and 
 * void* (user data). Given pointers to used data are stored and passed to corresponding
 * callbacks each time callback function is called.
 *
 * @param callback_notify a pointer to function to deal with notification callbacks
 * @param callback_newconnection a pointer to function to deal with new connections
 * @param notifydata Pointer to notify function data
 * @param newconndata Pointer to new connection function data
 *
 * @return pointer to new C_Callback object
 */
C_Callback* ph_c_create_callback(void(*callback_notify)(short, const char*, void*),
	void(*callback_newconnection)(const unsigned short, MAbstractConnection*, int, void*),
	void *notifydata,
	void *newconndata)
{
	return new C_Callback(callback_notify, callback_newconnection, notifydata, newconndata);
}

/**
 * @memo Delete the C_Callback
 * @doc Delete the callback.
 *
 * @param aCallback callback to be deleted.
 */
void ph_c_delete_callback(C_Callback* aCallback)
{
	if(aCallback) delete aCallback;
}
