/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2008 LUT. .
 *
 * @name WLANListener.cc
 * @memo Implementation of WLANListener.h for linux, uses ioctl.
 *
 * @version 0.1
 * date     14.4.2011
 * change   14.4.2011
 */

#include <cstdlib>
#include <stdlib.h>
#include "IFSearch.h"
#include "WLANListener.h"

using namespace std;

/**
 * @memo Default constructor - hidden
 * @doc Not in use
 */
CWLANListener::CWLANListener()
{
	iName = std::string("WLANListener");
	//iConverter = NULL;
}

/**
 * @memo Parametrizised constructor
 * @doc Constructor for creating a instance of this class, requires a reference to owner
 * component as parameter. The created instance of this class will be registered to that
 * given owner (WLAN plugin).
 * 
 * @param aConverter reference to owner
 */
CWLANListener::CWLANListener(MAbstractStateConverter* aConverter)
{
	if(aConverter == NULL) return;
	iConverter = aConverter;
	iName = std::string("WLANListener");
	iConverter->RegisterListener(this);
	time(&iPrevPoll); // Record time so it is always initialized
}

/**
 * @memo Default destructor
 * @doc Destructor, closes connection to D-Bus if not already closed.
 */
CWLANListener::~CWLANListener()
{
}

/**
 * @memo Connects listener to source and sets up the connection to D-Bus
 * @doc 
 *
 * @return bool true if success
 * 
 */
bool CWLANListener::Connect()
{
	return true;
}

/**
 * @memo Disconnects the listener
 * @doc 
 */
void CWLANListener::Disconnect()
{

}

/**
 * @memo Checks the initial state of the WLAN adapter
 * @doc 
 * 
 */
void CWLANListener::CheckInitialState()
{
	if(iConverter)
	{
		// Adapter exists
		if(iIfSearch.CheckInterfaceType(iConverter->GetAdapter()) != D_IF_NOT_FOUND)
		{
			time(&iPrevPoll);
			iConverter->SetState(iIfSearch.IsInterfaceUp(iConverter->GetAdapter()));
		}
		// No such adapter, select one
		else
		{
			// Select an interface (change to true if only "up" interfaces are allowed)
			bool select_up = false;
			const string* ifname = iIfSearch.SelectInterface(select_up);
			
			if(ifname)
			{
				// Get adapter id (last if digit, otherwise 0)
				int adapterId = 0;
				if(isdigit(ifname->at(ifname->length()-1))) adapterId = atoi(&(ifname->at(ifname->length()-1)));
				
				iConverter->SetAdapter(*ifname,adapterId);
				
				time(&iPrevPoll);
				iConverter->SetState(iIfSearch.IsInterfaceUp(iConverter->GetAdapter()));
				
				delete ifname;
			}
			// No interfaces
			else iConverter->SetState(false);
		}
	}
}

/**
 * @memo Checks the current state of WLAN adapter
 * @doc
 * 
 */
void CWLANListener::CheckState()
{
	if(iConverter)
	{
		time_t time_now;
		time(&time_now);
		
		// Poll via ioctl every 2 seconds
		if(difftime(time_now,iPrevPoll) >= _WL_POLL_INTERVAL)
		{
			time(&iPrevPoll);
			iConverter->SetState(iIfSearch.IsInterfaceUp(iConverter->GetAdapter()));
		}
	}
}

/**
 * @memo Returns listener name
 * @doc Returns the name of this listener (hardcoded atm.)
 * 
 * @return string Name of this listener
 */
const std::string& CWLANListener::GetName()
{
	return iName;
}

