/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

// Emacs, hail the mighty -*-C++-*-

/**
 * Copyright 2003-2011 LUT. .
 *
 * @name PHRTI.cc
 * @memo PeerHood's Run time information class implementation
 *
 * @version 0.11
 * date     19.04.2011
 * change   20.04.2011
 */

#include <fstream>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <net/if.h>

#include "PHRTI.h"

PHRTI* PHRTI::iInstance = NULL;

PHRTI* PHRTI::GetInstance()
{
	if(!iInstance) iInstance = new PHRTI();
	return iInstance;
}

PHRTI::PHRTI()
{
	struct stat str_stat;
	memset(&str_stat,0,sizeof(str_stat));
	
	// False by default
	iRTInfoAccess = false;
	
	// regular user
	if(geteuid()) 
	{
		// Can access directory
		if(stat(_RT_INFO_PATH,&str_stat) == 0)
		{
			if(CheckDirPermissions(&str_stat)) iRTInfoAccess = true;
		}
	}
	// root / super user
	else
	{
		// File/dir exists
		if(stat(_RT_INFO_PATH,&str_stat) == 0)
		{
			// Is not a dir?
			if(!S_ISDIR(str_stat.st_mode))
			{
				// If regular -> delete and create path
				if(S_ISREG(str_stat.st_mode))
				{
					if(remove(_RT_INFO_PATH) == 0)
					{
						if(CreatePath()) iRTInfoAccess = true;
					}
				}
			}
			// Is dir
			else
			{
				if(CheckDirPermissions(&str_stat)) iRTInfoAccess = true;
			}
		}
		// Doesn't exist
		else
		{
			if(CreatePath()) iRTInfoAccess = true;
		}

		// WLAN info file
		memset(&str_stat,0,sizeof(str_stat));
		if(stat(_RT_INFO_PATH_WLAN,&str_stat) == 0)
		{
			if(!CheckFilePermissions(&str_stat)) CorrectFilePermissions(std::string(_RT_INFO_PATH_WLAN));
		}
		else CreateFile(_RT_INFO_WLAN_IF);
		
		// Bluetooth info file
		memset(&str_stat,0,sizeof(str_stat));
		if(stat(_RT_INFO_PATH_BT,&str_stat) == 0)
		{
			if(!CheckFilePermissions(&str_stat)) CorrectFilePermissions(std::string(_RT_INFO_PATH_BT));
		}
		else CreateFile(_RT_INFO_BT_IF);
	}
}

PHRTI::~PHRTI()
{
	if(!geteuid())
	{
		remove(_RT_INFO_PATH_WLAN);
		remove(_RT_INFO_PATH_BT);
	}
}

bool PHRTI::AddRunTimeInfo(RTInfoType aType, const std::string& aInfo)
{
	std::string fpath = "";
	
	// Only for the not weak (root/superuser)
	if(geteuid()) return false;
	
	// Directory is not accessible
	if(!iRTInfoAccess) return false;
	
	switch(aType)
	{
		case _RT_INFO_WLAN_IF:
			fpath = std::string(_RT_INFO_PATH_WLAN);
			break;
		case _RT_INFO_BT_IF:
			fpath = std::string(_RT_INFO_PATH_BT);
			break;
		case _RT_INFO_GPRS_IF:
		default:
			return false;
	}
	
	if(fpath.length() == 0) return false;
	
	std::ofstream infofile (fpath.c_str(), std::ofstream::trunc);
	
	// Constructor should open the file, check if file is "good"
	if(infofile.is_open() && infofile.good())
	{
		infofile.write(aInfo.c_str(), aInfo.length());
		if(infofile.good())
		{
			infofile.close();
			return true;
		}
		else return false; // Write was not successfull
	}
	else return false;
}

const std::string* PHRTI::GetRunTimeInfo(RTInfoType aType)
{
	std::string fpath = "";
	
	// Directory is not accessible
	if(!iRTInfoAccess) return NULL;
	
	switch(aType)
	{
		case _RT_INFO_WLAN_IF:
			fpath = std::string(_RT_INFO_PATH_WLAN);
			break;
		case _RT_INFO_BT_IF:
			fpath = std::string(_RT_INFO_PATH_BT);
			break;
		case _RT_INFO_GPRS_IF:
		default:
			return NULL;
	}
	if(fpath.length() == 0) return NULL;
	
	char buffer[IFNAMSIZ+1];
	memset(&buffer,0,IFNAMSIZ+1);
	
	std::ifstream infofile (fpath.c_str());
	
	if(infofile.is_open())
	{
		long flength = 0;
		
		infofile.seekg(0, std::ios::end);
		
		// Too long to be interface name!
		if((flength = infofile.tellg()) > IFNAMSIZ) return NULL;
		// Too short
		if(flength < 3) return NULL;
		
		infofile.seekg(0,std::ios::beg);
		infofile.read(buffer,flength);
		infofile.close();
		
		if(buffer[flength-1] == '\n') buffer[flength-1] = '\0';
		
		return new std::string(buffer);
	}
	else return NULL;
}

bool PHRTI::CreatePath()
{
	// Only for the not weak (root/superuser)
	if(geteuid()) return false;
	
	if(mkdir(_RT_INFO_PATH,_RT_INFO_DIRMODE) == 0) return true;
	else return false;
}

bool PHRTI::CreateFile(RTInfoType aType)
{
	// Only for the not weak (root/superuser)
	if(geteuid()) return false;
	
	std::string fpath = "";
	switch(aType)
	{
		case _RT_INFO_WLAN_IF:
			fpath = std::string(_RT_INFO_PATH_WLAN);
			break;
		case _RT_INFO_BT_IF:
			fpath = std::string(_RT_INFO_PATH_BT);
			break;
		case _RT_INFO_GPRS_IF:
		default:
			break;
	}
	
	if(fpath.length() == 0) return false;
	
	if(mknod(fpath.c_str(),S_IFREG|_RT_INFO_FILEMODE,0) == 0) return true;
	else return false;
}

bool PHRTI::CheckFilePermissions(struct stat* aStat)
{
	if(aStat->st_mode & (S_IXUSR|S_IWGRP|S_IXGRP|S_IWOTH|S_IXOTH)) return false;
	else if(aStat->st_mode & (_RT_INFO_FILEMODE)) return true;
	else return false;
}

bool PHRTI::CheckDirPermissions(struct stat* aStat)
{
	if(aStat->st_mode & (S_IWGRP|S_IWOTH)) return false;
	else if(aStat->st_mode & (_RT_INFO_DIRMODE)) return true;
	else return false;
}

bool PHRTI::CorrectFilePermissions(const std::string& aPath)
{
	// Only for the not weak (root/superuser)
	if(geteuid()) return false;
	
	if(chmod(aPath.c_str(),_RT_INFO_FILEMODE) == 0) return true;
	else return false;
}

// TODO add file ownership check (uid & guid)
	
