/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2008 LUT. .
 *
 * @name ListenerFactory.cc
 * @memo Implementation of the abstract listener factory. This class is used
 * to create different listeners for different types of components.
 *
 * @version 0.02
 * date     25.08.2008
 * change   23.09.2008
 */

#include "ListenerFactory.h"

using namespace std;

// Initialize instance of self
ListenerFactory* ListenerFactory::iInstance = NULL;

/**
	 * @memo Constructor as private - no external calls 
	 * @doc Constructor set as private because only one instance needed. Used:
	 * Singleton pattern. Used only when there is no instance initialized.
	 */
ListenerFactory::ListenerFactory()
{
	iInstance = this;
	//std::list <MAbstractListenerCreator*> iCreators;
}

/**
 * @memo Destructor, clears instance and the list
 * @doc Deletes the instance and clears the list. Because creator
 * objects are static they will be automatically free'd.
 */
ListenerFactory::~ListenerFactory()
{
	delete iInstance; /** @TODO Check that this doesn't segfault */
	iInstance = NULL;
	iCreators.empty(); // Creator objects are static!
}

/**
 * @memo Singleton pattern instance-method
 * @doc Creates a instance of this class if not already created, then just returns reference to that instance.
 * 
 * @return Instance of this
 */
ListenerFactory* ListenerFactory::GetInstance()
{
	if(iInstance == NULL) new ListenerFactory;
	
	return iInstance;
}

/**
	 * @memo Register listener creator into factory
	 * @doc Register a listener creator, listener creator must implement the 
	 * MAbstractListenerCreator interface. Registered listener creator is added
	 * into iCreators list.
	 * 
	 * @param aCreator Reference to the concrete creator
	 */
void ListenerFactory::Register(MAbstractListenerCreator* aCreator)
{
	// If is an object
	if(aCreator != NULL)
	{
		iCreators.push_back(aCreator);
	}
}

/** 
	 * @memo Creates listener object
	 * @doc Creates a listener object if a suitable creator is found from the
	 * list of the creators, otherwise returns NULL. Given reference of caller 
	 * object will be passed to creator object (and further to plugin).
	 * 
	 * @param aName The name of the prototype that is going to be created
	 * @param aConverter Reference to object that will own the listener, CANNOT BE NULL!
	 * 
	 * @return int amount of listeners that could be found (and created)
	 */
int ListenerFactory::CreateListeners(const std::string& aName, MAbstractStateConverter* aConverter)
{
	MAbstractListener* listener = NULL;
	list<MAbstractListenerCreator*>::iterator iter;
	int created = 0;
	
	/* Go through the list and call every creator to create a listener with given name.
	 * The creator should return NULL if the prototype name is wrong or listener wasn't found.
	 */
	for(iter = iCreators.begin(); iter != iCreators.end(); ++iter)
	{
		if((listener = (*iter)->CreateListener(aName, aConverter)) != NULL) created++;
	}
	return created;
}
