/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

// Emacs, hail the mighty -*-C++-*-

/**
 * @name GPRSPlugin.h
 * @memo GPRS specific implementation of the PeerHood plugin interface.
 *
 * @version 0.1
 * date     21.08.2003
 * change   14.11.2003
 */

#ifndef __GPRSPLUGIN_H__
#define __GPRSPLUGIN_H__

#include <string>
#include <list>
#include <pthread.h>
#include <AbstractPlugin.h>
#include <DaemonDevice.h>

#include <TCPConnection.h>
#include <GPRSConnection.h>
#include <LocalConnection.h>
#include <ServiceCheck.h>
#include "AbstractListener.h"

class CGPRSThread;

/**
 * @memo GPRS specific implementation of the PeerHood plugin interface.
 * @doc This class is GPRS specific implementation of the PeerHood plugin
 * interface. Other PeerHood capable devices are located using the GPRS Gateway
 *
 * @see MAbstractPlugin
 */
class CGPRSPlugin : public MAbstractPlugin
{
 public:
  CGPRSPlugin();
  bool Advert();
  bool Unadvert();
  bool Start();
  void Stop();
  const std::string& GetName();
  void UpdateState();
  void LoadListeners();
  
  // Abstract state changer
  void RegisterListener(MAbstractListener* aListener);
  void SetState(bool aActive);
  void TriggerShutdown();
  void SetAdapter(const std::string& aInterface, int aId);
  const std::string& GetAdapter();

 private:
  static void* ThreadStarter(void *aArguments);
  static void* AdvertStarter(void *aArguments);
  static void *ServerDataThreadStarter(void *aArguments);
  static void *ClientDataThreadStarter(void *aArguments);

  void ClientDataThread(CGPRSThread& aThreadInfo);
  void ServerDataThread(CGPRSThread& aThreadInfo);

  bool AddDevice(unsigned int aId);
  bool AddService(unsigned int aId, CService* aService); 
  void AdvertThread();
  void InquiryThread();
  
  bool StartClientDataThread(CGPRSConnection* aLocalConnection);
  bool StartServerDataThread(unsigned int aOriginalId, const std::string& aServiceName, unsigned int aCommand);

  bool EncapsulateSignal(int aTargetFd, CGPRSSignal& aSignal);
  bool DecapsulateSignal(int aSourceFd, CGPRSSignal& aSignal);
  
  bool MakeOlder();
  bool LoadParameter(std::string& aParameter);
  bool GetLocalServicesL(CService *&aService, std::string& aServiceName);
  bool TestConnection(unsigned int aFd, unsigned int aBytes);
  
  void RemoveListeners();
  
  volatile bool iStarted;
  bool iInSdp, iService;
  bool iAdvertFailed, iInquiryFailed, iLocalFailed;
  bool iServiceReady;
  std::string iName;
  std::string iNDevice;
  pthread_t iInquiryThread, iAdvertThread;

  std::list<CDaemonDevice *> iDeviceList;
  std::list<CServiceCheck *> iServiceCheckList;
  
  // Listener list
  std::list<MAbstractListener*> iListenerList;
  
  int iServiceCheckInterval, iInquiryInterval;
  std::list<CGPRSInquiry *> iInquiryList;

  unsigned int iId;
  std::string iGateway;
  
  
};

class CGPRSThread
{
 public:
  CGPRSPlugin *iInstance;
  pthread_t iThread;
  std::string iServiceName;
  unsigned int iTargetId;
  unsigned int iCommand;
  CGPRSConnection* iLocalConnection;
};

#endif //__GPRSPLUGIN_H
