/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/* 
 * @name ServiceCWrapper.cc
 * @memo Wrapper for CService functions for C
 *
 * @version 0.1
 * date     18.02.2010
 * change   18.02.2010
 */


#include "Service.h"


/**
 * @memo Returns the process id of the parent library
 * @doc Returns the process id of the parent library. On the remote side this
 * function always returns -1.
 *
 * @param _aService pointer to CService object to use.
 *
 * @return the process id of the parent application
 */
int ph_c_service_get_pid(CService* _aService)
{
	return _aService->GetPid();
}

/**
 * @memo Returns the port of service's listening connection.
 * @doc Returns the port of service's listening connection.
 *
 * @param _aService pointer to CService object to use.
 *
 * @return the port of service's listening connection
 */
unsigned short ph_c_service_get_port(CService* _aService)
{
	return _aService->GetPort();
}

/**
 * @memo Returns service's attributes.
 * @doc Returns service's attributes. The format is "key value", with space 
 * between each key-value pair.
 *
 * @param _aService pointer to CService object to use.
 *
 * @return service's attribute string
 */
const char* ph_c_service_get_attributes(CService* _aService)
{
	return (_aService->GetAttributeString()).c_str();
}

/**
 * @memo Returns the value of one attribute.
 * @doc Returns the value of one attribute. If the given attribute is found
 * then its value is inserted to the first parameter. Otherwise the parameter
 * is left intact.
 *
 * @param _aService pointer to CService object to use.
 * @param _aKey The asked attribute. If the asked attribute is found then
 * it is inserted into this parameter upon return.
 * 
 * @return true if the given attribute was found
 */
const char* ph_c_service_get_attribute(CService* _aService, char* _aKey)
{
	std::string aKey = std::string(_aKey);
	
	// Parameter found and set
	if(_aService->GetAttribute(aKey)) return aKey.c_str();
	else return NULL;
}

/**
 * @memo Returns service's name.
 * @doc Returns service's name. Each service should have a unique name. If two 
 * services with the same name exist then the behavior is undefined.
 *
 * @param _aService pointer to CService object to use.
 *
 * @return service's name
 */
const char* ph_c_service_get_name(CService* _aService)
{
	return (_aService->GetName()).c_str();
}
