/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

// Emacs, hail the mighty -*-C++-*-

/**
 * Copyright 2003 LUT. .
 *
 * @name PeerHoodImpl.h
 * @memo Implementation of the MPeerHood interface.
 *
 * @version 0.2
 * date     19.05.2003
 * change   05.02.2010
 */

#ifndef __PEERHOODIMPL_H__
#define __PEERHOODIMPL_H__

#include <pthread.h>
#include <sys/types.h>

#include "PeerHood.h"
#include "Engine.h"
#include "AbstractPinger.h"
#include "AbstractConnection.h"
#include "AbstractMonitor.h"
#include "VirtualConnection.h"

class CMonitorInfo;
class CThreadInfo;
class CEngine; // TODO: type CEngine should be provided from include.
class CVirtualConnection; // TODO: CVirtualConnection should be provided from include.


/**
 * @memo Implementation of the MPeerHood interface.
 * @doc Implementation of the MPeerHood interface. This class separates the
 * definition of the PeerHood interface from its implementation. Application
 * developers should not use this class directly. Instead, they should use
 * the MPeerHood interface.
 *
 * @see MPeerHood
 */
class CPeerHoodImpl : public MPeerHood
{
 public:
  static MPeerHood* GetInstance(CBasicCallback* aCallback);
  static void Invoke(CPHObject* aObject, const char* aMethodName);
  ~CPeerHoodImpl();
  bool Init(int aArgc, char** aArgv);

  TDeviceList* GetDeviceListL();
  TDeviceList* GetDeviceListL(const std::string* aServiceName);

  TServiceList* GetLocalServiceListL();

  MAbstractConnection* Connect(TDeviceIterator& aDevice, const std::string aServiceName);
  MAbstractConnection* Connect(MAbstractDevice& aDevice, const std::string& aServiceName);
  MAbstractConnection* Connect(TServiceIterator& aService);
	MAbstractConnection* Connect(CService& aService);

  unsigned short RegisterService(const std::string& aName, const std::string& aAttributes, const std::string& aPort);
  unsigned short RegisterService(const std::string& aName, const std::string& aAttributes);
  bool UnregisterService(const std::string& aName);
  bool UnregisterService(const std::string& aName, const std::string& aPort);


  void SetPreferedPlugin(const char* aPluginName);

  bool MonitorDevice(TDeviceIterator& aDevice);
  bool UnmonitorDevice(TDeviceIterator& aDevice);
  bool SignalMonitorDevice(TDeviceIterator& aDevice);
  bool SignalUnmonitorDevice();

 protected:
  CPeerHoodImpl(CBasicCallback* aCallback = NULL);
  CPeerHoodImpl(const CPeerHoodImpl&);
  const CPeerHoodImpl& operator= (const CPeerHoodImpl&);

 private:
  void LoadPlugins();
  bool Read(void* aBuffer, int aLength);
  bool Write(const void *aBuffer, int aLength);
  bool LoadParameter(std::string& aParameter);
  bool LoadParameter(char* aName, int* aParameter);

  MAbstractConnection* Connect(const std::string& aPrototype,
			       const std::string& aAddress,
			       const int aPort,
			       const int aPid,
			       const std::string& aServiceName,
			       const std::string& aDeviceName,
			       unsigned int aChecksum);

  static void* ThreadStarter(void* aArguments);
  static void* SignalThreadStarter(void* aArguments);
  static void* RoamingThreadStarter(void* aArguments);

  // bool StartRoamingThread(CThreadInfo* aThreadInfo);
  bool StartRoamingThread();
  bool StopRoamingThread();

  void PingerThread();
  void MonitorThread();
  // void RoamingThread(CThreadInfo* aThreadInfo);
  void RoamingThread();

  inline void Lock();
  inline void Unlock();
  
  bool CheckPidfile();

  static CPeerHoodImpl* iInstance;

  //File descriptor for the daemon socket
  int iDaemonSocket;

  // ProcessID of the process using the library
  int iPid;

  int iMonitoringInterval;
  bool iConnected;
  volatile bool iPinging, iMonitoring, iRoaming;
  MAbstractMonitor* iSignalMonitor;

  // Core elements of the whole library
  CEngine* iEngine;
  CBasicCallback* iCallback;

  // List of local services
  std::list<CService*> iServiceList;
  std::list<MAbstractPinger*> iPingerList;

  std::string iPreferedPlugin;
  pthread_t iThread;
  pthread_t iSignalThread;
  pthread_t iRoamingThread;

  u_int8_t iConnectionId;
  CVirtualConnection *iVirtualConnection;
  bool iCopying;
  std::list<CThreadInfo*> iThreadList;
  pthread_mutex_t iLock;

  friend class CVirtualConnection;
};


class CThreadInfo
{
 public:
  CPeerHoodImpl *iInstance;
  std::string iDeviceName;
  std::string iServiceName;
  std::string iPrototype;
  std::string iAddress;
  int iPort;
  int iConnectionId;
  unsigned int iChecksum;
  int iLowCount;
};

#endif // __PEERHOODIMPL_H__
