/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name GPRSCreator.cc
 * @memo GPRS implementation of the MAbstractCreator interface.
 *
 * @version 0.1
 * date     21.08.2003
 * change   14.11.2003
 */

#include <GPRSCreator.h>
#include <GPRSConnection.h>
#include <GPRSPinger.h>
#include <GPRSMonitor.h>
#include <Factory.h>

/**
 * @memo Default constructor.
 * @doc Default constructor, sets internal variables to their default values
 * and insert creator to the PeerHood factory.
 *
 * @return none
 */
CGPRSCreator::CGPRSCreator()
{
  iProto = std::string("gprs-base");
  Factory::Register(this);
}


/**
 * @memo Creates a new GPRS specific connection object.
 * @doc Creates a new GPRS specific connection object. The object is
 * created dynamically so the caller is responsible of deletion of the object.
 *
 * @param aProto Prototype of the desired object's type.
 *
 * @return new CWLAConnection object or NULL if the prototype was not recognized
 */
MAbstractConnection* CGPRSCreator::CreateConnectionL(const std::string& aProto)
{
  if (iProto.compare(aProto) == 0) {
    return new CGPRSConnection();
  }

  return NULL;
}


/**
 * @memo Creates a new GPRS specific pinger object.
 * @doc Creates a new GPRS specific pinger object. The object is
 * created dynamically so the caller is responsible of deletion of the object.
 *
 * @param aProto Prototype of the desired object's type.
 * @param aAddress Remote device's address.
 *
 * @return new CGPRSPinger object or NULL if the prototype was not recognized
 */
MAbstractPinger* CGPRSCreator::CreatePingerL(const std::string& aProto, const std::string& aAddress)
{
  if (iProto.compare(aProto) == 0) {
    return new CGPRSPinger(aAddress);
  }

  return NULL;
}

/**
 * @memo Creates a new Bluetooth specific pinger object.
 * @doc Creates a new Bluetooth specific pinger object. The object is
 * created dynamically so the caller is responsible of deletion of the object.
 *
 * @param aProto Prototype of the desired object's type.
 * @param aAddress Remote device's address.
 *
 * @return new CBTPinger object or NULL if the prototype was not recognized
 */
MAbstractMonitor* CGPRSCreator::CreateMonitorL(const std::string& aProto, const std::string& aAddress)
{
  if (iProto.compare(aProto) == 0) {
    return new CGPRSMonitor(aAddress);
  }

  return NULL;
}


/** 
 * @memo Returns creator's unique id string.
 * @doc Returns creator's unique id string. GPRS creator's id <i>gprs-base
 * </i> is the same as GPRS plugin's id.
 *
 * @return creator's unique id string
 */
const std::string& CGPRSCreator::GetPrototype()
{
  return iProto;
}


static CGPRSCreator gprscreator;
