/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name UDPConnection.cc
 * @memo UDP implementation
 *
 * @version 0.11
 * date     04.07.2003
 * change   28.04.2010
 */

#include <iostream>
#include <stdio.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <string.h>

#include "UDPConnection.h"
#include <assert.h>

#warning "Temporary debug"
//temp
#include <syslog.h>

#define ERR(format, msg...) syslog(LOG_ERR, "ERROR: " format "\n" , ## msg)

#ifdef PH_DEBUG
#define DBG(format, msg...) syslog(LOG_DEBUG, format "\n" , ## msg)
#else
#define DBG( A... )
#endif
//temp


/**
 * @memo Default constructor.
 * @doc Default constructor, sets the initial vatiables to their default values
 * and creates a new WLAN socket.
 *
 * @return none
 */
CUDPConnection::CUDPConnection(const std::string& aIface)
{
  iConnected = false;
  iIsListening = false;
  iIface = aIface;
  
  CIFSearch search;
  iIfaces = search.GetIFInfo(aIface);

  iSocket = socket(AF_INET, SOCK_DGRAM, 0);
  assert(iSocket != -1);
}

/**
 * @memo Default destructor.
 * @doc Default destructor, deletes allocated variables
 *
 * @return none
 */
CUDPConnection::~CUDPConnection()
{
}

/**
 * @memo Explicitly closes the socket.
 * @doc Explicitly closes the socket without any additional state checking,
 * i.e. this function will execute even if there's no existing connection.
 * This function is used by the PeerHood engine to ensure that no file
 * descriptors are leaked.
 *
 * @return none
 */
void CUDPConnection::Close()
{
  close(iSocket);
}

/**
 * @memo Sends data to the other end of the connection.
 * @doc Sends data to the other end of the connection. Note that this
 * function is not endianess-safe.
 *
 * @param aOutBuf Buffer containing the data that should be written.
 * @param aLength The number of bytes to be written.
 * @param aPort Destination port
 * @param aMode Mode for sending. This can be "bcast" for broadcasting
 * and "ucast" for unicasting
 *
 * @return the number of bytes written or -1 in the case of an error
 */
int CUDPConnection::Write(const void* aOutBuf, int aLength, int aPort, const std::string& aAddress, const std::string& aMode)
{
  int written = 0;
  sockaddr_in address;
  in_addr *addr = new struct in_addr;
  
  if (iIfaces == NULL)
   {
 	  ERR("CUDPConnection:Listen: interface no present or not connected");
 	  return -1;
   }
  
  if (aMode.compare("bcast") == 0)
    address = iIfaces->iBcast;
  else
    {
      address = iIfaces->iUcast;
      if(inet_aton(aAddress.c_str(), addr) == 0)
	{
	  perror("CUDPConnection::Write : inet_aton failed");
	  return -1;
	}
      address.sin_addr = *addr;
    }

  address.sin_port = htons(aPort);
  written = sendto(iSocket, (char *)aOutBuf, aLength, 0, (struct sockaddr *)&address, sizeof(address));

  delete addr;
  return written;
}

/**
 * @memo Reads data from an open connection.
 * @doc Reads data from an open connection. Just like 
 * <code>Read</code> this function is not endianess-safe so it should be
 * treated more like a standard Posix sequential socket.
 *
 * @param aInBuf Buffer where the data is read to.
 * @param aLength The number of bytes that should be read.
 *
 * @return the number of bytes read or -1 in the case of an error
 */
int CUDPConnection::Read(void* aInBuf, int aLength)
{
  int retval = 0;
  struct sockaddr_in cliaddr;
  socklen_t clilen = sizeof(cliaddr);
  
  retval = recvfrom(iSocket, (char *)aInBuf, aLength, 0, (struct sockaddr *)&cliaddr, &clilen);
  
  if(retval > 0)
    iRemoteAddress = inet_ntoa(cliaddr.sin_addr);
  
  return retval;
}

/**
 * @memo Returns connection's remote address.
 * @doc Returns connection's remote address. Note that the existence of the
 * address doesn't mean that there's a connection established.
 *
 * @return connection's remote address
 */
const std::string& CUDPConnection::GetRemoteAddress()
{  
  return iRemoteAddress;
}

/**
 * @memo Returns connection's file descriptor.
 * @doc Returns connection's file descriptor. Other classes can have a direct
 * access to the connection's socket (or similar entity) via this function's
 * return value.
 *
 * @return connection's file descriptor or -1 if one isn't available
 */
int CUDPConnection::GetFd()
{
  return iSocket;
}


/**
 * @memo Sets the connection to the listening state.
 * @doc Sets the connection to the listening state. This function must be
 * called before any incoming connections targeted to this object can be
 * accepted.
 *
 * @param aPort The port that should be listened
 * @param aMode Listen mode. This can be "bcast" broadcast
 * or "ucast" for unicast mode.
 *
 * @return true if successfull
 */
bool CUDPConnection::Listen(int aPort, const std::string& aMode)
{
  sockaddr_in address;

  if (iIfaces == NULL)
  {
	  ERR("CUDPConnection:Listen: interface no present or not connected");
	  return false;
  }
  
  memset(&address,0,sizeof(struct sockaddr_in));
  
  if(aMode.compare("bcast") == 0)
    address = iIfaces->iBcast;
  else 
    address = iIfaces->iUcast;

  address.sin_port = htons(aPort);

  if (bind(iSocket, (struct sockaddr *)&address, sizeof(address)) == -1) {
    ERR("CUDPConnection::Listen : bind failed");
    return false;
  }
  iIsListening = true;
 
  return true;
}


/**
 * @memo Tells if the connection is listening.
 * @doc Tells if the connection is in the listening state or not.
 *
 * @return true if the connection is in the listening state
 */
bool CUDPConnection::IsListening()
{
  return iIsListening;
}

/**
 * @memo Returns broadcast and unicast addresses
 * @doc Returns broadcast and unicast addresses
 *
 * @return pointer to struct IFInfo containing addresses
 */
struct IFInfo *CUDPConnection::GetInterfaces()
{
  CIFSearch search;
  return search.GetIFInfo(iIface);
}

/**
 * @memo Tells if the connection has data waiting
 * @doc Checks if the connection has data to read.
 *
 * @return true if the data is ready
 */
bool CUDPConnection::HasData()
{
	struct timeval timeout;
	timeout.tv_sec = 0;
	timeout.tv_usec = 0;
  fd_set set;
  FD_ZERO(&set);
  FD_SET(GetFd(), &set);

  switch (select(GetFd() + 1, &set, NULL, NULL, &timeout)) {
  case -1:
  	ERR("CUDPConnection::HasData(): Select error");
  	break;
    
  case 0:
    break;

  default:
    if (FD_ISSET(GetFd(), &set)) {
      return true;
    }
    else {
      return false;
    }
  }
	return false;
}

unsigned int CUDPConnection::GetDeviceChecksum()
{
	return 0;
}
