/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name DeviceStorage.cc
 * @memo Implementation of the CDeviceStorage class.
 *
 * @version 0.1
 * date     08.04.2003
 * change   08.04.2003
 */

#include <DeviceStorage.h>
#include <DaemonDevice.h>

#warning "Temporary debug"
//temp
#include <syslog.h>

#define ERR(format, msg...) syslog(LOG_ERR, "ERROR: " format "\n" , ## msg)

#ifdef PH_DEBUG
#define DBG(format, msg...) syslog(LOG_DEBUG, format "\n" , ## msg)
#else
#define DBG( A... )
#endif
//temp

CDeviceStorage* CDeviceStorage::iInstance = NULL;

/** 
 * @memo Getter to the singleton instance of the class.
 * @doc Getter to the singleton instance of the class. If an old instance 
 * exists then it is returned. Otherwise a new instance is created and
 * returned. This function follows the idea of the singleton design pattern.
 *
 * @return me, myself and I (e.i. pointer to the singleton instance of the class)
 */
CDeviceStorage* CDeviceStorage::GetInstance()
{
  if (!iInstance) {
    iInstance = new CDeviceStorage;
  }

  return iInstance;
}


/** 
 * @memo Default constructor, initilizes the mutex used for thread-safety.
 * values.
 * @doc Default constructor, initializes the internal mutex used for 
 * thread-safety.
 *
 * @return none
 */
CDeviceStorage::CDeviceStorage()
{
  pthread_mutex_init(&iLock, NULL);  
}


/** 
 * @memo Updates the list of devices.
 * @doc Updates the internal list of devices. First, every device whose id 
 * matches the given id is removed from the list. Then every device in the list
 * given as a parameter is added to the internal list using the copy 
 * constructor of the device class.
 *
 * @param aProto Prototype id of the devices to be updated.
 * @param aList List of the devices to be updated.
 *
 * @return none
 */
void CDeviceStorage::Update(const char* aProto, const std::list<CDaemonDevice *>& aList)
{
  Lock();

  // This list size means the list size of device storage
  //  DBG("CDeviceStorage::Update : list size is %d", iDeviceList.size());

  // Goes through device storage and removes devices ?
  for (std::list<CDaemonDevice*>::iterator i = iDeviceList.begin();i != iDeviceList.end();) {
    //    DBG("Comparing protos %s = %s", (*i)->GetPrototype().c_str(), aProto);
    if ((*i)->GetPrototype().compare(aProto) == 0) {
      DBG("   CDeviceStorage::Update : %s deleted", (*i)->GetAddress().c_str());
      delete *i;
      std::list<CDaemonDevice*>::iterator sentenced = i++;
      iDeviceList.erase(sentenced);
    }
    else {
      i++;
    }
  }

  //  DBG("CDeviceStorage::Update : new devices: %d", aList.size());

  for (std::list<CDaemonDevice *>::const_iterator i = aList.begin();i != aList.end();++i) {
    DBG("CDeviceStorage::Update : Adding %s", (*i)->GetAddress().c_str());
    iDeviceList.push_back(new CDaemonDevice(*(*i)));
  }

  //  DBG("CDeviceStorage::Update : list size (end) is %d", iDeviceList.size());
  Unlock();
  //  DBG("CDeviceStorage::Update : update complete");
}


/** 
 * @memo Locks the internal thread-safety mutex.
 * @doc Locks the internal thread-safety mutex.
 *
 * @return none
 */
void CDeviceStorage::Lock()
{
  pthread_mutex_lock(&iLock);
}


/** 
 * @memo Unlocks the internal thread-safety mutex.
 * @doc Unlocks the internal thread-safety mutex.
 *
 * @return none
 */
void CDeviceStorage::Unlock()
{
  pthread_mutex_unlock(&iLock);
}


/**
 * @memo Returns an iterator to the first device.
 * @doc Returns an iterator to the first device. <b>Database's internal lock is
 * set within this function so the <code>ReleaseDeviceLock</code> must be
 * called also!</b>
 *
 * @return iterator pointing to the first device in the database
 */
TDEVICE_ITERATOR CDeviceStorage::FirstDevice()
{
  Lock();

  return iDeviceList.begin();
}


/**
 * @memo Returns an iterator to the last device.
 * @doc Returns an iterator to the last device.
 *
 * @return iterator pointing to the last device in the database
 */
TDEVICE_ITERATOR CDeviceStorage::LastDevice()
{
  return iDeviceList.end();
}


/**
 * @memo Releases database's internal lock.
 * @doc Releases database's internal lock. After this call is completed, new
 * commits to the database can be made.
 *
 * @return none
 */
void CDeviceStorage::ReleaseDeviceList()
{
  Unlock();
}


/**
 * @memo Returns a list containing all devices in the database.
 * @doc Returns a list containing all devices in the database. <b>Database's
 * internal lock is set within this function so the <code>ReleaseDeviceList
 * </code> must be called when the list is  no longer used!</b>
 *
 * @return list containing all devices in the database
 */
const TDEVICE_LIST CDeviceStorage::GetDeviceList()
{
  Lock();

  return iDeviceList;
}
