/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name BTPlugin.cc
 * @memo Bluetooth implementation of PeerHood's plugin interface.
 *
 * @version 0.1
 * date     08.04.2003
 * change   5.5.2004
 */

#include <sys/ioctl.h>
#include <cstdlib>
#include <ctime>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <fstream>
#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>
#include <bluetooth/l2cap.h>
#include <iostream>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "DeviceStorage.h"
#include "BTPlugin.h"
#include "Daemon.h"
#include "ListenerFactory.h"
#include "AbstractStateConverter.h"
#include "PluginStorage.h"

#define EXISTING_SOCKET 17
#define NO_SOCKET_CREATED 18
#define SOCKET_CREATED 19
#define RESET_SOCK_FAIL 20
#define RESET_DOWN_FAIL 21
#define RESET_UP_FAIL 22
#define RESET_OK 23

#warning check the sdp part naming conventions!
struct TSearchContext {
  char *svc;
  uuid_t group;
  uint32_t handle;
};

const char* PLUGIN_NAME = "bt-base";
/** @TODO Add the device into config */
const char* PLUGIN_DEVICE = "hci0";
const int PEERHOOD_SVCLASS_ID = 0x7744;
const int PEERHOOD_PROFILE_ID = 0x4377;

//counter for keeping devices on device-list
const int KEEP_ALIVE_STAMP = 10;

const int MINIMUM_INQUIRY_DELAY = 10;
const float INQUIRY_VARIATION = 10.0;

//default port for listening incoming service requests
const int BT_PLUGIN_PSM = 21000;


/** 
 * @memo Default constructor, sets the internal variables to their default
 * values.
 * @doc Default constructor, sets the internal variables to their default
 * values.
 *
 * @return none

 */
CBTPlugin::CBTPlugin()
{
  iStarted = false;
  iActive = true;
  iNeedsRestart = false;
  iResetRounds = 0;
  iInSdp = false;
  iListening = false;
  iName = std::string(PLUGIN_NAME);
  iNDevice = std::string(PLUGIN_DEVICE);
#ifdef __PH_OLD_PLUGIN_STYLE__
	pluginMap[PLUGIN_NAME] = this;
#else
	PluginStorage::GetInstance()->AddPlugin(PLUGIN_NAME, this);
#endif
  srand(time(NULL));
  
  // The device id - check from somewhere?
  iDeviceId = 0;
  
  // Set up the socket to be -1 when creating plugin
  iBTCtrlSock = -1;
  
  // Create a low level socket for bluetooth
  switch(CreateLowlevelSocket())
	{
		case NO_SOCKET_CREATED:
			DBG("CBTPlugin::CBTPlugin: Can't open HCI socket for BT ioctl (socket: %d).", iBTCtrlSock);
			iBTCtrlSock = -1;
			break;
		case SOCKET_CREATED:
			DBG("CBTPlugin::CBTPlugin: Created HCI socket for BT ioctl");
			break;
		case EXISTING_SOCKET:
			DBG("CBTPlugin::CBTPlugin: Existing socket? This should not ever happen!!!");
			break;
		default:
			break;
	}
}

CBTPlugin::~CBTPlugin()
{
	RemoveListeners();
	if(iBTCtrlSock > 0) close(iBTCtrlSock); // If the BT socket still open
	DBG("CBTPlugin::Destructor");
}
/** 
 * @memo Starts the plugin's inquiry routine.
 * @doc Starts the plugin's inquiry routine. After this method is called the
 * plugin starts to monitor its environment and tries to locate nearby 
 * PeerHood-capable devices. Whenever a change in the neighborhood status is
 * detected the plugin will update the device & service database . This plugin
 * uses Bluetooth's device discovery routine and also Service Discovery
 * Protocol to locate PeerHood-capable devices in the neighborhood.
 * This method loads monitoring and service checking interval variables from
 * configuration file.
 *
 * @return boolean telling if the plugin could be started
 */
bool CBTPlugin::Start()
{
	if(iStarted)
	{
		ERR("CBTPlugin::Start : inquiry thread already started.");
		return false;
	}
	
  iStarted = true;

  //  DBG("BTPlugin::Start() : before thread");

  if (pthread_create(&iInquiryThread, NULL, *CBTPlugin::ThreadStarter, this) != 0) {
    ERR("CBTPlugin::Start :failed to create a thread");
    iStarted = false;
    return false;
  }

  //  DBG("BTPlugin::Start() : after thread");

  std::string stringDelay = "BTMonitoringInterval";
  if(!LoadParameter(stringDelay))
    {
      ERR("CBTPlugin::Start : Can't load parameters, using defaults");
      iInquiryInterval = 10;
    } 
  else
    {
      iInquiryInterval = atoi(stringDelay.c_str());
    }

  stringDelay = "BTServiceCheckInterval";
  if(!LoadParameter(stringDelay))
    {
      ERR("CBTPlugin::Start : Can't load parameters, using defaults");
      iServiceCheckInterval = 5;
    } 
  else 
    {
      iServiceCheckInterval = atoi(stringDelay.c_str());
    }

  stringDelay = "BTDaemonPort";
  if (!LoadParameter(stringDelay))
    {
      ERR("CBTPlugin::Start : Can't load parameter BTDaemonPort, using default");
      iDaemonPort = BT_PLUGIN_PSM;
    } 
  else
    {
      iDaemonPort = atoi(stringDelay.c_str());
    }
  return true;
}

/**
 * @memo Loads a parameter from the configuration file.
 * @doc Loads a parameter from the configuration file. If the requested
 * parameter is found then the argument <code>aParameter</code> is replaced
 * with parameter's value.
 *
 * @param aParameter Parameter's name.
 *
 * @return true if the requested parameter was found
 */
bool CBTPlugin::LoadParameter(std::string& aParameter)
{
  char* home;
  std::string line;
  std::string path;

  if ((home = getenv("HOME")) == NULL) {
  	path = "/etc/default/phconfig";
    //return false;
  }
  else
  {
  	path = std::string(home);
  	path.append("/.phconfig");
  }
  
  struct stat st;
  
	if(stat(path.c_str(),&st) == -1)
  {
  	path.clear();
  	path = std::string("/etc/default/phconfig");
  }
  
  std::ifstream in(path.c_str());

  if (!in.is_open()) {
    ERR("Failed to open the configuration file");
    in.close();
    return false;
  }
  
  while (!in.eof()) {
    getline(in, line);
    if (line[0] == '#') continue;
    std::string::size_type position = line.find_first_of(':');
    
    if (position != std::string::npos) {
      std::string parameter = line.substr(0, position);
      if (parameter.compare(aParameter) == 0) {
        aParameter = line.substr(++position);
        in.close();
        return true;
      }
    }
  }

  in.close();
  return false;
}

/** 
 * @memo Stops the plugin's inquiry routine.
 * @doc Stops the plugin's inquiry routine. After this call is handled the 
 * plugin will stop monitoring the environment. This means that daemon's 
 * device & service database will be deprecated as the time passes.
 *
 * @return none
 */
void CBTPlugin::Stop()
{
  if (!iStarted) return;

  iStarted = false;

  if (pthread_join(iInquiryThread, NULL) != 0) {
    ERR("CBTPlugin::Stop : failed to join the inquiring thread");
  }
  
}


/** 
 * @memo Returns plugin's unique id string.
 * @doc Returns plugin's unique id string. The id of this plugin is 'bt-base'.
 *
 * @return plugin's unique id string ('bt-base')
 */
const std::string& CBTPlugin::GetName()
{
  return iName;
}

void CBTPlugin::UpdateState()
{
	// Check states of the listeners
	std::list<MAbstractListener*>::iterator iter;
	for(iter = iListenerList.begin(); iter != iListenerList.end(); ++iter)
	{
		(*iter)->CheckState();
	}
}

void CBTPlugin::RegisterListener(MAbstractListener* aListener)
{
	if(aListener != NULL)
	{
		iListenerList.push_back(aListener);
		DBG("CBTPlugin::RegisterListener: new listener %s.", aListener->GetName().c_str());
	}
}

void CBTPlugin::SetState(bool aActive)
{
	if(iActive != aActive)
	{
		iActive = aActive;
		DBG("CBTPlugin::SetState: new state: %s", aActive ? "active" : "passive");
	}
	else DBG("CBTPlugin::SetState: No change in activity state.");
	
	if(aActive == true && iInSdp == false && iStarted == true)
	{
		if(Advert()) DBG("CBTPlugin::SetState: active mode, adverting restarted");
		else DBG("CBTPlugin::SetState: active mode, cannot restart adverting");
	}
}

void CBTPlugin::TriggerShutdown()
{
	DBG("CBTPlugin::TriggerShutdown");
}

void CBTPlugin::LoadListeners()
{
	int listeners = 0;
	std::string listenerName = "bt";
	listeners = ListenerFactory::GetInstance()->CreateListeners(listenerName, (MAbstractStateConverter*)this);
	if(listeners == 0) DBG("CBTPlugin: No listeners for type \"%s\" found.", listenerName.c_str());

	// Connect listeners and check the initial state of the system, go through all listeners
	std::list<MAbstractListener*>::iterator iter;
	for(iter = iListenerList.begin(); iter != iListenerList.end(); ++iter)
	{
		if((*iter)->Connect())
		{
			(*iter)->CheckInitialState();
		}
	}
	DBG("CBTPlugin: Listeners started");
}
/** 
 * @memo Starts plugin's main (inquiry) thread.
 * @doc Starts plugin's main (inquiry) thread. This dummy function is required
 * because C++ doesn't allow non-static member functions to be started as a
 * thread. This function just calls plugin instance's real inquiry function.
 *
 * @param aArguments Pointer to the currently running instance.
 *
 * @return always NULL
 */
void *CBTPlugin::ThreadStarter(void *aArguments)
{
  CBTPlugin *fake = (CBTPlugin *)aArguments;
  fake->InquiryThread();
  return NULL;
}


/** 
 * @memo Starts plugin's advertising thread. 
 * @doc Starts plugin's advertising thread. This dummy function is required
 * because C++ doesn't allow non-static member functions to be started as a
 * thread. This function just calls plugin instance's real advertising
 * function.
 *
 * @param aArguments Pointer to the currently running instance.
 *
 * @return always NULL
 */
void *CBTPlugin::AdvertStarter(void *aArguments)
{
  CBTPlugin *fake = (CBTPlugin *)aArguments;
  fake->AdvertThread();
  return NULL;
}


/** 
 * @memo Plugin's main thread that performs BT inquiry at certain intervals.
 * @doc Plugin's main thread that performs BT inquiry at certain intervals. All
 * found devices are stored to the DeviceStorage shared with all plugins and
 * the PeerHood Daemon. This plugin declares a device to be present if it's 
 * found during inquiry. Otherwise the device is decided to be
 * absent.
 *
 * @return none
 */
void CBTPlugin::InquiryThread()
{
	int responses = 0, errors = 0;
	bool needsUpdate = false;
	inquiry_info* info = NULL;
	bdaddr_t address;
	char *str = NULL;
	//int dd;
	std::list<CDaemonDevice *>::iterator i;

	//dd = hci_open_dev(0);
	
	// If the restart is needed
	if(iNeedsRestart)
	{
		sleep(5); // Wait for a while
		
		// Try a reset 
		switch (ResetAdapter())
		{
			// Ok, no more restarting
			case RESET_OK:
				iNeedsRestart = false;
				break;
			// No socket was found and cannot create new one
			case RESET_SOCK_FAIL:
				iNeedsRestart = true;
				sleep(1);
				break;
			// Cannot put the bastard to sleep
			case RESET_DOWN_FAIL:
				iNeedsRestart = true;
				sleep(1);
				break;
			// Cannot bring up
			// TODO should this be tried later if now there is no success?
			case RESET_UP_FAIL:
				iNeedsRestart = false;
				break;
			default:
				break;
		}
		iResetRounds++;
	}
	
	// Reached the level of 100 restarts
	// TODO define a limit
	if(iResetRounds >= 100)
	{
		DBG("CBTPlugin::InquiryThread: Tried to reset the adapter %d times, inquiry thread stopping.",iResetRounds);
		return;
	}

	while(!iActive)
	{
		/** If the plugin was started but not active and while passive the plugin was set to
		 * shut down. Completely quit from loop. */
		if(!iStarted) return;
		else sleep(1);
	}
	
	DBG("CBTPlugin::InquiryThread: start");
	while (iStarted && iActive && !iNeedsRestart) {
		
		DBG("***MAIN INQUIRY LOOP START***");
		needsUpdate = false;

		/** TODO: Previously opened new connection to device at the beginning of the
		 * inquiryloop, handle wasn't used in any way -> removed / commented out. 
		 * hci_inquiry opens a handle to device when performing inquiry. **/
		/*if (dd < 0) {
    	dd = hci_open_dev(0);
    	if (dd < 0) {
    		perror("HCI device open failed");
    	}
    }*/

		/** TODO: There exists a feature either in bluetooth bluez-stack or in the
		 * hardwares of different devices which causes an error after several 
		 * hours of active inquiring. hci_inquiry reports only -1 in error situations
		 * and sets an error variable. This error condition should be somehow checked.**/
		
		// If there is no socket created to Bluetooth device
		if(iBTCtrlSock == -1)	responses = hci_inquiry(iDeviceId, 15, 20, NULL, &info, IREQ_CACHE_FLUSH);
		// Perform inquiry with own function
		else responses = DoInquiry(15, 20, NULL, &info, IREQ_CACHE_FLUSH);

		DBG("CBTPlugin::InquiryThread: Got %d responses", responses);

		if(responses < 0){
			errors++;
			ERR("BTPlugin::InquiryThread : HCI inquiry failed (%d)", errors);
			
			if(errno != 0)
			{
				perror("BTPlugin::InquiryThread ");
				DBG("CBTPlugin::Inquirythread : inquiry failure reason: %s.", strerror(errno));
			}
			
			// TODO specify different errors based on errno!
			switch(errno)
			{
				case 0:
					break;
				default:
					break;
			}
			
			// TODO define limit for error
			if(errors >= 10) iNeedsRestart = true;
			
		}
		else {
			for (int j = 0;j < responses;j++) {
				if (!iStarted || !iActive){
					break;
				}

				memset(&address, 0, sizeof(address));
				baswap(&address, &(info + j)->bdaddr);
				str = batostr(&address);

				// AddDevice adds device if new, otherwise ...
				if (AddDevice(str)) {
					needsUpdate = true;
				}

				delete str;
			}
		}	


		// After going through all responses see if need for an update
		if(MakeOlder() || (needsUpdate)) {
			// This is important phase as plugin based device list is moved to device storage
			DBG("CBTPlugin::InquiryThread: Updating Device Storage");
			CDeviceStorage::GetInstance()->Update(PLUGIN_NAME, iDeviceList);
			DBG("CBTPlugin::InquiryThread: Update complete");
		}
		else {
			DBG("CBTPlugin::InquiryThread: No need to update Device Storage");
		}

		free(info);
		info = NULL;

		DBG("***MAIN INQUIRY LOOP END***");

		for (int k = 0;k < iInquiryInterval;k++) {
			sleep(1);
			if (!iStarted || !iActive || iNeedsRestart) {
				break;
			}
		}
	}
	
	// If the plugin is set to passive but it should be running - restart
	if((!iActive && iStarted) || (iNeedsRestart))
	{
		DBG("CBTPlugin::InquiryThread: restarting..");
		InquiryThread();
	}
}

/** 
 * @memo Checks if the found device is new (i.e. not detected before).
 * @doc Checks if the found device is new (i.e. not detected before). If this
 * is the case then it is added to plugin's internal device list and the common
 * device & service database should be updated later on.
 * There two kinds of counters for device: timestamp is used for to mark "old" devices
 * and lastchecked is for performing service checking on specific interval.
 * Timestamp is zerod every time when device is found. If timestamp reaches specific value
 * device is erased. If device is seen, lastchecked value is decreased.  If lastchecked
 * reaches zero, device services are checked.
 * 
 * @param aDeviceName Name of the device to be added.
 *
 * @return true if the checked device is new
 */
bool CBTPlugin::AddDevice(const std::string& aDeviceName)
{
  CDaemonDevice* temp;
  std::list<CDaemonDevice *>::iterator i;
  std::list<CServiceCheck>::iterator j;
  bool found = false;
  int psm = BT_PLUGIN_PSM;

  for (i = iDeviceList.begin();i != iDeviceList.end();++i) {

    // This if handles existing devices
    if ((*i)->GetAddress().compare(aDeviceName) == 0){

      DBG("BTPlugin::AddDevice: Existing device %s", aDeviceName.c_str());

      (*i)->SetTimestamp(0);

      // Check the services in this device
      for (j = iServiceCheckList.begin();j != iServiceCheckList.end();++j) {

	if(aDeviceName.compare(j->iDeviceName) == 0) {

	  if(j->iLastChecked > iServiceCheckInterval){
	    j->iLastChecked = iServiceCheckInterval;
	  }

	  j->iLastChecked--;

	  // In case we have need to check the services
	  if(j->iLastChecked <= 0) {
	    j->iLastChecked = iServiceCheckInterval;

	    if ((found = (*i)->HasPeerHood()) != 0) {
	      // This should be set so that all information is retrieved
	      // If already peerhood then update only part
	    }
	    else {
	      // Here we could select what we want
	      // This means that get everything
	    }

	    found = HasPeerHood(aDeviceName, &psm);
	    (*i)->SetPeerHood(found);
	    
	    if (found) {
	      // Get all information required from the new device at once
	      if (!FetchInformation((*i), psm)) {
		DBG("CBTPlugin::AddDevice: Information not fetched correctly");      
		j->iLastChecked = 0;
		return false;
	      }
	      else {
		DBG("CBTPlugin::AddDevice: Information fetched correctly");
	      }
	    }
	    else {
	      DBG("CBTPlugin::AddDevice: Not a PeerHood capable device");
	    }
	    return true;
	  }
	  return false;
	}
      }
    }
  }

  // New device found, add that into lists

  DBG("BTPlugin::AddDevice: New device %s", aDeviceName.c_str());

  // Creates an empty device with only Devicename (=address) set
  temp = new CDaemonDevice(aDeviceName);
  temp->SetPrototype(PLUGIN_NAME);

  found = false;
  found = HasPeerHood(aDeviceName, &psm);

  temp->SetPeerHood(found);
  temp->SetTimestamp(0);
  
  if(found) {
    // Get all information required from the new device at once
    if (!FetchInformation(temp, psm)) {
      DBG("CBTPlugin::AddDevice: Information not fetched correctly");      
  
      delete temp;
      return false;
    }
    else {
      DBG("CBTPlugin::AddDevice: Information fetched correctly");
    }
  }
  else {
    DBG("CBTPlugin::AddDevice: Not a PeerHood capable device");
  }
  
  //  temp->DebugDevice();

  iDeviceList.push_back(temp);

  CServiceCheck tempCheck;
  tempCheck.iDeviceName = aDeviceName;
  tempCheck.iLastChecked = iServiceCheckInterval;
  iServiceCheckList.push_back(tempCheck);

  return true;
}


/** 
 * @memo Increases each device's timestamp by one and checks if some of them
 * exceeds the limit.
 * @doc Increases each device's timestamp value by one and checks if some of
 * them exceeds the limit defined by KEEP_ALIVE_STAMP. In that case the "old"
 * device is removed from the internal device list so the common device & service
 * database should be updated.
 *
 * @return true if some device was removed from the internal device list
 */
bool CBTPlugin::MakeOlder()
{
  bool retval = false;

  //  DBG("BTPlugin::MakeOlder: list size (begin) is %d", iDeviceList.size());

  for (std::list<CDaemonDevice *>::iterator i = iDeviceList.begin();i != iDeviceList.end();) {

    //     DBG("BTPlugin::MakeOlder: %s timestamp %d", (*i)->GetAddress().c_str(), (*i)->GetTimestamp());

    if ((*i)->GetTimestamp() > KEEP_ALIVE_STAMP){
      for (std::list<CServiceCheck>::iterator j = iServiceCheckList.begin();j != iServiceCheckList.end();++j) {
	if((*i)->GetAddress().compare(j->iDeviceName) == 0) {
	  iServiceCheckList.erase(j);
	  break;
	}
      }

      std::list<CDaemonDevice *>::iterator sentenced = i++;
      delete *sentenced;
      iDeviceList.erase(sentenced);
      retval = true;

    }
    else {
      (*i)->IncreaseTimestamp();

      // DBG("Timestamp increased to : %d", (*i)->GetTimestamp());

      i++;
    }
  }

  //  DBG("BTPlugin::MakeOlder: list size (end) is %d", iDeviceList.size());

  return retval;
}


/** 
 * @memo Inserts the PeerHood tag to the local SDP database.
 * @doc Inserts the PeerHood tag to the local SDP database. Other Bluetooth-
 * enabled PeerHood devices will look for this tag when they try to locate
 * the surrounding PeerHood devices. If this function fails other devices are
 * not capable of detecting PeerHood on this device.
 *
 * @return true if the advertising was started successfully
 */
bool CBTPlugin::Advert()
{
  sdp_record_t* record;
  sdp_list_t* root;
  sdp_list_t* serviceClassId;
  sdp_list_t* profileSeq;
  sdp_list_t* apSeq;
  sdp_list_t* protocol;
  sdp_list_t* aProtocol;
  sdp_profile_desc_t profile;
  uuid_t rootUuid;
  uuid_t ftrnUuid;
  uuid_t l2capUuid;
  sdp_data_t* psm;
  uint16_t localPort = iDaemonPort;

  //  uint16_t localPort = 22;

  // DBG("inserting PeerHood tag to the SDP database");
  
  if (iInSdp) {
    ERR("already in SDP");
    return false;
  }
  
  if ((iSDP = sdp_connect(BDADDR_ANY, BDADDR_LOCAL, 0)) == 0) {
    ERR("couldn't connect to the SDP daemon");
    return false;
  }
  
  if ((record = (sdp_record_t *)malloc(sizeof(sdp_record_t))) == NULL) {
    ERR("malloc");
    sdp_close(iSDP);
    return false;
  }

  bzero(record, sizeof(*record));
  record->handle = 0xffffffff;
  sdp_uuid16_create(&rootUuid, PUBLIC_BROWSE_GROUP);
  root = sdp_list_append(0, &rootUuid);
  sdp_set_browse_groups(record, root);
  
  // service class
  sdp_uuid16_create(&ftrnUuid, PEERHOOD_SVCLASS_ID);
  serviceClassId = sdp_list_append(0, &ftrnUuid);
  sdp_set_service_classes(record, serviceClassId);

  // profile
  sdp_uuid16_create(&profile.uuid, PEERHOOD_PROFILE_ID);
  profile.version = 0x0100;
  profileSeq = sdp_list_append(0, &profile);
  sdp_set_profile_descs(record, profileSeq);
  

  // protocol
  sdp_uuid16_create(&l2capUuid, L2CAP_UUID);
  protocol = sdp_list_append(0, &l2capUuid);
  psm = sdp_data_alloc(SDP_UINT16, &localPort);
  protocol = sdp_list_append(protocol, psm);
  apSeq = sdp_list_append(0, protocol);

  aProtocol = sdp_list_append(0, apSeq);
  sdp_set_access_protos(record, aProtocol);

  sdp_data_free(psm);
  sdp_list_free(protocol, 0);
  sdp_list_free(apSeq, 0);
  sdp_list_free(aProtocol, 0);
  sdp_list_free(root, 0);
  sdp_list_free(profileSeq, 0);
  sdp_list_free(serviceClassId, 0);
  sdp_set_info_attr(record, "VPeerHood", 0, 0);

  if (sdp_record_register(iSDP, record, SDP_RECORD_PERSIST) == -1) {
    ERR("registration to the SDP failed");
    sdp_record_free(record);
    sdp_close(iSDP);
    return false;
  }

  iSDPHandle = record->handle;
  sdp_record_free(record);

  iInSdp = true;

  //  DBG("PeerHood is now in the SDP database, handle 0x%x", iSDPHandle);

  if (pthread_create(&iAdvertThread, NULL, *CBTPlugin::AdvertStarter, this) != 0) {
    ERR("failed to create the advertising thread");
    Unadvert();
    return false;
  }
  
  //  DBG("advertising thread started");

  return true;
}


/** 
 * @memo Removes the PeerHood tag from the SDP database.
 * @doc Removes the PeerHood tag from the local SDP database. After this
 * function is invoked, other devices are not able to detect the existence of
 * the PeerHood service on this machine.
 *
 * @return true if the advertising was ended succesfully
 */
bool CBTPlugin::Unadvert()
{
  uint32_t range = 0x0000ffff;
  sdp_list_t *attributes;
  sdp_record_t *record;  

  //  DBG("removing PeerHood from the SDP database");

  if (!iInSdp) {
    ERR("not in sdp at all?!?");
    return false;
  }

  iInSdp = false;

  if (pthread_join(iAdvertThread, NULL) != 0) {
    ERR("failed to join the advertising thread");
  }

  attributes = sdp_list_append(0, &range);

  record = sdp_service_attr_req(iSDP, iSDPHandle, SDP_ATTR_REQ_RANGE, attributes);

  sdp_list_free(attributes, 0);

  if (!record) {
    ERR("PeerHood not found in the SDP database");
    return false;
  }

  if (sdp_record_unregister(iSDP, record)) {
    ERR("Removing PeerHood failed");
    return false;
  }

  sdp_close(iSDP);

  return true;
}


/** 
 * @memo Checks if the PeerHood tag can be found from remote device's SDP
 * database.
 * @doc Checks if the PeerHood tag can be found from remote device's SDP
 * database. If the tag if found then the remote device is assumed to be
 * PeerHood-capable without any further checking.
 *
 * @param aAddress The address of the remote device.
 * @param aPSM On return contains the PSM of the remote address if PeerHood is 
 * found, otherwise -1.%s", aActive ? "active" : "passive
 *
 * @return true if the PeerHood tag was found on remote device's SDP database
 */
bool CBTPlugin::HasPeerHood(const std::string& aAddress, int* aPSM)
{
  sdp_list_t *attrid, *search, *seq, *next, *proto;
  struct TSearchContext context;
  bdaddr_t bdaddr;
  sdp_session_t* session = NULL;
  uint32_t range = 0x0000ffff;

  // DBG("CBTPlugin::HasPeerHood : Performing SDP query on %s ", aAddress.c_str());
  
  bdaddr_t *tempBdaddr;
  tempBdaddr = strtoba(const_cast<char*>(aAddress.c_str()));
  baswap(&bdaddr, tempBdaddr);
  free(tempBdaddr);

  bzero(&context, sizeof(context));
  sdp_uuid16_create(&context.group, PEERHOOD_SVCLASS_ID);
  *aPSM = -1;

  if (!(session = sdp_connect(BDADDR_ANY, &bdaddr, 0))) {
    ERR("CBTPlugin::HasPeerHood : Device %s has no SDP", aAddress.c_str());
    if(errno != 0) DBG("CBTPlugin::HasPeerHood : Cannot connect to SDP of device %s, reason: %s",aAddress.c_str(), strerror(errno));
    free(context.svc);
    return false;
  }

  attrid = sdp_list_append(0, &range);
  search = sdp_list_append(0, &context.group);

  if (sdp_service_search_attr_req(session, search, SDP_ATTR_REQ_RANGE, attrid, &seq)) {
    ERR("CBTPlugin::HasPeerHood : SDP query failed");
    if(errno != 0) DBG("CBTPlugin::HasPeerHood : SDP query failed, reason: %s",strerror(errno));
    sdp_close(session);
    free(context.svc);
    return false;
  }

  sdp_list_free(attrid, 0);
  sdp_list_free(search, 0);

  for (;seq;seq = next) {
    sdp_record_t *rec = (sdp_record_t *)seq->data;
    sdp_record_print(rec);

    sdp_get_access_protos(rec, &proto);
    *aPSM = ((sdp_data_t*)((sdp_list_t*)proto->data)->data)->next->val.uint16;
    sdp_list_free(proto, free);
    
    next = seq->next;
    free(seq);
    free(context.svc);

    sdp_record_free(rec);
    sdp_close(session);
    free(context.svc);

    //   DBG("CBTPlugin::HasPeerHood : PeerHood found");

    return true;
  }

  sdp_close(session);  
  free(context.svc);

  //  DBG("CBTPlugin::HasPeerHood : No PeerHood in SDP");
  
  return false;
}

/** 
 * @memo Fetch the required information from other device
 * @doc 
 *
 * @return true if all information is successfully fetched
 */

bool CBTPlugin::FetchInformation(CDaemonDevice* aDevice, int aPSM)
{
  CBTConnection* connection = NULL;

  connection = new CBTConnection;

  // Here we need to set  flags for needed information
  int requestType = CDaemon::GetInstance()->GetFetchRequestType();

  DBG("--**-- ");

  // Create connection to the remote device
  if (!connection->Connect(aDevice->GetAddress(), aPSM)) {
    if(errno == 0) DBG("CBTPlugin::FetchInformation : failed to connect");
   	else DBG("CBTPlugin::FetchInformation : failed to connect, reason: %s",strerror(errno));
    delete connection;
    return false;
  }

  // Write the requesttype
  DBG("Requesting %d from %s", requestType, aDevice->GetAddress().c_str());

  requestType = htonl(requestType);

  if (connection->Write(&requestType, sizeof(requestType)) == -1) {
    ERR("CBTPlugin::FetchDeviceInfo : failed to write request type");
    connection->Disconnect();
    delete connection;
    return false;
  }
 
  requestType = ntohl(requestType);

  // Check here what information will be gathered (according to the requestType)
  // Take care of disconnects and deletes in here

  // Get the basic info
  if (requestType & D_GET_DEVICEINFO) {
    DBG("Requesting: device info");
    if(!FetchDeviceInfo(aDevice, connection)) {
      ERR("CBTPlugin::FetchDeviceInfo: Info not received correctly");
      return false;
    }
  }

  // Get services
  if (requestType & D_GET_SERVICEINFO) {
    DBG("Requesting: services");
    if(!FetchServices(aDevice, connection)) {    
      ERR("CBTPlugin::FetchServices: Info not received correctly");
      return false;
    }
  }

  // Get supported prototypes
  if (requestType & D_GET_PROTOINFO) {
    DBG("Requesting: prototypes");
    if(!FetchPrototypes(aDevice, connection)){    
      ERR("CBTPlugin::FetchPrototypes: Info not received correctly");
      return false;
    }
  }

  // Get neigborhoodinfo
  if (requestType & D_GET_NEIGHINFO) {
    DBG("Requesting: neighborhood");
    if (!FetchNeighbourhoodDevices(aDevice, connection)) {    
      ERR("CBTPlugin::FetchNeighborhoodDevices: Info not received correctly");
      return false;
    }
  }

  DBG("--**-- ");

  connection->Disconnect();
  delete connection;
  return true;
}

/** 
 * @memo Fetch services from the remote device
 * @doc Fetch services from remote device and add them to the list of this daemon device
 *
 * @return true if all information is successfully fetched
 */

bool CBTPlugin::FetchDeviceInfo(CDaemonDevice* aDevice, CBTConnection* connection)
{
  fd_set testSet;
  struct timeval timeVal;

  unsigned int nameLength = 0;
  unsigned int checksum = 0;
  char* buffer = NULL;

 
  FD_ZERO(&testSet);
  FD_SET(connection->GetFd(), &testSet);
  
  timeVal.tv_sec = 1;
  timeVal.tv_usec = 0;
  
  // Wait for the information
  if(select(connection->GetFd() + 1, &testSet, NULL, NULL, &timeVal) ==-1){
    ERR("CBTPlugin::FetchDeviceInfo : select failed");
    connection->Disconnect();
    return false;
  }
  
  if (FD_ISSET(connection->GetFd(), &testSet)) {
    if (connection->Read(&nameLength, sizeof(nameLength)) == -1) {
      DBG("CBTPlugin::FetchDeviceInfo : failed to read the device name length");
      connection->Disconnect();
      return false;
    }
  
    nameLength = ntohl(nameLength);
     
    if (FD_ISSET(connection->GetFd(), &testSet)) { 
      buffer = new char[nameLength];

      if (connection->Read(buffer, nameLength) == -1) {
	ERR("CBTPlugin::FetchDeviceInfo : failed to read the device name");
	connection->Disconnect();
	delete[] buffer;
	return false;
      }
      
      //      DBG("Got name %s", buffer);
      std::string deviceName(buffer);
      delete[] buffer;
      aDevice->SetDeviceName(deviceName);
	  
      if (connection->Read(&checksum, sizeof(checksum)) == -1) {
	ERR("CBTPlugin::FetchDeviceInfo : failed to read the device name checksum");
	connection->Disconnect();
	return false;
      }
	  
      aDevice->SetChecksum(ntohl(checksum));

      DBG("CBTPlugin::FetchDeviceInfo: OK");
      DBG(" ");

      return true;
    }  
    else {
      ERR("CBTPlugin::FetchDeviceInfo : FD_ISSET failed");
      if(errno != 0) DBG("CBTPlugin::FetchDeviceInfo: FD_ISSET failed, reason: %s", strerror(errno));
    }
  }
  else {
    ERR("CBTPlugin::FetchDeviceInfo : FD_ISSET failed");
    if(errno != 0) DBG("CBTPlugin::FetchDeviceInfo: outer FD_ISSET failed, reason: %s", strerror(errno));
  }

  return false;
}

/** 
 * @memo Fecthes a list of services available on the given remote device.
 * @doc Fetches a list of services available on the given remote device. This 
 * is done by establishing a new connection to the remote device using address
 * information given in the parameter <code>aDevice</code> and invoking the
 * service list request on the remote device. Founded services are added to
 * the device.
 *
 * @param aDevice The device whose service list should be fetched.
 *
 * @return none
 */
bool CBTPlugin::FetchServices(CDaemonDevice* aDevice)
{
  DBG("FetchServices without connection .. we should not have this");
  return false;
}


/** 
 * @memo Fetch services from the remote device
 * @doc Fetch services from remote device and add them to the list of this daemon device
 *
 * @return true if all information is successfully fetched
 */

bool CBTPlugin::FetchServices(CDaemonDevice* aDevice, CBTConnection* connection)
{
	fd_set testSet;
	struct timeval timeVal;

	char* buffer = NULL;

	unsigned short numberOfServices = 0;
	CService* temp = NULL;
	unsigned int size = 0;

	FD_ZERO(&testSet);
	FD_SET(connection->GetFd(), &testSet);

	timeVal.tv_sec = 1;
	timeVal.tv_usec = 0;

	bool foundService = false;

	// Wait for the information
	if(select(connection->GetFd() + 1, &testSet, NULL, NULL, &timeVal) ==-1) {
		DBG("CBTPlugin::FetchServices : select failed");
		return false;
	}

	if (FD_ISSET(connection->GetFd(), &testSet))
	{
		numberOfServices = 0;

		if (connection->Read(&numberOfServices, sizeof(numberOfServices)) == -1)
		{
			DBG("CBTPlugin::FetchServices : failed to read the number of services");
			return false;
		}
		
		numberOfServices = ntohs(numberOfServices);
		
		DBG("CBTPlugin::FetchServices: Got %d services", numberOfServices);
		
		// TODO find out why in some cases on n810/n900 25701 or 25717 services is received
		// and create a better fix. This is just a quick fix attempting to prevent crash.
		if(numberOfServices > 1000)
		{
			DBG("CBTPlugin::FetchServices : Amount of services is too big, must be a bug.");
			return false;
		}

		for (int i = 0;i < numberOfServices;i++)
		{
			foundService = false;

			if (FD_ISSET(connection->GetFd(), &testSet))
			{ 
				if (connection->Read(&size, sizeof(size)) == -1)
				{
					ERR("CBTPlugin::FetchServices : failed to read the size of the service");
					return false;
				}
		
				size = ntohl(size);

				if (FD_ISSET(connection->GetFd(), &testSet))
				{ 
					buffer = new char[size];

					if (connection->Read(buffer, size) == -1)
					{
						ERR("CBTPlugin::FetchServices : failed to read service info");
						delete[] buffer;
						return false;
					}

					temp = new CService(buffer);

					// Check is the service is already registered	
					foundService = aDevice->CheckService(temp->GetPort());

					if (!foundService)
					{
						DBG("FetchServices: Found new service %s for %s", temp->GetName().c_str(), aDevice->GetAddress().c_str());
						aDevice->AddService(temp);
					}
					else
					{
						DBG("FetchServices: Found old service %s for %s", temp->GetName().c_str(), aDevice->GetAddress().c_str());
						delete temp;
					}

					delete[] buffer;
				}
				else
				{
					ERR("CBTPlugin::FetchServices : FD_ISSET failed");
					if(errno != 0) DBG("CBTPlugin::FetchServices: FD_ISSET failed, reason: %s", strerror(errno));
					return false;
				}
			}
		} // for
		
		DBG("CBTPlugin::FetchServices: OK");
		DBG("Number of services in device is %d", aDevice->GetServiceListSize());
		DBG(" ");

		return true;
	}
	
	else
	{
		ERR("CBTPlugin::FetchServices : FD_ISSET failed");
		if(errno != 0) DBG("CBTPlugin::FetchServices: FD_ISSET failed, reason: %s", strerror(errno));
	}

	return false;
}

/** 
 * @memo Fetch prototypes from the remote device
 * @doc Fetch prototypes from remote device and add them to the list of this daemon device
 *
 * @return true if all information is successfully fetched
 */

bool CBTPlugin::FetchPrototypes(CDaemonDevice* aDevice, CBTConnection* connection)
{
  fd_set testSet;
  struct timeval timeVal;

  unsigned short numberOfPlugins = 0;
  unsigned short size = 0;
  char* newProto = NULL;

  FD_ZERO(&testSet);
  FD_SET(connection->GetFd(), &testSet);
  
  timeVal.tv_sec = 1;
  timeVal.tv_usec = 0;

  bool foundProto = false;

  // Wait for the information
  if(select(connection->GetFd() + 1, &testSet, NULL, NULL, &timeVal) ==-1){
    DBG("CBTPlugin::FetchPrototypes : select failed");
    return false;
  }
  
  if (FD_ISSET(connection->GetFd(), &testSet)) {
    numberOfPlugins = 0;

    if (connection->Read(&numberOfPlugins, sizeof(numberOfPlugins)) == -1) {
      DBG("CBTPlugin::FetchPrototypes : failed to read the number of plugins");
      return false;
    }
	      
    numberOfPlugins = ntohs(numberOfPlugins);

    DBG("FetchPrototypes: Number of protos %d", numberOfPlugins);
	      
    for (int i = 0;i < numberOfPlugins;i++) {
      foundProto = false;

      if (FD_ISSET(connection->GetFd(), &testSet)) { 
	if (connection->Read(&size, sizeof(size)) == -1) {
	  ERR("CBTPlugin::FetchPlugins : failed to read the size of the plugins");
	  return false;
	}
		    
	size = ntohs(size);
	    
	if (FD_ISSET(connection->GetFd(), &testSet)) { 

	  newProto = new char[size];

	  if (connection->Read(newProto, size) == -1) {
	    ERR("CBTPlugin::FetchPrototypes : failed to read plugin info");
	    delete[] newProto;
	    return false;
	  }

	  // Check is the service is already registered	
	  foundProto = aDevice->CheckPrototype(newProto);

	  if (!foundProto) {
	    DBG("FetchPrototypes: Found new proto %s for %s", newProto, aDevice->GetAddress().c_str());
	    aDevice->AddPrototype(newProto);
	  }
	  else {
	    DBG("FetchPrototypes: Found old proto %s for %s", newProto, aDevice->GetAddress().c_str());
	    delete [] newProto;
	  }
	}
	else {
	  ERR("CBTPlugin::FetchPrototypes : FD_ISSET failed");
	  if(errno != 0) DBG("CBTPlugin::FetchDeviceInfo: FD_ISSET failed, reason: %s", strerror(errno));
	  return false;
	}
      }
      else {
	ERR("CBTPlugin::FetchPrototypes : FD_ISSET failed");
	if(errno != 0) DBG("CBTPlugin::FetchDeviceInfo: FD_ISSET failed, reason: %s", strerror(errno));
	return false;
      }
    }

    DBG("CBTPlugin::FetchPrototypes: OK");
    DBG("Number of fetched prototypes %d", aDevice->GetProtoListSize());
    DBG(" ");

    return true;
  }
  else {
    ERR("CBTPlugin::FetchPrototypes : FD_ISSET failed");
    if(errno != 0) DBG("CBTPlugin::FetchDeviceInfo: FD_ISSET failed, reason: %s", strerror(errno));
  }

  return false;
}


bool CBTPlugin::FetchNeighbourhoodDevices(CDaemonDevice* aDevice, CBTConnection* connection)
{
  fd_set testSet;
  struct timeval timeVal;

  unsigned short numberOfNeighborhoodDevices = 0;
  CDaemonDevice* temp = NULL;

  char* newNeighbor = NULL;
  unsigned int size = 0;

  FD_ZERO(&testSet);
  FD_SET(connection->GetFd(), &testSet);
  
  timeVal.tv_sec = 1;
  timeVal.tv_usec = 0;

  bool foundNeighbor = false;
  
  if(select(connection->GetFd() + 1, &testSet, NULL, NULL, &timeVal) ==-1){
    ERR("CBTPlugin::FetchNeighbourhoodDevices : select failed");
    return false;
  }

  // Start reading data (number of devices and then their info)  
  if (FD_ISSET(connection->GetFd(), &testSet)) {
    if ((connection->Read(&numberOfNeighborhoodDevices, sizeof(numberOfNeighborhoodDevices))) == -1) {
      ERR("CBTPlugin::FetchNeighbourhoodDevices : failed to read the number of devices");
      return false;
    }
	      
    numberOfNeighborhoodDevices = ntohs(numberOfNeighborhoodDevices);

    DBG("FetchNeighbors: Number of neigh devices %d", numberOfNeighborhoodDevices);

    for (int i = 0;i < numberOfNeighborhoodDevices;i++) {
      foundNeighbor = false;

      if (FD_ISSET(connection->GetFd(), &testSet)) { 
	if ((connection->Read(&size, sizeof(size))) == -1) {
	  ERR("CBTPlugin::FetchNeighbourhoodDevices : failed to read the size of the device");
	  return false;
	}
		    
	size = ntohl(size);

	if (FD_ISSET(connection->GetFd(), &testSet)) { 
	  newNeighbor = new char[size];

	  if (connection->Read(newNeighbor, size) == -1) {
	    ERR("CBTPlugin::FetchNeighbourhoodDevices : failed to read device info");
	    delete[] newNeighbor;
	    return false;
	  }

	  temp = new CDaemonDevice(newNeighbor);
	  temp->SetReferrerName(aDevice->GetName());
			
	  foundNeighbor = aDevice->CheckNeighbors(temp->GetAddress());

	  if(!foundNeighbor) {
	    DBG("FetchNeighbors: Found new neighbor %s for %s", temp->GetAddress().c_str(), aDevice->GetAddress().c_str());
	    aDevice->AddNeighboringDevice(temp);
	  }
	  else {
	    DBG("FetchNeighbors: Found old neighbor %s for %s", temp->GetAddress().c_str(), aDevice->GetAddress().c_str());
	    delete temp;
	  }

	  delete[] newNeighbor;
	}
	else {
	  ERR("CBTPlugin::FetchNeighbourhoodDevices : FD_ISSET failed");
	  return false;
	}
      }
      else {
	ERR("CBTPlugin::FetchNeighbourhoodDevices : FD_ISSET failed");
	return false;
      }
    }

    DBG("Number of fetched neighbors %d", aDevice->GetNeighborListSize());    
    DBG("CBTPlugin::FetchNeighborhoodDevices: OK");
    DBG(" ");

    return true;
  }
  else {
    ERR("CBTPlugin::FetchNeighbourhoodDevices : FD_ISSET failed");
  }
    
  return false;
}


/** 
 * @memo Listens a PSM for incoming SERVICE LIST requests.
 * @doc Listens a PSM for incoming SERVICE LIST requests. If a new request is
 * received then a new CBTConnection object is created and passed to the
 * PeerHood Daemon that sends the list of available services. The created 
 * connection object is freed by the daemon. This function is ran on a thread
 * of its own.
 *
 * @return none
 */
void CBTPlugin::AdvertThread()
{
  fd_set fdSet;
  struct timeval timeVal;
  CBTConnection* connection;
  int serviceType;

  connection = new CBTConnection;

  while(!iActive)
  {
  	/** If the plugin was started but not active and while passive the plugin was set to
  	 * shut down. Completely quit from this loop. */
  	if(!iInSdp) return;
  	else sleep(1);
  }
  
  DBG("CBTPlugin::AdvertThread: start");
  while (iInSdp && iActive) {
  	
  	if(!iListening)
  	{
  		if (!connection->Listen(iDaemonPort)) {
  			ERR("CBTPlugin::AdvertThread : listening failed");
  			if(errno != 0) DBG("CBTPlugin::AdvertThread: listening failed, reason: %s", strerror(errno));
  			iListening = false;
  			delete connection;
  			if(iNeedsRestart || !iActive) continue; // if restart required or adapter was set to passive
  			else return;
  		}
  		else iListening = true;
  	}
  	
    timeVal.tv_sec = 1;
    timeVal.tv_usec = 0;

    FD_ZERO(&fdSet);
    FD_SET(connection->GetFd(), &fdSet);

    switch (select(connection->GetFd() + 1, &fdSet, NULL, NULL, &timeVal)) {
      
      // error
      case -1:
				ERR("CBTPlugin::AdvertThread : select failed");
				if(errno != 0) DBG("CBTPlugin::AdvertThread: select failed, reason: %s", strerror(errno));
				break;

      // timeout
      case 0:
				break;

      // data
      default:
	
				DBG("--**--");

				CBTConnection *newConnection = (CBTConnection *)connection->AcceptL();
	
				if (newConnection->Read(&serviceType, sizeof(serviceType)) == -1) {
					ERR("CBTPlugin::AdvertThread : failed to read request type");
					newConnection->Disconnect();
					delete newConnection;
					break;
				}

				serviceType = ntohl(serviceType);
				DBG("Got service request %d from %s", serviceType, newConnection->GetRemoteAddress().c_str());

				if (serviceType & D_GET_DEVICEINFO) {
					DBG("Sending device info");
					CDaemon::GetInstance()->SendDeviceInfo(newConnection);
				}
			 

				if (serviceType & D_GET_SERVICEINFO) {
					DBG("Sending service list");
					CDaemon::GetInstance()->SendServiceList(newConnection);
				}

				if (serviceType & D_GET_PROTOINFO) {
					DBG("Sending prototypes");
					CDaemon::GetInstance()->SendPrototypes(newConnection);
				}

				if (serviceType & D_GET_NEIGHINFO) {
					DBG("Sending neighbourhood list");
					CDaemon::GetInstance()->SendNeighborList(newConnection);
				}

				DBG("All information sent");
				DBG("--**--");

				newConnection->Disconnect();
				delete newConnection;
				break;
    	}
  }
  
	connection->Disconnect();
	connection->Close();
	
	delete connection;
	
	iListening = false;
  
  // If the plugin is set to passive state, but it should be running
	// Or if the restart is required (advert failed)
  // restart - start the function from beginning
  if(!iActive && iInSdp)
  {
  	DBG("CBTPlugin::AdvertThread: restarting..");
  	AdvertThread();
  }
}

void CBTPlugin::RemoveListeners()
{
	std::list<MAbstractListener*>::iterator iter;
	// Go through all listeners and disconnect, delete object and erase from list
	for(iter = iListenerList.begin(); iter != iListenerList.end();)
	{
		(*iter)->Disconnect();
		delete *iter;
		iter = iListenerList.erase(iter); // Returns the element that was after the erased element
	}
}

/**
 * @memo Try to reset (put down - bring up) adapter
 * @doc Tries to reset the adapter via ioctl calls to adapter.
 *
 * @param aAdapterId adapter to use
 *
 * @return RESET_OK when successful, RESET_SOCK/DOWN/UP_FAIL when errors.
 */
// TODO change to use the internal iDeviceId variable!
int CBTPlugin::ResetAdapter()
{
	int resetrounds = 0;
	
	// If the creation of socket was not successful in constructor
	if (iBTCtrlSock < 0)
	{
		if (CreateLowlevelSocket() == NO_SOCKET_CREATED)
		{
			DBG("CBTPlugin::ResetAdapter: No socket was available, can't open new HCI socket for adapter reset (%d).", iBTCtrlSock);
			return RESET_SOCK_FAIL;
		}
	}
	
	// Put HCI device down, if not successful return
	if(ioctl(iBTCtrlSock, HCIDEVDOWN, iDeviceId) < 0)
	{
		DBG("CBTPlugin::ResetAdapter: Cannot put adapter down. Unable to reset.");
		if(errno != 0)
		{
			perror("CBTPlugin::ResetAdapter device down");
			DBG("CBTPlugin::ResetAdapter: reason: %s.", strerror(errno));
		}
		return RESET_DOWN_FAIL;
	}
	else
	{
		DBG("CBTPlugin::ResetAdapter: Adapter down.");
		
		// Try to bring it up, max 10 tries.
		while(ioctl(iBTCtrlSock, HCIDEVUP, iDeviceId) < 0)
		{
			resetrounds++;
			if(resetrounds > 10) // TODO define reset round amount
			{
				DBG("CBTPlugin::ResetAdapter: Tried to bring adapter up %d times, no success.",resetrounds);
				return RESET_UP_FAIL;
			}
			DBG("CBTPlugin::ResetAdapter: Cannot bring adapter up. Waiting one second and trying again.");
			if(errno != 0)
			{
				perror("CBTPlugin::ResetAdapter device up");
				DBG("CBTPlugin::ResetAdapter: reason: %s.", strerror(errno));
			}
			sleep(1);
		}
		DBG("CBTPlugin::ResetAdapter: Adapter up (rounds: %d)",resetrounds);
		
		return RESET_OK;
	}
}

/**
 * @memo Create a low level socket for BT ioctl calls
 * @doc Creates a low level socket for BT ioctl calls.
 *
 * @return SOCKET_CREATED when successful, NO_SOCKET_CREATED when errors and EXISTING_SOCKET if socket already
 * created.
 */
int CBTPlugin::CreateLowlevelSocket()
{
	// Existing socket
	if(iBTCtrlSock != -1) return EXISTING_SOCKET;
	
	// Try to create
	if ((iBTCtrlSock = socket(AF_BLUETOOTH, SOCK_RAW, BTPROTO_HCI)) < 0)
	{
		DBG("CBTPlugin::CreateLowlevelSocket: No socket was available, can't open new HCI socket (%d).", iBTCtrlSock);
		if(errno != 0)
		{
			perror("CBTPlugin::CreateLowlevelSocket socket creation.");
			DBG("CBTPlugin::CreateLowlevelSocket: reason: %s.", strerror(errno));
		}
		return NO_SOCKET_CREATED;
	}
	else return SOCKET_CREATED;
}

/**
 * @memo "Own" inquiry function for bluetooth
 * @doc Copied from Bluez 4.32 sources (/lib/hci.c), almost identical to hci_inquiry() -function EXCEPT this can use an 
 * existing socket (!) and not to create new one with every inquiry.
 */
int CBTPlugin::DoInquiry(int aLen, int aNrsp, const uint8_t *aLap, inquiry_info **aIi, long aFlags)
{
	struct hci_inquiry_req *ir;
	uint8_t num_rsp = aNrsp;
	void *buf;
	int size, ret = -1;

	if (aNrsp <= 0) {
		num_rsp = 0;
		aNrsp = 255;
	}

	buf = malloc(sizeof(*ir) + (sizeof(inquiry_info) * (aNrsp)));
	if (!buf)
	{
		if(errno != 0)
		{
			DBG("CBTPlugin::DoInquiry: malloc failed, reason: %s",strerror(errno));
			perror("CBTPlugin::DoInquiry");
		}
		return -1;
	}

	ir = (hci_inquiry_req*)buf;
	ir->dev_id  = iDeviceId;
	ir->num_rsp = num_rsp;
	ir->length  = aLen;
	ir->flags   = aFlags;

	if (aLap) {
		memcpy(ir->lap, aLap, 3);
	} else {
		ir->lap[0] = 0x33;
		ir->lap[1] = 0x8b;
		ir->lap[2] = 0x9e;
	}

	ret = ioctl(iBTCtrlSock, HCIINQUIRY, (unsigned long) buf);
	if (ret < 0)
	{
		free(buf);
	}
	else
	{
		size = sizeof(inquiry_info) * ir->num_rsp;

		if (!*aIi)
			*aIi = (inquiry_info*)malloc(size);

		if (*aIi)
		{
			memcpy((inquiry_info*) *aIi, (char*)buf + sizeof(*ir), size);
			ret = ir->num_rsp;
		}
		else
			ret = -1;
	}

	return ret;
}

void CBTPlugin::SetAdapter(const std::string& aInterface, int aId)
{
	DBG("CBTPlugin::SetAdapter: called with parameters: interface = %s, id = %d.",aInterface.c_str(), aId);
	
	if(iNDevice.compare(aInterface) == 0 && iDeviceId == aId)
	{
		DBG("CBTPlugin::SetAdapter: no need to change values.");
	}
	else
	{
		// Set the plugin to inactive
		SetState(false);
		
		// Give threads some time to proceed to wait state
		sleep(1);
		
		// Set new values
		iNDevice = aInterface;
		iDeviceId = aId;
		
		// Set plugin to active
		SetState(true);
	}
}

const std::string& CBTPlugin::GetAdapter()
{
	return iNDevice;
}

static CBTPlugin btplugin;

