/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name BTMonitor.cc
 * @memo Bluetooth implementation of the MAbstractMonitor interface.
 *
 * @version 0.2
 * date     31.07.2003
 * change   23.03.2005
 */

#include <unistd.h>
#include <sys/socket.h>
#include <sys/poll.h>
#include <sys/ioctl.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>

#include <stdlib.h>
#include <asm/byteorder.h>
#include "BTMonitor.h"

#warning "Temporary debug"
//temp
#include <syslog.h>

#define ERR(format, msg...) syslog(LOG_ERR, "ERROR: " format "\n" , ## msg)

#ifdef PH_DEBUG
#define DBG(format, msg...) syslog(LOG_DEBUG, format "\n" , ## msg)
#else
#define DBG( A... )
#endif
//temp

/**
 * @memo Constructor.
 * @doc Constructor, initializes the monitor object so that it can be used
 * immediately after the construction has finished.
 *
 * @param aAddress Remote device's address.
 *
 * @return none
 */
CBTMonitor::CBTMonitor(const std::string& aAddress) : MAbstractMonitor(aAddress)
{
  iAddress = std::string(aAddress);
  iInRange = true;
}

CBTMonitor::~CBTMonitor()
{
}

/**
 * @memo Checks signal strength to a remote device
 * @doc Checks signal strength to a remote device
 *
 * @return 
 */
int CBTMonitor::Monitor()
{
  iInRange = false;
  struct hci_conn_info_req *cr;
  struct hci_request rq;	
  read_link_quality_rp rp;
  bdaddr_t bdaddr;
  int dd;
  
  char* addr;
  int quality;
  std::string address = iAddress;
  
  addr = (char *)iAddress.c_str();

  str2ba(addr, &bdaddr);
  
  dd = hci_open_dev(0);
  if (dd < 0) {
    ERR("CBTMonitor::Monitor : HCI device open failed");
    quality = -1;
    return quality;
  }
  
  
  cr = (struct hci_conn_info_req *)malloc(sizeof(*cr) + sizeof(struct hci_conn_info));
  if (!cr){
    ERR(" CBTMonitor::Monitor : Malloc failed");
    close(dd);
    quality = -1;
    return quality;
  }
	
  bacpy(&cr->bdaddr, &bdaddr);
  cr->type = ACL_LINK;
   
  if (ioctl(dd, HCIGETCONNINFO, (unsigned long) cr) < 0) {
    ERR("CBTMonitor::Monitor : Get connection info failed");
    
    quality = -1;
    close(dd);
    free(cr);
    return quality;
  }

  memset(&rq, 0, sizeof(rq));	
  rq.ogf = OGF_STATUS_PARAM;
  rq.ocf = OCF_READ_LINK_QUALITY;
  rq.cparam = &cr->conn_info->handle;
  rq.clen = 2;
  rq.rparam = &rp;
  rq.rlen = READ_LINK_QUALITY_RP_SIZE;
  
  if (hci_send_req(dd, &rq, 100) < 0) {
    ERR("CBTMonitor::Monitor : HCI_send failed");
    quality = -1;
    close(dd);
    free(cr);
    return quality;
  }
	

  if (rp.status) {
    ERR("CBTMonitor::Monitor : Link quality returned (error) status 0x%2.2X\n", rp.status);
    quality = 0;
    close(dd);
    free(cr);
    return quality;
  }
	
  quality = rp.link_quality;
  
  DBG("CBTMonitor::Monitor : Quality is %d\n", quality);
  close(dd);
  free(cr);
  
  iInRange = true;

  return quality;
 
}


/**
 * @memo Returns remote device's Bluetooth address.
 * @doc Return the Bluetooth address of the device under monitoring.
 *
 * @return remote device's address
 */
const std::string& CBTMonitor::GetAddress()
{
  return iAddress;
}


/**
 * @memo Tells whether a device is in range or not.
 * @doc Tells whether a device is in range or not.
 *
 * @return true if the device is in range
 */
bool CBTMonitor::InRange()
{
  return iInRange;
}

