/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2009,2010 LUT.
 *
 * @name AbstractDeviceCWrapper.cc
 * @memo Implementations of AbstractDevice class C-language API.
 *
 * @version 0.3
 * date     20.11.2009
 * change   18.02.2010
 */
 
#include "AbstractDevice.h"

using namespace std;

/**
 * @memo Returns device's name. 
 * @doc Returns the name of the device.
 *
 * @param _aDevice pointer to device object to use
 *
 * @return the name of the device
 */
const char* ph_c_device_get_name(MAbstractDevice* _aDevice)
{
	return (_aDevice->GetName()).c_str();
}

/**
 * @memo Returns device's name checksum.
 * @doc Returns the device name checksum
 *
 * @param _aDevice pointer to device object to use
 *
 * @return the checksum
 */
unsigned int ph_c_device_get_checksum(MAbstractDevice* _aDevice)
{
	return _aDevice->GetChecksum();
}

/**
 * @memo Returns device's prototype id.
 * @doc Returns device's prototype id. This id is unique for each network
 * technology.
 *
 * @param _aDevice pointer to device object to use
 *
 * @return the prototype id of the device
 */
const char* ph_c_device_get_prototype(MAbstractDevice* _aDevice)
{
	return (_aDevice->GetPrototype()).c_str();
}

/**
 * @memo Tells whether the device has PeerHood enabled or not.
 * @doc Tells whether the device has PeerHood enabled or not. If there's no
 * PeerHood then there are no services either. This function is somewhat
 * unnecessary since devices without PeerHood should be of no interest.
 * Anyway this function can be used to "sniff" the environment.
 *
 * @param _aDevice pointer to device object to use
 *
 * @return true if the remote device has PeerHood enabled
 */
int ph_c_device_has_peerhood(MAbstractDevice* _aDevice)
{
	return _aDevice->HasPeerHood() ? 1 : 0 ;
}

/**
 * @memo Used to query if the device has some particular service.
 * @doc Used to query if the device has some particular service present. The
 * service can be accessible through any networking technology; this function
 * doesn't make any difference between the technologies.
 *
 * @param _aDevice pointer to device object to use
 * @param _aServiceName The name of the service to be queried.
 *
 * @return true if the device has the given service present
 */
int ph_c_device_has_service(MAbstractDevice* _aDevice, const char* _aServiceName)
{
	return _aDevice->HasService(string(_aServiceName)) ? 1 : 0 ;
}

/**
 * @memo Returns all services available on the device.
 * @doc Returns all services available on the device. The returned variable
 * is a list that can be iterated via a <code>TServiceIterator</code> 
 * iterator. Note that caller is responsible of freeing the memory allocated
 * for the returned list.
 *
 * @param _aDevice pointer to device object to use
 *
 * @return list of all available services on the device
 */
TServiceList* ph_c_device_get_servicelist(MAbstractDevice* _aDevice)
{
	return _aDevice->GetServiceListL();
}

/**
 * @memo Returns the address used by the device.
 * @doc Returns the address used by the device. There's no formal shape for
 * the returned address, because the exact address format is depending on the
 * current transport medium.
 *
 * @param _aDevice pointer to device object to use
 *
 * @return the address used by the device
 */
const char* ph_c_device_get_address(MAbstractDevice* _aDevice)
{
	return (_aDevice->GetAddress()).c_str();
}
