/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name Engine.cc
 * @memo Implementation of the PeerHood engine class.
 *
 * @version 0.2
 * date     03.04.2003
 * change   28.04.2010
 */
#include <stdio.h>
#include <cassert>
#include <unistd.h>
#include <dlfcn.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <cassert>
#include <Engine.h>
#include <Factory.h>
#include <pdu.h>
#include <Logger.h>
#include <iostream>
using namespace std;

CEngine* CEngine::iInstance = NULL;


/**
 * @memo Creates a new instance of the CEngine class.
 * @doc Creates a new instance of the CEngine class. This method ensures that
 * at any given moment of time only one instance of this class is running (i.e.
 * it implements the singleton design pattern). If an instance exists already
 * then a pointer to it is returned.
 *
 * @param aCallback The interface where callback notifications should be sent
 * to.
 *
 * @return pointer to the active CEngine instance
 */
CEngine* CEngine::GetInstance(CBasicCallback* aCallback)
{
  if (!iInstance) {
    iInstance = new CEngine(aCallback);
  }

  return iInstance;
}


/**
 * @memo Destructor.
 * @doc Destructor, stops the listening thread & frees all allocated memory.
 *
 * @return none
 */
CEngine::~CEngine()
{
  iStop = true;
  assert(pthread_join(iThread, NULL) == 0);

  for (list<MAbstractConnection*>::iterator i = iConnectionList.begin();i != iConnectionList.end();++i) {
    (*i)->Close();
    delete *i;
  }

  iConnectionList.clear();
}


/**
 * @memo Constructor.
 * @doc Constructor. This constructor is a protected method so it cannot be
 * called directly. Instead, use the <code>GetInstance</code> method.
 *
 * @param aCallback The interface where callback notifications should be sent
 * to.
 *
 * @return none
 */
CEngine::CEngine(CBasicCallback* aCallback)
{
  iCallback = aCallback;
  iStop = false;
  pthread_mutex_init(&iLock, NULL);
}

/**
 * @memo Starts listening thread.
 * @doc Starts listening thread. This thread listens for incoming connection
 * requests.
 *
 * @return none
 */
void CEngine::StartListening()
{
  assert(pthread_create(&iThread, NULL, *CEngine::ThreadStarter, this) == 0);
}

/**
 * @memo Method creates a listening socket for every networking technology
 * @doc Method creates a listening socket for every networking technology.
 * These connections listen for incoming connection requests using processID
 * as port number.
 *
 * @param aLocalPid Process ID which is used as a port for listening
 * @return false if something fails (currently returns always true)
 */
bool CEngine::LoadConnections(const int aLocalPid)
{
  int count;
  string* pluginNames = Factory::GetPluginNamesL(&count);

  for (int i = 0;i < count;i++) {

    MAbstractConnection* connection = Factory::CreateConnectionL(pluginNames[i]);

    int on = 1;
    
    if((pluginNames[i] != "local") && (pluginNames[i] != "bt-base"))
		{
			if(setsockopt(connection->GetFd(), SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) == -1)
	    {
	    	char tmp[50];
	    	sprintf(tmp,"setsockopt failed  (%s)",pluginNames[i].c_str());
	      ERR(tmp);
	      delete connection;
	      continue;
	      //delete[] pluginNames;
	    }
    }
    
    if (!connection->Listen(aLocalPid))
    {
    	char tmp[200];
	    sprintf(tmp,"Listen() for a connection object failed  (%s)",pluginNames[i].c_str());
	    ERR(tmp);
      delete connection;
      //delete[] pluginNames;
    }
    
    else iConnectionList.push_back(connection);
  }
  
  delete[] pluginNames;
  return true;
}


/**
 * @memo Starts the listening thread.
 * @doc Starts the listening thread. This static function is used because the
 * <code>pthread</code> library doesn't allow running a member function in a
 * thread directly.
 *
 * @param aArguments Pointer to the current instance of the <code>CEngine
 * </code> class.
 *
 * @return always NULL
 */
void* CEngine::ThreadStarter(void* aArguments)
{
  CEngine* me = (CEngine *)aArguments;

  me->EternalThread();

  return NULL;
}


/**
 * @memo Listens for incoming requests.
 * @doc Listens for incoming requests and forwards them further. A 
 * request is forwarded to the registered callback interface or simply
 * discarded if the interface is not registered.
 *
 * @return none
 */
void CEngine::EternalThread()
{
  int maxFd;
  u_int8_t command;
  fd_set fdSet;
  struct timeval timeVal;
  MAbstractConnection* connection = NULL;
  //char nameBuf[256];
  int port;

  while (!iStop) {
    FD_ZERO(&fdSet);
    maxFd = 0;

    for (list<MAbstractConnection*>::iterator i = iConnectionList.begin();i != iConnectionList.end();++i) {
      FD_SET((*i)->GetFd(), &fdSet);
      if ((*i)->GetFd() > maxFd) maxFd = (*i)->GetFd();
    }
    
    timeVal.tv_sec = 3;
    timeVal.tv_usec = 0;
    maxFd++;

    switch (select(maxFd, &fdSet, NULL, NULL, &timeVal)) {
      // error
      case -1:
	ERR("CEngine::EternalThread: select");
	break;

	// timeout
      case 0:
	break;

	// data
      default:
	DBG("CEngine::EternalThread : got something...");
	for (list<MAbstractConnection*>::iterator i = iConnectionList.begin();i != iConnectionList.end();++i) {
	  if (FD_ISSET((*i)->GetFd(), &fdSet)) {

	    if ((*i)->IsListening()) {
	      assert((connection = (*i)->AcceptL()) != NULL);
	      iConnectionList.push_back(connection);
	      DBG("CEngine::EternalThread : accepted a new connection");
	      break;
	    }
	    
	    if((*i)->Read(&command, sizeof(command)) == -1)
	      {
		ERR("CEngine::EternalThread :read failed or connection closed (most likely unharmfull)");
		(*i)->Disconnect();
		iConnectionList.erase(i);
		delete *i;
		break;
	      }
	    
	    switch (command) {
	    case PH_DIRECT_DATA:
		DBG("CEngine::EternalThread : PH_DIRECT_DATA");
		iConnectionList.erase(i);
		if (iCallback) {
		 	    
		  if (connection->Read(&port, sizeof(port)) == -1) {
		    ERR("CEngine::EternalThread : failed to read port number");
		    delete connection;
		    break;
		    
		  }
		  
		    command = PH_OK;
		    if (connection->Write(&command, sizeof(command)) == -1) {
		      ERR("CEngine::EternalThread : failed to write PH_OK to a new connection");
		      delete connection;
		      break;
		    }
		    
		    DBG("CEngine::EternalThread : Sent PH_OK");
		    std::cerr << "PH_OK size is " << sizeof(command) << std::endl;

		    //int connectionId = 0;
		    u_int8_t connectionId = 0;
		    if (connection->Read(&connectionId, sizeof(connectionId)) == -1) {
		      ERR("CEngine::EternalThread : failed to read connection Id");
		      delete connection;
		      break;
		    }

		    connectionId = ntohs(connectionId);
		    
		    // EXTENDED SERVICE INFO : peer checksum
		    command = PH_GET_PEER_INFO;
		    if(connection->Write(&command, sizeof(command)) == -1) {
		    	ERR("CEngine::EternalThread : failed to send PH_GET_PEER_INFO");
		    	delete connection;
		    	break;
		    }
				
				unsigned int checksum = 0;
			
				// Read checksum - in network order
				if (connection->Read(&checksum, sizeof(checksum)) == -1) {
					ERR("CEngine::EternalThread : failed to read the device name checksum");
					delete connection;
					break;
				}

		    std::string connectionProto = "";
		    if(connection->GetRemoteAddress().find("BT:") < connection->GetRemoteAddress().size())
		      connectionProto = std::string("bt-base");
		    if(connection->GetRemoteAddress().find("WLAN:") < connection->GetRemoteAddress().size())
		      connectionProto = std::string("wlan-base");
		    if(connection->GetRemoteAddress().find("GPRS:") < connection->GetRemoteAddress().size())
		      connectionProto = std::string("gprs-base");
		    if(connection->GetRemoteAddress().find("LOCAL:") < connection->GetRemoteAddress().size())
		      connectionProto = std::string("local");

		    std::cerr << "CEngine::EternalThread : connection proto is then" << connection->GetRemoteAddress() << std::endl;
		    std::cerr << "CEngine::EternalThread : connection from device checksum: " << ntohl(checksum) << std::endl;
		    
		    iVirtualConnection = new CVirtualConnection(connectionProto);
		    iVirtualConnection->SetConnectionType(connection);
		    iVirtualConnection->SetDeviceChecksum(ntohl(checksum));
		    
		    CConnInfo *tempInfo = new CConnInfo;
		    tempInfo->iVirtualConnection = iVirtualConnection;
		    tempInfo->iId = connectionId;
		    iVirtualConnectionList.push_back(tempInfo);
		    
		    iCallback->NewConnection((unsigned short)port, iVirtualConnection, connectionId);
		    DBG("CEngine::EternalThread : callback notified");
		}
		else {
		  ERR("CEngine::EternalThread : Got a new PH_DIRECT_DATA but there's no callback registered");
		  command = PH_NO_CALLBACK;
		  
		  if (connection->Write(&command, sizeof(command)) == sizeof(command)) {
		    ERR("failed to send PH_NO_CALLBACK");
		    delete connection;
		    break;
		  }
		  
		  delete connection;
		}
		break;

	    case PH_REESTABLISH:
	      DBG("Engine::PH_REESTABLISH");
	      iConnectionList.erase(i);
	      if (iCallback) {
		 	    
		if (connection->Read(&port, sizeof(port)) == -1) {
		  ERR("CEngine::EternalThread : failed to read port number");
		  delete connection;
		  break;
		  
		}
		  
		DBG("CEngine::EternalThread : got port");

		command = PH_OK;
		if(connection->Write(&command, sizeof(command)) == -1) {
		  ERR("failed to write PH_OK to a new connection");
		  delete connection;
		  break;
		}
		
		int connectionId = 0;
		if (connection->Read(&connectionId, sizeof(connectionId)) == -1) {
		  ERR("failed to read service name");
		  delete connection;
		  break;
		}
		
		connectionId = ntohl(connectionId);
		
		iVirtualConnection->SetConnectionType(connection);
		list<CConnInfo*>::iterator i;
		bool found = false;
		
		for (i = iVirtualConnectionList.begin();i != iVirtualConnectionList.end();++i) {
		  if((*i)->iId == connectionId)
		    {
		      found = true;
		      break;
		    }
		}
		
		if(found)
		  (*i)->iVirtualConnection = iVirtualConnection;
		else
		  {
		    ERR("can't find connection object from list");
		    command = PH_NO_CALLBACK;
		    
		    if (connection->Write(&command, sizeof(command)) == sizeof(command)) {
		      ERR("failed to send PH_NO_CALLBACK");
		      delete connection;
		      break;
		    }
		  }
		
		iCallback->NewConnection((unsigned short)port, iVirtualConnection, connectionId);
		DBG("  callback notified");
	      }
	      else {
		ERR("Got a new PH_REESTABLISH but there's no callback registered");
		command = PH_NO_CALLBACK;
		
		if (connection->Write(&command, sizeof(command)) == sizeof(command)) {
		  ERR("failed to send PH_NO_CALLBACK");
		  delete connection;
		  break;
		}
		delete connection;
	      }
	      break;

		
	      default:
		ERR("Engine::unknown pdu, closing the violating connection");
		iConnectionList.erase(i);
		(*i)->Disconnect();
		delete *i;
		break;
	    }
	    
	    break;
	  }
	}
    }
  }
}
