/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name GPRSgw.cc
 * @memo Main file of the GPRS gateway
 *
 * @version 0.1
 * date     20.08.2003
 * change   20.08.2003
 */

#include <stdio.h>
#include <iostream>
#include <csignal>
#include <unistd.h>
#include <sys/ioctl.h>
#include <errno.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <peerhood/pdu.h>
#include "GPRSgw.h"

const int GATEWAY = 9000;

void SignalHandler(int aSignum);
void PipeSignalHandler(int aSignum);
CGPRSgw *gprsgw;

/**
 * @memo Main function of the GPRS Gateway service
 * @doc Main function of the GPRS Gateway service. This function creates a new
 * instance of the gprs gateway service.
 *
 * @parameter argc number of parameters
 * @parameter *argv[] pointer to array of parameters
 * @return system failure code if the daemon initialization fails
 */
int main(int argc, char *argv[])
{
  if(argc != 2)
    {
      std::cout << "USAGE: ./gprsgw <iface>" << std::endl;
      return -1;
    }

  signal(SIGTERM, SignalHandler);
  signal(SIGINT, SignalHandler);
  signal(SIGPIPE, PipeSignalHandler);
  gprsgw = new CGPRSgw(argv[1]);
  gprsgw->HandleSignalConnections();
  return 0;
}

/**
 * @memo Handler for the signals TERM and INT.
 * @doc Handles the signals TERM and INT so that the server is shut down in a
 * controlled manner.
 *
 * @param aSignum The number of the signal to be handled.
 *
 * @return none
 */

void SignalHandler(int aSignum)
{
  std::cout << "SignalHandler : Ouch! Hit by a fatal signal" << std::endl;
  gprsgw->iLoop = false;
}

/**
 * @memo Handler for the signal SIGPIPE.
 * @doc Handles the signal SIGPIPE
 *
 * @param aSignum The number of the signal to be handled.
 *
 * @return none
 */
void PipeSignalHandler(int aSignum)
{
}

/**
 * @memo Default constructor, sets the internal variables to their default
 * values.
 * @doc Default constructor, sets the internal variables to their default
 * values.
 *
 * @param aIface The network interface which is wanted to be used
 * @return none
 */
CGPRSgw::CGPRSgw(const std::string& aIface)
{
  iIface = aIface;
  iDataStarted = false;
  iDataFailed = false;
  iLoop = true;
}


/**
 * @memo Starts a new Data Thread
 * @doc Starts a new Data Thread.
 * Data Thread takes care of routing data in established
 * Data Connection.
 *
 * @param aArguments Pointer to GPRSgwThread class.
 *
 * @return always NULL
 */
void *CGPRSgw::NewThreadStarter(void *aArguments)
{
  CGPRSgwThread *tempgwThreadInfo = (CGPRSgwThread *)aArguments;
  CGPRSgw *fake = (CGPRSgw *)tempgwThreadInfo->iInstance;
  fake->NewThread((*tempgwThreadInfo));
  delete tempgwThreadInfo;
  return NULL;
}


/**
 * @memo Creates a new Data Thread
 * @doc Creates a new Data Thread and passes
 * thread information (two sockets) for the thread.
 *
 * @param aOriginalFd A socket
 * @param aReplyFd Other socket
 * @return boolean telling if the thread could be started
 */
bool CGPRSgw::StartNewThread(int aOriginalFd, int aReplyFd)
{
  pthread_t tempThread;
  CGPRSgwThread *tempgwThread = new CGPRSgwThread;
  tempgwThread->iInstance = this;
  tempgwThread->iOriginalFd = aOriginalFd;
  tempgwThread->iReplyFd = aReplyFd;

  if (pthread_create(&tempThread, NULL, *CGPRSgw::NewThreadStarter, (CGPRSgwThread* )tempgwThread) != 0) {
    std::cerr << "CGPRSgw::StartNewThread : failed to create a thread" << std::endl;
    return false;
  }
  pthread_detach(tempThread);
  return true;
}

/**
 * @memo Implementation of Data Thread
 * @doc Data Thread starts to route data between previously established
 * Data Connections. aThreadInfo contains two sockets which are the
 * end points of established connection. Data Thread simply reads incoming data
 * from the socket and forwards it to next socket. If socket is closed
 * Data Thread is ended. When Data Thread is ended, Data Connection connection
 * objects are destroyed.
 *
 * @param aThreadInfo Reference to GPRSgwThread class containing Data Connection sockets.
 * @return none
 */
void CGPRSgw::NewThread(CGPRSgwThread& aThreadInfo)
{
  fd_set dataSet;
  struct timeval timeVal;
  unsigned int maxFd = 0;

  while (iDataStarted) {

    FD_ZERO(&dataSet);
    FD_SET(aThreadInfo.iOriginalFd, &dataSet);
    FD_SET(aThreadInfo.iReplyFd, &dataSet);

    if ( aThreadInfo.iOriginalFd > maxFd)
      maxFd = aThreadInfo.iOriginalFd;
    if ( aThreadInfo.iReplyFd > maxFd)
      maxFd = aThreadInfo.iReplyFd;

      timeVal.tv_sec = 3;
      timeVal.tv_usec = 0;


      if(select(maxFd + 1, &dataSet, NULL, NULL, &timeVal) ==-1){
	close(aThreadInfo.iOriginalFd);
	close(aThreadInfo.iReplyFd);
	perror("CGPRSPlugin::DataThread : select failed");
	if(!DeleteDataConnectionObject(aThreadInfo.iOriginalFd))
	  std::cout << "CGPRSgw::NewThread : Failed to delete connection object" << std::endl;
	if(!DeleteDataConnectionObject(aThreadInfo.iReplyFd))
	  std::cout << "CGPRSgw::NewThread : Failed to delete connection object" << std::endl;
	return;
      }

      for(unsigned int i=0;i<=maxFd;i++){
	if (FD_ISSET(i, &dataSet))
	  {
	    int targetFd = 0;
	    unsigned int length = 0;
	    if(i==aThreadInfo.iOriginalFd)
	      targetFd = aThreadInfo.iReplyFd;


	    if(i==aThreadInfo.iReplyFd)
	      targetFd = aThreadInfo.iOriginalFd;

	    if(!TestConnection(i, sizeof(length)))
	      {
		/*
		close(aThreadInfo.iOriginalFd);
		close(aThreadInfo.iReplyFd);

		if(!DeleteDataConnectionObject(aThreadInfo.iOriginalFd))
		  std::cout << "CGPRSgw::NewThread : Failed to delete connection object" << std::endl;
		if(!DeleteDataConnectionObject(aThreadInfo.iReplyFd))
		  std::cout << "CGPRSgw::NewThread : Failed to delete connection object" << std::endl;
		return;
		*/
	      }

	    if(recv(i, &length, sizeof(length), MSG_WAITALL) <=0){
	      std::cout << "CGPRSgw::NewThread : Socket was closed while receiving data length" << std::endl;
	      close(aThreadInfo.iOriginalFd);
	      close(aThreadInfo.iReplyFd);

	      if(!DeleteDataConnectionObject(aThreadInfo.iOriginalFd))
		std::cout << "CGPRSgw::NewThread : Failed to delete connection object" << std::endl;
	      if(!DeleteDataConnectionObject(aThreadInfo.iReplyFd))
		std::cout << "CGPRSgw::NewThread : Failed to delete connection object" << std::endl;

	      return;
	    }

	    length = ntohl(length);

	    if (FD_ISSET(i, &dataSet))
	      {
		if(!TestConnection(i, length))
		  {
		    /*
		    close(aThreadInfo.iOriginalFd);
		    close(aThreadInfo.iReplyFd);

		    if(!DeleteDataConnectionObject(aThreadInfo.iOriginalFd))
		      std::cout << "CGPRSgw::NewThread : Failed to delete connection object" << std::endl;
		    if(!DeleteDataConnectionObject(aThreadInfo.iReplyFd))
		      std::cout << "CGPRSgw::NewThread : Failed to delete connection object" << std::endl;
		    return;
		    */

		  }

		char *buffer = new char[length];
		memset(buffer, 0, length);
		if(recv(i, buffer, length, MSG_WAITALL) <=0){
		  std::cout << "CGPRSgw::NewThread : Socket was closed while receiving data" << std::endl;
		  delete[] buffer;
		  close(aThreadInfo.iOriginalFd);
		  close(aThreadInfo.iReplyFd);
		  if(!DeleteDataConnectionObject(aThreadInfo.iOriginalFd))
		    std::cout << "CGPRSgw::NewThread : Failed to delete connection object" << std::endl;
		  if(!DeleteConnectionObject(aThreadInfo.iReplyFd))
		    std::cout << "CGPRSgw::NewThread : Failed to delete connection object" << std::endl;
		  return;
		}

		length = htonl(length);

		if(write(targetFd, &length, sizeof(length)) <= 0){
		  std::cout << "CGPRSgw::NewThread() : sending data length failed" << std::endl;
		  delete[] buffer;
		  close(aThreadInfo.iOriginalFd);
		  close(aThreadInfo.iReplyFd);
		  if(!DeleteDataConnectionObject(aThreadInfo.iOriginalFd))
		    std::cout << "CGPRSgw::NewThread : Failed to delete connection object" << std::endl;
		  if(!DeleteDataConnectionObject(aThreadInfo.iReplyFd))
		    std::cout << "CGPRSgw::NewThread : Failed to delete connection object" << std::endl;
		  return;
		}

		length = ntohl(length);

		if(write(targetFd, buffer, length) <= 0){
		  std::cout << "CGPRSgw::NewThread() : sending data failed" << std::endl;
		  delete[] buffer;
		  close(aThreadInfo.iOriginalFd);
		  close(aThreadInfo.iReplyFd);
		  if(!DeleteDataConnectionObject(aThreadInfo.iOriginalFd))
		    std::cout << "CGPRSgw::NewThread : Failed to delete connection object" << std::endl;
		  if(!DeleteDataConnectionObject(aThreadInfo.iReplyFd))
		    std::cout << "CGPRSgw::NewThread : Failed to delete connection object" << std::endl;
		  return;
		}

		delete[] buffer;
	      }//if (FD_ISSET(i, &dataSet))
	  }//if (FD_ISSET(i, &dataSet))
      }// for(unsigned int i=0;i<=maxFd;i++){
  }//while (iDataStarted) {
  close(aThreadInfo.iOriginalFd);
  close(aThreadInfo.iReplyFd);
  if(!DeleteDataConnectionObject(aThreadInfo.iOriginalFd))
    std::cout << "CGPRSgw::NewThread : Failed to delete connection object" << std::endl;
  if(!DeleteDataConnectionObject(aThreadInfo.iReplyFd))
    std::cout << "CGPRSgw::NewThread : Failed to delete connection object" << std::endl;
  return;
}


/**
 * @memo Starts Data Signalling thread.
 * @doc Starts plugin's Data Signalling thread. This dummy function is required
 * because C++ doesn't allow non-static member functions to be started as a
 * thread. This function just calls plugin instance's real inquiry function.
 *
 * @param aArguments Pointer to the currently running instance.
 *
 * @return always NULL
 */
void *CGPRSgw::DataThreadStarter(void *aArguments)
{
  CGPRSgw *fake = (CGPRSgw *)aArguments;
  fake->HandleDataConnections();
  return NULL;
}

/**
 * @memo Starts the Data Signalling routine.
 * @doc Starts the Data Signalling routine. After this method is called the
 * Gateway starts to listen a specific port for incoming Data Signalling
 * connections. These Data Signalling connections are used to negoatiate
 * a new Data Connection between GPRS -plugins. Protocol is following:
 * Device A creates a new Client Data thread which connects to Gateway using
 * new Data Connection and sends CONNECT -message containing the target device's ID -number.
 * Connection is handled by Data Signalling thread which forwards this CONNECT -message
 * to target device B using Data Signalling connection. Target device B receives this
 * CONNECT -message and creates a new Server Data Thread. Server Data Thread creates
 * a new Data Connection to Gateway and sends REPCONNECT -message. Again, this new
 * Data Connection is handled by Gateway Data Signalling thread. Data Signalling thread
 * forwards REPCONNECT message to device A and creates a new Data Thread. Data Thread
 * gets established Data Connection sockets as a parameter and starts to route data
 * between these sockets.
 *
 * @return boolean telling if the plugin could be started
 */
bool CGPRSgw::StartDataConnections( )
{
  if(iDataStarted)
    {
      std::cout << "CGPRSgw::StartDataConnections : data thread already started" << std::endl;
      return false;
    }

  std::cout << "CGPRSgw::StartDataConnections : starting data thread" << std::endl;

  iDataStarted = true;

  if (pthread_create(&iDataThread, NULL, *CGPRSgw::DataThreadStarter, this) != 0) {
    std::cerr << "CGPRSgw::StartDataConnections : failed to create a thread" << std::endl;
    iDataStarted = false;
    return false;
  }

  std::cout << "CGPRSgw::StartDataConnections : data thread started" << std::endl;

  return true;
}

/**
 * @memo Stops the Data Signalling Thread.
 * @doc Stops the Data Signalling Thread. After this call is handled the
 * there can't be established new Data Connections.
 *
 * @return none
 */
void CGPRSgw::StopDataConnections()
{
  if(!iDataStarted)
    {
      std::cout << "CGPRSgw::StopDataConnections : data thread is not running" << std::endl;
      return;
    }

  std::cout << "CGPRSgw::StopDataConnections : stopping the data thread" << std::endl;

  iDataStarted = false;

  if (pthread_join(iDataThread, NULL) != 0) {
    std::cerr << "CGPRSgw::StopDataConnections : failed to join the data thread" << std::endl;
  }

  std::cout << "CGPRSgw::StopDataConnections : data thread stopped" << std::endl;
}


/**
 * @memo Handles Data Signalling Connections.
 * @doc Handles Data Signalling Connections. Check the description
 * from StartDataConnections( ) -method.
 *
 * @return false on error situations
 */
bool CGPRSgw::HandleDataConnections()
{
  fd_set dataMasterSet, dataTempSet;
  struct timeval timeVal;
  unsigned int on = 1, maxFd = 0;

  CTCPConnection serviceConnection(iIface, "GPRS");
  CTCPConnection *newConnection;

  if(setsockopt(serviceConnection.GetFd(), SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) == -1)
    {
      std::cerr << "CGPRSgw::HandleDataConnections() : setsockopt failed" << std::endl;
      iDataFailed = true;
      return false;
    }

  if (!serviceConnection.Listen(GATEWAY+1)) {
    std::cerr << "CGPRSgw::HandleDataConnections() : listening failed" << std::endl;
    iDataFailed = true;
    return false;
  }

  FD_ZERO(&dataMasterSet);
  FD_ZERO(&dataTempSet);
  FD_SET(serviceConnection.GetFd(), &dataMasterSet);
  maxFd = serviceConnection.GetFd();

  while (iDataStarted) {
    std::cout << "CGPRSgw::HandleDataConnections : running..." << std::endl;

    timeVal.tv_sec = 3;
    timeVal.tv_usec = 0;
    dataTempSet = dataMasterSet;

    if(select(maxFd + 1, &dataTempSet, NULL, NULL, &timeVal) ==-1){
      std::cerr << "CGPRSgw::HandleDataConnections : select failed" << std::endl;
      iDataFailed = true;
      return false;
    }

    for(unsigned int i=0;i<=maxFd;i++){
      if (FD_ISSET(i, &dataTempSet))
	{
	  std::cout << "CGPRSgw::HandleDataConnections :: DataSignalConnection from " << i << std::endl;
	  //new connection
	  if(i==(unsigned int)serviceConnection.GetFd())
	    {
	      if(!iDataStarted)
		{
		  return true;
		}

	      if(!(newConnection = (CTCPConnection *)serviceConnection.AcceptL()))
		{
		  std::cerr << "CGPRSgw::HandleDataConnections : AcceptL failed" << std::endl;
		  iDataFailed = true;
		  return false;
		}
	    else
	      {
		std::cout << "CGPRSgw::HandleDataConnections :: Accepted newConnection from fd " << newConnection->GetFd() << std::endl;
		FD_SET(newConnection->GetFd(),&dataMasterSet);
		if((unsigned int)newConnection->GetFd() > maxFd)
		  maxFd=newConnection->GetFd();
		iDataConnectionObjects.push_back(newConnection);
	      }
	    }
	  //old connections
	  else
	    {
	      CGPRSSignal tempDataSignal;
	      if(!DecapsulateSignal(i, tempDataSignal))
		{
		  std::cout << "CGPRSgw::HandleDataConnections :: DecapsulateSignal failed" << std::endl;
		  close(i);
		  FD_CLR(i, &dataMasterSet);
		  NoticeDataError(i);
		  DeleteDataConnectionObject(i);
		  i = 0;
		  break;
		}

	      switch(tempDataSignal.iMessageType) {

		/*ADVERT is message for registering a new Data Signal Connection
		 *every plugin must send this message before new Data Connections
		 *can be established
		 */
	      case ADVERT:
		{
		  std::cout << "CGPRSgw::HandleDataConnections : ADVERT" << std::endl;

		  bool found = false;
		  pthread_mutex_lock(&iLock);
		  for (std::list<CGPRSDevice *>::iterator j = iDeviceList.begin();j != iDeviceList.end();++j)
		    {
		      if((*j)->iId == tempDataSignal.iOriginalId)
			{
			  (*j)->iDataFd = i;
			  found = true;
			  break;
			}
		    }
		  pthread_mutex_unlock(&iLock);

		  if(!found)
		    {
		      std::cout << "CGPRSgw::HandleDataConnections : Can't find id " << std::endl;
		      NoticeDataError(i);
		      close(i);
		      FD_CLR(i, &dataMasterSet);
		      DeleteDataConnectionObject(i);
		      break;
		    }

		  break;
		}

	      case CONNECT:
		{
		  std::cout <<  "CGPRSgw::HandleDataConnections : CONNECT from " << tempDataSignal.iOriginalId << " to " << tempDataSignal.iData << " serviceName is " << tempDataSignal.iServiceName << std::endl;

		  int originalDataFd = 0;
		  int targetDataFd = 0;
		  bool originalDataFound = false;
		  pthread_mutex_lock(&iLock);
		  for (std::list<CGPRSDevice *>::iterator j = iDeviceList.begin();j != iDeviceList.end();++j) {
		    if(tempDataSignal.iOriginalId == (*j)->iId)
		      {
			originalDataFound = true;
			originalDataFd = (*j)->iDataFd;
			break;
		      }
		  }
		  pthread_mutex_unlock(&iLock);

		  bool targetDataFound = false;
		  pthread_mutex_lock(&iLock);
		  for (std::list<CGPRSDevice *>::iterator j = iDeviceList.begin();j != iDeviceList.end();++j) {
		    if(tempDataSignal.iData == (*j)->iId)
		      {
			targetDataFound = true;
			targetDataFd = (*j)->iDataFd;
			break;
		      }
		  }
		  pthread_mutex_unlock(&iLock);

		  if((!originalDataFound) || (!targetDataFound))
		    {
		      if(!originalDataFound)
			std::cout << "CGPRSgw::HandleDataConnections : can't find originalDataFd" << std::endl;
		      if(!targetDataFound)
			std::cout << "CGPRSgw::HandleDataConnections : can't find targetDataFd" << std::endl;
		      /*
		      NoticeDataError(originalDataFd);
		      close(originalDataFd);
		      FD_CLR(originalDataFd, &dataMasterSet);
		      DeleteDataConnectionObject(originalDataFd);

		      NoticeDataError(targetDataFd);
		      close(targetDataFd);
		      FD_CLR(targetDataFd, &dataMasterSet);
		      DeleteDataConnectionObject(targetDataFd);
		      */

		      close(i);
		      FD_CLR(i, &dataMasterSet);
		      break;
		    }
		  else
		    {
		      if(!CheckPresence(targetDataFd))
			{
			  std::cout << "HandleDataConnections : Device" << tempDataSignal.iData << " is not available anymore" << std::endl;
			  /*
			   NoticeDataError(originalDataFd);
			   close(originalDataFd);
			   FD_CLR(originalDataFd, &dataMasterSet);
			   DeleteDataConnectionObject(originalDataFd);

			   NoticeDataError(targetDataFd);
			   close(targetDataFd);
			   FD_CLR(targetDataFd, &dataMasterSet);
			   DeleteDataConnectionObject(targetDataFd);
			  */

			   close(i);
			   FD_CLR(i, &dataMasterSet);
			   break;
			}


		      if(!EncapsulateSignal(targetDataFd, tempDataSignal))
			{
			  std::cout << "CGPRSgw::HandleDataConnections : Encapsulate CONNECT failed" << std::endl;
			  /*
			  NoticeDataError(originalDataFd);
			  close(originalDataFd);
			  FD_CLR(originalDataFd, &dataMasterSet);
			  DeleteDataConnectionObject(originalDataFd);

			   NoticeDataError(targetDataFd);
			   close(targetDataFd);
			   FD_CLR(targetDataFd, &dataMasterSet);
			   DeleteDataConnectionObject(targetDataFd);
			  */

			   close(i);
			   FD_CLR(i, &dataMasterSet);
			  break;
			}
		    }

		  CGPRSDataConnection tempDataConnection;
		  tempDataConnection.iOriginalId = tempDataSignal.iOriginalId;
		  tempDataConnection.iOriginalFd = i;
		  tempDataConnection.iServiceName = tempDataSignal.iServiceName;
		  iDataConnections.push_back(tempDataConnection);
		  break;
		}

	      case REPCONNECT:
		{
		  std::cout <<  "CGPRSgw::HandleDataConnections : REPCONNECT from " << tempDataSignal.iOriginalId << " to " << tempDataSignal.iData << " serviceName is " << tempDataSignal.iServiceName << std::endl;

		  int originalDataFd = 0;
		  int targetDataFd = 0;
		  int originalThreadFd = 0;

		  bool originalDataFound = false;
		  pthread_mutex_lock(&iLock);
		  for (std::list<CGPRSDevice *>::iterator j = iDeviceList.begin();j != iDeviceList.end();++j) {
		    if(tempDataSignal.iData == (*j)->iId)
		      {
			originalDataFound = true;
			originalDataFd = (*j)->iDataFd;
			break;
		      }
		  }
		  pthread_mutex_unlock(&iLock);

		  bool targetDataFound = false;
		  pthread_mutex_lock(&iLock);
		  for (std::list<CGPRSDevice *>::iterator j = iDeviceList.begin();j != iDeviceList.end();++j) {
		    if(tempDataSignal.iOriginalId == (*j)->iId)
		      {
			targetDataFound = true;
			targetDataFd = (*j)->iDataFd;
			break;
		      }
		  }
		  pthread_mutex_unlock(&iLock);

		  bool threadOriginalFound = false;
		  std::list<CGPRSDataConnection>::iterator j;
		  for (j = iDataConnections.begin();j != iDataConnections.end();++j) {
		    if(tempDataSignal.iData == j->iOriginalId)
		      {
			if(tempDataSignal.iServiceName.compare(j->iServiceName) == 0)
			  {
			    threadOriginalFound = true;
			    originalThreadFd = j->iOriginalFd;
			    break;
			  }
		      }
		  }

		  if((!originalDataFound) || (!targetDataFound) || (!threadOriginalFound))
		    {
		      if(!originalDataFound)
			std::cout << "CGPRSgw::HandleDataConnections : can't find original data fd" << std::endl;
		      if(!targetDataFound)
			std::cout << "CGPRSgw::HandleDataConnections : can't find target data fd" << std::endl;

		      if(!threadOriginalFound)
			std::cout << "CGPRSgw::HandleDataConnections : can't find thread original Id for REPCONNECT" << std::endl;
		      /*
		      NoticeDataError(originalDataFd);
		      close(originalDataFd);
		      FD_CLR(originalDataFd, &dataMasterSet);
		      DeleteDataConnectionObject(originalDataFd);

		      NoticeDataError(targetDataFd);
		      close(targetDataFd);
		      FD_CLR(targetDataFd, &dataMasterSet);
		      DeleteDataConnectionObject(targetDataFd);
		      */

		      close(originalThreadFd);
		      FD_CLR(originalThreadFd, &dataMasterSet);
		      close(i);
		      FD_CLR(i, &dataMasterSet);
		      break;
		    }
		  else
		    {

		      if(!CheckPresence(originalThreadFd))
			{
			  std::cout << "HandleDataConnections : Device " << tempDataSignal.iOriginalId << " is not available anymore" << std::endl;

			   NoticeDataError(originalDataFd);
			   close(originalDataFd);
			   FD_CLR(originalDataFd, &dataMasterSet);
			   DeleteDataConnectionObject(originalDataFd);

			   NoticeDataError(targetDataFd);
			   close(targetDataFd);
			   FD_CLR(targetDataFd, &dataMasterSet);
			   DeleteDataConnectionObject(targetDataFd);


			   close(originalThreadFd);
			   FD_CLR(originalThreadFd, &dataMasterSet);
			   close(i);
			   FD_CLR(i, &dataMasterSet);
			  break;
			}


		      if(!EncapsulateSignal(originalThreadFd, tempDataSignal))
			{
			  std::cout << "CGPRSgw::HandleDataConnections : Encapsulate REPCONNECT failed" << std::endl;
			  /*
			  NoticeDataError(originalDataFd);
			  close(originalDataFd);
			  FD_CLR(originalDataFd, &dataMasterSet);
			  DeleteDataConnectionObject(originalDataFd);

			  NoticeDataError(targetDataFd);
			  close(targetDataFd);
			  FD_CLR(targetDataFd, &dataMasterSet);
			  DeleteDataConnectionObject(targetDataFd);
			  */

			  close(originalThreadFd);
			  FD_CLR(originalThreadFd, &dataMasterSet);
			  close(i);
			  FD_CLR(i, &dataMasterSet);
			  break;
			}
		      else
			{
			  std::cout << "CGPRSgw::HandleDataConnections : creating new thread" << std::endl;
			  if(!StartNewThread(originalThreadFd, i))
			    {
			      std::cout << "CGPRSgw::HandleDataConnections : can't create thread" << std::endl;
			      return false;
			    }
			  FD_CLR(i, &dataMasterSet);
			  FD_CLR(originalThreadFd, &dataMasterSet);
			  iDataConnections.erase(j);
			}
		    }
		  break;
		}//case REPCONNECT:
	      }// switch(tempDataRoute.iMessageType)
	    }// else
	}// if (FD_ISSET(i, &dataTempSet))
    }//  for(unsigned int i=0;i<=maxFd;i++)
  }// while (iData) {
  std::cout << "CGPRSgw::HandleDataConnections : Ending DataConnection" << std::endl;
  return true;
}


/**
 * @memo Signal Thread routine.
 * @doc Signal Thread routine takes care of registering and deregistering
 * new Signal Connectionsa and sending devices and services. When a new
 * device connects it must register itself into device database. When device
 * is registered, Signal Thread routine inserts the device into database.
 * Device and service queries can be requested only when Gateway allows device
 * to do this. Gateway goes through the list of registered devices and allows
 * time for inquiries. This is notified as sending REQ -message to device.
 * When device gets this REQ -message it can perform device or service inquiries.
 * After the device has performed the tasks, it sends a REQCOMPLETE -message to
 * Gateway. While one device is performing inquiries, other devices are just
 * listening and ready for replying to service information requests. If device
 * doesn't send REQCOMPLETE -message after specific time, Gateway erases the
 * device and all it's connections.
  *
 * @return false if something failed
 */
bool CGPRSgw::HandleSignalConnections()
{
  fd_set tempSet;
  struct timeval timeVal;
  unsigned int on = 1, maxFd = 0, id = 1, request = 0, requestCount = 0;
  CTCPConnection connection(iIface, "GPRS");
  CTCPConnection *newConnection;

  if(setsockopt(connection.GetFd(), SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) == -1)
    {
      std::cerr << "CGPRSgw::HandleSignalConnections : setsockopt failed  " << std::endl;
      return false;
    }

  if (!connection.Listen(GATEWAY)) {
    std::cerr << "CGPRSgw::HandleSignalConnections : listening failed" << std::endl;
    return false;
  }

  FD_ZERO(&masterSet);
  FD_ZERO(&tempSet);
  FD_SET(connection.GetFd(), &masterSet);
  maxFd = connection.GetFd();

  iDeviceList.clear();
  iDeviceList.resize(0);
  if(!StartDataConnections())
    {
      std::cout << "CGPRSgw::HandleSignalConnections : can't start data connections" << std::endl;
      return false;
    }

  while (iLoop) {
    std::cout << "CGPRSgw::HandleSignalConnections : running..." << std::endl;
    if(iDataFailed)
      {
	iLoop = false;
	break;
      }

    pthread_mutex_lock(&iLock);
    for (std::list<CGPRSDevice *>::iterator i = iDeviceList.begin();i != iDeviceList.end();++i)
      {
	if((*i)->iStatus == 0)
	  {
	    if(request == (*i)->iId)
	      request = 0;
	    std::cout << "Trying to erase connection object" << std::endl;
	    DeleteConnectionObject((*i)->iSignalFd);
	    std::cout << "Trying to erase GPRS" << (*i)->iId << std::endl;
	    close((*i)->iSignalFd);
	    FD_CLR((*i)->iSignalFd, &masterSet);
	    std::list<CGPRSDevice *>::iterator sentenced = i++;
	    delete *sentenced;
	    iDeviceList.erase(sentenced);
	  }

      }
    pthread_mutex_unlock(&iLock);

    std::cout << "---Devices---" << std::endl;

    pthread_mutex_lock(&iLock);
    for (std::list<CGPRSDevice *>::iterator i = iDeviceList.begin();i != iDeviceList.end();++i)
      {
	std::cout << "GPRS" << (*i)->iId << " iSignalFd=" << (*i)->iSignalFd << " iDataFd=" << (*i)->iDataFd << " status=" << (*i)->iStatus << std::endl;

      }
    pthread_mutex_unlock(&iLock);


    if(request == 0)
      {
	std::cout << "CGPRSgw::HandleSignalConnections : request is not currently locked" << std::endl;
	bool found = false;
	pthread_mutex_lock(&iLock);
	for (std::list<CGPRSDevice *>::iterator i = iDeviceList.begin();i != iDeviceList.end();++i)
	  {
	    if(!(*i)->iChecked)
	      {
		std::cout << "CGPRSgw::HandleSignalConnections : sending REQ to GPRS " << (*i)->iId << std::endl;
		CGPRSSignal sendSignal;
		sendSignal.iMessageType = REQ;
		sendSignal.iOriginalId = 0;
		sendSignal.iData = 0;
		sendSignal.iData2 = 0;
		sendSignal.iServiceName = "";

		pthread_mutex_unlock(&iLock);
		if(!EncapsulateSignal((*i)->iSignalFd, sendSignal))
		  {
		    std::cout << "CGPRSgw::HandleSignalConnections : Encapsulate REQ failed" << std::endl;
		    NoticeError((*i)->iSignalFd);
		    pthread_mutex_lock(&iLock);
		    break;
		  }
		pthread_mutex_lock(&iLock);

		request = (*i)->iId;
		(*i)->iChecked = true;
		found = true;
		requestCount = 0;
		break;
	      }
	  }
	pthread_mutex_unlock(&iLock);

	if(!found)
	  {
	    pthread_mutex_lock(&iLock);
	    std::cout << "CGPRSgw::HandleSignalConnections : All devices have had a query" << std::endl;
	    for (std::list<CGPRSDevice *>::iterator i = iDeviceList.begin();i != iDeviceList.end();++i)
	      (*i)->iChecked = false;

	    pthread_mutex_unlock(&iLock);
	  }
      }
    else
      {
	std::cout << "CGPRSgw::HandleSignalConnections : request is locked by GPRS" << request << std::endl;
	if(requestCount > 10)
	  {
	    pthread_mutex_lock(&iLock);
	    for (std::list<CGPRSDevice *>::iterator i = iDeviceList.begin();i != iDeviceList.end();++i)
	      {
		if(request == (*i)->iId)
		  {
		    std::cout << "CGPRSgw::HandleSignalConnections : releasing the request..." << std::endl;
		    pthread_mutex_unlock(&iLock);
		    NoticeError((*i)->iSignalFd);
		    pthread_mutex_lock(&iLock);
		    break;
		  }
	      }
	    pthread_mutex_unlock(&iLock);
	    requestCount = 0;
	  }
	else
	  requestCount++;
      }

    timeVal.tv_sec = 3;
    timeVal.tv_usec = 0;
    tempSet = masterSet;

    if(select(maxFd + 1, &tempSet, NULL, NULL, &timeVal) ==-1){
      std::cerr << "CGPRSgw::HandleSignalConnections : select failed" << std::endl;
      iLoop = false;
      break;
    }

    for(unsigned int i=0;i<=maxFd;i++){

      if (FD_ISSET(i, &tempSet))
	{
	  //new connection
	  if(i==(unsigned int)connection.GetFd())
	    {
	      if(!iLoop)
		break;

	      if(!(newConnection = (CTCPConnection *)connection.AcceptL()))
		{
		  iLoop = false;
		  break;
		}
	    else
	      {
		std::cout << "CGPRSgw::HandleSignalConnections : Accepted newConnection from fd " << newConnection->GetFd() << std::endl;
		FD_SET(newConnection->GetFd(),&masterSet);
		if((unsigned int)newConnection->GetFd() > maxFd)
		  maxFd=newConnection->GetFd();
		iConnectionObjects.push_back(newConnection);
	      }
	    }
	  //old connections
	  else
	    {
	      std::cout << "CGPRSgw::HandleSignalConnections : SignalConnection from fd " << i << std::endl;

	      CGPRSSignal tempSignal;
	      if(!DecapsulateSignal(i, tempSignal))
		{
		  std::cout << "CGPRSgw::HandleSignalConnections : DecapsulateSignal failed!" << std::endl;
		  NoticeError(i);
		  break;
		}
	      else
		{
		  // switch(tempRoute.iSignal) {
		  switch(tempSignal.iMessageType) {

		    /* This message is for registering a new Signal Connection
		     * every device must send this before any service or device
		     * inquiries can be made
		     */
		  case GETID:
		    {
		      std::cout << "CGPRSgw::HandleSignalConnections : GETID" << std::endl;
		      //send REPID
		      CGPRSSignal sendSignal;
		      sendSignal.iMessageType = REPID;
		      sendSignal.iOriginalId = id;
		      sendSignal.iData2 = 0;
		      sendSignal.iServiceName = "";
		      sendSignal.iData = 0;

		      if(!EncapsulateSignal(i, sendSignal))
			{
			  std::cout << "CGPRSgw::HandleSignalConnections : Encapsulate REPID failed" << std::endl;
			  NoticeError(i);
			  break;
			}
		      std::cout << "Sent id " << id << std::endl;

		      CGPRSDevice *tempDevice = new CGPRSDevice;
		      tempDevice->iId = id;
		      tempDevice->iSignalFd = i;
		      tempDevice->iDataFd = 0;
		      tempDevice->iStatus = 1;
		      tempDevice->iChecked = false;

		      pthread_mutex_lock(&iLock);
		      iDeviceList.push_back(tempDevice);
		      pthread_mutex_unlock(&iLock);

		      id++;
		      break;
		    }

		  case REQCOMPLETE:
		    {
		      std::cout << "CGPRSgw::HandleSignalConnections : REQCOMPLETE from GPRS" << tempSignal.iOriginalId << std::endl;
		      request = 0;
		      break;
		    }

		  case GETDEV:
		    {
		      std::cout << "CGPRSgw::HandleSignalConnections : GETDEV" << std::endl;
		      //send device list
		      if(!SendDeviceList(i, tempSignal.iOriginalId, masterSet))
			{
			  std::cout << "CGPRSgw::HandleSignalConnections : SendDeviceList failed" << std::endl;
			   NoticeError(i);
			  break;
			}
		      break;
		    }

		  case GETSERV:
		    {
		      std::cout << "CGPRSgw::HandleSignalConnections : GETSERV" << std::endl;
		      int errorType = 0;
		      unsigned int targetId = tempSignal.iData;

		      if(!HandleServiceList(tempSignal.iData, i, errorType))
			{
			  //target can't be found from list
			  if(errorType == 0)
			    {
			      //send 0 services to device which initiated GETSERV
			      //check presence before
			      if(!CheckPresence(i))
				{
				  pthread_mutex_unlock(&iLock);
				  std::cout << "HandleSignalConnections : original Device is not available anymore" << std::endl;
				  NoticeError(i);
				  break;
				}
			      pthread_mutex_unlock(&iLock);

			      CGPRSSignal sendSignal;
			      sendSignal.iMessageType = DEVICEINFO;
			      sendSignal.iOriginalId = tempSignal.iData;
			      sendSignal.iData = 0;
			      sendSignal.iData2 = 0;
			      sendSignal.iServiceName = std::string("Generic Peerhood device");

			      if(!EncapsulateSignal(i, sendSignal))
				{
				  std::cout << "HandleSignalConnections : Encapsulate DEVICEINFO failed" << std::endl;
				  NoticeError(i);
				  break;
				}

			      //send 0 services
			      sendSignal.iMessageType = REPSERV;
			      sendSignal.iOriginalId = tempSignal.iData;
			      sendSignal.iData = 0;
			      sendSignal.iData2 = 0;
			      sendSignal.iServiceName = "";

			      if(!EncapsulateSignal(i, sendSignal))
				{
				  std::cout << "HandleSignalConnections : Encapsulate REPSERV failed" << std::endl;
				  NoticeError(i);
				  break;
				}

			    }

			  //source can't be found from list
			  if(errorType == 3)
			    NoticeError(i);

			  //target has failed
			  if(errorType == 1)
			    {
			      std::cout << "CGPRSgw::HandleSignalConnections : GETSERV, service target has failed" << std::endl;

			      unsigned int targetFd = 0;
			      bool found = false;
			      pthread_mutex_lock(&iLock);
			      //search targetId and remove
			      for (std::list<CGPRSDevice *>::iterator j = iDeviceList.begin();j != iDeviceList.end();++j) {
				if (((*j)->iId) == targetId)
				  {
				    targetFd = (*j)->iSignalFd;
				    found = true;
				    break;
				  }
			      }
			      pthread_mutex_unlock(&iLock);

			      if(found)
				NoticeError(targetFd);
			      else
				std::cout << "CGPRSgw::HandleSignalConnections : GETSERV, service target has failed, target can't be found anymore from the list" << std::endl;

			      //send 0 services to device which initiated GETSERV
			      //send probe before real data
			      if(!CheckPresence(i))
				{
				  std::cout << "HandleSignalConnections : Original device is not available anymore" << std::endl;
				  NoticeError(i);
				  break;
				}

			      CGPRSSignal sendSignal;
			      sendSignal.iMessageType = DEVICEINFO;
			      sendSignal.iOriginalId = tempSignal.iData;
			      sendSignal.iData = 0;
			      sendSignal.iData2 = 0;
			      sendSignal.iServiceName = std::string("Generic Peerhood device");

			      if(!EncapsulateSignal(i, sendSignal))
				{
				  std::cout << "HandleSignalConnections : Encapsulate DEVICEINFO failed" << std::endl;
				  NoticeError(i);
				  break;
				}

			      //send 0 services
			      sendSignal.iMessageType = REPSERV;
			      sendSignal.iOriginalId = tempSignal.iData;
			      sendSignal.iData = 0;
			      sendSignal.iData2 = 0;
			      sendSignal.iServiceName = "";

			      if(!EncapsulateSignal(i, sendSignal))
				{
				  std::cout << "HandleSignalConnections : Encapsulate REPSERV failed" << std::endl;
				  NoticeError(i);
				  break;
				}
			    }

			  //sender has failed
			  if(errorType == 2)
			    {
			      std::cout << "CGPRSgw::Start : SendServiceList failed" << std::endl;
			      NoticeError(i);
			    }
			  break;
			}
		      break;
		    }

		  }//switch case
		}//else
	    }//old connections else
	}//if (FD_ISSET(i

    }//for
  }//while (iLoop) {

  for (std::list<CGPRSDevice *>::iterator i = iDeviceList.begin();i != iDeviceList.end();++i)
    {
      std::list<CGPRSDevice *>::iterator sentenced = i++;
      delete *sentenced;
      iDeviceList.erase(sentenced);
    }

  StopDataConnections();
  std::cout << "CGPRSgw::HandleSignalConnections : stopping signal connections" << std::endl;
  return true;
}

/**
 * @memo Tests connection if there is enough bytes to be read
 * @doc Tests connection if there is enough bytes to be read. This method
 * test a TCP connection using ioctl to check how many bytes are pending
 * (available for reading). This must be checked because MSG_WAITALL
 * option is used on many situations. MSG_WAITALL is used because
 * WLAN transfers tend to fail without this.
 *
 * @param aFd Target file descriptor
 * @param aBytes Number of expected bytes
 * @return false on error situations
 */
bool CGPRSgw::TestConnection(unsigned int aFd, unsigned int aBytes)
{
  int nPeek = 0;
  int testBuffer = 0;
  nPeek = recv(aFd, &testBuffer, sizeof(testBuffer),  MSG_PEEK);
  std::cout << "CGPRSgw::TestConnection : MSG_PEEK gives " << nPeek << " bytes" << std::endl;

  unsigned long pending = 0;
  int errvalue = 0;
  if(ioctl(aFd, FIONREAD, &pending) == -1)
    {
      std::cerr << "CGPRSgw::TestConnection : ioctl failed " << std::endl;
      errvalue = errno;
      printf( "CGPRSgw::TestConnection : The error generated was %d\n", errvalue );
      printf( "CGPRSgw::TestConnection : That means: %s\n", strerror( errvalue ) );
      return false;
    }

  if(pending < aBytes)
    {
      std::cout << "CGPRSgw::TestConnection : there is only " << pending << " bytes pending while expecting " << aBytes << std::endl;
      return false;
    }

  std::cout << "CGPRSgw::TestConnection : there is " << pending << " bytes pending" << std::endl;
  return true;
}

/**
 * @memo Routine can be used to mark device as "bad".
 * @doc This routine goes through device list and
 * marks specific device (using Signalling Connection fd as identifier)
 * to be erased later.
 *
 * @param aFd Target file descriptor
 * @return none
 */
void CGPRSgw::NoticeError(int aFd)
{
  pthread_mutex_lock(&iLock);
  for (std::list<CGPRSDevice *>::iterator j = iDeviceList.begin();j != iDeviceList.end();++j) {
    if (((*j)->iSignalFd) == aFd)
      {
	(*j)->iStatus = 0;
	break;
      }
  }
  pthread_mutex_unlock(&iLock);
}

/**
 * @memo Routine can be used to mark device as "bad".
 * @doc This routine goes through device list and
 * marks specific device (using Data Signalling Connection fd as identifier)
 * to be erased later.
 *
 * @param aFd Target file descriptor
 * @return none
 */
void CGPRSgw::NoticeDataError(int aFd)
{
  pthread_mutex_lock(&iLock);
  for (std::list<CGPRSDevice *>::iterator j = iDeviceList.begin();j != iDeviceList.end();++j) {
    if (((*j)->iDataFd) == aFd)
      {
	(*j)->iStatus = 0;
	break;
      }
  }
  pthread_mutex_unlock(&iLock);
}


/**
 * @memo Routine for receiving signaling messages.
 * @doc Routine for receiving signaling messages.
 *
 *�@param aSourceFd Source file descriptor
 * @param aSignal Reference for CGPRSSignal class which will hold the received message
 * @return false if something failed
 */
bool CGPRSgw::DecapsulateSignal(int aSourceFd, CGPRSSignal& aSignal)
{
  int bufferSize = 0;
  char *buffer;
  fd_set testSet;
  struct timeval timeVal;

  pthread_mutex_lock(&iLock);
  for (std::list<CGPRSDevice *>::iterator i = iDeviceList.begin();i != iDeviceList.end();++i) {
    if (((*i)->iSignalFd) == aSourceFd)
      {
	if((*i)->iStatus == 0)
	  {
	    std::cout << "CGPRSgw::DecapsulateSignal : status is ZERO" << std::endl;
	    pthread_mutex_unlock(&iLock);
	    return false;
	  }
      }
  }
  pthread_mutex_unlock(&iLock);

  timeVal.tv_sec = 3;
  timeVal.tv_usec = 0;

  FD_ZERO(&testSet);
  FD_SET(aSourceFd, &testSet);

  if(select(aSourceFd + 1, &testSet, NULL, NULL, &timeVal) ==-1){
    std::cerr << "CGPRSgw::DecapsulateSignal : select failed" << std::endl;
    return false;
  }

  if(FD_ISSET(aSourceFd, &testSet))
    {
      if(!TestConnection(aSourceFd, sizeof(bufferSize)))
	{
	  std::cout << "CGPRSgw::DecapsulateSignal : TestConnection failed" << std::endl;
	  return false;
	}

      //receive data length
      if(recv(aSourceFd, &bufferSize, sizeof(bufferSize), MSG_WAITALL)<=0)
	{
	  std::cout << "CGPRSgw::DecapsulateSignal : receiving data length failed" << std::endl;
	  return false;
	}
            if(FD_ISSET(aSourceFd, &testSet))
	      {
		bufferSize = ntohl(bufferSize);

		if(!TestConnection(aSourceFd, bufferSize))
		  {
		    std::cout << "CGPRSgw::DecapsulateSignal : TestConnection failed" << std::endl;
		    return false;
		  }

		buffer = new char[bufferSize];
		memset(buffer, 0, bufferSize);

		//receive data
		if(recv(aSourceFd, buffer, bufferSize, MSG_WAITALL)<=0)
		  {
		    perror("CGPRSPlugin::DecapsulateSignal : receiving data failed");
		    delete[] buffer;
		    return false;
		  }

		memcpy(&aSignal.iMessageType, &buffer[0], sizeof(int));
		memcpy(&aSignal.iOriginalId, &buffer[sizeof(int)], sizeof(int));
		memcpy(&aSignal.iData, &buffer[2*sizeof(int)], sizeof(int));
		memcpy(&aSignal.iData2, &buffer[3*sizeof(int)], sizeof(int));
		aSignal.iServiceName = std::string(&buffer[4*sizeof(int)]);
		delete[] buffer;
		return true;
	      }
    }

  return false;
}

/**
 * @memo Routine for sending signaling messages.
 * @doc Routine for sending signaling messages.
 *
 *�@param aTargetFd Target file descriptor
 * @param aSignal Reference to CGPRSSignal class which holds the message to be sent
 * @return false if something failed
 */
bool CGPRSgw::EncapsulateSignal(int aTargetFd, CGPRSSignal& aSignal)
{
  int bufferSize = 0;
  char *buffer;

  pthread_mutex_lock(&iLock);
  for (std::list<CGPRSDevice *>::iterator i = iDeviceList.begin();i != iDeviceList.end();++i) {
    if (((*i)->iSignalFd) == aTargetFd)
      {
	if((*i)->iStatus == 0)
	  {
	    std::cout << "CGPRSgw::DecapsulateSignal : status is ZERO" << std::endl;
	    pthread_mutex_unlock(&iLock);
	    return false;
	  }
      }
  }
  pthread_mutex_unlock(&iLock);

  bufferSize = 4*sizeof(int);
  bufferSize += aSignal.iServiceName.length()+1;

  buffer = new char[bufferSize];
  memset(buffer, 0, bufferSize);
  memcpy(&buffer[0], &aSignal.iMessageType, sizeof(int));
  memcpy(&buffer[sizeof(int)], &aSignal.iOriginalId, sizeof(int));
  memcpy(&buffer[2*sizeof(int)], &aSignal.iData, sizeof(int));
  memcpy(&buffer[3*sizeof(int)], &aSignal.iData2, sizeof(int));
  strcpy(&buffer[4*sizeof(int)], aSignal.iServiceName.c_str());
  strcat(buffer, "\0");

  //send length
  bufferSize = htonl(bufferSize);

  if(write(aTargetFd, &bufferSize, sizeof(bufferSize)) <= 0){
    perror("CGPRSPlugin::EncapsulateSignal : failed to send length");
    delete[] buffer;
    return false;
  }

  bufferSize = ntohl(bufferSize);

  //send data
  if(write(aTargetFd, buffer, bufferSize) <= 0){
    perror("CGPRSPlugin::EncapsulateSignal : failed to send");
    delete[] buffer;
    return false;
  }

  delete[] buffer;
  return true;
}

/**
 * @memo Sends service list when requested.
 * @doc Sends service list when requested. Routine takes care of requesting
 * service list from target device and forwarding the list to the device which
 * initiated the service query.
 *
 * @param aTargetId Target devices ID (the device which is wanted to be discovered)
 * @param aSourceFd Source devices Signalling Connection file descriptor (the device which initiated the discovery)
 * @param aErrorType errortype which happened during service query
 * @return false if something failed
 */
bool CGPRSgw::HandleServiceList(unsigned int aTargetId, unsigned int aSourceFd, int &aErrorType)
{
  unsigned short numberOfServices = 0;
  char *msgBuf;
  unsigned int length = 0;
  bool found = false;
  struct timeval timeVal;
  fd_set testSet;
  int targetFd = 0;

    //search targetFd
  pthread_mutex_lock(&iLock);
  for (std::list<CGPRSDevice *>::iterator i = iDeviceList.begin();i != iDeviceList.end();++i) {
    std::cout << "searching for " << (*i)->iId << " from " << aTargetId << std::endl;
    if (((*i)->iId) == aTargetId)
      {

	if((*i)->iSignalFd > 0)
	  {
	    targetFd = (*i)->iSignalFd;
	    found =true;
	    break;
	  }
      }
  }
  pthread_mutex_unlock(&iLock);

  if(found == false)
    {
      std::cout << "CGPRSgw::HandleServiceList : ID " << aTargetId << " was not found for that fd!!" << std::endl;
      aErrorType = 0;
      return false;
    }


  if(!CheckPresence(targetFd))
    {
      std::cout << "CGPRSgw::HandleServiceList : Device " <<  aTargetId << " is not available anymore" << std::endl;
      aErrorType = 1;
      return false;
    }


  CGPRSSignal sendSignal;
  sendSignal.iMessageType = GETSERV;
  sendSignal.iOriginalId = 0;
  sendSignal.iData = 0;
  sendSignal.iData2 = 0;
  sendSignal.iServiceName = "";

  if(!EncapsulateSignal(targetFd, sendSignal))
    {
      std::cout << "HandleServiceList : Encapsulate failed" << std::endl;
      aErrorType = 1;
      return false;
    }

  timeVal.tv_sec = 3;
  timeVal.tv_usec = 0;

  FD_ZERO(&testSet);
  FD_SET(targetFd, &testSet);

  if(select(targetFd + 1, &testSet, NULL, NULL, &timeVal) ==-1){
    std::cerr << "CGPRSgw::HandleServiceList : select failed" << std::endl;
    aErrorType = 1;
    return false;
  }

  if(FD_ISSET(targetFd, &testSet))
    {
      CGPRSSignal tempSignal;
      if(!DecapsulateSignal(targetFd, tempSignal))
	{
	  std::cout << "CGPRSPlugin::HandleServiceList : Decapsulate GETSERVREP failed" << std::endl;
	  aErrorType = 1;
	  return false;
	}

      if(tempSignal.iMessageType != GETSERVREP)
	{
	  std::cout << "CGPRSPlugin::HandleServiceList : Decapsulate GETSERVREP failed, received " << tempSignal.iMessageType << std::endl;
	  aErrorType = 1;
	  return false;
	}

      if(FD_ISSET(targetFd, &testSet))
	{
	  int realLength = 0;

	  if(!TestConnection(targetFd, sizeof(realLength)))
	    {
	      std::cout << "CGPRSgw::HandleServiceList : TestConnection failed" << std::endl;
	      aErrorType = 1;
	      return false;
	    }

	  if(recv(targetFd, &realLength, sizeof(realLength), MSG_WAITALL) <= 0)
	    {
	      std::cout << "CGPRSPlugin::HandleServiceList : receiving device name length failed" << std::endl;
	      aErrorType = 1;
	      return false;
	    }

	  realLength = ntohl(realLength);

	  if(FD_ISSET(targetFd, &testSet))
	    {
	      if(!TestConnection(targetFd, realLength))
		{
		  std::cout << "CGPRSgw::HandleServiceList : TestConnection failed" << std::endl;
		  aErrorType = 1;
		  return false;
		}

	      if(recv(targetFd, &length, realLength, MSG_WAITALL) <= 0)
		{
		  std::cout << "CGPRSPlugin::HandleServiceList : receiving device name length failed" << std::endl;
		  aErrorType = 1;
		  return false;
		}

	      if(FD_ISSET(targetFd, &testSet))
		{
		  if(!TestConnection(targetFd, sizeof(realLength)))
		    {
		      std::cout << "CGPRSgw::HandleServiceList : TestConnection failed" << std::endl;
		      aErrorType = 1;
		      return false;
		    }

		  length = ntohl(length);
		  msgBuf = new char[length];
		  memset(msgBuf, 0, length);

		  if(recv(targetFd, &realLength, sizeof(realLength), MSG_WAITALL) <= 0)
		    {
		      std::cout << "CGPRSPlugin::HandleServiceList : receiving device name failed" << std::endl;
		      aErrorType = 1;
		      delete[] msgBuf;
		      return false;
		    }

		  realLength = ntohl(realLength);

		  if(FD_ISSET(targetFd, &testSet))
		    {
		      if(!TestConnection(targetFd, realLength))
			{
			  std::cout << "CGPRSgw::HandleServiceList : TestConnection failed" << std::endl;
			  aErrorType = 1;
			  return false;
			}

		      if(recv(targetFd, msgBuf, realLength, MSG_WAITALL) <= 0)
			{
			  std::cout << "CGPRSPlugin::HandleServiceList : receiving device name failed" << std::endl;
			  aErrorType = 1;
			  delete[] msgBuf;
			  return false;
			}

		      std::string deviceName(msgBuf);
		      delete[] msgBuf;

		      std::cout << "CGPRSPlugin::HandleServiceList : received device name " << deviceName << std::endl;

		      if(FD_ISSET(targetFd, &testSet))
			{
			  if(!TestConnection(targetFd, sizeof(realLength)))
			    {
			      std::cout << "CGPRSgw::HandleServiceList : TestConnection failed" << std::endl;
			      aErrorType = 1;
			      return false;
			    }

			   if(recv(targetFd, &realLength, sizeof(realLength), MSG_WAITALL) <= 0)
			     {
			       std::cout << "CGPRSPlugin::HandleServiceList : receiving device name checksum length failed" << std::endl;
			       aErrorType = 1;
			       return false;
			     }

			   realLength = ntohl(realLength);

			   if(FD_ISSET(targetFd, &testSet))
			     {
			       if(!TestConnection(targetFd, realLength))
				 {
				   std::cout << "CGPRSgw::HandleServiceList : TestConnection failed" << std::endl;
				   aErrorType = 1;
				   return false;
				 }

			       if(recv(targetFd, &length, realLength, MSG_WAITALL) <= 0)
				 {
				   std::cout << "CGPRSPlugin::HandleServiceList : receiving device name checksum" << std::endl;
				   aErrorType = 1;
				   return false;
				 }

			       length = ntohl(length);

			       std::cout << "CGPRSPlugin::HandleServiceList : received device name checksum " << length << std::endl;

			       //send probe before real data
			       if(!CheckPresence(aSourceFd))
				 {
				   aErrorType = 2;
				   return false;
				 }

			       sendSignal.iMessageType = DEVICEINFO;
			       sendSignal.iOriginalId = aTargetId;
			       sendSignal.iData = length;
			       sendSignal.iData2 = 0;
			       sendSignal.iServiceName = std::string(deviceName);

			       if(!EncapsulateSignal(aSourceFd, sendSignal))
				 {
				   std::cout << "HandleServiceList : Encapsulate DEVICEINFO failed" << std::endl;
				   aErrorType = 2;
				   return false;
				 }

			       if(FD_ISSET(targetFd, &testSet))
				 {
				   if(!TestConnection(targetFd, sizeof(realLength)))
				     {
				       std::cout << "CGPRSgw::HandleServiceList : TestConnection failed" << std::endl;
				       aErrorType = 1;
				       return false;
				     }

				   //receive number of services
				   realLength = 0;
				   if(recv(targetFd, &realLength, sizeof(realLength), MSG_WAITALL) <= 0)
				     {
				       std::cout << "CGPRSPlugin::HandleServiceList : receiving realLength failed" << std::endl;
				       aErrorType = 1;
				       return false;
				     }

				   if(FD_ISSET(targetFd, &testSet))
				     {
				       realLength = ntohl(realLength);

				       if(!TestConnection(targetFd, realLength))
					 {
					   std::cout << "CGPRSgw::HandleServiceList : TestConnection failed" << std::endl;
					   aErrorType = 1;
					   return false;
					 }

				       if(recv(targetFd, &numberOfServices, realLength, MSG_WAITALL) <=0)
					 {
					   std::cout << "CGPRSPlugin::HandleServiceList : receiving the numberOfServices realLength failed :" << std::endl;
					   aErrorType = 1;
					   return false;
					 }

				       numberOfServices = ntohs(numberOfServices);
				       std::cout << "Received " << numberOfServices << " services " << std::endl;
				       //send probe before real data

				       if(!CheckPresence(aSourceFd))
					 {
					   std::cout << "CGPRSgw::HandleServiceList : original device is not available anymore" << std::endl;
					   aErrorType = 2;
					   return false;
					 }


				       sendSignal.iMessageType = REPSERV;
				       sendSignal.iOriginalId = aTargetId;
				       sendSignal.iData = numberOfServices;
				       sendSignal.iData2 = 0;
				       sendSignal.iServiceName = "";

				       if(!EncapsulateSignal(aSourceFd, sendSignal))
					 {
					   std::cout << "HandleServiceList : Encapsulate REPSERV failed" << std::endl;
					   aErrorType = 2;
					   return false;
					 }

				       //receive services
				       for(unsigned short j=0; j<numberOfServices; j++)
					 {
					   if(FD_ISSET(targetFd, &testSet))
					     {
					       if(!TestConnection(targetFd, sizeof(realLength)))
						 {
						   std::cout << "CGPRSgw::HandleServiceList : TestConnection failed" << std::endl;
						   aErrorType = 1;
						   return false;
						 }

					       if(recv(targetFd, &realLength, sizeof(realLength), MSG_WAITALL) <= 0)
						 {
						   std::cout << "HandleServiceList, receiving realLength failed" << std::endl;
						   aErrorType = 1;
						   return false;
						 }

					       realLength = ntohl(realLength);

					       if(FD_ISSET(targetFd, &testSet))
						 {
						   if(!TestConnection(targetFd, realLength))
						     {
						       std::cout << "CGPRSgw::HandleServiceList : TestConnection failed" << std::endl;
						       aErrorType = 1;
						       return false;
						     }

						   if(recv(targetFd, &length, realLength, MSG_WAITALL)<=0)
						     {
						       std::cout << "HandleServiceList, receiving the service length failed :" << std::endl;
						       aErrorType = 1;
						       return false;
						     }

						   length = ntohl(length);
						   std::cout << "Received service size " << length << std::endl;

						   if(FD_ISSET(targetFd, &testSet))
						     {
						       if(!TestConnection(targetFd, sizeof(realLength)))
							 {
							   std::cout << "CGPRSgw::HandleServiceList : TestConnection failed" << std::endl;
							   aErrorType = 1;
							   return false;
							 }

						       //receive service
						       msgBuf = new char[length];
						       memset(msgBuf, 0, length);

						       if(recv(targetFd, &realLength, sizeof(realLength), MSG_WAITALL)<=0)
							 {
							   std::cerr << "HandleServiceList, receiving the numberOfServices realLength failed :" << std::endl;
							   aErrorType = 1;
							   return false;
							 }

						       realLength = ntohl(realLength);

						       if(FD_ISSET(targetFd, &testSet))
							 {

							   if(!TestConnection(targetFd, realLength))
							     {
							       std::cout << "CGPRSgw::HandleServiceList : TestConnection failed" << std::endl;
							       aErrorType = 1;
							       return false;
							     }


							   if(recv(targetFd, msgBuf, realLength, MSG_WAITALL)<=0)
							     {
							       std::cerr << "HandleServiceList, receiving the service failed :" << std::endl;
							       aErrorType = 1;
							       delete[] msgBuf;
							       return false;
							     }

							   //send probe before real data
							   if(!CheckPresence(aSourceFd))
							     {
							       std::cout << "CGPRSgw::HandleServiceList : original device is not available anymore" << std::endl;
							       aErrorType = 2;
							       return false;
							     }

							   CGPRSSignal sendSignal;
							   sendSignal.iMessageType = SERVICE;
							   sendSignal.iOriginalId = aTargetId;
							   sendSignal.iData = realLength;
							   sendSignal.iData2 = 0;
							   sendSignal.iServiceName = "";

							   if(!EncapsulateSignal(aSourceFd, sendSignal))
							     {
							       std::cout << "HandleServiceList : Encapsulate failed" << std::endl;
							       aErrorType = 2;
							       return false;
							     }

							   //send service
							   if(write(aSourceFd, msgBuf, realLength)<=0)
							     {
							       std::cerr << "HandleServiceList, sending the service failed :" << std::endl;
							       aErrorType = 2;
							       delete[] msgBuf;
							       return false;
							     }

							   delete[] msgBuf;

							 }
						       else
							 {
							   std::cout << "HandleServiceList : timeout while receiving msgBuf" << std::endl;
							   delete[] msgBuf;
							 }
						     }
						   else
						     {
						       std::cout << "HandleServiceList : timeout while receiving realLength for msgBuf" << std::endl;
						     }
						 }
					       else
						 {
						   std::cout << "HandleServiceList : timeout while receiving Length for msgBuf" << std::endl;
						 }
					     }
					   else
					     {
					       std::cout << "HandleServiceList : timeout while receiving realLength for length" << std::endl;
					     }
					 }// for(unsigned short j=0; j<numberOfServices; j++)
				       return true;

				     }
				   else
				     {
				       std::cout << "HandleServiceList : timeout while receiving number of services" << std::endl;
				     }
				 }
			       else
				 {
				   std::cout << "HandleServiceList : timeout while receiving realLength for number of services" << std::endl;
				 }
			     }
			   else
			     {
			       std::cout << "HandleServiceList : timeout while receiving checksum" << std::endl;
			     }
			 }
		       else
			 {
			   std::cout << "HandleServiceList : timeout while receiving realLength for checksum" << std::endl;
			 }
		    }
		  else
		    {
		      std::cout << "HandleServiceList : timeout while receiving device name" << std::endl;
		      delete[] msgBuf;
		    }
		}
	      else
		{
		  std::cout << "HandleServiceList : timeout while receiving realLength for device name" << std::endl;
		}
	    }
	  else
	    {
	      std::cout << "HandleServiceList : timeout while receiving device name length" << std::endl;
	    }
	}
      else
	{
	  std::cout << "HandleServiceList : timeout while receiving realLength for device name length" << std::endl;
	}
    }
  else
    {
      std::cout << "HandleServiceList : timeout while receiving GETSERVREP signal" << std::endl;
    }

  aErrorType = 1;
  return false;
}


/**
 * @memo Checks if the found device is new (i.e. not detected before).
 * @doc Checks if the found device is new (i.e. not detected before). If this
 * is the case then it is added to plugin's internal device list and the common
 * device & service database should be updated later on.
 *
 * @param aId ID -number of the device to be added.
 * @param aFd FileDescriptor of device (Signalling Connection)
 *
 * @return true if the checked device is new
 */
bool CGPRSgw::AddDevice(unsigned int aId, unsigned int aFd)
{
  pthread_mutex_lock(&iLock);
  std::list<CGPRSDevice *>::iterator i;

  for (i = iDeviceList.begin();i != iDeviceList.end();++i) {
    if (((*i)->iId) == aId)
      {
	pthread_mutex_unlock(&iLock);
	return false;
      }
  }
  pthread_mutex_unlock(&iLock);

  CGPRSDevice *tempDevice = new CGPRSDevice;
  tempDevice->iId = aId;
  tempDevice->iSignalFd = aFd;
  tempDevice->iDataFd = 0;

  pthread_mutex_lock(&iLock);
  iDeviceList.push_back(tempDevice);
  pthread_mutex_unlock(&iLock);

  return true;
}

/**
 * @memo Removes the device from device database.
 * @doc Removes the device from device database.
 *
 * @param aFd The filedescriptor (Signalling Connection) of the device.
 *
 * @return false if device is not found
 */
bool CGPRSgw::RemoveDevice(unsigned int aFd)
{
  std::cout << "CGPRSgw::RemoveDevice : starting..." << std::endl;
  pthread_mutex_lock(&iLock);
  std::list<CGPRSDevice *>::iterator i;
  for (i = iDeviceList.begin();i != iDeviceList.end();++i) {
    if (((*i)->iSignalFd) == (int)aFd)
      {
	close((*i)->iSignalFd);
	FD_CLR((*i)->iSignalFd, &masterSet);

	/*
	std::cout << "deleting signal object..." << std::endl;
	if(!DeleteConnectionObject(i->iSignalFd))
	  std::cout << "CGPRSgw::RemoveDevice : Failed to delete signal connection object" << std::endl;
	std::cout << "deleting service object..." << std::endl;
	if(!DeleteConnectionObject(i->iServiceFd))
	  std::cout << "CGPRSgw::RemoveDevice : Failed to delete service connection object" << std::endl;
	*/

	iDeviceList.erase(i);
	pthread_mutex_unlock(&iLock);
	return true;
      }
  }
  pthread_mutex_unlock(&iLock);
  return false;
}

/**
 * @memo Deletes a connection object
 * @doc Deletes a connection object.
 *
 * @param aFd The filedescriptor (Signalling Connection) of the device.
 *
 * @return false if device is not found
 */
bool CGPRSgw::DeleteConnectionObject(unsigned int aFd)
{
  std::list<MAbstractConnection*>::iterator j;
  for (j = iConnectionObjects.begin();j != iConnectionObjects.end();++j) {
    if((unsigned int)(*j)->GetFd() == aFd)
      {
	delete (*j);
	iConnectionObjects.erase(j);
	std::cout << "CGPRSgw::DeleteConnectionObject : removed connection object fd " << aFd << std::endl;
	return true;
      }
  }
  std::cout << "CGPRSgw::DeleteConnectionObjec : can't find connection object" << std::endl;
  return false;
}

/**
 * @memo Deletes a connection object
 * @doc Deletes a connection object.
 *
 * @param aFd The filedescriptor (Data Signalling Connection) of the device.
 *
 * @return false if device is not found
 */
bool CGPRSgw::DeleteDataConnectionObject(unsigned int aFd)
{
  std::list<MAbstractConnection*>::iterator j;
  for (j = iDataConnectionObjects.begin();j != iDataConnectionObjects.end();++j) {
    if((unsigned int)(*j)->GetFd() == aFd)
      {
	delete (*j);
	iDataConnectionObjects.erase(j);
	std::cout << "CGPRSgw::DeleteDataConnectionObject : removed data connection object fd " << aFd << std::endl;
	return true;
      }
  }
  std::cout << "CGPRSgw::DeleteDataConnectionObject : can't find data connection object" << std::endl;
  return false;
}


/**
 * @memo Sends the device list
 * @doc Sends the device list. If device requests device list, this routine
 * checks for the state of the connections found from the device database. If
 * connection is bad it is not sended.
 *
 * @param aTargetFd File descriptor of the device which initiated the device inquiry
 * @param aTargetId ID of the device which initiated the device inquiry
 * @param aTempSet Fd_set for testing connections.
 * service list. This parameter cannot be NULL.
 *
 * @return true if the services were sent succesfully
 */
bool CGPRSgw::SendDeviceList(unsigned int aTargetFd, unsigned int aTargetId, fd_set &aTempSet)
{
  std::list<int> tempList;

  //test connections and mark bad if broken
  pthread_mutex_lock(&iLock);
  for(std::list<CGPRSDevice *>::iterator j = iDeviceList.begin();j != iDeviceList.end();++j) {
    if((*j)->iId != aTargetId)
      {
	pthread_mutex_unlock(&iLock);
	if(!CheckPresence((*j)->iSignalFd))
	  {
	    pthread_mutex_lock(&iLock);
	    std::cout << "CGPRSgw::SendDeviceList : Device " <<  (*j)->iId << " is not available anymore" << std::endl;
	    (*j)->iStatus = 0;
	    pthread_mutex_unlock(&iLock);
	  }
	else
	  {
	    pthread_mutex_lock(&iLock);
	    tempList.push_back((*j)->iId);
	    pthread_mutex_unlock(&iLock);
	  }
	pthread_mutex_lock(&iLock);
      }
  }
  pthread_mutex_unlock(&iLock);


  //check original sender connection
  pthread_mutex_lock(&iLock);
  for(std::list<CGPRSDevice *>::iterator j = iDeviceList.begin();j != iDeviceList.end();++j) {
    if((*j)->iId == aTargetId)
      {
	pthread_mutex_unlock(&iLock);
	if(!CheckPresence((*j)->iSignalFd))
	  {
	    std::cout << "CGPRSgw::SendDeviceList : Device " <<  (*j)->iId << " is not available anymore" << std::endl;
	    (*j)->iStatus = 0;
	    return false;
	  }
	pthread_mutex_lock(&iLock);
	break;
      }
  }
  pthread_mutex_unlock(&iLock);


  //send number of devices
  CGPRSSignal sendSignal;
  sendSignal.iMessageType = REPDEV;
  sendSignal.iOriginalId = 0;
  sendSignal.iData = tempList.size();
  sendSignal.iData2 = 0;
  sendSignal.iServiceName = "";

  std::cout << "Sending REPDEV to " << aTargetFd << std::endl;

  if(!EncapsulateSignal(aTargetFd, sendSignal))
    {
      std::cout << "CGPRSgw::SendDeviceList : Encapsulate REPDEV failed" << std::endl;
      return false;
    }

  //send devices
  for (std::list<int>::iterator j = tempList.begin();j != tempList.end();++j)
    {
      std::cout << "CGPRSgw::SendDeviceList : Sending device " << (*j) << " to id " << aTargetId << std::endl;
      sendSignal.iMessageType = DEVICE;
      sendSignal.iOriginalId = 0;
      sendSignal.iData = (*j);
      sendSignal.iData2 = 0;
      sendSignal.iServiceName = "";

      if(!EncapsulateSignal(aTargetFd, sendSignal))
	{
	  std::cout << "CGPRSgw::SendDeviceList : Encapsulate DEVICE failed" << std::endl;
	  return false;
	}
    }

    return true;
}

/**
 * @memo Checks the presence of the device
 * @doc Checks the presence of the device. This routine checks the connections and devices
 * using PROBE -messages. If the device doesn't respond in specific time using REPPROBE
 * device and connection are considered as not-valid.
 *
 * @param aTargetFd File Descriptor of the device (Signalling Connection)
 * service list. This parameter cannot be NULL.
 *
 * @return true if the services were sent succesfully
 */
bool CGPRSgw::CheckPresence(int aTargetFd)
{
 struct timeval timeVal;
 fd_set testSet;

 pthread_mutex_lock(&iLock);
 for (std::list<CGPRSDevice *>::iterator i = iDeviceList.begin();i != iDeviceList.end();++i) {
   if (((*i)->iSignalFd) == aTargetFd)
     {
       if((*i)->iStatus == 0)
	 {
	   std::cout << "CGPRSgw::CheckPresence : status is ZERO" << std::endl;
	   pthread_mutex_unlock(&iLock);
	   return false;
	 }

       if((*i)->iSignalFd == 0)
	 {
	   std::cout << "CGPRSgw::CheckPresence : this fd is not yet ready" << std::endl;
	   pthread_mutex_unlock(&iLock);
	   return true;
	 }
     }
 }
 pthread_mutex_unlock(&iLock);

 CGPRSSignal tempSignal;
 tempSignal.iMessageType = PROBE;
 tempSignal.iOriginalId = 0;
 tempSignal.iData = 0;
 tempSignal.iData2 = 0;
 tempSignal.iServiceName = "";

 std::cout << "CGPRSgw::CheckPresence : sending PROBE to " << aTargetFd << std::endl;

 if(!EncapsulateSignal(aTargetFd, tempSignal))
   {
     std::cout << "CGPRSgw::CheckPresence : Encapsulate PROBE failed" << std::endl;
     return false;
   }

 timeVal.tv_sec = 10;
 timeVal.tv_usec = 0;

 FD_ZERO(&testSet);
 FD_SET(aTargetFd, &testSet);

 if(select(aTargetFd + 1, &testSet, NULL, NULL, &timeVal) ==-1){
   std::cerr << "CGPRSgw::CheckPresence : select failed" << std::endl;
   return false;
 }

 int bufferSize = 0;
 int testBuffer = 0;
 if(FD_ISSET(aTargetFd, &testSet))
   {
     if(!TestConnection(aTargetFd, sizeof(bufferSize)))
       {
	 std::cerr << " CGPRSgw::CheckPresence : TestConnection ailed" << std::endl;
	 return false;
       }

     if(recv(aTargetFd, &bufferSize, sizeof(bufferSize),  MSG_WAITALL)<=0)
       {
	 std::cerr << " CGPRSgw::CheckPresence : receiving data length failed" << std::endl;
	 return false;
       }

     bufferSize = ntohl(bufferSize);

     if(FD_ISSET(aTargetFd, &testSet))
       {
	 if(!TestConnection(aTargetFd, bufferSize))
	   {
	     std::cerr << " CGPRSgw::CheckPresence : TestConnection ailed" << std::endl;
	     return false;
	   }

	 char *buffer = new char[bufferSize];
	 memset(buffer, 0, bufferSize);

	 //receive data
	 if(recv(aTargetFd, buffer, bufferSize, MSG_WAITALL)<=0)
	   {
	     std::cerr << " CGPRSgw::CheckPresence : receiving data failed" << std::endl;
	     delete[] buffer;
	     return false;
	   }

	 memcpy(&tempSignal.iMessageType, &buffer[0], sizeof(int));
	 memcpy(&tempSignal.iOriginalId, &buffer[sizeof(int)], sizeof(int));
	 memcpy(&tempSignal.iData, &buffer[2*sizeof(int)], sizeof(int));
	 memcpy(&tempSignal.iData2, &buffer[3*sizeof(int)], sizeof(int));
	 tempSignal.iServiceName = std::string(&buffer[4*sizeof(int)]);
	 delete[] buffer;

	 if(tempSignal.iMessageType == REPPROBE)
	   {
	     std::cout << "CGPRSgw::CheckPresence : received REPPROBE from" << aTargetFd << std::endl;
	     return true;
	   }
	 else
	   {
	     std::cout << "CGPRSgw::CheckPresence : expected REPPROBE, received " << tempSignal.iMessageType << std::endl;
	     return false;
	   }
       }
   }

 std::cout << "CGPRSgw::CheckPresence : timeout" << std::endl;
 return false;
}
