/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/


// Emacs, hail the mighty -*-C++-*-

/**
 * Copyright 2003 LUT. .
 *
 * @name Daemon.h
 * @memo PeerHood's Daemon class. This defines the PeerHood Daemon that is
 * constantly running on the background and gathering information about the
 * device's neigborhood.
 *
 * @version 0.11
 * date     03.04.2003
 * change   23.09.2008
 */

#include <list>
#include "Service.h"

#include <stdio.h>

#include <AbstractConnection.h>
#include "AbstractListener.h"
#include "AbstractStateConverter.h"

#ifndef __DAEMON_H__
#define __DAEMON_H__

/* AARGH */
/*
#define ERR(format, msg...) syslog(LOG_ERR, "ERROR: " format "\n" , ## msg)
*/
#define ERR(format, msg...) printf(format "\n",## msg)


#ifdef PH_DEBUG
#define DBG(format, msg...) syslog(LOG_DEBUG, format "\n" , ## msg)
#else
#define DBG( A... )
#endif

// Structure to hold one client's data
struct ClientData {
  int iPid;
  int iFd;
};

// enumeration for daemon fetched information, define here and implement in different plugins
#define D_GET_NOINFO 0x0000
#define D_GET_DEVICEINFO 0x0001
#define D_GET_SERVICEINFO 0x0002
#define D_GET_PROTOINFO 0x0004
#define D_GET_NEIGHINFO 0x0008


/**
 * @memo PeerHood's Daemon class. This daemon is constantly running on the
 * background and gathering information about nearby devices.
 * @doc This class forms the base part of the PeerHood Daemon that monitors
 * device's neighborhood all the time. The actual monitoring is carried out by
 * several different plugins, one for each networking technology. Information
 * gathered by the plugins is stored in this daemon and delivered to PeerHood
 * objects when requested. The communication between the daemon and PeerHood
 * objects is done via a local socket.
 *
 * @see MAbstractPlugin
 * @see CPeerHood
 */
class CDaemon : public MAbstractStateConverter
{
public:
  CDaemon();
  static CDaemon* GetInstance();
  bool MakeDaemon();
  bool Init();
  void Run();
  void DaemonRevertMode(bool aMode);

  bool SendDeviceInfo(MAbstractConnection* aConnection);
  bool SendServiceList(MAbstractConnection* aConnection);
  bool SendPrototypes(MAbstractConnection* aConnection);
  bool SendNeighborList(MAbstractConnection* aConnection);

  // To get the information what needs to be fecthed
  int GetFetchRequestType();
  
  /******** <MAbstractStateConverter INTERFACE> *************/
  /** 
   * @memo Register system state register 
   * @doc Register a new listener into daemon.
   * 
   * @param aListener The actual listener to be registered
   */
  void RegisterListener(MAbstractListener* aListener);
  
  /**
   * @memo Change activity of daemon
   * @doc Change the state of the daemon from active to passive.
   * Currently used by listeners -> into interface for listeners to use?
   * 
   * @param aActive New state (true - active, false - passive)
   */
  void SetState(bool aActive);
  
  /**
   * @memo Trigger daemon shutdown
   * @doc Set daemon into shutdown state. Currently used by listeners
   * -> into interface for listeners to use?
   */
  void TriggerShutdown();
  
  void SetAdapter(const std::string& aInterface, int aId);
  const std::string& GetAdapter();

  /******** <\MAbstractStateConverter INTERFACE> *************/
  
 private:
  static void SignalHandler(int aSignal);

  bool CreateLocalSocket();
  bool IsUnique();
  int LoadPlugins();
  void SetUpPlugins();
  void StartPlugins();
  void StopPlugins();
  void Shutdown();

  void HandleGetDeviceList(const int aFd);
  void HandleGetLocalServiceList(const int aFd);
  void HandleInsertService(const int aFd, int aPid);
  void HandleRemoveService(const int aFd);
  void HandleGetFreePort(const int aFd);

  bool Write(const int aFd, void *aBuffer, int aLength);

  bool LoadParameter(std::string& aParameter);

  /**
   * @memo Create listener objects
   * @doc
   */
  int CreateListeners(const std::string& aType);
  
  
  /**
   * @memo Connect listener objects to their sources
   * @doc
   */
  int ConnectListeners();
  
 /**
  * @memo Remove event listeners
  * @doc
  */
  void RemoveListeners();
    
// Data concerning this device
  static CDaemon* iInstance;

  // Information concerning this daemon
  std::string iDeviceName;
  unsigned int iChecksum;
  unsigned int iDeviceNameLength;

  // Connection to ?
  int iServerSocket;

  // Plugins in this device
  int iNumberOfPlugins;
  std::list<char*> iSupportedPlugins;

  // Clients using this daemon
  std::list<ClientData*> iClients;

  // Services registered into this daemon
  std::list<CService*> iServiceList;
  
  // Event listeners registered into this daemon
  std::list<MAbstractListener*> iListenerList;
  // NOTE that DeviceStorage contains information concerning neighboring devices
  // DeviceStorage is independent element

  // This info is for deamon to know what to fetch
  int iFetchRequestType;
  
  /* STATES */
  /** @TODO Create a suitable state system & state managing system (enum?) */
  // Plugins running - StartPlugins() should set this true and StopPlugins() false
  bool iPluginsRunning;
  
  // State of the daemon (triggered by device) - set by listener object via callback-function
  bool iDaemonActive;
    
  // Daemon is shutting down
  bool iDaemonShutdown;
  
  // Variable indicating if daemon should stay passive when no clients
  bool iRevertToPassive;
  
  // user mode
  bool iDaemonMode;
};

#endif // __DAEMON_H__
