/**************************************************************************
*   Copyright (C) 2010 by Michael Carpenter (malcom2073)                  *
*   mcarpenter@interforcesystems.com                                      *
*                                                                         *
*   This file is a part of libobd                                         *
*                                                                         *
*   libobd is free software: you can redistribute it and/or modify        *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation, either version 3 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   libobd is distributed in the hope that it will be useful,             *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with libobd.  If not, see <http://www.gnu.org/licenses/>.       *
***************************************************************************/

#include "ObdThread.h"
#include <QVector>
#include <QDateTime>
#include <QMetaType>
#include <QStringList>
ObdThread::ObdThread(QObject *parent) : QThread(parent)
{
	qRegisterMetaType<QList<QString> >("QList<QString>");
	m_obd = new obdLib();
	m_threadRunning = false;
	m_requestLoopRunning = false;
	loopType = 0;
	m_set = 0;
	//threadLockMutex.lock();
	m_obdConnected = false;
	m_baud = 0;
	m_port = "";
	m_obdInfo = new ObdInfo();
	start();
}
void ObdThread::connect()
{
	threadLockMutex.lock();
	RequestClass req;
	req.type = CONNECT;
	m_reqClassList.append(req);
	threadLockMutex.unlock();
}
void ObdThread::start()
{
	m_threadRunning = true;
	QThread::start();
}
void ObdThread::disconnect()
{
	threadLockMutex.lock();
	RequestClass req;
	req.type = DISCONNECT;
	m_reqClassList.append(req);
	threadLockMutex.unlock();

}
void ObdThread::clearReqList()
{
	removePidMutex.lock();
	m_reqClassListThreaded.clear();
	removePidMutex.unlock();
}
void ObdThread::requestTroubleCodes()
{
	qDebug() << "Trouble codes request...";
	threadLockMutex.lock();
	RequestClass req;
	req.type = TROUBLE_CODES;
	m_reqClassList.append(req);
	threadLockMutex.unlock();
}
void ObdThread::singleShotRequest(QByteArray request)
{
	threadLockMutex.lock();
	RequestClass req;
	req.type = RAW_REQUEST;
	req.custom = request;
	m_reqClassList.append(req);
	threadLockMutex.unlock();
}
void ObdThread::blindSingleShotRequest(QByteArray request)
{
	threadLockMutex.lock();
	RequestClass req;
	req.type = RAW_BLIND_REQUEST;
	req.custom = request;
	m_reqClassList.append(req);
	threadLockMutex.unlock();
}
void ObdThread::findObdPort()
{
	threadLockMutex.lock();
	RequestClass req;
	req.type = FIND_PORT;
	m_reqClassList.append(req);
	threadLockMutex.unlock();
}
void ObdThread::removeRequest(int mode, int pid, int priority)
{
	threadLockMutex.lock();
	RequestClass req;
	req.pid = pid;
	req.mode = mode;
	req.priority = priority;
	req.type = MODE_PID;
	m_reqClassRemoveList.append(req);
	threadLockMutex.unlock();
}

void ObdThread::run()
{
	int cycleCounter =0;
	bool first = true;
	m_requestLoopRunning = true;
	std::vector<unsigned char> replyVectorString;
	std::vector<unsigned char> replyVector;
	while (m_threadRunning)
	{
		removePidMutex.lock();
		threadLockMutex.lock();
		for (int i=0;i<m_reqClassRemoveList.count();i++)
		{
			//qDebug() << "Removing pid:" << m_reqClassRemoveList[i].mode << m_reqClassRemoveList[i].pid;
			for (int j=0;j<m_reqClassListThreaded.count();j++)
			{
				if ((m_reqClassRemoveList[i].mode == m_reqClassListThreaded[j].mode) && (m_reqClassRemoveList[i].pid == m_reqClassListThreaded[j].pid) && (m_reqClassRemoveList[i].type == m_reqClassListThreaded[j].type))
				{
					m_reqClassFailureMap.remove(&m_reqClassListThreaded[i]);
					m_reqClassListThreaded.removeAt(j);
					m_reqClassRemoveList.removeAt(i);
					j--;
					i--;
				}
			}
		}
		for (int i=0;i<m_reqClassList.count();i++)
		{
			m_reqClassListThreaded.prepend(m_reqClassList[i]);
			m_reqClassFailureMap[&m_reqClassList[i]] = 0;
			m_whiteList[&m_reqClassList[i]]=false;

		}
		m_reqClassList.clear();
		threadLockMutex.unlock();
		first = true;
		for (int i=0;i<m_reqClassListThreaded.count();i++)
		{
			//Handle request here
			if (m_reqClassListThreaded[i].type == CONNECT)
			{
				if (!m_connect())
					continue;
			}
			else if (m_reqClassListThreaded[i].type == DISCONNECT)
			{
				qDebug() << "Disconnected from ELM";
				m_obd->closePort();
				m_obdConnected = false;
				emit disconnected();

			}
			else if (m_reqClassListThreaded[i].type == CLEAR_TROUBLE_CODES)
			{
				consoleMessage("Trouble Codes Requested...");
				if (!m_obdConnected)
				{
					if (!m_connect())
						continue;
				}
				m_obd->sendObdRequest("04\r",3);
				if (!m_obdConnected)
				{
					emit consoleMessage(QString("Disconnected"));
					m_obd->closePort();
					emit disconnected();
				}
			}
			else if (m_reqClassListThreaded[i].type == TROUBLE_CODES)
			{
				consoleMessage("Trouble Codes Requested...");
				if (!m_obdConnected)
				{
					if (!m_connect())
						continue;
				}
				else
				{
				}

				setHeaders(true);
				std::vector<unsigned char> replyVector;
				//troubleCodes(QList<QString> codes)
				QString vect = "";
				if (!m_obd->sendObdRequestString("ATDPN\r",6,&replyVector))
				{
					qDebug() << "Error retreiving protocol";
				}
				else
				{

					for (int i=0;i<replyVector.size();i++)
					{
						vect += replyVector[i];
					}
					qDebug() << "Protcol Number:" << vect;
				}
				if (!m_obd->sendObdRequestString("03\r",3,&replyVector))
				{
					qDebug() << "Error retreiving trouble codes";
					if (m_obd->lastError() == obdLib::NODATA)
					{
						emit consoleMessage("No trouble codes");
						qDebug() << "NODATA";
					}
					//Error
				}
				if (vect == "A6")
				{
					bool inHeader = true;
					QString num = "";
					for (int i=0;i<replyVector.size()-1;i++)
					{
						num += QString((char)replyVector.at(i));
					}
					QStringList numsplit = num.split(" ");
					QString currentHeader;
					numsplit.removeAt(3);
					for (int i=0;i<numsplit.count()-1;i++)
					{
						if (inHeader)
						{
							currentHeader = numsplit[i];
							inHeader = false;
							numsplit.removeAt(i);
							numsplit.removeAt(i);
							numsplit.removeAt(i);
							i--;
							i--;
							qDebug() << "Header:" << currentHeader;
						}
						else
						{
							if (numsplit[i+1].length() > 2)
							{
								//This is a header message;

								inHeader = true;
							}
							else
							{
								QString code = numsplit[i] + numsplit[i+1];
								i++;
								qDebug() << "Code:" << code;
							}
						}
					}
					//7E8

					//11 bit can
				}
				/*QString rett = "";
				for (int i=0;i<replyVector.size();i++)
				{
					rett += (char)replyVector[i] + " ";
				}
				qDebug() << rett;
				*/
				if (!m_obdConnected)
				{
					emit consoleMessage(QString("Disconnected"));
					m_obd->closePort();
					emit disconnected();
				}
				//continue;
				//qDebug() << rett;
				QList<QString> codes;
				bool inCode = false;
				int count = 0;
				unsigned char ecu = 0;
				QString codetype;
				for (int i=0;i<replyVector.size();i++)
				{
					if (replyVector[i] == 0x43)
					{
						inCode = true;
						i++;
					}
					if (inCode)
					{
						count++;
						//qDebug() << "Val:" << replyVector[i];
						if (!((replyVector[i] >> 7) & 1))
						{
							if (!((replyVector[i] >> 6) & 1))
							{
								//qDebug() << "Powertrain code";
								//codes.append(QString::number(ecu) + ":P");
								codetype = "P";
							}
							else if (((replyVector[i] >> 6) & 1))
							{
								//codes.append(QString::number(ecu) + ":C");
								codetype = "C";
								//qDebug () << "Chassis Code";
							}
						}
						else if ((replyVector[i] >> 7) & 1)
						{
							if (!((replyVector[i] >> 6) & 1))
							{
								//codes.append(QString::number(ecu) + ":B");
								codetype = "B";
								//qDebug() << "Body code";
							}
							else if (((replyVector[i] >> 6) & 1))
							{
								//codes.append(QString::number(ecu) + ":U");
								codetype = "U";
								//qDebug () << "Network Code";
							}
						}
						unsigned char second = (replyVector[i] << 4) & 3;
						unsigned char third = (replyVector[i]) & 0xF;
						unsigned char fourth = (replyVector[i+1]) & 0xF;
						unsigned char fifth = (replyVector[i+1] << 4) & 0xF;
						//qDebug () << (int)second << (int) third << (int) fourth << (int)fifth;
						if (second == 0 && third == 0 && fourth == 0 && fifth == 0)
						{

						}
						else
						{
							codes.append(QString::number(ecu).append(":").append(codetype).append(QString::number(second)).append(QString::number(third)).append(QString::number(fourth)).append(QString::number(fifth)));

						}
						i++;
						if (count >= 3)
						{
							count = 0;
							inCode = false;
							i++; // Pop past the checksum byte
						}
					}
					else
					{
						//Read headers here
						//qDebug() << "Header:" << replyVector[i] << replyVector[i+1] << replyVector[i+2];
						ecu = replyVector[i+2];
						i+=2;
					}
				}
				troubleCodes(codes);
				//qDebug() <<
				if (!m_obdConnected)
				{
					emit consoleMessage(QString("Disconnected"));
					m_obd->closePort();
					emit disconnected();
				}
				/*QString replyString = QString((char*)replyVectorString.data());
				if (replyString.contains("NODATA"))
				{
					qDebug() << "NODATA";
				}
				*/
				/*QString rep = "";
				for (int i=0;i<replyVector.size();i++)
				{
					rep += QString::number(replyVector[i],16).toUpper() + " ";
				}
			qDebug() << rep;*/

			}
			else if (m_reqClassListThreaded[i].type == RAW_REQUEST)
			{
				if (!m_obdConnected)
				{
					if (!m_connect())
						continue;
				}
				emit consoleMessage("Raw request");
				qDebug() << "Raw Request";
				std::vector<unsigned char> reply;
				QByteArray replyArray;

				if (!m_obd->sendObdRequestString(m_reqClassListThreaded[i].custom,m_reqClassListThreaded[i].custom.length(),&reply,2000))
				{
					emit consoleMessage("Unable to send custom request");
					qDebug() << "Error with custom request";
				}
				for (unsigned int i=0;i<reply.size();i++)
				{
					replyArray.append(reply[i]);
				}
				emit singleShotResponse(m_reqClassListThreaded[i].custom,replyArray);
				if (!m_obdConnected)
				{
					emit consoleMessage(QString("Disconnected"));
					m_obd->closePort();
					emit disconnected();
				}
			}
			else if (m_reqClassListThreaded[i].type == RAW_BLIND_REQUEST)
			{
				if (!m_obdConnected)
				{
					if (!m_connect())
						continue;
				}
				emit consoleMessage("Raw blind request");
				qDebug() << "Raw blind Request";
				std::vector<unsigned char> reply;
				QByteArray replyArray;

				if (!m_obd->sendObdRequestString(m_reqClassListThreaded[i].custom,m_reqClassListThreaded[i].custom.length(),&reply,0,0))
				{
					emit consoleMessage("Unable to send custom blind request");
					qDebug() << "Error with custom blind request";
				}
				if (!m_obdConnected)
				{
					emit consoleMessage(QString("Disconnected"));
					m_obd->closePort();
					emit disconnected();
				}
			}
			else if (m_reqClassListThreaded[i].type == START_REQ_LOOP)
			{

			}
			else if (m_reqClassListThreaded[i].type == STOP_REQ_LOOP)
			{

			}
			else if (m_reqClassListThreaded[i].type == SCAN_ALL)
			{
				//Full scan for supported pids
				qDebug() << "Beginning full pid scan";
				if (!m_obdConnected)
				{
					if (!m_connect())
						continue;
				}
				QString pid = "";
				std::vector<unsigned char> reply;
				QList<QString> pidList;
				for (int i=1;i<=9;i++)
				{
					//pidList.append(QList<int>());
					if (i != 3 && i != 4 && i != 2)
					{
						for (int j=0;j<0x20;j++)
						{
							pid = QString("0").append(QString::number(i)).append(((j < 16) ? "0" : "")).append(QString::number(j,16));
							if (!m_obd->sendObdRequest(QString(pid + "\r").toStdString().c_str(),pid.length()+1,&reply))
							{
								if (m_obd->lastError() == obdLib::NODATA)
								{
									//pidList[i-1].append(0);
								}
								else
								{
									//pidList[i-1].append(0);

								}
							}
							else
							{
								//pidList[i-1].append(1);
								pidList.append(pid);
							}
							msleep(100);
						}
					}
				}
				supportedPids(pidList);
				if (!m_obdConnected)
				{
					emit consoleMessage(QString("Disconnected"));
					qDebug() << "ELM Disconnected";
					m_obd->closePort();
					emit disconnected();
				}

			}
			else if (m_reqClassListThreaded[i].type == FIND_PORT)
			{
				qDebug() << "Finding OBD port...";
				QStringList portList;
				portList << "/dev/ttyS0";
				portList << "/dev/ttyS1";
				portList << "/dev/ttyUSB0";
				portList << "/dev/ttyUSB1";
				portList << "/dev/ttyUSB2";
				portList << "/dev/ttyUSB3";
				portList << "/dev/ttyUSB4";
				//portList << "/dev/pts/0";
				//portList << "/dev/pts/1";
				//portList << "/dev/pts/2";
				//portList << "/dev/pts/3";
				QList<int> baudList;
				baudList << -1;
				baudList << 9600;
				baudList << 38400;
				if (!m_obdConnected)
				{
					bool found = false;
					QString foundport = "";
					for (int j=0;j<baudList.count();j++)
					{
						for (int i=0;i<portList.count();i++)
						{
							if (baudList[j] != -1)
							{
								m_obd->openPort(portList[i].toStdString().c_str(),baudList[j]);
							}
							else
							{
								m_obd->openPort(portList[i].toStdString().c_str());
							}
							if (!initElm())
							{
								qDebug() << "Error";
								m_obd->closePort();
							}
							else
							{
								qDebug() << "Port Found!" << portList[i];
								m_obd->closePort();
								foundport = portList[i];
								i = portList.count();
								found = true;
							}
						}
					}
					if (found)
					{
						emit obdPortFound(foundport);
					}
				}

			}
			else if (m_reqClassListThreaded[i].type == REQ_SUPPORTED_PIDS)
			{
				if (!m_obdConnected)
				{
					if (!m_connect())
						continue;
				}
				emit consoleMessage("Requesting supported pids");
				std::vector<unsigned char> replyVector;
				if (!m_obd->sendObdRequest("0100\r",5,&replyVector))
				{
					qDebug() << "Error in requesting supported pids";
					continue;
				}
				QString rep = "Reply: ";
				for (unsigned int i=0;i<replyVector.size();i++)
				{
					rep += QString::number(replyVector[i],16).toUpper() + " ";
				}
				//qDebug() << "String reply:" << rep;
				unsigned var = (((unsigned int)replyVector[2] << 24) + ((unsigned int)replyVector[3] << 16) + ((unsigned int)replyVector[4] << 8) + (unsigned int)replyVector[5]);
				//qDebug() << "00 PID Reply:" << var;
				for (int i=2;i<6;i++)
				{
					//qDebug() << replyVector[i];
				}
				QList<QString> pids;
				for (int i=1;i<0x20;i++)
				{
					if (((var >> (0x20-i)) & 1))
					{
						//qDebug() << "Pid " << i << "Supported";
						pids.append(QString("01").append((i < 16) ? "0" : "").append(QString::number(i,16).toUpper()));
						qDebug() << "Pid" << pids[pids.count()-1] << "supported";
						//pids[0].append(1);
					}
					else
					{
						//pids[0].append(0);
					}
				}
				emit supportedPids(pids);

				if (!m_obdConnected)
				{
					m_obd->closePort();
					m_obdConnected = false;
					emit disconnected();
				}



			}
			else if (m_reqClassListThreaded[i].type == MODE_PID)
			{
				if (m_obdConnected)
				{
					if (first)
					{
						first = false;
						cycleCounter++;
					}
					if ((cycleCounter % m_reqClassListThreaded[i].priority) == 0)
					{
						//qDebug() << "Making req" << m_reqList[currentReqNum].mid(0,m_reqList[currentReqNum].length()-1) << m_set << m_reqPriority[currentReqNum] << currentReqNum;
						//if (!m_obd->sendObdRequest(m_reqList[currentReqNum].toStdString().c_str(),m_reqList[currentReqNum].length(),&replyVector))
						QString req = (m_reqClassListThreaded[i].mode < 16) ? "0" : "";
						req += QString::number(m_reqClassListThreaded[i].mode,16).toUpper();
						req += (m_reqClassListThreaded[i].pid < 16) ? "0" : "";
						req += QString::number(m_reqClassListThreaded[i].pid,16).toUpper();
						//qDebug() << "Current request:" << req;
						QString pidreq = req;
						if (m_reqClassListThreaded[i].wait != 0)
						{
							req += " " + QString::number(m_reqClassListThreaded[i].wait);
						}
						req += "\r";
						if (!m_obd->sendObdRequest(req.toStdString().c_str(),req.length(),&replyVectorString))
						{
							if (m_obd->lastError() == obdLib::NODATA)
							{
								qDebug() << "OBD Request failed: NODATA for request" << req.mid(0,req.length()-1) << m_reqClassFailureMap[&m_reqClassListThreaded[i]];
								m_reqClassFailureMap[&m_reqClassListThreaded[i]]++;

								bool allislost=true;
								foreach(RequestClass* req,m_reqClassFailureMap.keys())
								{
									allislost &= m_reqClassFailureMap[req] > 0;
								}

								if (!allislost && !m_whiteList[&m_reqClassListThreaded[i]] && m_reqClassFailureMap[&m_reqClassListThreaded[i]] > 20)
								{
									qDebug() << "Reached Maximum errors for pid. Disabling: " << req.mid(0,req.length()-1);
									emit consoleMessage(QString("Reached maximum errors for pid ") + req.mid(0,req.length()-1) + QString(". Removing from request list."));
									m_reqClassFailureMap.remove(&m_reqClassListThreaded[i]);
									m_reqClassListThreaded.removeAt(i);
								}
								//m_errorCount[currentReqNum]++;
								//if (m_errorCount[currentReqNum] > 20)
								//{
								//	qDebug() << "Reached Maximum errors for pid" << m_reqList[currentReqNum].mid(0,m_reqList[currentReqNum].length()-1) << "shutting down requests for that pid";
								//	m_reqList.removeAt(currentReqNum);
								//	m_reqPriority.removeAt(currentReqNum);
								//	m_errorCount.removeAt(currentReqNum);
								//	emit consoleMessage(QString("Readed maximum errors for pid") + m_reqList[currentReqNum].mid(0,m_reqList[currentReqNum].length()-1) + "shutting down requests for that pid");
								//}

							}
							else if (m_obd->lastError() == obdLib::SERIALREADERROR || m_obd->lastError() == obdLib::SERIALWRITEERROR)
							{
								//errorCount++;
								qDebug() << "Serial read/write error on request" << req.mid(0,req.length()-1);
								emit consoleMessage("Serial read/write error on request" + req.mid(0,req.length()-1));
								//m_requestLoopRunning = false;

							}
							//Error here
						}
						else
						{
							m_reqClassFailureMap[&m_reqClassListThreaded[i]] = 0;
							m_whiteList[&m_reqClassListThreaded[i]] = true;
							//errorCount = 0;
							/*replyVector.clear();
							for (unsigned int i=0;i<replyVectorString.size()-1;i++)
							{
								replyVector.push_back(m_obd->byteArrayToByte(replyVectorString[i],replyVectorString[i+1]));
								i++;
							}
							*/
							ObdInfo::Pid *pid = m_obdInfo->getPidFromString(pidreq);
							if (pid)
							{
								/*QString rrtest="";
								for (int i=0;i<replyVectorString.size();i++)
								{
									rrtest += QString::number(replyVectorString.at(i),16).toUpper() + " ";
								}
								qDebug() << rrtest;
								*/
								QString func = pid->function;
								if (func.contains("A"))
								{
									func = func.replace(QString("A"),QString::number(replyVectorString[2]));
								}
								if (func.contains("B"))
								{
									func = func.replace(QString("B"),QString::number(replyVectorString[3]));
								}
								QString done = parse(func);

								pidReceived(pidreq,done,cycleCounter,(((float)QDateTime::currentDateTime().toTime_t()) + ((float)QDateTime::currentDateTime().time().msec() / 1000.0)));
							}
							else
							{
								qDebug() << "Invalid Pid returned:" << req.mid(0,req.length()-1);
								emit consoleMessage(QString("Invalid PID returned from ObdInfo::getPidFromString(): ") + req.mid(0,req.length()-1));
							}
						}

					}
				}
			}
			if (!m_reqClassListThreaded[i].repeat)
			{
				m_reqClassFailureMap.remove(&m_reqClassListThreaded[i]);
				m_reqClassListThreaded.removeAt(i);
				i--;
			}
			if (!m_requestLoopRunning)
			{
				m_requestLoopRunning = true;
				i = m_reqClassListThreaded.count();
				continue;
			}
		}
		removePidMutex.unlock();
		msleep(100);
	}
}


void ObdThread::setPort(QString port)
{
	m_port = port;
}
void ObdThread::setBaud(int baud)
{
	m_baud = baud;
}
void ObdThread::addRequest(int mode, int pid, int priority,int wait)
{
	threadLockMutex.lock();
	RequestClass req;
	req.mode = mode;
	req.pid = pid;
	req.priority = priority;
	req.repeat = true;
	req.type = MODE_PID;
	req.wait = wait;
	m_reqClassList.append(req);
	threadLockMutex.unlock();
}

void ObdThread::getSupportedPids()
{
	threadLockMutex.lock();
	RequestClass req;
	req.type = REQ_SUPPORTED_PIDS;
	m_reqClassList.append(req);
	threadLockMutex.unlock();
}
void ObdThread::fullPidScan()
{
	threadLockMutex.lock();
	RequestClass req;
	req.type = SCAN_ALL;
	m_reqClassList.append(req);
	threadLockMutex.unlock();
}
void ObdThread::clearTroubleCodes()
{
	threadLockMutex.lock();
	RequestClass req;
	req.type = CLEAR_TROUBLE_CODES;
	m_reqClassList.append(req);
	threadLockMutex.unlock();
}

QString ObdThread::parse(QString str)
{
	if ((str[0] != '(' && str[str.length()-1] != ')'))
	{
		str = QString("(") + str + ")";
	}
	QString done = calc(str);
	//qDebug() << "Final Answer:" << done;
	return done;
}
bool ObdThread::initElm()
{
	if (!resetElm())
	{
		emit consoleMessage("Error resetting ELM Device");
		qDebug() << "Error resetting ELM device";
		return false;
	}
	if (!echoOff())
	{
		emit consoleMessage("Error turning echo off");
		qDebug() << "Error turning echo off";
		return false;
	}
	if (!setHeaders(false))
	{
		emit consoleMessage("Error turning headers off");
		qDebug() << "Error turning headers off";
		return false;
	}
	if (!lineFeedOff())
	{
		emit consoleMessage("Error turning linefeeds off");
		qDebug() << "Error turning linefeeds off";
		return false;
	}
	std::vector<unsigned char> replyVector;
	if (!m_obd->sendObdRequestString("0100\r",5,&replyVector,5000))
	{
		emit consoleMessage("Error in detecting protocol");
		qDebug() << "Error in finding protocol";
		return false;
	}
	return true;
}
bool ObdThread::resetElm()
{
	std::vector<unsigned char> replyVector;
	QString reply="";
	if (!m_obd->sendObdRequestString("atz\r",4,&replyVector,20,5))
	{
		qDebug() << "Error in reset sent";
	}
	for (unsigned int i=0;i<replyVector.size();i++)
	{
		reply += replyVector[i];
	}
	if (reply.contains("ELM"))
	{
		return true;
	}
	else if (reply.contains("atz"))
	{
		//Echoing here?
		qDebug() << "Reset echoed:" << reply;
		m_obd->flush();
		usleep(10000);
		return true;
	}
	else
	{
		qDebug() << "Reset:" << reply;
		return false;
	}
}
bool ObdThread::echoOff()
{
	std::vector<unsigned char> replyVector;
	QString reply="";
	m_obd->sendObdRequestString("ate0\r",5,&replyVector,20,3);
	reply = "";
	for (unsigned int i=0;i<replyVector.size();i++)
	{
		reply += replyVector[i];
	}
	if (reply.contains("OK"))
	{
		return true;
	}
	qDebug() << "Bad Echo:" << reply;
	return false;
}
bool ObdThread::setHeaders(bool on)
{
	std::vector<unsigned char> replyVector;
	QString reply="";
	if (on)
	{
		m_obd->sendObdRequestString("ath1\r",5,&replyVector,20,3);
	}
	else
	{
		m_obd->sendObdRequestString("ath0\r",5,&replyVector,20,3);
	}
	reply = "";
	for (unsigned int i=0;i<replyVector.size();i++)
	{
		reply += replyVector[i];
	}
	if (reply.contains("OK"))
	{
		return true;
	}
	return false;
}
bool ObdThread::lineFeedOff()
{
	std::vector<unsigned char> replyVector;
	QString reply="";
	m_obd->sendObdRequestString("atl0\r",5,&replyVector,20,3);
	reply = "";
	for (unsigned int i=0;i<replyVector.size();i++)
	{
		reply += replyVector[i];
	}
	if (reply.contains("OK"))
	{
		return true;
	}
	return false;
}
QString ObdThread::getElmVersion()
{
	std::vector<unsigned char> replyVector;
	QString reply="";
	m_obd->sendObdRequestString("ati\r",4,&replyVector,20,3);
	reply = "";
	for (unsigned int i=0;i<replyVector.size();i++)
	{
		reply += replyVector[i];
	}
	//qDebug() << "Reply String:" << reply.replace("ati","");
	return reply;
}
QString ObdThread::calc(QString str)
{
	//qDebug() << str;
	int start=-1;
	int stop=-1;
	for (int i=0;i<str.length();i++)
	{
		if (str[i] == '(')
		{
			if (start == -1)
				start = i+1;
		}
		if (str[i] == ')')
		{
				stop = i;
			//i = str.length();
		}
	}
	if (start != -1 && stop != -1)
	{
		QString tmp = calc(str.mid(start,stop-start));
		//qDebug() << "Old:" << str;
		str.replace(start-1,(stop-start)+2,tmp);
		//qDebug() << "New:" << str;
	}
	else
	{
		//qDebug() << "Inner function:" << str;
	}
	start = 0;
	stop = -1;
	//int op = 0;
	//int current;
	//"5+3*2+2-5*4/8;
	//5 3*2 2-5*4/8   + split
	//2 5*4/8     - split
	//5*4 8    / split
	//5 4   * split

	//"5+3*2+2-5*4/8;
	//5+3*2+2 5*4/8  - split
	//5 3*2 2-5*4/8   + split
	//5+3 2+2-5 4/8  * split

	for (int k=0;k<2;k++)
	{
	for (int i=0;i<str.length();i++)
	{
		if (k == 1)
		{
		if (str[i] == '+')
		{
			stop = -1;
			start = i;
			for (int j=i-1;j>=0;j--)
			{
				if (str[j] == '+' || str[j] == '-' || str[j] == '/' || str[j] == '*')
				{
					stop = j;
					j=0;
				}
			}
			if (stop == -1)
			{
				stop = 0;
			}
			int replacer = stop;
			QString tmp = str.mid(stop,start-stop);
			//qDebug() << "Found:" <<i<<stop<< tmp;
			stop = -1;
			for (int j=i+1;j<str.length();j++)
			{
				if (str[j] == '+' || str[j] == '-' || str[j] == '/' || str[j] == '*')
				{
					stop = j;
					j = str.length();
				}
			}
			if (stop == -1)
			{
				stop = str.length();
			}
			QString tmp2 = str.mid(start+1,stop-(start+1));
			//qDebug() << "Found2:" <<start<<stop<< tmp2;
			float first = tmp.toFloat();
			float second = tmp2.toFloat();
			str = str.replace(replacer,tmp.length()+tmp2.length()+1,QString::number(first+second));
			//qDebug() << str;
			i = 0;
		}
		if (str[i] == '-')
		{
			stop = -1;
			start = i;
			for (int j=i-1;j>=0;j--)
			{
				if (str[j] == '+' || str[j] == '-' || str[j] == '/' || str[j] == '*')
				{
					stop = j;
					j=0;
				}
			}
			if (stop == -1)
			{
				stop = 0;
			}
			int replacer = stop;
			QString tmp = str.mid(stop,start-stop);
			//qDebug() << "Found:" <<i<<stop<< tmp;
			stop = -1;
			for (int j=i+1;j<str.length();j++)
			{
				if (str[j] == '+' || str[j] == '-' || str[j] == '/' || str[j] == '*')
				{
					stop = j;
					j = str.length();
				}
			}
			if (stop == -1)
			{
				stop = str.length();
			}
			QString tmp2 = str.mid(start+1,stop-(start+1));
			//qDebug() << "Found2:" << tmp2;
			float first = tmp.toFloat();
			float second = tmp2.toFloat();
			str = str.replace(replacer,tmp.length()+tmp2.length()+1,QString::number(first-second));
			//qDebug() << str;
			i = 0;
		}
		}
		if (k == 0)
		{
		if (str[i] == '*')
		{
			stop = -1;
			start = i;
			for (int j=i-1;j>=0;j--)
			{
				if (str[j] == '+' || str[j] == '-' || str[j] == '/' || str[j] == '*')
				{
					stop = j;
					j=0;
				}
			}
			if (stop == -1)
			{
				stop = 0;
			}
			int replacer = stop;
			QString tmp = str.mid(stop,start-stop);
			//qDebug() << "Found:" << tmp;
			stop = -1;
			for (int j=i+1;j<str.length();j++)
			{
				if (str[j] == '+' || str[j] == '-' || str[j] == '/' || str[j] == '*')
				{
					stop = j;
					j = str.length();
				}
			}
			if (stop == -1)
			{
				stop = str.length();
			}
			QString tmp2 = str.mid(start+1,stop-(start+1));
			//qDebug() << "Found2:" << tmp2;
			float first = tmp.toFloat();
			float second = tmp2.toFloat();
			str = str.replace(replacer,tmp.length()+tmp2.length()+1,QString::number(first*second));
			//qDebug() << str;
			i = 0;
		}
		if (str[i] == '/')
		{
			stop = -1;
			start = i;
			for (int j=i-1;j>=0;j--)
			{
				if (str[j] == '+' || str[j] == '-' || str[j] == '/' || str[j] == '*')
				{
					stop = j;
					j=0;
				}
			}
			if (stop == -1)
			{
				stop = 0;
			}
			int replacer = stop;
			QString tmp = str.mid(stop,start-stop);
			//qDebug() << "Found:" << tmp;
			stop = -1;
			for (int j=i+1;j<str.length();j++)
			{
				if (str[j] == '+' || str[j] == '-' || str[j] == '/' || str[j] == '*')
				{
					stop = j;
					j = str.length();
				}
			}
			if (stop == -1)
			{
				stop = str.length();
			}
			QString tmp2 = str.mid(start+1,stop-(start+1));
			//qDebug() << "Found2:" << tmp2;
			float first = tmp.toFloat();
			float second = tmp2.toFloat();
			str = str.replace(replacer,tmp.length()+tmp2.length()+1,QString::number(first/second));
			//qDebug() << str;
			i = 0;
		}
		}
		//usleep(100000);
	}
	}
	return str;
}
bool ObdThread::m_connect()
{
	qDebug() << "Connecting...";
	m_obd->openPort(m_port.toStdString().c_str(),m_baud);
	if (!initElm())
	{
		emit consoleMessage("Error in ELM init, port not opened");
		m_obd->closePort();
		emit disconnected();
		//continue;
		return false;
	}
	m_obdConnected = true;
	QString version = getElmVersion();
	emit consoleMessage(QString("Elm found. Version: ").append(version));
	qDebug() << "Connected to ELM version" << version;
	setProtocol(0,false);
	QString protocol = getProtocolName();
	qDebug() << "Connected protocol:" << protocol;
	emit protocolFound(protocol);
	emit connected(version);
	return true;
}
QString ObdThread::getProtocolName()
{
	std::vector<unsigned char> replyVector;
	QString reply="";
	m_obd->sendObdRequestString("0100\r",5,&replyVector,20,5);
	m_obd->sendObdRequestString("ATDP\r",5,&replyVector,100);
	reply = "";
	for (unsigned int i=0;i<replyVector.size();i++)
	{
		reply += replyVector[i];
	}
	//qDebug() << "Reply String:" << reply.replace("ati","");
	return reply;

}
void ObdThread::setProtocol(int num, bool autosearch)
{

	if (!m_obd->sendObdRequest("ATSP00\r",7,20))
	{
		qDebug() << "Error setting auto-protocol";
	}
}
