/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation, or under the terms of the GNU Lesser General
 *  Public License version 2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <microfeed-common/microfeedfile.h>
#include <microfeed-common/microfeedmisc.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <sys/file.h>

typedef uint32_t Offset;
typedef uint32_t Size;

#define PAGE_SIZE 4096
#define MINIMUM_USER_SIZE 16

#define BLOCK(f,o) ((Block*)((void*)((f)->header) + (o)))
#define NEXT_BLOCK(b) ((Block*)((void*)(b) + (b)->size))
#define OFFSET(f,b) ((Offset)((void*)(b) - (void*)((f)->header)))

typedef struct  {
	Size header_size;
	Offset free_block_offset;
	MicrofeedFileIndex offsets_length;
	Offset offsets[1];
} Header;

typedef struct {
	Size size;
	Size user_size;
	MicrofeedFileIndex index;
} Block;

struct _MicrofeedFile {
	int fd;

	size_t size;
	Header* header;
};

static Block* allocate_block(MicrofeedFile* file, Size size, MicrofeedFileIndex index);
static void check_state(MicrofeedFile* file);

MicrofeedFile* microfeed_file_new(const char* path, int create) {
	MicrofeedFile* file = NULL;
	int fd;
	struct flock lock;
	struct stat stat_buffer;
	void* p;
	unsigned int i;
	
	if ((fd = open(path, (create ? O_CREAT : 0) | O_RDWR, 0666)) >= 0) {
		lock.l_type = F_WRLCK;
		lock.l_whence = SEEK_SET;
		lock.l_start = 0;
		lock.l_len = 0;
		if (fcntl(fd, F_SETLK, &lock) < 0 ||
		    fstat(fd, &stat_buffer) < 0 ||
		    (stat_buffer.st_size < 2 * PAGE_SIZE && ftruncate(fd, 2 * PAGE_SIZE) < 0) ||
		    (p = mmap(NULL, (stat_buffer.st_size < 2 * PAGE_SIZE ? 2 * PAGE_SIZE : stat_buffer.st_size), PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0)) == MAP_FAILED) { 
			close(fd);
		} else {
			file = microfeed_memory_allocate(MicrofeedFile);
			file->fd = fd;
			file->size = (stat_buffer.st_size < 2 * PAGE_SIZE ? 2 * PAGE_SIZE : stat_buffer.st_size);
			file->header = (Header*)p;
			
			if (!file->header->header_size) {
				file->header->header_size = PAGE_SIZE;
				file->header->offsets_length = (file->header->header_size - sizeof(Size) - sizeof(Offset) - sizeof(MicrofeedFileIndex)) / sizeof(Offset);
				memset(file->header->offsets, 0, file->header->header_size - sizeof(Size) - sizeof(Offset) - sizeof(MicrofeedFileIndex));
				file->header->free_block_offset = (Offset)file->header->header_size;
				BLOCK(file, file->header->free_block_offset)->size = file->size - file->header->header_size;
				BLOCK(file, file->header->free_block_offset)->index = MICROFEED_FILE_INDEX_INVALID;
			}
		}
	}
	
	check_state(file);
	
	return file;
}

void microfeed_file_free(MicrofeedFile* file) {
	
	check_state(file);

	msync(file->header, file->size, MS_INVALIDATE);
	munmap(file->header, file->size);
	close(file->fd);
	microfeed_memory_free(file);
}

void* microfeed_file_allocate_block_impl(MicrofeedFile* file, size_t size) {
	Size real_size;
	MicrofeedFileIndex index;
	Block* b;
	
	real_size = ((Size)size + 3) / 4 * 4 + sizeof(Block);
	for (index = 0; index < file->header->offsets_length; index++) {
		if (!file->header->offsets[index]) {
			break;
		}
	}
	if (file->header->offsets[index]) {
		for (index = 0; index < file->header->offsets_length; index++) {
			if (file->header->offsets[index]) {
				file->header->offsets[index] += PAGE_SIZE;
			}
		}
		if (ftruncate(file->fd, file->size + PAGE_SIZE) < 0 ||
		    (file->header = (Header*)mremap(file->header, file->size, file->size + PAGE_SIZE, MREMAP_MAYMOVE)) == MAP_FAILED) { 
			abort();
		}
		memmove(((void*)file->header) + file->header->header_size + PAGE_SIZE, ((void*)file->header) + file->header->header_size, file->size - file->header->header_size);
		memset(((void*)file->header) + file->header->header_size, 0, PAGE_SIZE);
		file->header->header_size += PAGE_SIZE;
		file->header->free_block_offset += PAGE_SIZE;
		file->header->offsets_length = (file->header->header_size - sizeof(Size) - sizeof(Offset) - sizeof(MicrofeedFileIndex)) / sizeof(Offset);
		file->size += PAGE_SIZE;
	}
	
	b = allocate_block(file, real_size, index);
	b->user_size = (Size)size;

	check_state(file);
	
	return ((void*)b) + sizeof(Block);
}

void* microfeed_file_resize_block_impl(MicrofeedFile* file, void* block, size_t size) {
	Block* b;
	Size real_size;
	Block* free_b;
	MicrofeedFileIndex index;
	void* buffer;
	size_t buffer_size;

	b = (Block*)(block - sizeof(Block));
	if (b->user_size != size) {
		real_size = ((Size)size + 3) / 4 * 4 + sizeof(Block);
		if (b->size >= real_size) {
			if (b->size > real_size + sizeof(Block) + MINIMUM_USER_SIZE) {
				free_b = (Block*)(((void*)b) + real_size);
				free_b->size = b->size - real_size;
				free_b->index = MICROFEED_FILE_INDEX_INVALID;
				b->size = real_size;
				if (free_b < BLOCK(file, file->header->free_block_offset)) {
					file->header->free_block_offset = OFFSET(file, free_b);
				}
			}
		} else {
			index = b->index;
			buffer_size = b->user_size;
			b->index = MICROFEED_FILE_INDEX_INVALID;
			if (b < BLOCK(file, file->header->free_block_offset)) {
				file->header->free_block_offset = OFFSET(file, b);
			}
			buffer = malloc(buffer_size);
			memcpy(buffer, block, buffer_size);
			b = allocate_block(file, real_size, index);
			memcpy(((void*)b) + sizeof(Block), buffer, buffer_size);
			free(buffer);
			b->index = index;
			b->size = real_size;
			file->header->offsets[index] = OFFSET(file, b);
		}
		b->user_size = size;
	}

	check_state(file);
	
	return ((void*)b) + sizeof(Block);
}

void microfeed_file_free_block(MicrofeedFile* file, void* block) {
	Block* b;
	Offset o;
	
	b = (Block*)(block - sizeof(Block));
	file->header->offsets[b->index] = 0;
	b->index = MICROFEED_FILE_INDEX_INVALID;

	o = OFFSET(file, b);
	if (o < file->header->free_block_offset) {
		file->header->free_block_offset = o;
	}

	check_state(file);
}

void microfeed_file_free_index(MicrofeedFile* file, MicrofeedFileIndex index) {
	void* block;
	
	block = microfeed_file_get_block_impl(file, index);
	microfeed_assert(block != NULL);
	microfeed_file_free_block(file, block);
}

MicrofeedFileIndex microfeed_file_get_index(MicrofeedFile* file, void* block) {
	Block* b;
	
	b = ((void*)block) - sizeof(Block);

	return (MicrofeedFileIndex)b->index;
}

void* microfeed_file_get_block_impl(MicrofeedFile* file, MicrofeedFileIndex index) {
	void* block = NULL;
	
	if (index < file->header->offsets_length && file->header->offsets[index]) {	
		block = ((void*)BLOCK(file, file->header->offsets[index])) + sizeof(Block);
	}
	
	return block;
}

size_t microfeed_file_get_index_size(MicrofeedFile* file, MicrofeedFileIndex index) {
	size_t size;
	
	if (index < file->header->offsets_length && file->header->offsets[index]) {	
		size = BLOCK(file, file->header->offsets[index])->user_size;
	}

	return size;
}

void microfeed_file_print_statistics(MicrofeedFile* file) {
	MicrofeedFileIndex index;
	Block* b;
	Size free_size = 0;
	Size user_size = 0;
	MicrofeedFileIndex block_count = 0;
	
	printf("File size: %ld\n", (long)file->size);
	printf("Header size: %ld (%.2lf %%)\n", (long)file->header->header_size, (double)(100.0 * file->header->header_size / file->size));
	printf("Offset table length: %ld\n", (long)file->header->offsets_length);
	printf("First free block offset: %ld\n", (long)file->header->free_block_offset);

	printf("Offsets:\n");
	for (index = 0; index < file->header->offsets_length; index++) {
		if (file->header->offsets[index]) {
			printf("    %ld => %ld\n", (long)index, file->header->offsets[index]);
		}
	}
	printf("Blocks:\n");
	for (b = BLOCK(file, file->header->header_size); b < BLOCK(file, file->size); b = NEXT_BLOCK(b)) {
		block_count++;
		if (b->index == MICROFEED_FILE_INDEX_INVALID) {
			printf("    %ld: %ld bytes free\n", (long)OFFSET(file, b), (long)b->size);
			free_size += b->size;
		} else {
			printf("    %ld: %ld bytes (%ld bytes)\n", (long)OFFSET(file, b), (long)b->size, (long)b->user_size);
			user_size += b->user_size;
		}
	}
	printf("Total number of blocks: %ld\n", (long)block_count);
	printf("Total size of user data: %ld (%.2lf %%)\n", (long)user_size, (double)(100.0 * user_size / file->size));
	printf("Total size of free blocks: %ld (%.2lf %%)\n", (long)free_size, (double)(100.0 * free_size / file->size));
}

size_t microfeed_file_block_get_size(void* block) {
	Block* b;
	
	b = ((void*)block) - sizeof(Block);

	return (size_t)b->user_size;
}

static Block* allocate_block(MicrofeedFile* file, Size size, MicrofeedFileIndex index) {
	Block* b;
	Size free_block_size;
	Block* nb;
	unsigned pages;

	b = BLOCK(file, file->header->free_block_offset);
	while (b->index != MICROFEED_FILE_INDEX_INVALID || b->size < size + sizeof(Block) + MINIMUM_USER_SIZE) {
		free_block_size = b->size;
		nb = NEXT_BLOCK(b);
		if (nb < BLOCK(file, file->size)) {
			if (nb->index == MICROFEED_FILE_INDEX_INVALID) {
				b->size += nb->size;
			} else {
				memmove(b, nb, nb->size);
				file->header->offsets[b->index] = OFFSET(file, b);
				b = NEXT_BLOCK(b);
				b->size = free_block_size;
				b->index = MICROFEED_FILE_INDEX_INVALID;
				file->header->free_block_offset = OFFSET(file, b);
			}
		} else {
			pages = (size + sizeof(Block)) / PAGE_SIZE + 1;
			if (ftruncate(file->fd, file->size + pages * PAGE_SIZE) < 0 ||
			    (file->header = (Header*)mremap(file->header, file->size, file->size + pages * PAGE_SIZE, MREMAP_MAYMOVE)) == MAP_FAILED) {
				abort();
			}
			nb = BLOCK(file, file->size);
			nb->size = pages * PAGE_SIZE;
			nb->index = MICROFEED_FILE_INDEX_INVALID;
			file->size += pages * PAGE_SIZE;
			b = BLOCK(file, file->header->free_block_offset);
		}
	}
	
	free_block_size = b->size;
	b->size = size;
	b->index = index;
	file->header->offsets[index] = OFFSET(file, b);

	nb = NEXT_BLOCK(b);
	nb->size = free_block_size - b->size;
	nb->index = MICROFEED_FILE_INDEX_INVALID;
	if (BLOCK(file, file->header->free_block_offset) == b) {
		file->header->free_block_offset = OFFSET(file, nb);
	}

	return b;
}

static void check_state(MicrofeedFile* file) {
	MicrofeedFileIndex index;
	Block* b;
	
	microfeed_assert(file->header->header_size < file->size);
	microfeed_assert(file->header->free_block_offset < file->size);
	microfeed_assert(BLOCK(file, file->header->free_block_offset)->index == MICROFEED_FILE_INDEX_INVALID);
	
	for (index = 0; index < file->header->offsets_length; index++) {
		if (file->header->offsets[index]) {
			microfeed_assert(BLOCK(file, file->header->offsets[index])->index == index);
		}
	}
	for (b = BLOCK(file, file->header->header_size); b < BLOCK(file, file->size); b = NEXT_BLOCK(b)) {
		if (b->index != MICROFEED_FILE_INDEX_INVALID) {
			microfeed_assert(file->header->offsets[b->index] == OFFSET(file, b));
		} else {
			microfeed_assert(b >= BLOCK(file, file->header->free_block_offset));
		}
	}
	microfeed_assert(b == BLOCK(file, file->size));
}
