/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version
 *  2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <microfeed-subscriber/microfeedsubscriber.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeedconfiguration.h>
#include <microfeed-common/microfeedprotocol.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <dirent.h>

/* TODO: When a publisher (a provider) disconnects, try to reconnect (relaunch) and resubscribe all the feeds automatically. */

struct _MicrofeedSubscriber {
	char* identifier;
	DBusConnection* connection;
	char* object_path;
	MicrofeedConfiguration* configuration;
	unsigned int max_retries;
	
	MicrofeedStore* publishers_by_identifier;
	MicrofeedStore* providers_by_bus_name;
	MicrofeedStore* providers_by_unique_connection_name;
};

typedef struct {
	MicrofeedSubscriber* subscriber;
	char* bus_name;
	char* unique_connection_name;
	MicrofeedStore* publishers_by_object_path;
} Provider;

typedef struct _DataStoredCallback DataStoredCallback;
struct _DataStoredCallback {
	DataStoredCallback* next;
	DataStoredCallback* previous;
	MicrofeedSubscriberDataCallback callback;
	void* user_data;
};

typedef struct {
	unsigned int reference_count;
	Provider* provider;
	char* identifier;
	char* object_path;
	time_t last_activity;
	MicrofeedStore* feeds;
	DataStoredCallback* data_stored_callbacks;
} Publisher;

typedef struct _Subscription Subscription;
struct _Subscription {
	Subscription* next;
	Subscription* previous;
	MicrofeedSubscriberCallbacks* callbacks;
	void* user_data;
};

typedef struct {
	unsigned int reference_count;
	Publisher* publisher;
	char* uri;
	char* name;
	Subscription* subscriptions;
	int updating : 1;
	int republishing : 1;
} Feed;

typedef struct {
	const char* name;
	void (*callback)(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
} SignalCallback;

typedef struct {
	Publisher* publisher;
	char* uri;
	char* uid;
	DBusMessage* message;
	unsigned int retry_counter;
	MicrofeedSubscriberErrorCallback callback;
	void* user_data;
} MethodReturnData;

static Provider* provider_new(MicrofeedSubscriber* subscriber, const char* bus_name);
static void provider_free(Provider* provider);
static const char* provider_get_bus_name(Provider* provider);
static const char* provider_get_unique_connection_name(Provider* provider);
static void provider_set_unique_connection_name(Provider* provider, const char* unique_connection_name);

static Publisher* publisher_new(Provider* provider, const char* identifier);
static void publisher_free(Publisher* publisher);
static Publisher* publisher_ref(Publisher* publisher);
static void publisher_unref(Publisher* publisher);
static const char* publisher_get_identifier(Publisher* publisher);
static const char* publisher_get_object_path(Publisher* publisher);

static Feed* feed_new(Publisher* publisher, const char* uri);
static void feed_free(Feed* feed);
static const char* feed_get_uri(Feed* feed);
static void feed_add_subscription(Feed* feed, MicrofeedSubscriberCallbacks* callbacks, void* user_data);
static void feed_remove_subscription(Feed* feed, MicrofeedSubscriberCallbacks* callbacks, void* user_data);

static void call_publisher_method(Publisher* publisher, const char* uri, const char* uid, DBusMessage* message, MicrofeedSubscriberErrorCallback callback, void* user_data);
static MicrofeedItem* parse_item_from_message(DBusMessageIter* iter);
static int parse_item_signal(Publisher* publisher, DBusMessage* message, Feed** feed_return, MicrofeedItem** item_return);
static int parse_feed_signal(Publisher* publisher, DBusMessage* message, Feed** feed_return);
static Publisher* get_publisher(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberErrorCallback callback, void* user_data);
static Feed* get_feed(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberErrorCallback callback, void* user_data);
static void object_unregister(DBusConnection* connection, void* user_data);
static DBusHandlerResult object_message(DBusConnection* connection, DBusMessage* message, void* user_data);

static void signal_feed_update_started(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_feed_update_ended(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_feed_republishing_started(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_feed_republishing_ended(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_item_added(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_item_changed(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_item_republished(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_item_removed(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_item_status_changed(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_data_stored(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);

static SignalCallback signal_callbacks[] = {
	{ MICROFEED_SIGNAL_NAME_FEED_UPDATE_STARTED, signal_feed_update_started },
	{ MICROFEED_SIGNAL_NAME_FEED_UPDATE_ENDED, signal_feed_update_ended },
	{ MICROFEED_SIGNAL_NAME_FEED_REPUBLISHING_STARTED, signal_feed_republishing_started },
	{ MICROFEED_SIGNAL_NAME_FEED_REPUBLISHING_ENDED, signal_feed_republishing_ended },
	{ MICROFEED_SIGNAL_NAME_ITEM_ADDED, signal_item_added },
	{ MICROFEED_SIGNAL_NAME_ITEM_CHANGED, signal_item_changed },
	{ MICROFEED_SIGNAL_NAME_ITEM_REPUBLISHED, signal_item_republished },
	{ MICROFEED_SIGNAL_NAME_ITEM_REMOVED, signal_item_removed },
	{ MICROFEED_SIGNAL_NAME_ITEM_STATUS_CHANGED, signal_item_status_changed },
	{ MICROFEED_SIGNAL_NAME_DATA_STORED, signal_data_stored },
	{ NULL, NULL }
};

static DBusObjectPathVTable object_vtable = {
	object_unregister,
	object_message
};	

static unsigned int subscriber_count = 0;

MicrofeedSubscriber* microfeed_subscriber_new(const char* identifier, DBusConnection* connection) {
	MicrofeedSubscriber* subscriber;
	char buffer[1024];
	
	subscriber = microfeed_memory_allocate(MicrofeedSubscriber);
	subscriber->identifier = strdup(identifier);
	subscriber->connection = connection;
	snprintf(buffer, 1024, "%s%u", MICROFEED_DBUS_OBJECT_PATH_PREFIX_SUBSCRIBER, subscriber_count++);
	subscriber->object_path = strdup(buffer);
	subscriber->configuration = microfeed_configuration_new();
	subscriber->max_retries = 3;
	
	subscriber->publishers_by_identifier = microfeed_store_new_sorted_data((MicrofeedStoreCompareKeysFunction)strcmp,
	                                                                       (MicrofeedStoreGetKeyFunction)publisher_get_identifier);
	subscriber->providers_by_bus_name = microfeed_store_new_sorted_data((MicrofeedStoreCompareKeysFunction)strcmp,
	                                                                    (MicrofeedStoreGetKeyFunction)provider_get_bus_name);
	subscriber->providers_by_unique_connection_name = microfeed_store_new_sorted_data((MicrofeedStoreCompareKeysFunction)strcmp,
	                                                                                  (MicrofeedStoreGetKeyFunction)provider_get_unique_connection_name);

	if (!dbus_connection_register_object_path(subscriber->connection, subscriber->object_path, &object_vtable, subscriber)) {
	
		return NULL;
	}
	if (!dbus_connection_add_filter(subscriber->connection, object_message, subscriber, NULL)) {

		return NULL;
	}

	snprintf(buffer, 1024, "type='signal',interface='%s',destination='%s'", MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION, dbus_bus_get_unique_name(subscriber->connection));
	dbus_bus_add_match(subscriber->connection, buffer, NULL);
	snprintf(buffer, 1024, "type='signal',interface='%s',destination ='%s'", MICROFEED_DBUS_INTERFACE_ERROR_TO_DESTINATION, dbus_bus_get_unique_name(subscriber->connection));
	dbus_bus_add_match(subscriber->connection, buffer, NULL);
	
	return subscriber;
}

/**
 * TODO: This function is not yet implemented.
 */
void microfeed_subscriber_free(MicrofeedSubscriber* subscriber) {
	char buffer[1024];

	snprintf(buffer, 1024, "type='signal',interface='%s',destination='%s'", MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION, dbus_bus_get_unique_name(subscriber->connection));
	dbus_bus_remove_match(subscriber->connection, buffer, NULL);
	snprintf(buffer, 1024, "type='signal',interface='%s',destination ='%s'", MICROFEED_DBUS_INTERFACE_ERROR_TO_DESTINATION, dbus_bus_get_unique_name(subscriber->connection));
	dbus_bus_remove_match(subscriber->connection, buffer, NULL);

	/* TODO */
}

const char* microfeed_subscriber_get_identifier(MicrofeedSubscriber* subscriber) {

	return subscriber->identifier;
}

int microfeed_subscriber_add_item(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, MicrofeedItem* item, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	DBusMessage* message;
	DBusMessageIter iter;
	const char* uid;
	MicrofeedItemIterator* iterator;
	const char* key;
	const char* value;
	
	if ((publisher = get_publisher(subscriber, publisher_identifier, uri, microfeed_item_get_uid(item), callback, user_data))) {
		message = dbus_message_new_method_call(publisher->provider->bus_name, publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_ADD_ITEM);
		dbus_message_iter_init_append(message, &iter);
		uid = microfeed_item_get_uid(item);
		dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &uri);
		dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &uid);
		for (iterator = microfeed_item_iterate_properties(item, NULL);
		     microfeed_item_iterator_get(iterator, &key, &value);
		     microfeed_item_iterator_next(iterator)) {
			dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &key);
			dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &value);
		}
		microfeed_item_iterator_free(iterator);
		call_publisher_method(publisher, uri, microfeed_item_get_uid(item), message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}

	return retvalue;
}

int microfeed_subscriber_modify_item(MicrofeedSubscriber*subscriber, const char* publisher_identifier, const char* uri, MicrofeedItem* item, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	DBusMessage* message;
	DBusMessageIter iter;
	const char* uid;
	MicrofeedItemIterator* iterator;
	const char* key;
	const char* value;
	
	if ((publisher = get_publisher(subscriber, publisher_identifier, uri, microfeed_item_get_uid(item), callback, user_data))) {
		message = dbus_message_new_method_call(publisher->provider->bus_name, publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_MODIFY_ITEM);
		dbus_message_iter_init_append(message, &iter);
		uid = microfeed_item_get_uid(item);
		dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &uri);
		dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &uid);
		for (iterator = microfeed_item_iterate_properties(item, NULL);
		     microfeed_item_iterator_get(iterator, &key, &value);
		     microfeed_item_iterator_next(iterator)) {
			dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &key);
			dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &value);
		}
		microfeed_item_iterator_free(iterator);
		call_publisher_method(publisher, uri, microfeed_item_get_uid(item), message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}

	return retvalue;
}

int microfeed_subscriber_remove_item(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberReplyCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	DBusMessage* message;
	
	if ((publisher = get_publisher(subscriber, publisher_identifier, uri, uid, callback, user_data))) {
		message = dbus_message_new_method_call(publisher->provider->bus_name, publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_REMOVE_ITEM);
		dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID);
		call_publisher_method(publisher, uri, uid, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}

	return retvalue;

}

int microfeed_subscriber_subscribe_feed(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, MicrofeedSubscriberCallbacks* callbacks, void* callbacks_user_data, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	Feed* feed;
	DBusMessage* message;
	char buffer[1024];

	if ((publisher = get_publisher(subscriber, publisher_identifier, uri, NULL, callback, user_data))) {
		if ((feed = microfeed_store_get(publisher->feeds, uri, Feed))) {
			feed_add_subscription(feed, callbacks, callbacks_user_data);
			if (callback) {
				callback(subscriber, publisher_identifier, uri, NULL, NULL, NULL, user_data);
			}
		} else {
			feed = feed_new(publisher, uri);
			feed_add_subscription(feed, callbacks, callbacks_user_data);

			snprintf(buffer, 1024, "type='signal',path='%s',interface='%s',arg0='%s'", publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, uri);
			dbus_bus_add_match(subscriber->connection, buffer, NULL);
			snprintf(buffer, 1024, "type='signal',path='%s',interface='%s',arg0='%s'", publisher->object_path, MICROFEED_DBUS_INTERFACE_ERROR, uri);
			dbus_bus_add_match(subscriber->connection, buffer, NULL);
		
			message = dbus_message_new_method_call(feed->publisher->provider->bus_name, feed->publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_SUBSCRIBE_FEED);
			dbus_message_append_args(message, DBUS_TYPE_STRING, &feed->uri, DBUS_TYPE_INVALID);
			call_publisher_method(publisher, uri, NULL, message, callback, user_data);
			dbus_message_unref(message);
			retvalue = 1;
		}
	}
	
	return retvalue;
}

int microfeed_subscriber_unsubscribe_feed(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, MicrofeedSubscriberCallbacks* callbacks, void* callbacks_user_data, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Feed* feed;
	Publisher* publisher;
	DBusMessage* message;
	char buffer[1024];
	
	if ((feed = get_feed(subscriber, publisher_identifier, uri, NULL, callback, user_data))) {
		publisher = feed->publisher;
		feed_remove_subscription(feed, callbacks, callbacks_user_data);
		if (feed->subscriptions) {
			if (callback) {
				callback(subscriber, publisher_identifier, uri, NULL, NULL, NULL, user_data);
			}
		} else {
			snprintf(buffer, 1024, "type='signal',path='%s',interface='%s',arg0='%s'", publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, uri);
			dbus_bus_remove_match(subscriber->connection, buffer, NULL);
			snprintf(buffer, 1024, "type='signal',path='%s',interface='%s',arg0='%s'", publisher->object_path, MICROFEED_DBUS_INTERFACE_ERROR, uri);
			dbus_bus_remove_match(subscriber->connection, buffer, NULL);

			message = dbus_message_new_method_call(feed->publisher->provider->bus_name, feed->publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_UNSUBSCRIBE_FEED);
			dbus_message_append_args(message, DBUS_TYPE_STRING, &feed->uri, DBUS_TYPE_INVALID);
			call_publisher_method(feed->publisher, feed->uri, NULL, message, callback, user_data);
			dbus_message_unref(message);
			retvalue = 1;

			publisher = feed->publisher;
			feed_free(feed);
			if (microfeed_store_get_size(publisher->feeds) == 0) {
				publisher_unref(publisher);
			}
		}
	}
	
	return retvalue;
}

int microfeed_subscriber_update_feed(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Feed* feed;
	DBusMessage* message;
	
	if ((feed = get_feed(subscriber, publisher_identifier, uri, NULL, callback, user_data))) {
		message = dbus_message_new_method_call(feed->publisher->provider->bus_name, feed->publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_UPDATE_FEED);
		dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_INVALID);
		call_publisher_method(feed->publisher, feed->uri, NULL, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}
	
	return retvalue;
}

int microfeed_subscriber_republish_items(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* start_uid, const char* end_uid, unsigned int max_count, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Feed* feed;
	DBusMessage* message;
	const char* start_uid_dbus;
	const char* end_uid_dbus;
	dbus_uint16_t max_count_dbus;

	if ((feed = get_feed(subscriber, publisher_identifier, uri, start_uid, callback, user_data))) {
		message = dbus_message_new_method_call(feed->publisher->provider->bus_name, feed->publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_REPUBLISH_ITEMS);
		start_uid_dbus = (start_uid ? start_uid : "");
		end_uid_dbus = (end_uid ? end_uid : "");
		max_count_dbus = max_count;
		dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &start_uid_dbus, DBUS_TYPE_STRING, &end_uid_dbus, DBUS_TYPE_UINT16, &max_count_dbus, DBUS_TYPE_INVALID);
		call_publisher_method(feed->publisher, feed->uri, start_uid, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}
	
	return retvalue;
}

int microfeed_subscriber_store_data(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* url, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	DBusMessage* message;
	
	if ((publisher = get_publisher(subscriber, publisher_identifier, url, NULL, callback, user_data))) {
		message = dbus_message_new_method_call(publisher->provider->bus_name, publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_STORE_DATA);
		dbus_message_append_args(message, DBUS_TYPE_STRING, &url, DBUS_TYPE_INVALID);
		call_publisher_method(publisher, url, NULL, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}
	
	return retvalue;
}

void microfeed_subscriber_add_data_stored_callback(MicrofeedSubscriber* subscriber, const char* publisher_identifier, MicrofeedSubscriberDataCallback callback, void* user_data) {
	Publisher* publisher;
	DataStoredCallback* data_stored_callback;
	char buffer[1024];
	
	if ((publisher = get_publisher(subscriber, publisher_identifier, NULL, NULL, NULL, NULL))) {
		data_stored_callback = microfeed_memory_allocate(DataStoredCallback);
		data_stored_callback->callback = callback;
		data_stored_callback->user_data = user_data;
		data_stored_callback->next = publisher->data_stored_callbacks;
		publisher->data_stored_callbacks = data_stored_callback;

		snprintf(buffer, 1024, "type='signal',path='%s',interface='%s',member='%s'", publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_SIGNAL_NAME_DATA_STORED);
		dbus_bus_add_match(subscriber->connection, buffer, NULL);
	}
}

void microfeed_subscriber_remove_data_stored_callback(MicrofeedSubscriber* subscriber, const char* publisher_identifier, MicrofeedSubscriberDataCallback callback, void* user_data) {
	Publisher* publisher;
	DataStoredCallback* data_stored_callback;
	char buffer[1024];
	
	if ((publisher = get_publisher(subscriber, publisher_identifier, NULL, NULL, NULL, NULL))) {
		for (data_stored_callback = publisher->data_stored_callbacks; data_stored_callback; data_stored_callback = data_stored_callback->next) {
			if (data_stored_callback->callback == callback && data_stored_callback->user_data == user_data) {
				if (data_stored_callback->next) {
					data_stored_callback->next->previous = data_stored_callback->previous;
				}
				if (data_stored_callback->previous) {
					data_stored_callback->previous->next = data_stored_callback->next;
				} else {
					publisher->data_stored_callbacks = data_stored_callback->next;
				}

				microfeed_memory_free(data_stored_callback);

				snprintf(buffer, 1024, "type='signal',path='%s',interface='%s',member='%s'", publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_SIGNAL_NAME_DATA_STORED);
				dbus_bus_remove_match(subscriber->connection, buffer, NULL);

				break;
			}
		}
	}
}

int microfeed_subscriber_create_publisher(MicrofeedSubscriber* subscriber, const char* publisher_identifier, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	DBusMessage* message;
	
	microfeed_configuration_invalidate(subscriber->configuration);
	if (microfeed_configuration_get_publisher_directory(subscriber->configuration, publisher_identifier)) {
		if (callback) {
			callback(subscriber, publisher_identifier, NULL, NULL, MICROFEED_ERROR_PUBLISHER_ALREADY_EXISTS, "Trying to create a publisher that already exists.", user_data);
		}		
	} else if ((publisher = get_publisher(subscriber, publisher_identifier, NULL, NULL, callback, user_data))) {
		message = dbus_message_new_method_call(publisher->provider->bus_name, publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_CREATE_PUBLISHER);
		call_publisher_method(publisher, NULL, NULL, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}

	return retvalue;
}

int microfeed_subscriber_destroy_publisher(MicrofeedSubscriber* subscriber, const char* publisher_identifier, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	DBusMessage* message;
	
	if ((publisher = get_publisher(subscriber, publisher_identifier, NULL, NULL, callback, user_data))) {
		message = dbus_message_new_method_call(publisher->provider->bus_name, publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_DESTROY_PUBLISHER);
		call_publisher_method(publisher, NULL, NULL, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}
	
	return retvalue;
}

int microfeed_subscriber_mark_item(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	DBusMessage* message;
	
	if ((publisher = get_publisher(subscriber, publisher_identifier, NULL, NULL, callback, user_data))) {
		message = dbus_message_new_method_call(publisher->provider->bus_name, publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_MARK_ITEM);
		dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID);
		call_publisher_method(publisher, uri, uid, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}
	
	return retvalue;
}

int microfeed_subscriber_unmark_item(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	DBusMessage* message;
	
	if ((publisher = get_publisher(subscriber, publisher_identifier, NULL, NULL, callback, user_data))) {
		message = dbus_message_new_method_call(publisher->provider->bus_name, publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_UNMARK_ITEM);
		dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID);
		call_publisher_method(publisher, uri, uid, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}
	
	return retvalue;	
}

int microfeed_subscriber_read_item(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberReplyCallback callback, void* user_data) {
	int retvalue = 0;
	Feed* feed;
	DBusMessage* message;

	if ((feed = get_feed(subscriber, publisher_identifier, uri, uid, callback, user_data))) {
		message = dbus_message_new_method_call(feed->publisher->provider->bus_name, feed->publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_READ_ITEM);
		dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID);
		call_publisher_method(feed->publisher, feed->uri, uid, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}
	
	return retvalue;
}

MicrofeedConfiguration* microfeed_subscriber_get_configuration(MicrofeedSubscriber* subscriber) {
	
	return subscriber->configuration;
}

int microfeed_subscriber_handle_configured_subscriptions(MicrofeedSubscriber* subscriber, MicrofeedSubscriberConfiguredSubscribeCallback configured_subscribe, MicrofeedSubscriberConfiguredUnsubscribeCallback configured_unsubscribe, void* user_data) {
	int retvalue = 0;
	const char** subscriptions;
	const char** s;
	MicrofeedStoreIterator* publishers_iterator;
	Publisher* publisher;
	MicrofeedStoreIterator* feeds_iterator;
	Feed* feed;
	Subscription* subscription;
	
	microfeed_configuration_invalidate(subscriber->configuration);
	if ((subscriptions = microfeed_configuration_get_subscriptions(subscriber->configuration, subscriber->identifier))) {
		retvalue = 1;

		/* Subscribe new publishers. */
		for (s = subscriptions; *s; s++) {
			if (!microfeed_store_get(subscriber->publishers_by_identifier, *s, Publisher)) {
				configured_subscribe(subscriber, *s, user_data);
			}
		}

		/* Unsubscribe non-existing feeds. */
		for (publishers_iterator = microfeed_store_iterate(subscriber->publishers_by_identifier, NULL);
		     (publisher = microfeed_store_iterator_get(publishers_iterator, Publisher));
		     microfeed_store_iterator_next(publishers_iterator)) {
			for (s = subscriptions; *s; s++) {
				if (!strcmp(publisher->identifier, *s)) {
					break;
				}		
			}
			if (!*s) {
				for (feeds_iterator = microfeed_store_iterate(publisher->feeds, NULL);
				     (feed = microfeed_store_iterator_get(feeds_iterator, Feed));
				     microfeed_store_iterator_next(feeds_iterator)) {
					for (subscription = feed->subscriptions; subscription; subscription = subscription->next) {
						configured_unsubscribe(subscriber, publisher->identifier, feed->uri, subscription->callbacks, subscription->user_data, user_data);
					}
				}
			}
		}
	}

	return retvalue;
}

static Provider* provider_new(MicrofeedSubscriber* subscriber, const char* bus_name) {
	Provider* provider;
	
	provider = microfeed_memory_allocate(Provider);
	provider->subscriber = subscriber;
	provider->bus_name = strdup(bus_name);
	provider->publishers_by_object_path = microfeed_store_new_sorted_data((MicrofeedStoreCompareKeysFunction)strcmp,
	                                                                      (MicrofeedStoreGetKeyFunction)publisher_get_object_path);

	microfeed_store_insert(subscriber->providers_by_bus_name, provider);
	
	return provider;
}

static void provider_free(Provider* provider) {
	microfeed_store_remove(provider->subscriber->providers_by_bus_name, provider);
	provider_set_unique_connection_name(provider, NULL);

	free(provider->bus_name);
	free(provider->unique_connection_name);
	microfeed_store_free(provider->publishers_by_object_path);
	provider->subscriber = NULL;
	provider->bus_name = NULL;
	provider->unique_connection_name = NULL;
	provider->publishers_by_object_path = NULL;
	microfeed_memory_free(provider);
}

static const char* provider_get_bus_name(Provider* provider) {

	return provider->bus_name;
}

static const char* provider_get_unique_connection_name(Provider* provider) {

	return (provider->unique_connection_name ? provider->unique_connection_name : "");
}

static void provider_set_unique_connection_name(Provider* provider, const char* unique_connection_name) {
	char buffer[1024];

	if (unique_connection_name) {
		if (provider->unique_connection_name) {
			provider_set_unique_connection_name(provider, NULL);
		}
		provider->unique_connection_name = strdup(unique_connection_name);
		microfeed_store_insert(provider->subscriber->providers_by_unique_connection_name, provider);

		snprintf(buffer, 1024, "type='signal',sender='org.freedesktop.DBus',interface='org.freedesktop.DBus',member='NameOwnerChanged',arg0='%s'", provider->unique_connection_name);
		dbus_bus_add_match(provider->subscriber->connection, buffer, NULL);
	} else if (provider->unique_connection_name) {
		snprintf(buffer, 1024, "type='signal',sender='org.freedesktop.DBus',interface='org.freedesktop.DBus',member='NameOwnerChanged',arg0='%s'", provider->unique_connection_name);
		dbus_bus_remove_match(provider->subscriber->connection, buffer, NULL);

		microfeed_store_remove(provider->subscriber->providers_by_unique_connection_name, provider);
		free(provider->unique_connection_name);
		provider->unique_connection_name = NULL;	
	}
}

static Publisher* publisher_new(Provider* provider, const char* identifier) {
	Publisher* publisher = NULL;
	char* separator;
	
	publisher = microfeed_memory_allocate(Publisher);
	publisher->reference_count = 1;
	publisher->identifier = strdup(identifier);
	if ((separator = strchr(publisher->identifier, MICROFEED_PUBLISHER_IDENTIFIER_SEPARATOR_CHAR))) {
		*separator = 0;
		publisher->object_path = microfeed_util_string_concatenate(MICROFEED_DBUS_OBJECT_PATH_PREFIX_PUBLISHER, publisher->identifier, NULL);
		*separator = MICROFEED_PUBLISHER_IDENTIFIER_SEPARATOR_CHAR;
	} else {
		publisher->object_path = strdup(identifier);
	}
	publisher->feeds = microfeed_store_new_sorted_data((MicrofeedStoreCompareKeysFunction)strcmp,
		                                           (MicrofeedStoreGetKeyFunction)feed_get_uri);
	publisher->provider = provider;

	microfeed_store_insert(provider->publishers_by_object_path, publisher);
	microfeed_store_insert(provider->subscriber->publishers_by_identifier, publisher);
	
	return publisher;
}

static void publisher_free(Publisher* publisher) {
	microfeed_store_remove(publisher->provider->subscriber->publishers_by_identifier, publisher);
	microfeed_store_remove(publisher->provider->publishers_by_object_path, publisher);
	if (microfeed_store_get_size(publisher->provider->publishers_by_object_path) == 0) {
		provider_free(publisher->provider);
	}
	
	free(publisher->identifier);
	free(publisher->object_path);
	publisher->provider = NULL;
	publisher->identifier = NULL;
	publisher->object_path = NULL;
	microfeed_store_free(publisher->feeds);
	microfeed_memory_free(publisher);
}

static Publisher* publisher_ref(Publisher* publisher) {
	publisher->reference_count++;
	
	return publisher;
}

static void publisher_unref(Publisher* publisher) {
	publisher->reference_count--;
	if (publisher->reference_count == 0) {
		publisher_free(publisher);
	}
}

static const char* publisher_get_identifier(Publisher* publisher) {
	
	return publisher->identifier;
}

static const char* publisher_get_object_path(Publisher* publisher) {
	
	return publisher->object_path;
}

static Feed* feed_new(Publisher* publisher, const char* uri) {
	Feed* feed;
	
	feed = microfeed_memory_allocate(Feed);
	feed->reference_count = 1;
	feed->publisher = publisher;
	feed->uri = strdup(uri);

	microfeed_store_insert(publisher->feeds, feed);
	
	return feed;
}
	
static void feed_free(Feed* feed) {
	microfeed_store_remove(feed->publisher->feeds, feed);

	free(feed->uri);
	free(feed->name);
	feed->publisher = NULL;
	feed->uri = NULL;
	feed->name = NULL;
	microfeed_memory_free(feed);
}
/*
Feed* feed_ref(Feed* feed) {
	feed->reference_count++;
	
	return feed;
}

void feed_unref(Feed* feed) {
	feed->reference_count--;
	if (feed->reference_count == 0) {
		feed_free(feed);
	}
}
*/

static const char* feed_get_uri(Feed* feed) {

	return feed->uri;
}

static void feed_add_subscription(Feed* feed, MicrofeedSubscriberCallbacks* callbacks, void* user_data) {
	Subscription* s;
	
	for (s = feed->subscriptions; s; s = s->next) {
		if (s->callbacks == callbacks && s->user_data == user_data) {
			break;
		}
	}
	if (!s) {
		s = microfeed_memory_allocate(Subscription);
		s->callbacks = callbacks;
		s->user_data = user_data;
		
		s->next = feed->subscriptions;
		feed->subscriptions = s;
	}
}

static void feed_remove_subscription(Feed* feed, MicrofeedSubscriberCallbacks* callbacks, void* user_data) {
	Subscription* s;
	
	for (s = feed->subscriptions; s; s = s->next) {
		if (s->callbacks == callbacks && s->user_data == user_data) {
			break;
		}
	}
	if (s) {
		if (s->next) {
			s->next->previous = s->previous;
		}
		if (s->previous) {
			s->previous->next = s->next;
		} else {
			feed->subscriptions = s->next;
		}

		microfeed_memory_free(s);
	}
}

static void signal_feed_update_started(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	Feed* feed;
	Subscription* s;
	
	if (parse_feed_signal(publisher, message, &feed)) {
		feed->updating = 1;
		for (s = feed->subscriptions; s; s = s->next) {
			if (s->callbacks->feed_update_started) {
				s->callbacks->feed_update_started(subscriber, publisher->identifier, feed->uri, s->user_data);
			}
		}
	}
}

static void signal_feed_update_ended(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	Feed* feed;
	Subscription* s;
	
	if (parse_feed_signal(publisher, message, &feed)) {
		feed->updating = 0;
		for (s = feed->subscriptions; s; s = s->next) {
			if (s->callbacks->feed_update_ended) {
				s->callbacks->feed_update_ended(subscriber, publisher->identifier, feed->uri, s->user_data);
			}
		}
	}
}

static void signal_feed_republishing_started(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	Feed* feed;
	Subscription* s;
	
	if (parse_feed_signal(publisher, message, &feed)) {
		feed->republishing = 1;
		for (s = feed->subscriptions; s; s = s->next) {
			if (s->callbacks->feed_republishing_started) {
				s->callbacks->feed_republishing_started(subscriber, publisher->identifier, feed->uri, s->user_data);
			}
		}
	}
}

static void signal_feed_republishing_ended(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	Feed* feed;
	Subscription* s;
	
	if (parse_feed_signal(publisher, message, &feed)) {
		feed->republishing = 0;
		for (s = feed->subscriptions; s; s = s->next) {
			if (s->callbacks->feed_republishing_ended) {
				s->callbacks->feed_republishing_ended(subscriber, publisher->identifier, feed->uri, s->user_data);
			}
		}
	}
}

static void signal_item_added(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	Feed* feed;
	MicrofeedItem* item;
	Subscription* s;
	
	if (parse_item_signal(publisher, message, &feed, &item)) {
		for (s = feed->subscriptions; s; s = s->next) {
			if (s->callbacks->item_added) {
				s->callbacks->item_added(subscriber, publisher->identifier, feed->uri, item, s->user_data);
			}
		}
		microfeed_item_free(item);
	}
}

static void signal_item_changed(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	Feed* feed;
	MicrofeedItem* item;
	Subscription* s;
	
	if (parse_item_signal(publisher, message, &feed, &item)) {
		for (s = feed->subscriptions; s; s = s->next) {
			if (s->callbacks->item_changed) {
				s->callbacks->item_changed(subscriber, publisher->identifier, feed->uri, item, s->user_data);
			}
		}
		microfeed_item_free(item);
	}
}

static void signal_item_republished(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	Feed* feed;
	MicrofeedItem* item;
	Subscription* s;
	
	if (parse_item_signal(publisher, message, &feed, &item)) {
		for (s = feed->subscriptions; s; s = s->next) {
			if (s->callbacks->item_republished) {
				s->callbacks->item_republished(subscriber, publisher->identifier, feed->uri, item, s->user_data);
			}
		}
		microfeed_item_free(item);
	}
}

static void signal_item_removed(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	DBusError error;
	char* uri;
	char* uid;
	Feed* feed;
	Subscription* s;
	
	dbus_error_init(&error);
	if (dbus_message_get_args(message, &error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID) &&
	    (feed = microfeed_store_get(publisher->feeds, uri, Feed))) {
		for (s = feed->subscriptions; s; s = s->next) {
			if (s->callbacks->item_removed) {
				s->callbacks->item_removed(subscriber, publisher->identifier, feed->uri, uid, s->user_data);
			}
		}
	}
}

static void signal_item_status_changed(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	DBusError error;
	char* uri;
	char* uid;
	char status;
	Feed* feed;
	Subscription* s;
	
	dbus_error_init(&error);
	if (dbus_message_get_args(message, &error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_BYTE, &status, DBUS_TYPE_INVALID) &&
	    (feed = microfeed_store_get(publisher->feeds, uri, Feed))) {
		for (s = feed->subscriptions; s; s = s->next) {
			if (s->callbacks->item_status_changed) {
				s->callbacks->item_status_changed(subscriber, publisher->identifier, feed->uri, uid, (MicrofeedItemStatus)status, s->user_data);
			}
		}
	}
}

static void signal_data_stored(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	DBusError error;
	char* url;
	char* file;
	char* path = NULL;
	DataStoredCallback* data_stored_callback;

	dbus_error_init(&error);
	if (dbus_message_get_args(message, &error, DBUS_TYPE_STRING, &url, DBUS_TYPE_STRING, &file, DBUS_TYPE_INVALID) && publisher->data_stored_callbacks) {
		path = microfeed_util_string_concatenate(microfeed_configuration_get_publisher_directory(subscriber->configuration, publisher->identifier), "/", publisher->identifier, "/data/", file, NULL);
		for (data_stored_callback = publisher->data_stored_callbacks; data_stored_callback; data_stored_callback = data_stored_callback->next) {
			data_stored_callback->callback(subscriber, publisher->identifier, url, path, data_stored_callback->user_data);
		}
		free(path);	
	}
}

static Publisher* get_publisher(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	Publisher* publisher;
	const char* bus_name;
	Provider* provider;
	
	if (!(publisher = microfeed_store_get(subscriber->publishers_by_identifier, publisher_identifier, Publisher))) {
		if ((bus_name = strchr(publisher_identifier, MICROFEED_PUBLISHER_IDENTIFIER_SEPARATOR_CHAR)) && bus_name[1] != 0) {
			bus_name++;
			if (!(provider = microfeed_store_get(subscriber->providers_by_bus_name, bus_name, Provider))) {
				microfeed_configuration_invalidate(subscriber->configuration);
				if (microfeed_configuration_get_provider_name(subscriber->configuration, bus_name)) {
					provider = provider_new(subscriber, bus_name);
				} else if (callback) {
					callback(subscriber, publisher_identifier, uri, uid, MICROFEED_ERROR_NO_SUCH_PROVIDER, "Provider for the publisher does not exist.", user_data);
				}
			}
			if (provider) {
				publisher = publisher_new(provider, publisher_identifier);
			}
		} else if (callback) {
			callback(subscriber, publisher_identifier, uri, uid, MICROFEED_ERROR_INVALID_PUBLISHER_IDENTIFIER, "Publisher identifier was invalid.", user_data);
		}	    
	}

	return publisher;
}

static Feed* get_feed(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	Publisher* publisher;
	Feed* feed = NULL;
	
	if (!(publisher = microfeed_store_get(subscriber->publishers_by_identifier, publisher_identifier, Publisher))) {
		if (callback) {
			callback(subscriber, publisher_identifier, uri, uid, MICROFEED_ERROR_NO_SUCH_PUBLISHER, "Publisher does not exist.", user_data);
		}	    
	} else if (!(feed = microfeed_store_get(publisher->feeds, uri, Feed))) {
		if (callback) {
			callback(subscriber, publisher_identifier, uri, uid, MICROFEED_ERROR_FEED_NOT_SUBSCRIBED, "Trying to access a feed that is not subscribed.", user_data);
		}
	}
	
	return feed;
}

static MicrofeedItem* parse_item_from_message(DBusMessageIter* iter) {
	MicrofeedItem* item = NULL;
	char* uid;
	dbus_uint64_t timestamp;
	unsigned char status;
	char* key;
	char* value;

	if (dbus_message_iter_next(iter) && dbus_message_iter_get_arg_type(iter) == DBUS_TYPE_STRING) {
		dbus_message_iter_get_basic(iter, &uid);
		if (dbus_message_iter_next(iter) && dbus_message_iter_get_arg_type(iter) == DBUS_TYPE_UINT64) {
			dbus_message_iter_get_basic(iter, &timestamp);
			if (dbus_message_iter_next(iter) && dbus_message_iter_get_arg_type(iter) == DBUS_TYPE_BYTE) {
				dbus_message_iter_get_basic(iter, &status);
				item = microfeed_item_new_with_status(uid, (time_t)timestamp, (MicrofeedItemStatus)status);
				while (dbus_message_iter_next(iter) && dbus_message_iter_get_arg_type(iter) == DBUS_TYPE_STRING) {
					dbus_message_iter_get_basic(iter, &key);
					if (dbus_message_iter_next(iter) && dbus_message_iter_get_arg_type(iter) == DBUS_TYPE_STRING) {
						dbus_message_iter_get_basic(iter, &value);
						microfeed_item_set_property(item, key, value);
					}
				}
			}
		} else {
			item = microfeed_item_new(uid, 0);
		}
	}
	
	return item;
}

static int parse_item_signal(Publisher* publisher, DBusMessage* message, Feed** feed_return, MicrofeedItem** item_return) {
	int retvalue = 0;
	DBusMessageIter iter;
	char* uri;
	Feed* feed;
	MicrofeedItem* item;
	
	if (dbus_message_iter_init(message, &iter) && dbus_message_iter_get_arg_type(&iter) == DBUS_TYPE_STRING) {
		dbus_message_iter_get_basic(&iter, &uri);
		if ((feed = microfeed_store_get(publisher->feeds, uri, Feed)) && 
		    (item = parse_item_from_message(&iter))) {
			*feed_return = feed;
			*item_return = item;
			retvalue = 1;
		}
	}

	return retvalue;
}

static int parse_feed_signal(Publisher* publisher, DBusMessage* message, Feed** feed_return) {
	int retvalue = 0;
	DBusError error;
	const char* uri;
	Feed* feed;
	
	dbus_error_init(&error);
	if (dbus_message_get_args(message, &error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_INVALID)) {
		if ((feed = microfeed_store_get(publisher->feeds, uri, Feed))) {
			*feed_return = feed;
			retvalue = 1;
		}
	} else {
		dbus_error_free(&error);
	}
	
	return retvalue;
}

static void handle_publisher_method_return(DBusPendingCall* pending, void* user_data) {
	MethodReturnData* data;
	DBusMessage* reply;
	DBusError error;
	const char* error_name;
	const char* error_message;
	int retrying = 0;
	DBusPendingCall* pending_call;
	
	data = (MethodReturnData*)user_data;
	reply = dbus_pending_call_steal_reply(pending);
	if (!reply) {
		if (data->callback) {
			data->callback(data->publisher->provider->subscriber, data->publisher->identifier, data->uri, data->uid, MICROFEED_ERROR_DBUS_MESSAGE_FAILED, "No reply from the publisher.", data->user_data);
		}
	} else if (dbus_message_get_type(reply) == DBUS_MESSAGE_TYPE_ERROR &&
	           !microfeed_util_string_starts_with(dbus_message_get_error_name(reply), MICROFEED_DBUS_INTERFACE_PREFIX)) {
		if (data->retry_counter < data->publisher->provider->subscriber->max_retries) {
			retrying = 1;
			if (data->retry_counter) {
				/* TODO: Some pause would be nice, but sleep(data->retry_counter); is not possible. */
			}
			data->retry_counter++;
			
			dbus_connection_send_with_reply(data->publisher->provider->subscriber->connection, data->message, &pending_call, MICROFEED_SUBSCRIBER_DBUS_METHOD_CALL_TIMEOUT);
			if (pending_call) {
				dbus_pending_call_set_notify(pending_call, handle_publisher_method_return, data, NULL);
			}
		} else if (data->callback) {
			if (!(error_name = dbus_message_get_error_name(reply))) {
				error_name = MICROFEED_ERROR_UNKNOWN;
			}
			dbus_error_init(&error);
			if (!dbus_message_get_args(reply, &error, DBUS_TYPE_STRING, &error_message, DBUS_TYPE_INVALID)) {
				error_message = NULL;
			}
			data->callback(data->publisher->provider->subscriber, data->publisher->identifier, data->uri, data->uid, error_name, error_message, data->user_data);
		}
	} else {
		if (!data->publisher->provider->unique_connection_name) {
			provider_set_unique_connection_name(data->publisher->provider, dbus_message_get_sender(reply));
		}
		if (data->callback) {
			data->callback(data->publisher->provider->subscriber, data->publisher->identifier, data->uri, data->uid, NULL, NULL, data->user_data);
		}
	}
	if (reply) {
		dbus_message_unref(reply);
	}
		
	if (!retrying) {
		dbus_message_unref(data->message);
		publisher_unref(data->publisher);	
		free(data->uri);
		free(data->uid);
		microfeed_memory_free(data);		
	}
}

static void call_publisher_method(Publisher* publisher, const char* uri, const char* uid, DBusMessage* message, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	DBusPendingCall* pending_call;
	MethodReturnData* data;
	
	dbus_connection_send_with_reply(publisher->provider->subscriber->connection, message, &pending_call, MICROFEED_SUBSCRIBER_DBUS_METHOD_CALL_TIMEOUT);
	if (pending_call) {
		data = microfeed_memory_allocate(MethodReturnData);
		data->message = dbus_message_copy(message);
		data->publisher = publisher_ref(publisher);
		if (uri) {
			data->uri = strdup(uri);
		} else {
			data->uri = NULL;
		}
		if (uid) {
			data->uid = strdup(uid);
		} else {
			data->uid = NULL;
		}
		data->callback = callback;
		data->user_data = user_data;
		dbus_pending_call_set_notify(pending_call, handle_publisher_method_return, data, NULL);
	}
}

static void object_unregister(DBusConnection* connection, void* user_data) {
}

static DBusHandlerResult object_message(DBusConnection* connection, DBusMessage* message, void* user_data) {
	DBusHandlerResult result = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	MicrofeedSubscriber* subscriber;
	const char* name;
	const char* old_owner;
	const char* new_owner;
	MicrofeedStoreIterator* publisher_iterator;
	MicrofeedStoreIterator* feed_iterator;
	Feed* feed;
	int i;
	const char* path;
	Provider* provider;
	Publisher* publisher;
	DBusMessage* reply;
	DBusError error;
	const char* error_name;
	char* error_message;
	char* uri;
	char* uid;
	Subscription* s;

	subscriber = (MicrofeedSubscriber*)user_data;
	if (dbus_message_is_signal(message, "org.freedesktop.DBus", "NameOwnerChanged") &&
	    dbus_message_get_args(message, NULL, DBUS_TYPE_STRING, &name, DBUS_TYPE_STRING, &old_owner, DBUS_TYPE_STRING, &new_owner, DBUS_TYPE_INVALID) &&
	    name[0] == ':' && new_owner[0] == 0 && !strcmp(name, old_owner) && (provider = microfeed_store_get(subscriber->providers_by_unique_connection_name, name, Provider))) {
		for (publisher_iterator = microfeed_store_iterate(provider->publishers_by_object_path, NULL);
		     (publisher = microfeed_store_iterator_get(publisher_iterator, Publisher));
		     microfeed_store_iterator_next(publisher_iterator)) {
			for (feed_iterator = microfeed_store_iterate(publisher->feeds, NULL);
			     (feed = microfeed_store_iterator_get(feed_iterator, Feed));
			     microfeed_store_iterator_next(feed_iterator)) {
				for (s = feed->subscriptions; s; s = s->next) {
					if (feed->republishing && s->callbacks->feed_republishing_ended) {
						s->callbacks->feed_republishing_ended(subscriber, publisher->identifier, feed->uri, s->user_data);
					}
					if (feed->updating && s->callbacks->feed_update_ended) {
						s->callbacks->feed_update_ended(subscriber, publisher->identifier, feed->uri, s->user_data);
					}
					if (s->callbacks->error_occured) {
						s->callbacks->error_occured(subscriber, publisher->identifier, feed->uri, NULL, MICROFEED_ERROR_PROVIDER_CLOSED_CONNECTION, "Provider closed connection", s->user_data);	
					}
				}
				/* TODO: Think! Think! Think! feed_free(feed); */
			}
			microfeed_store_iterator_free(feed_iterator);
			/* TODO: I am quite sure that this should not be here: publisher_unref(publisher); */
		}
		provider_set_unique_connection_name(provider, NULL);
		microfeed_store_iterator_free(publisher_iterator);
	}

	if ((path = dbus_message_get_path(message)) &&
	    (provider = microfeed_store_get(subscriber->providers_by_unique_connection_name, dbus_message_get_sender(message), Provider)) &&
	    (publisher = microfeed_store_get(provider->publishers_by_object_path, path, Publisher))) {
		publisher->last_activity = time(NULL);
		if (dbus_message_is_method_call(message, MICROFEED_DBUS_INTERFACE_SUBSCRIBER, MICROFEED_METHOD_NAME_PING)) {
			reply = dbus_message_new_method_return(message);
			dbus_connection_send(connection, reply, NULL);
			dbus_message_unref(reply);
			result = DBUS_HANDLER_RESULT_HANDLED;
		} else if (dbus_message_get_type(message) == DBUS_MESSAGE_TYPE_SIGNAL && (dbus_message_has_interface(message, MICROFEED_DBUS_INTERFACE_ERROR) || dbus_message_has_interface(message, MICROFEED_DBUS_INTERFACE_ERROR_TO_DESTINATION))) {
			dbus_error_init(&error);
			if ((error_name = dbus_message_get_member(message)) && dbus_message_get_args(message, &error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_STRING, &error_message, DBUS_TYPE_INVALID)) {
				if ((feed = microfeed_store_get(publisher->feeds, uri, Feed))) { 
					for (s = feed->subscriptions; s; s = s->next) {
						if (s->callbacks->error_occured) {
							s->callbacks->error_occured(subscriber, publisher->identifier, uri, uid, error_name, error_message, s->user_data);
						}
					}
				}
			}
		} else {
			for (i = 0; signal_callbacks[i].name; i++) {
				if (dbus_message_is_signal(message, MICROFEED_DBUS_INTERFACE_PUBLISHER, signal_callbacks[i].name) ||
				    dbus_message_is_signal(message, MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION, signal_callbacks[i].name)) {
					signal_callbacks[i].callback(subscriber, publisher, message);
					result = DBUS_HANDLER_RESULT_HANDLED;
					break;
				}
			}
		}
	}
	
	return result;
}
