
#include <microfeed-common/microfeedstore.h>
#include <microfeed-common/microfeedmisc.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#define DELTA 10

struct _MicrofeedStore {
	void** data;
	unsigned int size;
	unsigned int reserved;
	MicrofeedStoreCompareKeysFunction compare_keys;
	MicrofeedStoreGetKeyFunction get_key;
	MicrofeedStoreIterator* iterators;
	int unsorted;
};

struct _MicrofeedStoreIterator {
	MicrofeedStoreIterator* previous;
	MicrofeedStoreIterator* next;
	
	MicrofeedStore* store;
	unsigned int index;
	void* current_data;
};

static int get_index(MicrofeedStore* store, const void* key, unsigned int* index);

MicrofeedStore* microfeed_store_new_sorted(MicrofeedStoreCompareKeysFunction compare_keys, MicrofeedStoreGetKeyFunction get_key) {
	MicrofeedStore* store;
	
	store = microfeed_memory_allocate(MicrofeedStore);
	store->compare_keys = compare_keys;
	store->get_key = get_key;
	
	return store;
}

MicrofeedStore* microfeed_store_new_unsorted(MicrofeedStoreCompareKeysFunction compare_keys, MicrofeedStoreGetKeyFunction get_key) {
	MicrofeedStore* store;
	
	store = microfeed_store_new_sorted(compare_keys, get_key);
	store->unsorted = 1;
	
	return store;
}

void microfeed_store_free(MicrofeedStore* store) {
	MicrofeedStoreIterator* iterator;
	
	for (iterator = store->iterators; iterator; iterator = iterator->next) {
		iterator->store = NULL;
	}	
	free(store->data);
	store->data = NULL;
	store->iterators = NULL;
	store->size = store->reserved = 0;
	free(store);
}

int microfeed_store_is_sorted(MicrofeedStore* store) {
	
	return !store->unsorted;
}

void microfeed_store_foreach(MicrofeedStore* store, MicrofeedStoreForeachFunction foreach, void* user_data) {
	int i;
	
	for (i = 0; i < store->size; i++) {
		foreach(store->data[i], user_data);
	}
}

void* microfeed_store_get_impl(MicrofeedStore* store, const void* key) {
	void* data = NULL;
	unsigned int index;

	if (get_index(store, key, &index)) {
		data = store->data[index];
	}

	return data;
}

void* microfeed_store_get_index_impl(MicrofeedStore* store, unsigned int index) {
	void* data = NULL;
	
	if (index < store->size) {
		data = store->data[index];
	}
	
	return data;	
}

unsigned int microfeed_store_get_size(MicrofeedStore* store) {

	return store->size;
}

int microfeed_store_insert(MicrofeedStore* store, void* data) {
	int retvalue = 0;
	unsigned int index;
	MicrofeedStoreIterator* iterator;

	if (!get_index(store, store->get_key(data), &index)) {
		if (store->size == store->reserved) {
			store->reserved += DELTA;
			store->data = realloc(store->data, store->reserved * sizeof(void*));
		}
	
		if (index < store->size) {
			memmove(store->data + index + 1, store->data + index,
			        (store->size - index) * sizeof(void*));
		}
		store->size++;
		store->data[index] = data;

		retvalue = 1;
		
		for (iterator = store->iterators; iterator; iterator = iterator->next) {
			if (iterator->index <= index) {
				iterator->index++;
			}
		}
	}
	
	return retvalue;
}

void* microfeed_store_replace_impl(MicrofeedStore* store, void* data) {
	unsigned int index;
	void* existing;
	
	if (get_index(store, store->get_key(data), &index)) {
		existing = store->data[index];
	} else {
		existing = NULL;
		if (store->size == store->reserved) {
			store->reserved += DELTA;
			store->data = realloc(store->data, store->reserved * sizeof(void*));
		}

		if (index < store->size) {
			memmove(store->data + index + 1, store->data + index,
		        	(store->size - index) * sizeof(void*));
		}
		store->size++;
	}
	store->data[index] = data;
	
	return existing;
}

int microfeed_store_remove(MicrofeedStore* store, const void* data) {
	int retvalue = 0;
	unsigned int index;
	MicrofeedStoreIterator* iterator;

	if (get_index(store, store->get_key(data), &index)) {
		if (index + 1 < store->size) {
			memmove(store->data + index, store->data + index + 1,
		        	(store->size - index - 1) * sizeof(void*));
		}

		store->size--;
		if (store->size + 2 * DELTA == store->reserved) {
			store->reserved -= DELTA;
			store->data = realloc(store->data, store->reserved * sizeof(void*));
		}

		retvalue = 1;

		for (iterator = store->iterators; iterator; iterator = iterator->next) {
			if (iterator->index <= index) {
				iterator->index--;
			}
		}
	}

	return retvalue;
}

void* microfeed_store_remove_key_impl(MicrofeedStore* store, const void* key) {
	void* data = NULL;
	unsigned int index;
	MicrofeedStoreIterator* iterator;

	if (get_index(store, key, &index)) {
		data = store->data[index];
		if (index + 1 < store->size) {
			memmove(store->data + index, store->data + index + 1,
		        	(store->size - index - 1) * sizeof(void*));
		}

		store->size--;
		if (store->size + 2 * DELTA == store->reserved) {
			store->reserved -= DELTA;
			store->data = realloc(store->data, store->reserved * sizeof(void*));
		}

		for (iterator = store->iterators; iterator; iterator = iterator->next) {
			if (iterator->index <= index) {
				iterator->index--;
			}
		}
	}

	return data;
}

void* microfeed_store_remove_index_impl(MicrofeedStore* store, unsigned int index) {
	void* data = NULL;
	MicrofeedStoreIterator* iterator;

	if (index < store->size) {
		data = store->data[index];
		if (index + 1 < store->size) {
			memmove(store->data + index, store->data + index + 1,
		        	(store->size - index - 1) * sizeof(void*));
		}

		store->size--;
		if (store->size + 2 * DELTA == store->reserved) {
			store->reserved -= DELTA;
			store->data = realloc(store->data, store->reserved * sizeof(void*));
		}

		for (iterator = store->iterators; iterator; iterator = iterator->next) {
			if (iterator->index <= index) {
				iterator->index--;
			}
		}
	}

	return data;
}

void microfeed_store_remove_and_free_all(MicrofeedStore* store, MicrofeedStoreFreeDataFunction free_data) {
	while (store->size > 0) {
		free_data(store->data[store->size - 1]);
		store->size--;
	}
	if (store->reserved > DELTA) {
		store->reserved = DELTA;
		store->data = realloc(store->data, store->reserved * sizeof(void*));
	}
}

MicrofeedStoreIterator* microfeed_store_iterate(MicrofeedStore* store, const void* start_key) {
	MicrofeedStoreIterator* iterator;
	unsigned int index;
	
	iterator = microfeed_memory_allocate(MicrofeedStoreIterator);
	iterator->store = store;
	if (start_key) {
		if (get_index(store, start_key, &index)) {
			iterator->index = index;
		} else {
			iterator->index = store->size;
		}
	} else {
		iterator->index = 0;
	}
	if (store->iterators) {
		store->iterators->previous = iterator;
		iterator->next = store->iterators;
	} else {
		iterator->next = NULL;
	}
	iterator->previous = NULL;
	store->iterators = iterator;
	
	return iterator;
}

void microfeed_store_sort(MicrofeedStore* store, MicrofeedStoreCompareDatasFunction compare_datas, void* user_data) {
	unsigned int i;
	int j;
	void* data;
	
	if (store->unsorted) {
		for (i = 1; i < store->size; i++) {	
			data = store->data[i];
			for (j = i - 1; j >= 0 && compare_datas(store->data[j], data, j, i, user_data) > 0; j = j - 1) {
				store->data[j + 1] = store->data[j];
			}
			store->data[j + 1] = data;
		}
	}
}

void microfeed_store_iterator_free(MicrofeedStoreIterator* iterator) {
	if (iterator->next) {
		iterator->next->previous = iterator->previous;
	}
	if (iterator->previous) {
		iterator->previous->next = iterator->next;
	} else if (iterator->store) {
		iterator->store->iterators = iterator->next;
	}

	iterator->store = NULL;
	iterator->current_data = NULL;
	microfeed_memory_free(iterator);
}

void* microfeed_store_iterator_get_impl(MicrofeedStoreIterator* iterator) {
	if (!iterator->current_data && iterator->store && iterator->index < iterator->store->size) {
		iterator->current_data = iterator->store->data[iterator->index];
	}
	
	return iterator->current_data;
}

void microfeed_store_iterator_next(MicrofeedStoreIterator* iterator) {
	if (iterator->store && (iterator->index < iterator->store->size || iterator->index == (unsigned int)-1)) {
		iterator->index++;
	}
	iterator->current_data = NULL;
}

int microfeed_store_compare_keys_direct(const void* key1, const void* key2) {

	return (key1 == key2 ? 0 : (key1 < key2 ? -1 : 1));
}

const void* microfeed_store_get_key_direct(const void* data) {
	
	return data;
}

static int get_index(MicrofeedStore* store, const void* key, unsigned int* index) {
	int retval = 0;
	const void* k;
	int result;
	unsigned int i, min, max;

	if (store->unsorted) {
		*index = store->size;
		if (key) {
			for (i = 0; i < store->size; i++) {
				k = store->get_key(store->data[i]);
				if (k && store->compare_keys(key, k) == 0) {
					*index = i;
					retval = 1;
				}
			}
		}
	} else {
		if (store->size == 0) {
			*index = 0;
		} else if ((result = store->compare_keys(key, store->get_key(store->data[0]))) == 0) {
			*index = 0;
			retval = 1;	
		} else if (result < 0) {
			*index = 0;
		} else if (store->size == 1) {
			*index = 1;
		} else if ((result = store->compare_keys(key, store->get_key(store->data[store->size - 1]))) == 0) {
			*index = store->size -1;
			retval = 1;
		} else if (result > 0) {
			*index = store->size;
		} else if (store->size == 2) {
			*index = store->size - 1;
		} else {
			min = i = 0;
			max = store->size - 1;
			
			while (min <= max) {
				i = (min + max) / 2;
				if ((result = store->compare_keys(key, store->get_key(store->data[i]))) == 0) {
					retval = 1;
					break;
				} else if (result < 0) {
					max = i - 1;
				} else {
					i++;
					min = i;
				}
			}
			
			*index = i;
		}
	}

	return retval;
}
