#ifndef MICROFEEDITEM_H
#define MICROFEEDITEM_H

#include <microfeed/microfeedstore.h>

#include <sys/types.h>

/**
 * @addtogroup MicrofeedItem
 *
 * This module implementing an item in a feed is used both in the publisher and in the subscriber side.
 *
 * MicrofeedItem is not thread-safe, so keep one item in one thread only. However, usually the lifetime
 * of an item is short, so there is even no need to access an item from multiple threads. Thus, there is
 * no reference counting available. If you have to access the same item content in multiple threads,
 * use #microfeed_item_duplicate.
 *
 * @{
 */

/**
 * Opaque data type representing Microfeed item.
 */
typedef struct _MicrofeedItem MicrofeedItem;

/**
 * Opaque data type representing Microfeed item iterator that iterates over the properties of the item.
 */
typedef struct _MicrofeedItemIterator MicrofeedItemIterator;

typedef enum {
	MICROFEED_ITEM_STATUS_NONE = 0,
	MICROFEED_ITEM_STATUS_ACTIVE = 1 << 0,
	MICROFEED_ITEM_STATUS_UNREAD = 1 << 1,
	MICROFEED_ITEM_STATUS_MARKED = 1 << 2,
	MICROFEED_ITEM_STATUS_SET = 1 << 7
} MicrofeedItemStatus;

#define MICROFEED_ITEM_STATUS_NEW (MICROFEED_ITEM_STATUS_ACTIVE | MICROFEED_ITEM_STATUS_UNREAD)

MicrofeedItem* microfeed_item_new_temporary(void);
MicrofeedItem* microfeed_item_new(const char* uid, time_t timestamp);
MicrofeedItem* microfeed_item_new_with_status(const char* uid, time_t timestamp, MicrofeedItemStatus status);
void microfeed_item_free(MicrofeedItem* item);
MicrofeedItem* microfeed_item_duplicate(MicrofeedItem* item);

int microfeed_item_demarshal_properties(MicrofeedItem* item, const void* data, size_t size);
const char* microfeed_item_get_uid(MicrofeedItem* item);
time_t microfeed_item_get_timestamp(MicrofeedItem* item);
void microfeed_item_set_timestamp(MicrofeedItem* item, time_t timestamp);
MicrofeedItemStatus microfeed_item_get_status(MicrofeedItem* item);
const char* microfeed_item_get_property(MicrofeedItem* item, const char* key);
MicrofeedItemIterator* microfeed_item_iterate_properties(MicrofeedItem* item, const char* start_key);
void microfeed_item_marshal_properties(MicrofeedItem* item, const void** data_pointer, size_t* size_pointer);
void microfeed_item_set_property(MicrofeedItem* item, const char* key, const char* value);
void microfeed_item_set_property_with_length(MicrofeedItem* item, const char* key, const char* value, size_t value_length);
void microfeed_item_set_property_full(MicrofeedItem* item, const char* key, size_t key_length, const char* value, size_t value_length);
char* microfeed_item_get_properties_as_string(MicrofeedItem* item);
int microfeed_item_set_properties_from_string(MicrofeedItem* item, const char* string);

void microfeed_item_iterator_free(MicrofeedItemIterator* iterator);
int microfeed_item_iterator_get(MicrofeedItemIterator* iterator, const char** key, const char** value);
void microfeed_item_iterator_next(MicrofeedItemIterator* iterator);

/** @} */

#endif
