
#include <microfeed-common/microfeedconfiguration.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeedstore.h>

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <libgen.h>

struct _MicrofeedConfiguration {
	char* providers_environment_directory;
	char* providers_home_directory;
	char* publishers_environment_directory;
	char* publishers_home_directory;
	char* subscribers_environment_directory;
	char* subscribers_home_directory;
	const char** providers;
	const char** provider_directories;
	const char** provider_names;
	const char*** provider_feeds;
	const char** publishers;
	const char** publisher_directories;
	MicrofeedStore* subscriptions;
};

typedef struct {
	char* application_identifier;
	char* filename;
	const char** subscriptions;
} Subscription;

static Subscription* subscription_new(const char* application_identifier, const char* filename, const char** subscriptions);
static void subscription_free(Subscription* subscription);
static const char* subscription_get_application_identifier(Subscription* subscription);
static void subscription_save(Subscription* subscription);

static ssize_t read_line(int fd, size_t offset, char* buffer, size_t buffer_size);
static void get_providers(MicrofeedConfiguration* configuration, const char* directory, unsigned int* length_pointer);
static void get_publishers(MicrofeedConfiguration* configuration, const char* directory, unsigned int* length_pointer);
static void update_providers(MicrofeedConfiguration* configuration);
static void update_publishers(MicrofeedConfiguration* configuration);
static int update_and_get_provider_index(MicrofeedConfiguration* configuration, const char* provider_identifier);
static Subscription* update_and_get_subscription(MicrofeedConfiguration* configuration, const char* application_identifier);

MicrofeedConfiguration* microfeed_configuration_new(void) {
	MicrofeedConfiguration* configuration;
	char* directory;
	size_t length;
	
	configuration = microfeed_memory_allocate(MicrofeedConfiguration);
	if ((directory = getenv("MICROFEED_PROVIDERS_DIRECTORY"))) {
		configuration->providers_environment_directory = strdup(directory);
	}
	if ((directory = getenv("MICROFEED_PUBLISHERS_DIRECTORY"))) {
		configuration->publishers_environment_directory = strdup(directory);
	}
	if ((directory = getenv("MICROFEED_SUBSCRIBERS_DIRECTORY"))) {
		configuration->subscribers_environment_directory = strdup(directory);
	}
	if ((directory = getenv("HOME"))) {
		length = strlen(directory);
		configuration->providers_home_directory = microfeed_memory_allocate_bytes(length + 22); /* Remember to update. */
		memcpy(configuration->providers_home_directory, directory, length);
		memcpy(configuration->providers_home_directory + length, "/.microfeed/providers", 22); /* Remember the NULL byte */
		configuration->publishers_home_directory = microfeed_memory_allocate_bytes(length + 23); /* Remember to update. */
		memcpy(configuration->publishers_home_directory, directory, length);
		memcpy(configuration->publishers_home_directory + length, "/.microfeed/publishers", 23); /* Remember the NULL byte */
		configuration->subscribers_home_directory = microfeed_memory_allocate_bytes(length + 24); /* Remember to update. */
		memcpy(configuration->subscribers_home_directory, directory, length);
		memcpy(configuration->subscribers_home_directory + length, "/.microfeed/subscribers", 24); /* Remember the NULL byte */
	}
	configuration->subscriptions = microfeed_store_new_sorted((MicrofeedStoreCompareKeysFunction)strcmp, (MicrofeedStoreGetKeyFunction)subscription_get_application_identifier);
	
	return configuration;
}

void microfeed_configuration_free(MicrofeedConfiguration* configuration) {
	microfeed_configuration_invalidate(configuration);
	free(configuration->providers_environment_directory);
	free(configuration->publishers_environment_directory);
	free(configuration->subscribers_environment_directory);
	microfeed_memory_free(configuration->providers_home_directory);
	microfeed_memory_free(configuration->publishers_home_directory);
	microfeed_memory_free(configuration->subscribers_home_directory);
	microfeed_store_free(configuration->subscriptions);
	
	microfeed_memory_free(configuration);
}

void microfeed_configuration_invalidate(MicrofeedConfiguration* configuration) {	
	int i;
	int k;
	
	if (configuration->providers) {
		for (i = 0; configuration->providers[i]; i++) {
			free((void*)configuration->providers[i]);
			/* The items in provider_directories point to strings in configuration (no free). */
			free((void*)configuration->provider_names[i]);
			if (configuration->provider_feeds[i]) {
				for (k = 0; configuration->provider_feeds[i][k]; k++) {
					free((void*)configuration->provider_feeds[i][k]);
				}
				free(configuration->provider_feeds[i]);
			}
		}
		free(configuration->providers);
		free(configuration->provider_names);
		free(configuration->provider_feeds);
		free(configuration->provider_directories);
		configuration->providers = NULL;
		configuration->provider_names = NULL;
		configuration->provider_feeds = NULL;
		configuration->provider_directories = NULL;
	}
		
	if (configuration->publishers) {
		for (i = 0; configuration->publishers[i]; i++) {
			free((void*)configuration->publishers[i]);
			/* The items in publisher_directories point to strings in configuration (no free). */
		}
		free(configuration->publishers);
		free(configuration->publisher_directories);
		configuration->publishers = NULL;
		configuration->publisher_directories = NULL;
	}
	microfeed_store_remove_and_free_all(configuration->subscriptions, (MicrofeedStoreFreeDataFunction)subscription_free);	
}

const char**  microfeed_configuration_get_providers(MicrofeedConfiguration* configuration) {
	if (!configuration->providers) {
		update_providers(configuration);
	}
	
	return configuration->providers;
}

const char* microfeed_configuration_get_provider_name(MicrofeedConfiguration* configuration, const char* provider_identifier) {
	const char* name = NULL;
	int index;

	if ((index = update_and_get_provider_index(configuration, provider_identifier)) != -1) {
		name = configuration->provider_names[index];
	}

	return name;
}

const char* microfeed_configuration_get_provider_directory(MicrofeedConfiguration* configuration, const char* provider_identifier) {
	const char* directory = NULL;
	unsigned int index;
	int difference = 1;
	
	if (!configuration->providers) {
		update_providers(configuration);
	}
	if (configuration->providers) {
		for (index = 0; configuration->providers[index] && (difference = strcmp(configuration->providers[index], provider_identifier)) < 0; index++) {
		}
		if (!difference) {
			directory = configuration->provider_directories[index];
		}
	}
	
	return directory;
}

const char** microfeed_configuration_get_publishers(MicrofeedConfiguration* configuration) {
	if (!configuration->publishers) {
		update_publishers(configuration);
	}
	
	return configuration->publishers;	
}

const char* microfeed_configuration_get_publisher_directory(MicrofeedConfiguration* configuration, const char* publisher_identifier) {
	const char* directory = NULL;
	unsigned int index;
	int difference = 1;
	
	if (!configuration->publishers) {
		update_publishers(configuration);
	}
	if (configuration->publishers) {
		for (index = 0, difference = 1; configuration->publishers[index] && (difference = strcmp(configuration->publishers[index], publisher_identifier)) < 0; index++) {
		}
		if (!difference) {
			directory = configuration->publisher_directories[index];
		}
	}
	
	return directory;
}

const char* microfeed_configuration_get_default_publisher_directory(MicrofeedConfiguration* configuration) {
	
	return (configuration->publishers_environment_directory ? configuration->publishers_environment_directory : configuration->publishers_home_directory);
}

const char** microfeed_configuration_get_providers_predefined_feed_uris(MicrofeedConfiguration* configuration, const char* provider_identifier) {
	const char** feeds = NULL;
	int index;

	if ((index = update_and_get_provider_index(configuration, provider_identifier)) != -1) {
		feeds = configuration->provider_feeds[index];
	}

	return feeds;
}

const char** microfeed_configuration_get_subscriptions(MicrofeedConfiguration* configuration, const char* application_identifier) {
	Subscription* subscription;
	
	if (!(subscription = microfeed_store_get(configuration->subscriptions, application_identifier, Subscription))) {
		subscription = update_and_get_subscription(configuration, application_identifier);
	}

	return (subscription ? subscription->subscriptions : NULL);
}

void microfeed_configuration_add_subscription(MicrofeedConfiguration* configuration, const char* application_identifier, const char* publisher_identifier) {
	Subscription* subscription;
	int i;
	
	if (!(subscription = microfeed_store_get(configuration->subscriptions, application_identifier, Subscription))) {
		subscription = update_and_get_subscription(configuration, application_identifier);
	}
	if (subscription) {
		for (i = 0; subscription->subscriptions[i]; i++) {
			/* Nothing */
		}
		subscription->subscriptions = (const char**)realloc(subscription->subscriptions, (i + 2) * sizeof(char*));
		subscription->subscriptions[i] = strdup(publisher_identifier);
		subscription->subscriptions[i + 1] = NULL;

		subscription_save(subscription);
	}
}

void microfeed_configuration_remove_subscription(MicrofeedConfiguration* configuration, const char* application_identifier, const char* publisher_identifier) {
	Subscription* subscription;
	int i;
	int index;
	
	if (!(subscription = microfeed_store_get(configuration->subscriptions, application_identifier, Subscription))) {
		subscription = update_and_get_subscription(configuration, application_identifier);
	}
	if (subscription) {
		index = -1;
		for (i = 0; subscription->subscriptions[i]; i++) {
			if (!strcmp(subscription->subscriptions[i], publisher_identifier)) {
				index = i;
			}
		}
		if (index != -1) {
			subscription->subscriptions = (const char**)realloc(subscription->subscriptions, i * sizeof(char*));
			memmove(subscription->subscriptions + index, subscription->subscriptions + index + 1, (i - index) * sizeof(const char**));

			subscription_save(subscription);
		}
	}
}

static Subscription* subscription_new(const char* application_identifier, const char* filename, const char** subscription_strings) {
	Subscription* subscription;
	
	subscription = microfeed_memory_allocate(Subscription);
	subscription->application_identifier = strdup(application_identifier);
	subscription->filename = strdup(filename);
	subscription->subscriptions = subscription_strings;
	
	return subscription;
}
	
static void subscription_free(Subscription* subscription) {
	int i;
	
	free(subscription->application_identifier);
	free(subscription->filename);
	for (i = 0; subscription->subscriptions[i]; i++) {
		free((void*)subscription->subscriptions[i]);
	}
	free(subscription->subscriptions);
}

static const char* subscription_get_application_identifier(Subscription* subscription) {

	return subscription->application_identifier;
}

static void subscription_save(Subscription* subscription) {
	char* file;
	char* directory;
	int fd;
	int i;
	
	file = strdup(subscription->filename);
	directory = dirname(file);
	microfeed_util_create_directory_recursively(directory);
	free(file);

	if ((fd = open(subscription->filename, O_WRONLY | O_CREAT | O_TRUNC, 0666)) != -1) {
		for (i = 0; subscription->subscriptions[i]; i++) {
			write(fd, subscription->subscriptions[i], strlen(subscription->subscriptions[i]));
			write(fd, "\n", 1);
		}
		close(fd);
	}	
}

static ssize_t read_line(int fd, size_t offset, char* buffer, size_t buffer_size) {
	ssize_t length = -1;
	size_t i;
	
	if (lseek(fd, offset, SEEK_SET) != -1) {
		if ((length = read(fd, buffer, buffer_size)) > 0) {
			for (i = 0; i < length && buffer[i] != '\n' && buffer[i] != '\r'; i++) {
			}
			if (i == buffer_size) {
				length = 0;
			} else {
				buffer[i] = 0;
				length = i + 1;
			}
		}
	}

	return length;
}

static void get_providers(MicrofeedConfiguration* configuration, const char* directory, unsigned int* length_pointer) {
	DIR* dir;
	struct dirent* dirent;
	int i;
	int order;
	
	if ((dir = opendir(directory))) {
		while ((dirent = readdir(dir))) {
			if (dirent->d_name[0] != '.') {
				for (i = 0; i < *length_pointer; i++) {
					if ((order = strcmp(configuration->providers[i], dirent->d_name)) >= 0) {
						break;
					}
				}
				if (i == *length_pointer || order != 0) {
					(*length_pointer)++;
					configuration->providers = (const char**)realloc(configuration->providers, (*length_pointer + 1) * sizeof(char*));
					configuration->provider_directories = (const char**)realloc(configuration->provider_directories, *length_pointer  * sizeof(char*));
					configuration->provider_names = (const char**)realloc(configuration->provider_names, *length_pointer  * sizeof(char*));
					configuration->provider_feeds = (const char***)realloc(configuration->provider_feeds, *length_pointer * sizeof(char**));
					if (i < *length_pointer) {
						memmove(configuration->providers + i + 1, configuration->providers + i, (*length_pointer - i - 1) * sizeof(char*));
					}
					configuration->providers[i] = strdup(dirent->d_name);	
					configuration->provider_directories[i] = directory;
					configuration->provider_names[i] = NULL;
					configuration->provider_feeds[i] = NULL;
					configuration->providers[*length_pointer] = NULL;
				}
			}
		}
		closedir(dir);
	}
}
	
static void get_publishers(MicrofeedConfiguration* configuration, const char* directory, unsigned int* length_pointer) {
	DIR* dir;
	struct dirent* dirent;
	int i;
	int order;
	
	if ((dir = opendir(directory))) {
		while ((dirent = readdir(dir))) {
			if (dirent->d_name[0] != '.') {
				for (i = 0; i < *length_pointer; i++) {
					if ((order = strcmp(configuration->publishers[i], dirent->d_name)) >= 0) {
						break;
					}
				}
				if (i == *length_pointer || order != 0) {
					(*length_pointer)++;
					configuration->publishers = (const char**)realloc(configuration->publishers, (*length_pointer + 1) * sizeof(char*));
					configuration->publisher_directories = (const char**)realloc(configuration->publisher_directories, *length_pointer * sizeof(char*));
					if (i < *length_pointer) {
						memmove(configuration->publishers + i + 1, configuration->publishers + i, (*length_pointer - i - 1) * sizeof(char*));
						memmove(configuration->publisher_directories + i + 1, configuration->publisher_directories + i, (*length_pointer - i - 1) * sizeof(char*));
					}
					configuration->publishers[i] = strdup(dirent->d_name);
					configuration->publisher_directories[i] = directory;
					configuration->publishers[*length_pointer] = NULL;
				}
			}
		}
		closedir(dir);
	}
}

static Subscription* get_subscription(MicrofeedConfiguration* configuration, const char* application_identifier, const char* directory) {
	Subscription* subscription = NULL;
	const char** strings = NULL;
	char* filename;
	int fd;
	char buffer[1024];
	size_t offset;
	ssize_t length;
	int i;
	
	strings = (const char**)malloc(sizeof(char*));
	strings[0] = NULL;
	filename = microfeed_util_string_concatenate(directory, "/", application_identifier, NULL);
	if ((fd = open(filename, O_RDONLY)) != -1) {
		offset = 0;
		for (i = 0; (length = read_line(fd, offset, buffer, 1024)) > 0; i++) {
			strings = (const char**)realloc(strings, (i + 2) * sizeof(char*));
			strings[i] = strdup(buffer);
			offset += length;
		}
		strings[i] = NULL;
		close(fd);		
	}
	subscription = subscription_new(application_identifier, filename, strings);
	free(filename);

	return subscription;
}

static void update_providers(MicrofeedConfiguration* configuration) {
	unsigned int length;

	length = 0;
	if (configuration->providers_environment_directory) {
		get_providers(configuration, configuration->providers_environment_directory, &length);
	}
	if (configuration->providers_home_directory) {
		get_providers(configuration, configuration->providers_home_directory, &length);
	}
	get_providers(configuration, "/usr/local/lib/microfeed/providers", &length);
	get_providers(configuration, "/usr/lib/microfeed/providers", &length);
}

static void update_publishers(MicrofeedConfiguration* configuration) {
	unsigned int length;
	
	length = 0;
	if (configuration->publishers_environment_directory) {
		get_publishers(configuration, configuration->publishers_environment_directory, &length);
	}
	if (configuration->publishers_home_directory) {
		get_publishers(configuration, configuration->publishers_home_directory, &length);
	}
	get_publishers(configuration, "/usr/local/lib/microfeed/publishers", &length);
	get_publishers(configuration, "/usr/lib/microfeed/publishers", &length);
}

static int update_and_get_provider_index(MicrofeedConfiguration* configuration, const char* provider_identifier) {
	int index = -1;
	char* filename;
	int difference = -1;
	int fd;
	char buffer[1024];
	size_t offset;
	ssize_t length;
	int i;

	if (!configuration->providers) {
		update_providers(configuration);
	}
	if (configuration->providers) {
		for (index = 0; configuration->providers[index] && (difference = strcmp(configuration->providers[index], provider_identifier)) < 0; index++) {
		}
		if (!difference) {
			if (!configuration->provider_names[index]) {
				filename = microfeed_util_string_concatenate(configuration->provider_directories[index], "/", provider_identifier, NULL);
				if ((fd = open(filename, O_RDONLY)) != -1) {
					offset = 0;
					if ((length = read_line(fd, offset, buffer, 1024)) > 0) {
						configuration->provider_names[index] = strdup(buffer);
					}
					offset += length;
					for (i = 0; (length = read_line(fd, offset, buffer, 1024)) > 0; i++) {
						configuration->provider_feeds[index] = (const char**)realloc(configuration->provider_feeds[index], (i + 2) * sizeof(char*));
						configuration->provider_feeds[index][i] = strdup(buffer);
						offset += length;
					}
					configuration->provider_feeds[index][i] = NULL;
					close(fd);
				}
				free(filename);
			}
		} else {
			index = -1;
		}
	}

	return index;
}

static Subscription* update_and_get_subscription(MicrofeedConfiguration* configuration, const char* application_identifier) {
	Subscription* subscription = NULL;
	
	if (configuration->subscribers_environment_directory) {
		subscription = get_subscription(configuration, application_identifier, configuration->subscribers_environment_directory);
	} else if (configuration->subscribers_home_directory) {
		subscription = get_subscription(configuration, application_identifier, configuration->subscribers_home_directory);
	}
		
	return subscription;
}
