#include <hildon/hildon.h>
#include "he-menu-store.h"
#include "he-menu-view-column.h"
#include "he-menu-view.h"
#include "he-menu-view_priv.h"

struct _HeMenuView
{
	GtkFixed __parent_instance__;

	GQueue *menu_q;
	GtkMenu *menu;
	GtkWidget *button;

	gboolean head_is_new;
	gboolean resizing;
	gboolean froze_updates;

	GtkWidget *menu_widget;
};

struct _HeMenuViewClass
{
	GtkFixedClass __parent_class__;
};

enum {
	HE_MENU_VIEW_MENU_PROPERTY = 1,
	HE_MENU_VIEW_MENU_WIDGET_PROPERTY
};

G_DEFINE_TYPE(HeMenuView, he_menu_view, GTK_TYPE_FIXED);

static void hmvc_notify_menu_widget(GtkWidget *hmvc, GParamSpec *pspec, HeMenuView *hmv);
static gboolean my_toggle_menu(HildonWindow *self, guint button, guint32 time);

static void
start_resizing(HeMenuView *hmv)
{
	if (!(hmv->resizing)) {
		GdkWindow *wnd = GTK_WIDGET(hmv)->window;

		hmv->resizing = TRUE;
		if (wnd && !hmv->froze_updates) {
			gdk_window_freeze_updates(wnd);
	//		g_print("size_allocate: ----- disable updates -----\n");
			hmv->froze_updates = TRUE;
		}
	}
}

static void
stop_resizing(HeMenuView *hmv)
{
	if (hmv->resizing) {	
		GdkWindow *wnd = GTK_WIDGET(hmv)->window;
	
		hmv->resizing = FALSE ;
		if (hmv->froze_updates && wnd) {
			gdk_window_thaw_updates(wnd);
//			g_print("size_allocate: +++++ enable updates +++++\n");
			hmv->froze_updates = FALSE;
		}
	}
}

static GtkWidget *
add_new_column(HeMenuView *hmv, GtkMenu *menu)
{
	GtkWidget *hmvc;

	start_resizing(hmv);

	hmvc = g_object_new(HE_TYPE_MENU_VIEW_COLUMN, "visible", TRUE, "menu", menu, NULL);
	gtk_container_add(GTK_CONTAINER(hmv), hmvc);
	g_signal_connect(G_OBJECT(hmvc), "notify::menu-widget", (GCallback)hmvc_notify_menu_widget, hmv);

	return hmvc;
}

static void
hmvc_notify_menu_widget(GtkWidget *hmvc, GParamSpec *pspec, HeMenuView *hmv)
{
	GtkMenuItem *menu_item = NULL;

	g_object_get(G_OBJECT(hmvc), "menu-widget", &menu_item, NULL);

	if (menu_item) {
		GtkWidget *submenu = gtk_menu_item_get_submenu(menu_item);

		if (submenu) {
			while(hmv->menu_q->tail->data != hmvc) {
				gtk_widget_destroy(GTK_WIDGET(hmv->menu_q->tail->data));
				g_queue_pop_tail(hmv->menu_q);
			}

			g_queue_push_tail(hmv->menu_q, add_new_column(hmv, GTK_MENU(submenu)));
			hmv->head_is_new = FALSE;
		}
		else {
			hmv->menu_widget = GTK_WIDGET(menu_item);
			g_object_notify(G_OBJECT(hmv), "menu-widget");
		}
	}
}

static void
button_clicked(GtkWidget *button, HeMenuView *hmv)
{
	if (hmv->menu_q->head) {
		GtkWidget *widget = _hmv_private_menu_get_parent_menu(he_menu_view_column_get_menu(HE_MENU_VIEW_COLUMN(hmv->menu_q->head->data)));

		if (widget) {
			g_queue_push_head(hmv->menu_q, add_new_column(hmv, GTK_MENU(widget)));
			hmv->head_is_new = TRUE;

			if (!_hmv_private_menu_get_parent_menu(he_menu_view_column_get_menu(HE_MENU_VIEW_COLUMN(hmv->menu_q->head->data))) && GTK_WIDGET_VISIBLE(hmv->button)) {
				gtk_widget_hide(hmv->button);
				return;
			}
		}
	}
}

static int
do_single_allocate(GtkContainer *me, GtkWidget *widget, gboolean is_last, int x, int y, int max_width, int left_over_part, int height, gboolean *p_is_dirty)
{
	GtkRequisition rq_child;
	int x_child;

	GtkAllocation alloc = {
		.x      = x,
		.y      = y,
		.height = height
	};

	gtk_widget_get_child_requisition(widget, &rq_child);
	gtk_container_child_get(me, widget, "x", &x_child, NULL);
	if (x_child != x) {
		(*p_is_dirty) = TRUE;
		gtk_container_child_set(me, widget, "x", x, NULL);
	}

	alloc.width = is_last ? max_width : MIN(max_width, rq_child.width + left_over_part);

	/* Don't bother allocating if there's gonna be another size-allocate anyway */
	if (!(*p_is_dirty))
		gtk_widget_size_allocate(widget, &alloc);

	return alloc.width;
}

/* Destroy columns starting from the tail or starting from the head */
static gboolean
destroy_excess_columns(HeMenuView *hmv, int alloc_width, gboolean tail_is_significant, GList **p_beg, GList **p_end, int advance_offset, gpointer (*pop_fn)(GQueue *))
{
	gboolean keep_looping;
	GList *ll_itr;
	int width_left = alloc_width;
	GtkRequisition rq_child;

	if (GTK_WIDGET_VISIBLE(hmv->button)) {
		gtk_widget_get_child_requisition(GTK_WIDGET(hmv->button), &rq_child);
		width_left -= rq_child.width;
	}

	/* start from head (or tail) and go next (or prev) */
	for (ll_itr = (*p_beg) ; ll_itr ; ll_itr = G_STRUCT_MEMBER(GList *, ll_itr, advance_offset)) {
		gtk_widget_get_child_requisition(GTK_WIDGET(ll_itr->data), &rq_child);
		width_left -= rq_child.width;
		if (width_left < 0)
			break;
	}

	if (ll_itr) {
		if (tail_is_significant && !GTK_WIDGET_VISIBLE(hmv->button)) {
			gtk_widget_show(hmv->button);
			/* Need another size-allocate to account for the button before we can start the actual destruction */
			return FALSE;
		}

		do {
			keep_looping = (ll_itr != (*p_end));
			gtk_widget_destroy(GTK_WIDGET((*p_end)->data));
			(*pop_fn)(hmv->menu_q);
		} while (keep_looping);
	}

	return TRUE;
}

static void
size_allocate(GtkWidget *widget, GtkAllocation *alloc)
{
	gboolean is_dirty = FALSE;
	GtkRequisition hmv_rq, rq_child;
	int x = 0;
	HeMenuView *hmv = HE_MENU_VIEW(widget);
	GtkContainer *hmv_container = GTK_CONTAINER(widget);
	GList *ll_itr;
	int width_left = alloc->width;
	int width_gone = 0;
	int width_accum = 0;
	int left_over_part = 0;

//	g_print("\nHMV: size_allocate (%d,%d)[%dx%d]\n", alloc->x, alloc->y, alloc->width, alloc->height);

	widget->allocation = *alloc;

	gtk_widget_get_child_requisition(widget, &hmv_rq);

	if (hmv_rq.width > alloc->width) {
		if (hmv->head_is_new)
			destroy_excess_columns(hmv, alloc->width, FALSE, &(hmv->menu_q->head), &(hmv->menu_q->tail), G_STRUCT_OFFSET(GList, next), g_queue_pop_tail);
		else
		if (!destroy_excess_columns(hmv, alloc->width, TRUE, &(hmv->menu_q->tail), &(hmv->menu_q->head), G_STRUCT_OFFSET(GList, prev), g_queue_pop_head))
				return;
	}

	if (GTK_WIDGET_VISIBLE(hmv->button)) {
		width_gone = do_single_allocate(hmv_container, hmv->button, FALSE, 0, 0, width_left, 0, alloc->height, &is_dirty);
		x += width_gone;
		width_left -= width_gone;
	}

	for (ll_itr = hmv->menu_q->head ; ll_itr ; ll_itr = ll_itr->next) {
		gtk_widget_get_child_requisition(GTK_WIDGET(ll_itr->data), &rq_child);
		width_accum += rq_child.width;
	}

	if (hmv->menu_q->length > 0)
		left_over_part = (MAX(width_left - width_accum, 0) / hmv->menu_q->length);
//	g_print("HMV: size_allocate: left_over_part = (%d - %d) / %d = %d\n", width_left, width_accum, hmv->menu_q->length, left_over_part);

	for (ll_itr = hmv->menu_q->head ; ll_itr ; ll_itr = ll_itr->next) {
		width_gone = do_single_allocate(hmv_container, GTK_WIDGET(ll_itr->data), ll_itr->next == NULL, x, 0, width_left, left_over_part, alloc->height, &is_dirty);
		x += width_gone;
		width_left -= width_gone;
	}

	if (!is_dirty) {
		stop_resizing(hmv);
		hmv->head_is_new = TRUE;
	}
}

static void
finalize(GObject *obj)
{
	g_queue_free(HE_MENU_VIEW(obj)->menu_q);
}

static void
get_property(GObject *obj, guint property_id, GValue *val, GParamSpec *pspec)
{
	switch(property_id) {

		case HE_MENU_VIEW_MENU_PROPERTY:
			g_value_set_object(val, G_OBJECT(he_menu_view_get_menu(HE_MENU_VIEW(obj))));
			break;

		case HE_MENU_VIEW_MENU_WIDGET_PROPERTY:
			g_value_set_object(val, HE_MENU_VIEW(obj)->menu_widget);
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id, pspec);
			break;
	}
}

static void
set_property(GObject *obj, guint property_id, const GValue *val, GParamSpec *pspec)
{
	switch(property_id) {
		case HE_MENU_VIEW_MENU_PROPERTY:
			he_menu_view_set_menu(HE_MENU_VIEW(obj), GTK_MENU(g_value_get_object(val)));
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id, pspec);
			break;
	}
}

static void
he_menu_view_class_init(HeMenuViewClass *hmv_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS(hmv_class);
	GtkWidgetClass *gtkwidget_class = GTK_WIDGET_CLASS(hmv_class);

	gobject_class->finalize     = finalize;
	gobject_class->set_property = set_property;
	gobject_class->get_property = get_property;

	gtkwidget_class->size_allocate = size_allocate;

	g_object_class_install_property(gobject_class, HE_MENU_VIEW_MENU_PROPERTY,
		g_param_spec_object("menu", "Menu", "The menu to display",
			GTK_TYPE_MENU, G_PARAM_READWRITE));

	g_object_class_install_property(gobject_class, HE_MENU_VIEW_MENU_WIDGET_PROPERTY,
		g_param_spec_object("menu-widget", "Menu Widget", "The last menu widget that was activated",
			GTK_TYPE_WIDGET, G_PARAM_READABLE));
}

static void
he_menu_view_init(HeMenuView *hmv)
{
	hmv->menu_widget   = NULL;
	hmv->head_is_new   = TRUE;
	hmv->resizing      = FALSE;
	hmv->froze_updates = FALSE;
	hmv->menu_q        = g_queue_new();
	hmv->button        =
		g_object_new(GTK_TYPE_BUTTON, "image",
			g_object_new(GTK_TYPE_IMAGE, "visible", TRUE, "stock", GTK_STOCK_GO_BACK, "icon-size", GTK_ICON_SIZE_MENU, NULL),
			NULL);

	g_signal_connect(G_OBJECT(hmv->button), "clicked", (GCallback)button_clicked, hmv);
	gtk_container_add(GTK_CONTAINER(hmv), hmv->button);
}

void
he_menu_view_set_menu(HeMenuView *hmv, GtkMenu *menu)
{
	g_return_if_fail(HE_IS_MENU_VIEW(hmv));
	g_return_if_fail((NULL == menu) || GTK_IS_MENU(menu));

	if (menu != hmv->menu) {
		if (hmv->menu) {
			g_object_unref(hmv->menu);
			hmv->menu = NULL;
		}

		g_queue_foreach(hmv->menu_q, (GFunc)gtk_widget_destroy, NULL);
		g_queue_clear(hmv->menu_q);

		if (menu) {
			hmv->menu = g_object_ref_sink(G_OBJECT(menu));
			g_queue_push_head(hmv->menu_q, add_new_column(hmv, GTK_MENU(menu)));
		}

	g_object_notify(G_OBJECT(hmv), "menu");
	}
}

GtkWidget *
he_menu_view_get_menu(HeMenuView *hmv)
{
	return hmv->menu ? GTK_WIDGET(hmv->menu) : NULL;
}

static gboolean
dialog_ignore_close(GObject *dlg)
{
	g_object_set_data(dlg, "no-close-timeout-id", GINT_TO_POINTER(0));
	return FALSE;
}

static gboolean
dlg_ignore_delete(GObject *dlg, GdkEvent *event, gpointer null)
{
	return ((gboolean)GPOINTER_TO_INT(g_object_get_data(dlg, "no-close-timeout-id")));
}

static void
unset_borders(GtkWidget *widget, gpointer null)
{
	if (GTK_IS_CONTAINER(widget)) {
		g_object_set(G_OBJECT(widget), "border-width", 0, NULL);
		if (GTK_IS_BOX(widget))
			g_object_set(G_OBJECT(widget), "spacing", 0, NULL);
		else
		if (GTK_IS_ALIGNMENT(widget))
			g_object_set(G_OBJECT(widget),
				"xscale", 1.0, "yscale", 1.0,
				"left-padding", 0, "right-padding", 0, 
				"top-padding", 0, "bottom-padding", 0, NULL);
		gtk_container_foreach(GTK_CONTAINER(widget), (GtkCallback)unset_borders, NULL);
	}
}

static void
pop_dialog(GtkWindow *parent, GtkMenu *menu, gboolean run)
{
	GtkWidget *hmv;
	GtkWidget *dlg;

	hmv = g_object_new(HE_TYPE_MENU_VIEW, "visible", TRUE, "menu", menu, NULL);

	dlg = g_object_new(GTK_TYPE_DIALOG, "visible", TRUE, "transient-for", parent, NULL);
	g_object_set_data_full(G_OBJECT(dlg), "no-close-timeout-id", 
		GINT_TO_POINTER(g_timeout_add_seconds(1, (GSourceFunc)dialog_ignore_close, dlg)),
		(GDestroyNotify)g_source_remove);
	g_signal_connect(G_OBJECT(dlg), "delete-event", (GCallback)dlg_ignore_delete, NULL);

	unset_borders(GTK_BIN(dlg)->child, NULL);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dlg)->vbox), hmv);

	g_signal_connect_swapped(G_OBJECT(hmv), "notify::menu-widget", (GCallback)gtk_widget_destroy, dlg);

	if (run)
		gtk_dialog_run(GTK_DIALOG(dlg));
	else
		gtk_widget_show(dlg);
}

static gboolean (*toggle_menu_orig)(HildonWindow *self, guint button, guint32 time) = my_toggle_menu;
static gboolean
my_toggle_menu(HildonWindow *self, guint button, guint32 time)
{
	GtkMenu *menu = hildon_window_get_main_menu(self);

	if (menu) {
		pop_dialog(GTK_WINDOW(self), menu, FALSE);

		return TRUE;
	}

	return toggle_menu_orig(self, button, time);
}

void
he_menu_view_popup(GtkMenu *menu, GtkWindow *parent)
{
	g_return_if_fail(menu != NULL);
	g_return_if_fail(GTK_IS_MENU(menu));
	if (parent)
		g_return_if_fail(GTK_IS_WINDOW(parent));

	pop_dialog(parent, menu, TRUE);
}

void 
he_menu_view_handle_hildon_windows()
{
	if (G_UNLIKELY(toggle_menu_orig == my_toggle_menu)) {
		toggle_menu_orig = HILDON_WINDOW_CLASS(g_type_class_peek(HILDON_TYPE_WINDOW))->toggle_menu;
		HILDON_WINDOW_CLASS(g_type_class_peek(HILDON_TYPE_WINDOW))->toggle_menu = my_toggle_menu;
	}
}
