#include <string.h>
#include "he-menu-store.h"
#include "he-menu-view_priv.h"

struct _HeMenuStore
{
	GObject __parent_instance__;
	GtkMenu *menu;
	gint stamp;
	gboolean is_deep;
};

struct _HeMenuStoreClass
{
	GObjectClass __parent_class__;
};

typedef struct {
	gint stamp;
	GtkMenuItem *menu_item;
	void *user_data2;
	void *user_data3;
} MenuStoreIter;

enum {
	HE_MENU_STORE_MENU_PROPERTY = 1,
	HE_MENU_STORE_DEEP_PROPERTY,
};

#define IS_VALID_MENU_ITEM(widget) \
	(!(GTK_IS_SEPARATOR_MENU_ITEM((widget)) ||  \
	   GTK_IS_TEAROFF_MENU_ITEM((widget))   || \
	  (GTK_IS_MENU_ITEM((widget)) && (NULL == _hmv_private_menu_item_get_label(GTK_WIDGET((widget)), NULL)))))

static void he_menu_store_tree_model_init(GtkTreeModelIface *tree_model_iface);

G_DEFINE_TYPE_EXTENDED(HeMenuStore, he_menu_store, G_TYPE_OBJECT, 0, G_IMPLEMENT_INTERFACE(GTK_TYPE_TREE_MODEL, he_menu_store_tree_model_init));

static GtkTreeModelFlags
he_menu_store_get_flags(GtkTreeModel *tm)
{
	return GTK_TREE_MODEL_ITERS_PERSIST;
}

static gint
he_menu_store_get_n_columns(GtkTreeModel *tm)
{
	return HE_MENU_STORE_N_COLUMNS;
}

static GType
he_menu_store_get_column_type(GtkTreeModel *tm, gint index)
{
	return 
		HE_MENU_STORE_IMAGE_PIXBUF    == index ? GDK_TYPE_PIXBUF :
//	HE_MENU_STORE_IMAGE_GICON     == index ? G_TYPE_ICON     :
//	HE_MENU_STORE_IMAGE_ICON_NAME == index ? G_TYPE_STRING   :
//	HE_MENU_STORE_IMAGE_STOCK_ID  == index ? G_TYPE_STRING   :
//	HE_MENU_STORE_IMAGE_ICON_SIZE == index ? G_TYPE_UINT     :
	  HE_MENU_STORE_IS_IMAGE_ITEM   == index ? G_TYPE_BOOLEAN  :
	  HE_MENU_STORE_IS_CHECK_ITEM   == index ? G_TYPE_BOOLEAN  :
	  HE_MENU_STORE_IS_RADIO_ITEM   == index ? G_TYPE_BOOLEAN  :
		HE_MENU_STORE_TEXT            == index ? G_TYPE_STRING   :
		HE_MENU_STORE_SENSITIVE       == index ? G_TYPE_BOOLEAN  :
		HE_MENU_STORE_ACTIVE          == index ? G_TYPE_BOOLEAN  :
		HE_MENU_STORE_INCONSISTENT    == index ? G_TYPE_BOOLEAN  :
		HE_MENU_STORE_VISIBLE         == index ? G_TYPE_BOOLEAN  :
		HE_MENU_STORE_HAS_CHILDREN    == index ? G_TYPE_BOOLEAN  :
//	HE_MENU_STORE_ACCEL           == index ? G_TYPE_STRING   :
		HE_MENU_STORE_MENU_WIDGET     == index ? G_TYPE_OBJECT   :
			G_TYPE_INVALID;
}

/*
 * Iterator structure:
 * stamp:      menu_store->stamp
 * user_data:  menu_item
 * user_data2: unused
 * user_data3: unused
 */

static gboolean
assign_iterator(HeMenuStore *hms, MenuStoreIter *itr, int depth, int idx, int *indices, GtkMenu *menu)
{
	GList *ll_children = gtk_container_get_children(GTK_CONTAINER(menu));
	GList *ll_itr = ll_children;
	int Nix;
	GtkWidget *submenu = NULL;
	gboolean ret = FALSE;

	for (Nix = 0; Nix < indices[idx] && ll_itr ; ll_itr = ll_itr->next)
		if (ll_itr)
			if (IS_VALID_MENU_ITEM(ll_itr->data))
				Nix++;

	if (ll_itr) {
		if (idx == depth - 1) {
			itr->stamp = hms->stamp;
			itr->menu_item = GTK_MENU_ITEM(ll_itr->data);
			ret = TRUE;
		}
		else
			submenu = gtk_menu_item_get_submenu(GTK_MENU_ITEM(ll_itr->data));
	}

	g_list_free(ll_children);
	return submenu ? assign_iterator(hms, itr, depth, idx + 1, indices, GTK_MENU(submenu)) : ret;
}

static gboolean
he_menu_store_get_iter(GtkTreeModel *tm, GtkTreeIter *iter, GtkTreePath *tp)
{
	HeMenuStore *hms = HE_MENU_STORE(tm);
	gint *indices;
	gint depth;

	indices = gtk_tree_path_get_indices(tp);
	depth = gtk_tree_path_get_depth(tp);

	g_return_val_if_fail(depth > 0, FALSE);
	g_return_val_if_fail(depth < 2 || hms->is_deep, FALSE);

	return assign_iterator(hms, (MenuStoreIter *)iter, depth, 0, indices, hms->menu);
}

static gboolean
fill_indices(GtkTreePath *tp, GtkMenuItem *menu_item, gboolean is_deep)
{
	int Nix;
	gboolean ret = FALSE;
	GtkMenuItem *attach_widget = NULL;
	GtkContainer *parent = GTK_CONTAINER(gtk_widget_get_parent(GTK_WIDGET(menu_item)));
	GList *ll_children = gtk_container_get_children(GTK_CONTAINER(parent)), *ll_itr;

	for (Nix = 0, ll_itr = ll_children ; ll_itr && menu_item != ll_itr->data ; ll_itr = ll_itr->next);
		if (ll_itr)
			if (IS_VALID_MENU_ITEM(ll_itr->data))
				Nix++;

	if (ll_itr) {
		attach_widget = _hmv_private_get_attach_widget(GTK_WIDGET(parent));

		gtk_tree_path_prepend_index(tp, Nix);
		ret = TRUE;
	}

	g_list_free(ll_children);
	return (attach_widget && is_deep) ? fill_indices(tp, attach_widget, is_deep) : ret;
}

static GtkTreePath *
he_menu_store_get_path(GtkTreeModel *tm, GtkTreeIter *iter)
{
	MenuStoreIter *msi = (MenuStoreIter *)iter;
	GtkMenuItem *menu_item = msi->menu_item;
	HeMenuStore *hms = HE_MENU_STORE(tm);
	GtkTreePath *tp;

	g_return_val_if_fail(iter->stamp == hms->stamp, NULL);

	tp = gtk_tree_path_new();

	if (menu_item)
		if (!fill_indices(tp, menu_item, hms->is_deep)) {
			gtk_tree_path_free(tp);
			tp = NULL;
		}

	return tp;
}

static char *
strip_underscores(char *text)
{
	if (text) {
		char *itr, *end = &text[strlen(text)];

		for (itr = text ; itr < end ; )
			if ('_' == (*itr)) {
				if (itr == end - 1)
					(*itr) = 0;
				else
					memmove(itr, itr + 1, end - itr - 1);
				end--;
				(*end) = 0;
			}
			else
				itr++;
	}

	return text;
}

/*
[X] GTK_IMAGE_EMPTY,
[ ] GTK_IMAGE_PIXMAP,
[ ] GTK_IMAGE_IMAGE,
[X] GTK_IMAGE_PIXBUF,
[ ] GTK_IMAGE_STOCK,
[ ] GTK_IMAGE_ICON_SET,
[ ] GTK_IMAGE_ANIMATION,
[ ] GTK_IMAGE_ICON_NAME,
[ ] GTK_IMAGE_GICON
*/

static GdkPixbuf *
image_get_pixbuf(GtkImage *image)
{
	GdkPixbuf *pb = NULL;

	switch (gtk_image_get_storage_type(image)) {
		case GTK_IMAGE_PIXBUF:
			pb = gtk_image_get_pixbuf(image);
			if (pb)
				pb = gdk_pixbuf_copy(pb);
			break;

		case GTK_IMAGE_STOCK:
			{
				char *stock_id;
				GtkIconSize size;

				gtk_image_get_stock(image, &stock_id, &size);
				if (stock_id)
					pb = gtk_widget_render_icon(GTK_WIDGET(image), stock_id, size, NULL);
			}
			break;

		default:
			break;
	}

	return pb;
}

static void
he_menu_store_get_value(GtkTreeModel *tm, GtkTreeIter *iter, gint column, GValue *val)
{
	HeMenuStore *hms = HE_MENU_STORE(tm);
	MenuStoreIter *msi = (MenuStoreIter *)iter;
	GtkMenuItem *menu_item = msi->menu_item;

	g_return_if_fail(hms->stamp == iter->stamp);

	g_value_init(val, he_menu_store_get_column_type(tm, column));

	switch (column) {
#if (0)
		case HE_MENU_STORE_IMAGE_GICON:
			if (GTK_IS_IMAGE_MENU_ITEM(msi->menu_item)) {
				GtkWidget *image = gtk_image_menu_item_get_image(GTK_IMAGE_MENU_ITEM(msi->menu_item));
				if (image)
					g_object_get_property(G_OBJECT(image), "gicon", val);
			}
			break;

		case HE_MENU_STORE_IMAGE_ICON_NAME:
			if (GTK_IS_IMAGE_MENU_ITEM(msi->menu_item)) {
				GtkWidget *image = gtk_image_menu_item_get_image(GTK_IMAGE_MENU_ITEM(msi->menu_item));
				if (image)
					g_object_get_property(G_OBJECT(image), "icon-name", val);
			}
			break;

		case HE_MENU_STORE_IMAGE_STOCK_ID:
			if (GTK_IS_IMAGE_MENU_ITEM(msi->menu_item)) {
				GtkWidget *image = gtk_image_menu_item_get_image(GTK_IMAGE_MENU_ITEM(msi->menu_item));
				if (image) {
					char *stock;

					g_object_get(G_OBJECT(image), "stock", &stock, NULL);
					g_print("stock: %s\n", stock);
					g_free(stock);
					g_object_get_property(G_OBJECT(image), "stock", val);
				}
			}
			break;

		case HE_MENU_STORE_IMAGE_ICON_SIZE:
			if (GTK_IS_IMAGE_MENU_ITEM(msi->menu_item)) {
				GtkWidget *image = gtk_image_menu_item_get_image(GTK_IMAGE_MENU_ITEM(msi->menu_item));
				if (image)
					g_object_get_property(G_OBJECT(image), "icon-size", val);
			}
			break;
#endif /* (0) */
		case HE_MENU_STORE_IMAGE_PIXBUF:
			if (GTK_IS_IMAGE_MENU_ITEM(menu_item)) {
				GtkWidget *image = gtk_image_menu_item_get_image(GTK_IMAGE_MENU_ITEM(menu_item));
				if (image) {
					GdkPixbuf *pb = image_get_pixbuf(GTK_IMAGE(image));

					g_value_set_object(val, pb);
					g_object_unref(pb);
				}
			}
			break;

		case HE_MENU_STORE_IS_IMAGE_ITEM:
			g_value_set_boolean(val, GTK_IS_IMAGE_MENU_ITEM(menu_item));
			break;

		case HE_MENU_STORE_IS_CHECK_ITEM:
			g_value_set_boolean(val, GTK_IS_CHECK_MENU_ITEM(menu_item));
			break;

		case HE_MENU_STORE_IS_RADIO_ITEM:
			g_value_set_boolean(val, 
				 GTK_IS_RADIO_MENU_ITEM(menu_item) ||
				(GTK_IS_CHECK_MENU_ITEM(menu_item) && gtk_check_menu_item_get_draw_as_radio(GTK_CHECK_MENU_ITEM(menu_item))));
			break;

		case HE_MENU_STORE_SENSITIVE:
			g_object_get_property(G_OBJECT(menu_item), "sensitive", val);
			break;

		case HE_MENU_STORE_ACTIVE:
			if (GTK_IS_CHECK_MENU_ITEM(menu_item))
				g_object_get_property(G_OBJECT(menu_item), "active", val);
			else
				g_value_set_boolean(val, FALSE);
			break;

		case HE_MENU_STORE_INCONSISTENT:
			if (GTK_IS_CHECK_MENU_ITEM(menu_item))
				g_object_get_property(G_OBJECT(menu_item), "inconsistent", val);
			else
				g_value_set_boolean(val, FALSE);
			break;

		case HE_MENU_STORE_VISIBLE:
			g_object_get_property(G_OBJECT(menu_item), "visible", val);
			break;

		case HE_MENU_STORE_HAS_CHILDREN:
			g_value_set_boolean(val, (NULL != gtk_menu_item_get_submenu(menu_item)));
			break;

		case HE_MENU_STORE_TEXT:
			{
				GtkWidget *label = NULL;
				char *text = g_strdup(_hmv_private_menu_item_get_label(GTK_WIDGET(menu_item), &label));

				if (gtk_label_get_use_underline(GTK_LABEL(label)))
					text = strip_underscores(text);
				g_value_set_string(val, text);
				g_free(text);
			}
			break;

		case HE_MENU_STORE_MENU_WIDGET:
			g_value_set_object(val, menu_item);
			break;
	}
}

static gboolean
he_menu_store_iter_next(GtkTreeModel *tm, GtkTreeIter *iter)
{
	HeMenuStore *hms = HE_MENU_STORE(tm);
	gboolean ret = FALSE;
	GList *ll_children, *ll_itr;
	MenuStoreIter *msi = (MenuStoreIter *)iter;
	GtkContainer *parent;

	g_return_val_if_fail(hms->stamp == iter->stamp, FALSE);

	msi->stamp = 0;

	parent = GTK_CONTAINER(gtk_widget_get_parent(GTK_WIDGET(msi->menu_item)));

	ll_children = gtk_container_get_children(parent);
	ll_itr = g_list_find(ll_children, msi->menu_item);

	for (ll_itr = ll_itr->next; ll_itr; ll_itr = ll_itr->next)
		if (ll_itr)
			if (IS_VALID_MENU_ITEM(ll_itr->data)) {
				msi->stamp = hms->stamp;
				msi->menu_item = ll_itr->data;
				ret = TRUE;
				break;
			}

	g_list_free(ll_children);

	return ret;
}

static gboolean
he_menu_store_iter_has_child(GtkTreeModel *tm, GtkTreeIter *iter)
{
	HeMenuStore *hms = HE_MENU_STORE(tm);
	g_return_val_if_fail(iter->stamp == hms->stamp, FALSE);

	return hms->is_deep && (gtk_menu_item_get_submenu(GTK_MENU_ITEM(((MenuStoreIter *)iter)->menu_item)) != NULL);
}

static gint
he_menu_store_iter_n_children(GtkTreeModel *tm, GtkTreeIter *iter)
{
	int ret = 0;
	HeMenuStore *hms = HE_MENU_STORE(tm);
	GtkWidget *submenu = NULL;
	MenuStoreIter *msi = (MenuStoreIter *)iter;

	if (iter)
		g_return_val_if_fail(iter->stamp == HE_MENU_STORE(tm)->stamp, 0);

	submenu = msi ? gtk_menu_item_get_submenu(msi->menu_item) : GTK_WIDGET(hms->menu);
	if (submenu) {
		GList *ll_children = gtk_container_get_children(GTK_CONTAINER(submenu));

		ret = g_list_length(ll_children);
		g_list_free(ll_children);
	}

	return ret;
}

static gboolean
he_menu_store_iter_nth_child(GtkTreeModel *tm, GtkTreeIter *iter, GtkTreeIter *parent, gint n)
{
	HeMenuStore *hms = HE_MENU_STORE(tm);
	MenuStoreIter *msi = (MenuStoreIter *)iter, *msi_parent = (MenuStoreIter *)parent;
	gboolean ret = FALSE;
	GtkWidget *submenu;

	if (msi_parent != NULL)
		g_return_val_if_fail(msi_parent->stamp == hms->stamp, FALSE);

	msi->stamp = 0;

	submenu = msi_parent ? gtk_menu_item_get_submenu(msi_parent->menu_item) : GTK_WIDGET(hms->menu);

	if (submenu) {
			GList *ll_children = gtk_container_get_children(GTK_CONTAINER(submenu)), *ll_itr;
			int Nix;

			for (Nix = 0, ll_itr = ll_children; ll_itr && Nix < n ; ll_itr = ll_itr->next)
				if (ll_itr)
					if (IS_VALID_MENU_ITEM(ll_itr->data))
						Nix++;

			if (ll_itr) {
				if (Nix == n) {
					msi->stamp = hms->stamp;
					msi->menu_item = GTK_MENU_ITEM(ll_itr->data);
					ret = TRUE;
				}
			}

			g_list_free(ll_children);
	}

	return ret;
}

static gboolean
he_menu_store_iter_children(GtkTreeModel *tm, GtkTreeIter *iter, GtkTreeIter *parent)
{
	return he_menu_store_iter_nth_child(tm, iter, parent, 0);
}

static gboolean
he_menu_store_iter_parent(GtkTreeModel *tm, GtkTreeIter *iter, GtkTreeIter *child)
{
	HeMenuStore *hms = HE_MENU_STORE(tm);
	gboolean ret = FALSE;

	if (hms->is_deep) {
		GtkWidget *container;
		GtkMenuItem *attach_widget;
		MenuStoreIter *msi = (MenuStoreIter *)iter, *msi_child = (MenuStoreIter *)child;

		g_return_val_if_fail(child->stamp == hms->stamp, FALSE);

		msi->stamp = 0;

		container = gtk_widget_get_parent(GTK_WIDGET(msi_child->menu_item));
		attach_widget = _hmv_private_get_attach_widget(container);
		if (attach_widget) {
			msi->stamp = msi_child->stamp;
			msi->menu_item = attach_widget;
			ret = TRUE;
		}
	}

	return ret;
}

static void
he_menu_store_tree_model_init(GtkTreeModelIface *tree_model_iface)
{
	tree_model_iface->get_flags       = he_menu_store_get_flags;
	tree_model_iface->get_n_columns   = he_menu_store_get_n_columns;
	tree_model_iface->get_column_type = he_menu_store_get_column_type;
	tree_model_iface->get_iter        = he_menu_store_get_iter;
	tree_model_iface->get_path        = he_menu_store_get_path;
	tree_model_iface->get_value       = he_menu_store_get_value;
	tree_model_iface->iter_next       = he_menu_store_iter_next;
	tree_model_iface->iter_children   = he_menu_store_iter_children;
	tree_model_iface->iter_has_child  = he_menu_store_iter_has_child;
	tree_model_iface->iter_n_children = he_menu_store_iter_n_children;
	tree_model_iface->iter_nth_child  = he_menu_store_iter_nth_child;
	tree_model_iface->iter_parent     = he_menu_store_iter_parent;
}

static void
he_menu_store_init(HeMenuStore *hms)
{
	hms->menu = NULL;
	hms->is_deep = FALSE;
	for (hms->stamp = g_random_int(); 0 == hms->stamp; hms->stamp = g_random_int());
}

static void
menu_item_changed(GObject *menu_item, GParamSpec *null, HeMenuStore *hms)
{
	GtkTreePath *tp = NULL;
	MenuStoreIter msi = {
		.stamp     = hms->stamp,
		.menu_item = GTK_MENU_ITEM(menu_item)
	};

	tp = he_menu_store_get_path(GTK_TREE_MODEL(hms), (GtkTreeIter *)&msi);

	if (tp) {
		char *str = gtk_tree_path_to_string(tp);
//		g_print("HMS: menu_item_changed: %s: emitting row-changed: %s\n", he_menu_store_menu_item_get_label(GTK_WIDGET(msi.menu_item)), str);
		g_free(str);
		gtk_tree_model_row_changed(GTK_TREE_MODEL(hms), tp, (GtkTreeIter *)&msi);
		gtk_tree_path_free(tp);
	}
}

/* Unfortunately radio menu items do not emit notify::active when toggled, so we need to do this manually */
static void
menu_item_toggled(GObject *menu_item, HeMenuStore *hms)
{
	menu_item_changed(menu_item, NULL, hms);
}

static void
unhook_signals(GObject *menu_item, HeMenuStore *hms)
{
	g_signal_handlers_disconnect_by_func(menu_item, (GCallback)menu_item_changed, hms);
	if (GTK_IS_RADIO_MENU_ITEM(menu_item))
		g_signal_handlers_disconnect_by_func(menu_item, (GCallback)menu_item_toggled, hms);
	if (hms->is_deep) {
		GtkWidget *submenu = gtk_menu_item_get_submenu(GTK_MENU_ITEM(menu_item));

		if (submenu)
			gtk_container_foreach(GTK_CONTAINER(submenu), (GtkCallback)unhook_signals, hms);
	}
}

static void
hook_up_signals(GObject *menu_item, HeMenuStore *hms)
{
	g_signal_connect(menu_item, "notify", (GCallback)menu_item_changed, hms);
	if (GTK_IS_RADIO_MENU_ITEM(menu_item))
		g_signal_connect(menu_item, "toggled", (GCallback)menu_item_toggled, hms);
	if (hms->is_deep) {
		GtkWidget *submenu = gtk_menu_item_get_submenu(GTK_MENU_ITEM(menu_item));

		if (submenu)
			gtk_container_foreach(GTK_CONTAINER(submenu), (GtkCallback)hook_up_signals, hms);
	}
}

static void
set_menu(HeMenuStore *hms, GtkMenu *menu)
{
	g_return_if_fail(GTK_IS_MENU(menu));

	if (hms->menu != menu) {
		if (hms->menu)
			g_object_unref(G_OBJECT(hms->menu));
		hms->menu = menu ? g_object_ref_sink(menu) : NULL;
		if (menu) {
//			g_print("HMS: set_menu: hms = %p\n", hms);
			gtk_container_foreach(GTK_CONTAINER(menu), (GtkCallback)hook_up_signals, hms);
		}
		g_object_notify(G_OBJECT(hms), "menu");
	}
}

static void
set_property(GObject *obj, guint property_id, const GValue *val, GParamSpec *pspec)
{
	switch (property_id) {
		case HE_MENU_STORE_MENU_PROPERTY:
			set_menu(HE_MENU_STORE(obj), GTK_MENU(g_value_get_object(val)));
			break;

		case HE_MENU_STORE_DEEP_PROPERTY:
			HE_MENU_STORE(obj)->is_deep = g_value_get_boolean(val);
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id, pspec);
			break;
	}
}

static void
get_property(GObject *obj, guint property_id, GValue *val, GParamSpec *pspec)
{
	switch (property_id) {
		case HE_MENU_STORE_MENU_PROPERTY:
			g_value_set_object(val, G_OBJECT(HE_MENU_STORE(obj)->menu));
			break;

		case HE_MENU_STORE_DEEP_PROPERTY:
			g_value_set_boolean(val, HE_MENU_STORE(obj)->is_deep);
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id, pspec);
			break;
	}
}

static void
finalize(GObject *obj)
{
	HeMenuStore *hms = HE_MENU_STORE(obj);

	if (hms->menu) {
		gtk_container_foreach(GTK_CONTAINER(hms->menu), (GtkCallback)unhook_signals, hms);
		g_object_unref(hms->menu);
		hms->menu = NULL;
	}
}

static void
he_menu_store_class_init(HeMenuStoreClass *menu_store_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS(menu_store_class);

	gobject_class->finalize     = finalize;
	gobject_class->set_property = set_property;
	gobject_class->get_property = get_property;

	g_object_class_install_property(gobject_class, HE_MENU_STORE_MENU_PROPERTY,
		g_param_spec_object("menu", "Menu", "The menu widget", GTK_TYPE_MENU,
			G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	g_object_class_install_property(gobject_class, HE_MENU_STORE_DEEP_PROPERTY,
		g_param_spec_boolean("deep", "Deep", "Whether this model reveals submenus as children", FALSE,
			G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
}

GtkTreeModel *
he_menu_store_new(GtkMenu *menu)
{
	return g_object_new(HE_TYPE_MENU_STORE, "menu", menu, NULL);
}
