/*
 * This file is part of libdicto
 *
 * Copyright (C) 2007-2009 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

/**
 * SECTION:dicto
 * @short_description: Audio dictation gobject
 * @stability: unstable
 * @see_also: #DictoWidget
 *
 * #Dicto is object that simplifies creating audio dictations in applications, 
 * it can be combined with #DictoWidget to create a simple and automatic UI.
 *
 * It supports some common audio formats, suitable for speech:
 * 8Khz, 16-bit, Mono
 * 11Khz, 8-bit, Mono
 * 11Khz, 16-bit, Mono
 * Speex compressed
 * A-Law
 * mu-law
 *
 * The object takes care of a list of notes in the given base directory. Signals are
 * emited when playing, recording, stopping, seeking and note list changes.
 *
 */

#include "config.h"

#include <string.h>
#include <math.h>

#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <gst/gst.h>

#include "dicto.h"

#define GST_TIME_TO_SECS(t) \
	(gdouble) (((gdouble)(t)) / (gdouble) GST_SECOND) /* GST_SECOND should be 1e9 */

#define POS_INT_DEFAULT 250
#define POS_INT_TIMEOUT 100

G_DEFINE_TYPE(Dicto, dicto, G_TYPE_OBJECT);

/* Signal IDs */
enum {
	SIGNAL_0,
	SIGNAL_POSITION,
	SIGNAL_PLAYING,
	SIGNAL_RECORDING,
	SIGNAL_STOPPED_PLAY,
	SIGNAL_STOPPED_RECORD,
	SIGNAL_ERROR,
	SIGNAL_NOTES_REFRESH,
	SIGNAL_READY,
	SIGNAL_SEEK,
	SIGNAL_LEVEL,
	LAST_SIGNAL
};

/* Property IDs */
enum {
	PROP_0,
	PROP_SOURCE,
	PROP_SINK,
	PROP_BASEDIR,
	PROP_PREFIX,
	PROP_SUFFIX,
	PROP_CURFILE,
	PROP_FORMAT,
	PROP_POS_INTERVAL,
};

typedef struct _dicto_pipeline dicto_pipeline;
struct _dicto_pipeline {
	GstElement *pipeline;
	GstElement *src;
	GstElement *sink;
	GstElement *filter;
	GstElement *muxer;
	GstElement *encoder;
	GstElement *caps;
	GstElement *level;
	GstCaps	*srccaps;
	gboolean active;
	gboolean rec;
	GObject *owner;
};

#define GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DICTO_TYPE, DictoPrivate))

typedef struct _DictoPrivate DictoPrivate;
struct _DictoPrivate {
	gchar *cfile;
	gchar *basedir;
	gchar *prefix;
	gchar *suffix;
	gchar *ext;
	gchar *gst_sink;
	gchar *gst_source;
	gchar *timefmt;
	DictoFormat format;
	gint pos_sid;
	guint pos_int;
	gdouble pos;
	gdouble len;
	gdouble rms;
	gdouble rforsecs;
	GHashTable *notes;
	dicto_pipeline play;
	dicto_pipeline record;
};

static guint signals[LAST_SIGNAL]={ 0 };

#define UNREF_ELEMENT(e) { if (np->e) { gst_object_unref(np->e); np->e=NULL; } }

static void
dicto_dispose(GObject *object)
{
G_OBJECT_CLASS(dicto_parent_class)->dispose(object);
}

static void
dicto_pipeline_free(dicto_pipeline *np)
{
UNREF_ELEMENT(pipeline);
UNREF_ELEMENT(src);
UNREF_ELEMENT(filter);
UNREF_ELEMENT(muxer);
UNREF_ELEMENT(level);
UNREF_ELEMENT(sink);
UNREF_ELEMENT(encoder);
UNREF_ELEMENT(srccaps);
g_object_unref(np->owner);
np->owner=NULL;
}

static void
dicto_finalize(GObject *object)
{
Dicto *d=DICTO(object);
DictoPrivate *p=GET_PRIVATE(d);

g_return_if_fail(d);

if (p->pos_sid!=0)
	g_source_remove(p->pos_sid);

gst_element_set_state(p->play.pipeline, GST_STATE_NULL);
gst_element_set_state(p->record.pipeline, GST_STATE_NULL);

dicto_pipeline_free(&p->play);
dicto_pipeline_free(&p->record);

G_OBJECT_CLASS(dicto_parent_class)->finalize(object);
}

static void
dicto_set_property(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
Dicto *d=DICTO(object);
DictoPrivate *p=GET_PRIVATE(d);

switch (prop_id) {
	case PROP_BASEDIR:
		dicto_set_basedir(d, g_value_get_string(value));
	break;
	case PROP_PREFIX:
		if (p->prefix)
			g_free(p->prefix);
		p->prefix=g_value_dup_string(value);
	break;
	case PROP_SUFFIX:
		if (p->suffix)
			g_free(p->suffix);
		p->suffix=g_value_dup_string(value);
	break;
	case PROP_SOURCE:
		p->gst_source=g_value_dup_string(value);
		g_debug("Source set to: %s", p->gst_source);
	break;
	case PROP_SINK:
		p->gst_sink=g_value_dup_string(value);
		g_debug("Sink set to: %s", p->gst_sink);
	break;
	case PROP_FORMAT:
		p->format=g_value_get_uint(value);
	break;
	case PROP_POS_INTERVAL:
		p->pos_int=g_value_get_uint(value);
	break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
	break;
}
}

static void
dicto_get_property(GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
Dicto *d=DICTO(object);
DictoPrivate *p=GET_PRIVATE(d);

switch (prop_id) {
	case PROP_BASEDIR:
		g_value_set_string(value, p->basedir);
	break;
	case PROP_PREFIX:
		g_value_set_string(value, p->prefix);
	break;
	case PROP_SUFFIX:
		g_value_set_string(value, p->suffix);
	break;
	case PROP_SOURCE:
		g_value_set_string(value, p->gst_source);
	break;
	case PROP_SINK:
		g_value_set_string(value, p->gst_sink);
	break;
	case PROP_FORMAT:
		g_value_set_uint(value, p->format);
	break;
	case PROP_CURFILE:
		g_value_set_string(value, p->cfile);
	break;
	case PROP_POS_INTERVAL:
		g_value_set_uint(value, p->pos_int);
	break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
	break;
}
}

static void
dicto_class_init(DictoClass *klass)
{
GParamSpec *pspec;
GObjectClass *object_class=G_OBJECT_CLASS(klass);

object_class->dispose=dicto_dispose;
object_class->finalize=dicto_finalize;
object_class->set_property=dicto_set_property;
object_class->get_property=dicto_get_property;

g_type_class_add_private(object_class, sizeof (DictoPrivate));

/**
 * Dicto:basedir:
 */
pspec=g_param_spec_string("basedir", "Base directory", "Base directory to store audio notes in", "/tmp", G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
g_object_class_install_property(object_class, PROP_BASEDIR, pspec);

/**
 * Dicto:prefix:
 */
pspec=g_param_spec_string("prefix", "Base directory","Prefix for audio notes", "an-", G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
g_object_class_install_property(object_class, PROP_PREFIX, pspec);

/**
 * Dicto:suffix:
 */
pspec=g_param_spec_string("suffix", "File suffix", "Suffix for audio notes, before file extension", "", G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
g_object_class_install_property(object_class, PROP_SUFFIX, pspec);

/**
 * Dicto:format:
 */
pspec=g_param_spec_uint("format", "Audio format", "Audio encoding format", FORMAT_DEFAULT, FORMAT_LAST, FORMAT_DEFAULT, G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY);
g_object_class_install_property(object_class, PROP_FORMAT, pspec);

/**
 * Dicto:pos_int:
 */
pspec=g_param_spec_uint("pos_int", "Position interval", "Report position this often, in ms", 50, 1000, POS_INT_DEFAULT, G_PARAM_READWRITE);
g_object_class_install_property(object_class, PROP_POS_INTERVAL, pspec);

/**
 * Dicto:source:
 *
 * gstreamer source element used for recording.
 */
pspec=g_param_spec_string("source", "Audio source", "Gstreamer audio source element", AUDIO_SINK, G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY);
g_object_class_install_property(object_class, PROP_SOURCE, pspec);

/**
 * Dicto:sink:
 *
 * gstreamer sink element used for playback.
 */
pspec=g_param_spec_string("sink", "Audio sink", "Gstreamer audio sink element", AUDIO_SRC, G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY);
g_object_class_install_property(object_class, PROP_SINK, pspec);

/**
 * Dicto:current-file:
 */
pspec=g_param_spec_string("current-file", "Active file", "The active file", "", G_PARAM_READABLE);
g_object_class_install_property(object_class, PROP_CURFILE, pspec);

signals[SIGNAL_POSITION]=g_signal_new("position", G_OBJECT_CLASS_TYPE(object_class),
	G_SIGNAL_RUN_FIRST, G_STRUCT_OFFSET(DictoClass, position_update),
	NULL, NULL, g_cclosure_marshal_VOID__DOUBLE, G_TYPE_NONE, 1, G_TYPE_DOUBLE);

signals[SIGNAL_PLAYING]=g_signal_new("playing", G_OBJECT_CLASS_TYPE(object_class),
	G_SIGNAL_RUN_FIRST, G_STRUCT_OFFSET(DictoClass, started_play),
	NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
signals[SIGNAL_STOPPED_PLAY]=g_signal_new("stopped-play", G_OBJECT_CLASS_TYPE(object_class),
	G_SIGNAL_RUN_FIRST, G_STRUCT_OFFSET(DictoClass, stopped_play),
	NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

signals[SIGNAL_RECORDING]=g_signal_new("recording", G_OBJECT_CLASS_TYPE(object_class),
	G_SIGNAL_RUN_FIRST, G_STRUCT_OFFSET(DictoClass, started_record),
	NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
signals[SIGNAL_STOPPED_RECORD]=g_signal_new("stopped-record", G_OBJECT_CLASS_TYPE(object_class),
	G_SIGNAL_RUN_FIRST, G_STRUCT_OFFSET(DictoClass, stopped_record),
	NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

signals[SIGNAL_ERROR]=g_signal_new("error", G_OBJECT_CLASS_TYPE(object_class),
	G_SIGNAL_RUN_FIRST, G_STRUCT_OFFSET(DictoClass, error),
	NULL, NULL, g_cclosure_marshal_VOID__POINTER, G_TYPE_NONE, 1, G_TYPE_POINTER);

signals[SIGNAL_NOTES_REFRESH]=g_signal_new("refresh", G_OBJECT_CLASS_TYPE(object_class),
	G_SIGNAL_RUN_FIRST, G_STRUCT_OFFSET(DictoClass, refresh),
	NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

signals[SIGNAL_READY]=g_signal_new("ready", G_OBJECT_CLASS_TYPE(object_class),
	G_SIGNAL_RUN_FIRST, G_STRUCT_OFFSET(DictoClass, ready),
	NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

signals[SIGNAL_SEEK]=g_signal_new("seeked", G_OBJECT_CLASS_TYPE(object_class),
	G_SIGNAL_RUN_FIRST, G_STRUCT_OFFSET(DictoClass, ready),
	NULL, NULL, g_cclosure_marshal_VOID__DOUBLE, G_TYPE_NONE, 1, G_TYPE_DOUBLE);

signals[SIGNAL_LEVEL]=g_signal_new("level", G_OBJECT_CLASS_TYPE(object_class),
	G_SIGNAL_RUN_FIRST, G_STRUCT_OFFSET(DictoClass, level),
	NULL, NULL, g_cclosure_marshal_VOID__DOUBLE, G_TYPE_NONE, 1, G_TYPE_DOUBLE);

}

static void 
dicto_init(Dicto *d)
{
DictoPrivate *p=GET_PRIVATE(d);

p->rms=0;
p->timefmt=NULL;
}

static inline gdouble db2norm(gdouble db)
{
return pow(10, db/20);
}

static gboolean
dicto_position_cb(gpointer data)
{
Dicto *d=(Dicto *)data;
DictoPrivate *p;

g_return_val_if_fail(IS_DICTO(d), FALSE);
p=GET_PRIVATE(d);

/* Are we recording for a specific time ? */
if (p->rforsecs>0) {
	if (!dicto_get_position(d, NULL, &p->pos)) {
		g_warning("Failed to get position for timeout recording");
		return TRUE;
	}

	if (p->pos>=p->rforsecs)
		dicto_stop(d);
	else
		g_signal_emit(G_OBJECT(data), signals[SIGNAL_POSITION], 0, p->rforsecs-p->pos, NULL);

	return TRUE;
}

g_signal_emit(G_OBJECT(data), signals[SIGNAL_POSITION], 0, 0, NULL);
return TRUE;
}

static void
dicto_position_update(Dicto *d, gboolean ena)
{
DictoPrivate *p=GET_PRIVATE(d);
g_return_if_fail(d);

if (p->pos_sid!=0 && !ena) {
	g_source_remove(p->pos_sid);
	p->pos_sid=0;
}
if (ena && p->pos_sid==0)
	p->pos_sid=g_timeout_add(p->pos_int, (GSourceFunc)dicto_position_cb, d);
}

/**
 * dicto_set_basedir:
 * 
 * Set base directory to use to for recordings.
 * @d: A #Dicto object.
 * @basedir: A UNIX path to notes, it will be created if it does not exists.
 *
 */
gboolean
dicto_set_basedir(Dicto *d, const gchar *basedir)
{
DictoPrivate *p=GET_PRIVATE(d);

g_return_val_if_fail(d, FALSE);
g_return_val_if_fail(basedir, FALSE);
g_return_val_if_fail(g_path_is_absolute(basedir), FALSE);

if (p->basedir)
	g_free(p->basedir);
p->basedir=g_strdup(basedir);
if (g_mkdir_with_parents(p->basedir, 0775)==-1) {
	g_warning("Failed to create basedirectory: %s", p->basedir);
	return FALSE;
}

g_debug("Basedir: %s", p->basedir);

return TRUE;
}

/**
 * dicto_filename_time_new:
 * @d: A #Dicto object
 * @t: A unix timestamp, or 0 for current time.
 *
 * Returns: A string containg a path+filename with the given or current timestamp using format "%Y-%m-%d-%H%M%S"
 */
gchar *
dicto_filename_time_new(Dicto *d, time_t t)
{
gchar buffer[64];
struct tm *tmp;
DictoPrivate *p=GET_PRIVATE(d);

g_return_val_if_fail(d, NULL);
g_return_val_if_fail(p->basedir, NULL);

if (t==0)
	t=time(NULL);
tmp=localtime(&t);
if (tmp==NULL)
	return NULL;

/* Bah, stupid FAT can't use : in filenames */
strftime(buffer, sizeof(buffer), p->timefmt ? p->timefmt : "%Y-%m-%d-%H-%M-%S", tmp);
return g_strdup_printf("%s/%s%s%s.wav", p->basedir, p->prefix, p->suffix, buffer);
}

static void
dicto_load_tag(const GstTagList *list, const gchar *tag, gpointer data)
{
g_debug("Tag: %s", tag);
}

static gdouble
dicto_level_get_rms(Dicto *d, const GstStructure *s, gdouble *rms_dB, gdouble *peak_dB, gdouble *decay_dB)
{
gint channels;
gdouble rms=0;
const GValue *list;
gint i;
DictoPrivate *p=GET_PRIVATE(d);

list=gst_structure_get_value(s, "rms");
channels=gst_value_list_get_size(list);

for (i=0;i<channels;++i) {
	*rms_dB=g_value_get_double(gst_value_list_get_value(gst_structure_get_value(s, "rms"), i));
	*peak_dB=g_value_get_double(gst_value_list_get_value (gst_structure_get_value (s, "peak"), i));
	*decay_dB=g_value_get_double(gst_value_list_get_value (gst_structure_get_value (s, "decay"), i));
	g_debug("RMS: %f dB, peak: %f dB, decay: %f dB", *rms_dB, *peak_dB, *decay_dB);
	rms+=db2norm(*rms_dB);
}

p->rms=rms/channels;
return p->rms;
}

static gboolean
dicto_bus_cb(GstBus *bus, GstMessage *msg, gpointer data)
{
GstState oldstate, newstate, pending;
gchar *debug;
GError *err;
GstTagList *tags;
const GstStructure *s;
const gchar *name;

dicto_pipeline *np=(dicto_pipeline *)data;

switch (GST_MESSAGE_TYPE(msg)) {
	case GST_MESSAGE_EOS:
		np->active=FALSE;
   		g_debug("EOS");
		if (np->rec) {
			gst_element_set_state(np->pipeline, GST_STATE_NULL);
			dicto_position_update(DICTO(np->owner), FALSE);
			g_signal_emit(np->owner, signals[SIGNAL_STOPPED_RECORD], 0, NULL);
		} else {
			gst_element_set_state(np->pipeline, GST_STATE_PAUSED);
			dicto_position_update(DICTO(np->owner), FALSE);
			g_signal_emit(np->owner, signals[SIGNAL_STOPPED_PLAY], 0, NULL);
			dicto_seek(DICTO(np->owner), 0.0);
		}
	break;
	case GST_MESSAGE_ERROR:
		gst_message_parse_error(msg, &err, &debug);
		g_warning("GstBusError: %s", err->message);
		g_free(debug);
		np->active=FALSE;
		dicto_position_update(DICTO(np->owner), FALSE);
		g_signal_emit(np->owner, signals[SIGNAL_ERROR], 0, err, NULL);
		g_error_free(err);
		gst_element_set_state(np->pipeline, GST_STATE_NULL);
	break;
	case GST_MESSAGE_STATE_CHANGED:
		if (GST_MESSAGE_SRC(msg)!=GST_OBJECT(np->pipeline))
			return TRUE;
		gst_message_parse_state_changed(msg, &oldstate, &newstate, &pending);
		if (newstate==GST_STATE_PAUSED && oldstate==GST_STATE_READY)
			g_signal_emit(DICTO(np->owner), signals[SIGNAL_POSITION], 0, NULL);
		if (np->rec && newstate==GST_STATE_READY && oldstate==GST_STATE_PLAYING) {
			gst_element_set_state(np->pipeline, GST_STATE_NULL);
			g_object_set(G_OBJECT(np->sink), "location", "", NULL);
			g_signal_emit(np->owner, signals[SIGNAL_STOPPED_RECORD], 0, NULL);
		}
		g_debug("GST: [%d] state: O:%d N:%d P:%d", np->rec, oldstate, newstate, pending);
	break;
	case GST_MESSAGE_TAG:
		gst_message_parse_tag(msg, &tags);
		if (tags) {
			gst_tag_list_foreach(tags, (GstTagForeachFunc)dicto_load_tag, np->owner);
			gst_tag_list_free(tags);
		} else {
			g_warning("Unable to parse tags");
		}
	break;
	case GST_MESSAGE_ELEMENT:
		s=gst_message_get_structure(msg);
		name=gst_structure_get_name(s);
		/* Handle level events, but calculate only if someone is listening to us */
		if (g_str_equal (name, "level") && g_signal_has_handler_pending(np->owner, signals[SIGNAL_LEVEL], 0, TRUE)) {
			gdouble r,p,d,nr;

			nr=dicto_level_get_rms(DICTO(np->owner), s, &r, &p, &d);
			g_signal_emit(np->owner, signals[SIGNAL_LEVEL], 0, nr, NULL);
		}
	break;
	default:
		g_debug("GST: %s", gst_message_type_get_name(GST_MESSAGE_TYPE(msg)));
	break;
	}
return TRUE;
}

static gboolean
dicto_new_pad_cb(GstElement *wavparse, GstPad *new_pad, gpointer data)
{
GstElement *sink=(GstElement *)data;

if (!gst_element_link(wavparse, sink)) {
	g_warning("Failed to link wavparse to sink.");
}

gst_element_sync_state_with_parent(sink);

return FALSE;
}

static GstCaps *
dicto_get_src_caps(const gchar *type, gint depth, gint rate, gint ch)
{
GstCaps *c;

if (depth>0) {
	c=gst_caps_new_simple(type,
		"depth", G_TYPE_INT, depth,
		"signed", G_TYPE_BOOLEAN, depth==16 ? TRUE : FALSE, 
		"width", G_TYPE_INT, depth,
		"rate", G_TYPE_INT, rate,
		"channels", G_TYPE_INT, ch,
		NULL);
} else {
	c=gst_caps_new_simple(type,
		"rate", G_TYPE_INT, rate,
		"channels", G_TYPE_INT, ch,
		NULL);
}
g_return_val_if_fail(c, NULL);
return c;
}

static gboolean
dicto_create_record_pipeline(Dicto *d)
{
GstBus *bus;
GstElement *enext;
DictoPrivate *p=GET_PRIVATE(d);
dicto_pipeline *np=&p->record;

g_object_ref(d);
np->owner=G_OBJECT(d);

g_debug("GST: Creating record pipeline");
g_return_val_if_fail(p->gst_source, FALSE);

np->encoder=NULL;
np->muxer=NULL;
np->filter=NULL;

g_debug("Format: %d", p->format);
switch (p->format) {
	case FORMAT_WAV_11K_8B_M:
		np->srccaps=dicto_get_src_caps("audio/x-raw-int", 8, 11025, 1);
	break;
	case FORMAT_WAV_22K_8B_M:
		np->srccaps=dicto_get_src_caps("audio/x-raw-int", 8, 22050, 1);
	break;
	case FORMAT_WAV_22K_16B_M:
		np->srccaps=dicto_get_src_caps("audio/x-raw-int", 16, 22050, 1);
	break;
	case FORMAT_WAV_44K_8B_M:
		np->srccaps=dicto_get_src_caps("audio/x-raw-int", 8, 44100, 1);
	break;
	case FORMAT_WAV_44K_16B_M:
		np->srccaps=dicto_get_src_caps("audio/x-raw-int", 16, 44100, 1);
	break;
	case FORMAT_WAV_44K_16B_S:
		np->srccaps=dicto_get_src_caps("audio/x-raw-int", 16, 44100, 2);
	break;
	case FORMAT_WAV_8K_16B_M:
	case FORMAT_DEFAULT:
		np->srccaps=dicto_get_src_caps("audio/x-raw-int", 16, 8000, 1);
	break;
	case FORMAT_WAV_11K_16B_M:
		np->srccaps=dicto_get_src_caps("audio/x-raw-int", 16, 11025, 1);
	break;
	case FORMAT_SPX_SPEEX:
		np->encoder=gst_element_factory_make("speexenc", "encoder");
		if (!np->encoder)
			goto error_rec;
		np->srccaps=dicto_get_src_caps("audio/x-raw-int", 16, 11025, 1);
		np->muxer=gst_element_factory_make("oggmux", "muxer");
		if (!np->muxer)
			goto error_rec;
	break;
	case FORMAT_WAV_ALAW:
		if (strcmp(p->gst_source, "dsppcmsrc")==0) {
			np->srccaps=dicto_get_src_caps("audio/x-alaw", 0, 8000, 1);
		} else {
			np->encoder=gst_element_factory_make("alawenc", "encoder");
			if (!np->encoder)
				goto error_rec;
			np->srccaps=dicto_get_src_caps("audio/x-raw-int", 16, 11025, 1);
		}
	break;
	case FORMAT_WAV_MULAW:
		if (strcmp(p->gst_source, "dsppcmsrc")==0) {
			np->srccaps=dicto_get_src_caps("audio/x-mulaw", 0, 8000, 1);
		} else {
			np->encoder=gst_element_factory_make("mulawenc", "encoder");
			if (!np->encoder)
				goto error_rec;
			np->srccaps=dicto_get_src_caps("audio/x-raw-int", 16, 11025, 1);
		}
	break;
	case FORMAT_VORBIS:
		np->filter=gst_element_factory_make("audioconvert", "ac");
		if (!np->filter)
			goto error_rec;
		np->encoder=gst_element_factory_make("vorbisenc", "encoder");
		if (!np->encoder)
			goto error_rec;
		np->muxer=gst_element_factory_make("oggmux", "muxer");
		if (!np->muxer)
			goto error_rec;
	break;
	default:
		g_warning("Invalid recording format");
		goto error_rec;
	break;
}

if (!np->srccaps)
	goto error_rec;

np->pipeline=gst_pipeline_new("rpipeline");
if (!np->pipeline)
	goto error_rec;

np->src=gst_element_factory_make(p->gst_source, "asrc");
if (!np->src)
	goto error_rec;

if (!np->muxer) {
	np->muxer=gst_element_factory_make("wavenc", "wf");
	if (!np->muxer)
		goto error_rec;
}

np->sink=gst_element_factory_make("filesink", "fsink");
if (!np->sink)
	goto error_rec;

np->level=gst_element_factory_make("level", "level");
if (!np->level)
	goto error_rec;

gst_bin_add_many(GST_BIN(np->pipeline), np->src, np->level, np->muxer, np->sink, NULL);

if (!gst_element_link_filtered(np->src, np->level, np->srccaps)) {
	g_warning("Failed to link source to level");
	goto error_rec;
}
g_debug("src->level");
enext=np->level;

if (np->filter) {
	gst_bin_add(GST_BIN(np->pipeline), np->filter);
	/* level ! filter */
	if (!gst_element_link(enext, np->filter)) {
		g_warning("Failed to link to filter");
		goto error_rec;
	}
	g_debug("->filter");
	enext=np->filter;
} 

if (np->encoder) {
	gst_bin_add(GST_BIN(np->pipeline), np->encoder);
	if (!gst_element_link(enext, np->encoder)) {
		g_warning("Failed to link to encoder");
		goto error_rec;
	}
	g_debug("->encoder");
	enext=np->encoder;
}

/* filter/encoder ! caps ! muxer */ 
if (!gst_element_link(enext, np->muxer)) {
	g_warning("Failed to link to muxer");
	goto error_rec;
}
g_debug("->muxer (caps)");
gst_caps_unref(np->srccaps);

/* muxer ! sink */
if (!gst_element_link(np->muxer, np->sink)) {
	g_warning("Failed to link muxer to sink");
	goto error_rec;
}
g_debug("->sink");

bus=gst_pipeline_get_bus(GST_PIPELINE(np->pipeline));
gst_bus_add_watch(bus, dicto_bus_cb, np);
gst_object_unref(bus);
np->rec=TRUE;
np->active=FALSE;
return TRUE;

error_rec: ;

dicto_pipeline_free(np);
return FALSE;
}

static gboolean
dicto_create_play_playbin_pipeline(Dicto *d)
{
GstBus *bus;
DictoPrivate *p=GET_PRIVATE(d);
dicto_pipeline *np=&p->play;

np->pipeline=gst_element_factory_make("playbin", "playbin");
if (!np->pipeline)
	goto error_play;

np->sink=gst_element_factory_make(p->gst_sink, "asink");
if (!np->sink)
	goto error_play;

g_object_set(G_OBJECT(np->pipeline), "audio-sink", np->sink, NULL);

bus=gst_pipeline_get_bus(GST_PIPELINE(np->pipeline));
gst_bus_add_watch(bus, dicto_bus_cb, np);
gst_object_unref(bus);
np->rec=FALSE;
np->active=FALSE;

return TRUE;

error_play: ;
return FALSE;
}

static gboolean
dicto_create_play_pipeline(Dicto *d)
{
GstBus *bus;
DictoPrivate *p=GET_PRIVATE(d);
dicto_pipeline *np=&p->play;

g_debug("GST: Creating playback pipeline");
g_return_val_if_fail(p->gst_sink, FALSE);

g_object_ref(d);
np->owner=G_OBJECT(d);

np->pipeline=gst_pipeline_new("ppipeline");
if (!np->pipeline)
	goto error_play;

np->src=gst_element_factory_make("filesrc", "fsource");
if (!np->src)
	goto error_play;

np->filter=gst_element_factory_make("wavparse", "wavparse");
if (!np->filter)
	goto error_play;

np->level=gst_element_factory_make("level", "level");
if (!np->level)
	goto error_play;

np->sink=gst_element_factory_make(p->gst_sink, "asink");
if (!np->sink)
	goto error_play;

gst_bin_add_many(GST_BIN(np->pipeline), np->src, np->filter, np->level, np->sink, NULL);

if (!gst_element_link(np->src, np->filter))
	goto error_play;

if (!gst_element_link(np->level, np->sink))
	goto error_play;

g_signal_connect(np->filter, "pad_added", G_CALLBACK(dicto_new_pad_cb), np->level);

bus=gst_pipeline_get_bus(GST_PIPELINE(np->pipeline));
gst_bus_add_watch(bus, dicto_bus_cb, np);
gst_object_unref(bus);
np->rec=FALSE;
np->active=FALSE;

return TRUE;

error_play:;

dicto_pipeline_free(np);
return FALSE;
}

/**
 * Set the given elements (filesrc or filesink) file location
 */
static gboolean
dicto_set_filename(Dicto *d, GstElement *e, const gchar *file)
{
DictoPrivate *p=GET_PRIVATE(d);
gchar *tmp, *ecf;

g_return_val_if_fail(d, FALSE);
g_return_val_if_fail(e, FALSE);
g_return_val_if_fail(p->basedir, FALSE);

if (!file) {
	g_debug("Clearing file");
	g_object_set(G_OBJECT(e), "location", "", NULL);
	if (p->cfile)
		g_free(p->cfile);
	p->cfile=NULL;
	return TRUE;
}

if (g_path_is_absolute(file))
	tmp=g_strdup(file);
else
	tmp=g_build_path("/", p->basedir, file, NULL);

/* Is it already set, if so skip */
g_object_get(G_OBJECT(e), "location", &ecf, NULL);
if (p->cfile && ecf && strcmp(tmp, p->cfile)==0 && strcmp(tmp, ecf)==0) {
	g_debug("File is already set to: %s", tmp);
	g_free(ecf);
	g_free(tmp);
	return TRUE;
}
g_free(ecf);

if (p->cfile)
	g_free(p->cfile);

p->cfile=tmp;

gst_element_set_state(p->play.pipeline, GST_STATE_NULL);
g_object_set(G_OBJECT(e), "location", p->cfile, NULL);
return TRUE;
}

/**
 * 
 */
static gboolean
dicto_notes_load_list(Dicto *d, gboolean s)
{
GDir *ddir;
DictoPrivate *p=GET_PRIVATE(d);
const gchar *f;

g_return_val_if_fail(d, FALSE);

if (!p->notes)
	p->notes=g_hash_table_new(g_str_hash, g_str_equal);
else
	g_hash_table_remove_all(p->notes);

ddir=g_dir_open(p->basedir, 0, NULL);
if (!ddir)
	return FALSE;
while ((f=g_dir_read_name(ddir))!=NULL) {
	if (g_str_has_suffix(f, ".wav")) {
		g_debug("File: %s", f);
		g_hash_table_insert(p->notes, g_strdup(f), g_strdup(f));
	}
}
g_dir_close(ddir);
if (s)
	g_signal_emit(d, signals[SIGNAL_NOTES_REFRESH], 0, NULL);
return TRUE;
}

GHashTable *
dicto_notes_get_list(Dicto *d)
{
DictoPrivate *p=GET_PRIVATE(d);

g_return_val_if_fail(d, FALSE);

return p->notes;
}

/**
 * dicto_refresh:
 * @d: A #Dicto object
 *
 * Ask the Dicto object to reload file list from basedir.
 * Will send a "refresh" signal when done.
 * 
 * Returns: TRUE if reloading was successfull, FALSE on error.
 */
gboolean
dicto_refresh_list(Dicto *d)
{
g_return_val_if_fail(d, FALSE);
return dicto_notes_load_list(d, TRUE);
}

static gboolean
dicto_refresh_idle_cb(gpointer data)
{
dicto_refresh_list(DICTO(data));
return FALSE;
}

/**
 * dicto_play:
 * @d: A #Dicto object
 * @file: A file to play
 *
 * Plays given audio note file. Does nothing if already playing. Clears current note on error.
 *
 * Returns: %TRUE if playback started, %FALSE on error and if recording.
 */
gboolean
dicto_play(Dicto *d, const gchar *file)
{
DictoPrivate *p=GET_PRIVATE(d);

g_return_val_if_fail(d, FALSE);
g_return_val_if_fail(file, FALSE);

if (p->play.active==TRUE)
	return TRUE;

if (p->record.active==TRUE)
	return FALSE;

p->rforsecs=-1;

dicto_set_filename(d, p->play.src, file);

if (gst_element_set_state(p->play.pipeline, GST_STATE_PLAYING)==GST_STATE_CHANGE_FAILURE) {
	g_warning("Failed to play file %s\n", p->cfile);
	dicto_set_filename(d, p->play.src, NULL);
	p->play.active=FALSE;
	return FALSE;
}

g_debug("Playing: %s", p->cfile);
p->play.active=TRUE;
dicto_position_update(d, TRUE);
g_signal_emit(d, signals[SIGNAL_PLAYING], 0, NULL);
return TRUE;
}

/**
 * dicto_prepare:
 * @d: A #Dicto object
 * @file: A file to prepare for playback 
 *
 * Prepares given file for playback but does not start playback. Use dicto_play() with same filename to start playback.
 * Does nothing if we are already playing. Clears current file if error happens.
 *
 * Returns: %TRUE if preparation was successfull, %FALSE otherwise.
 */
gboolean
dicto_prepare(Dicto *d, const gchar *file)
{
DictoPrivate *p=GET_PRIVATE(d);

g_return_val_if_fail(d, FALSE);

if (p->play.active==TRUE)
	return FALSE;

if (p->record.active==TRUE)
	return FALSE;

g_return_val_if_fail(file, FALSE);

gst_element_set_state(p->play.pipeline, GST_STATE_NULL);
dicto_set_filename(d, p->play.src, file);
if (gst_element_set_state(p->play.pipeline, GST_STATE_PAUSED)==GST_STATE_CHANGE_FAILURE) {
	g_warning("Failed to go to ready state for file %s\n", p->cfile);
	dicto_set_filename(d, p->play.src, NULL);
	return FALSE;
}

g_debug("Prepared: %s", p->cfile);
g_signal_emit(d, signals[SIGNAL_READY], 0, NULL);
return TRUE;
}

/**
 * dicto_seek:
 * @d: A #Dicto object
 * @pos: Position to seek to
 *
 * Seeks to given position in the current playback note. The note must have been 
 * prepared with dicto_prepare() or beeing played back by dicto_play().
 *
 * Returns: %TRUE if seeking was successfull.
 */
gboolean dicto_seek(Dicto *d, gdouble pos)
{
DictoPrivate *p=GET_PRIVATE(d);
GstState current;
GstState pending;

g_return_val_if_fail(d, FALSE);
g_return_val_if_fail(p->play.pipeline, FALSE);
g_return_val_if_fail(pos>=0.0, FALSE);

gst_element_get_state(p->play.pipeline, &current, &pending, GST_CLOCK_TIME_NONE);
if (current==GST_STATE_NULL) {
	g_debug("Not in seekable state: %d", current);
	return FALSE;
}

if (!gst_element_seek(p->play.pipeline, 1.0, GST_FORMAT_TIME, GST_SEEK_FLAG_FLUSH, GST_SEEK_TYPE_SET, (gint64)(pos*GST_SECOND), GST_SEEK_TYPE_NONE, GST_CLOCK_TIME_NONE)) {
	g_warning("Seek failed");
	return FALSE;
}

g_debug("Seek to %f", pos);
g_signal_emit(d, signals[SIGNAL_SEEK], 0, pos, NULL);

return TRUE;
}

/**
 * dicto_record_timeout:
 * @d: A #Dicto object
 * @f: Destination filename to record to.
 * @secs: How many seconds to record, max (Not accurate, depend on update period). Give -1 for no limit.
 *
 * Records to given file for a minimum of @secs seconds. Emits #Dicto::recording when recording starts.
 *
 * Returns: TRUE if recording started, FALSE on error.
 */
gboolean
dicto_record_timeout(Dicto *d, const gchar *file, gdouble secs)
{
DictoPrivate *p=GET_PRIVATE(d);
g_return_val_if_fail(d, FALSE);
g_return_val_if_fail(file, FALSE);

if (p->record.active==TRUE)
	return FALSE;

if (p->play.active==TRUE)
	return FALSE;

if (p->cfile) {
	g_free(p->cfile);
	p->cfile=NULL;
}

p->rforsecs=secs;
gst_element_set_state(p->record.pipeline, GST_STATE_NULL);
dicto_set_filename(d, p->record.sink, file);
if (gst_element_set_state(p->record.pipeline, GST_STATE_PLAYING)==GST_STATE_CHANGE_FAILURE) {
	g_warning("Failed to record to file %s\n", file);
	p->record.active=FALSE;
	return FALSE;
}
p->record.active=TRUE;

dicto_position_update(d, TRUE);
g_signal_emit(d, signals[SIGNAL_RECORDING], 0, NULL);
g_idle_add(dicto_refresh_idle_cb, d);

g_debug("Recording");
return TRUE;
}

/**
 * dicto_record:
 * @d: A #Dicto object
 * @f: Destination filename to record to.
 *
 * Records to given file without a limit. Emits #Dicto::recording when recording starts.
 *
 * Returns: TRUE if recording started, FALSE on error.
 */
gboolean
dicto_record(Dicto *d, const gchar *file)
{
return dicto_record_timeout(d, file, -1);
}

/**
 * dicto_delete:
 * @d: A #Dicto object
 * @file: A note to delete, must be an existing note file name
 *
 * Returns: %TRUE if note was deleted. %FALSE on error.
 */
gboolean
dicto_delete(Dicto *d, const gchar *file)
{
DictoPrivate *p=GET_PRIVATE(d);
gchar *tmp;
gint r;

g_return_val_if_fail(d, FALSE);
g_return_val_if_fail(p->notes, FALSE);
g_return_val_if_fail(p->basedir, FALSE);
g_return_val_if_fail(file, FALSE);

if (g_path_is_absolute(file))
	return FALSE;

if (g_hash_table_lookup(p->notes, file)==NULL)
	return FALSE;

tmp=g_build_path("/", p->basedir, file, NULL);
g_return_val_if_fail(tmp, FALSE);
r=g_unlink(tmp);
if (r!=0)
	g_warning("Unlink of note %s (%s) failed", file, tmp);
g_debug("Removed: %s", tmp);
g_free(tmp);

if (g_hash_table_remove(p->notes, file)==FALSE) {
	g_warning("Failed to remove file from cache, reloading list");
	dicto_notes_load_list(d, TRUE);
	return TRUE;
}

/* Make sure we can change the location */
gst_element_set_state(p->play.pipeline, GST_STATE_NULL);
dicto_set_filename(d, p->play.src, NULL);

g_signal_emit(d, signals[SIGNAL_NOTES_REFRESH], 0, NULL);
return TRUE;
}

/**
 * dicto_stop:
 * @d: A #Dicto object
 *
 * Stop recording or playback. Emits signals #Dicto::stopped_play or #Dicto::stopped_record if pipelines where stopped.
 *
 * Returns: TRUE if a pipeline was stopped.
 */
gboolean
dicto_stop(Dicto *d)
{
DictoPrivate *p=GET_PRIVATE(d);
GstState current;
GstState pending;

gst_element_get_state(p->record.pipeline, &current, &pending, GST_CLOCK_TIME_NONE);
if (current==GST_STATE_PLAYING) {
	g_debug("Stop: recording");
	dicto_position_update(d, FALSE);
	gst_element_set_state(p->record.pipeline, GST_STATE_READY);
	p->record.active=FALSE;
	p->rms=0;
	g_signal_emit(d, signals[SIGNAL_STOPPED_RECORD], 0, NULL);
	return TRUE;
}

gst_element_get_state(p->play.pipeline, &current, &pending, GST_CLOCK_TIME_NONE);
if (current==GST_STATE_PLAYING || current==GST_STATE_PAUSED) {
	g_debug("Stop: playing");
	dicto_position_update(d, FALSE);
	gst_element_set_state(p->play.pipeline, GST_STATE_READY);
	p->play.active=FALSE;
	p->rms=0;
	g_signal_emit(d, signals[SIGNAL_STOPPED_PLAY], 0, NULL);
	return TRUE;
}

return FALSE;
}

/**
 * dicto_toggle_record:
 * @d: A #Dicto object
 * @file: Filename to use for new recording.
 *
 * Toggles recording: stops if recording, records if stopped.
 *
 * Returns: %TRUE if successfull, %FALSE if playing or other error.
 */
gboolean
dicto_toggle_record(Dicto *d, const gchar *file)
{
DictoPrivate *p=GET_PRIVATE(d);

g_return_val_if_fail(d, FALSE);

if (p->play.active)
	return FALSE;

return (p->record.active) ? dicto_stop(d) : dicto_record(d, file);
}

/**
 * dicto_get_position:
 * @d: A #Dicto object
 * @length: Pointer to a gdouble, to store length of the currently playing recording, dummy if recording.
 * @position: Pointer to a gdouble, to store the current position of playing recording or length to current recording.
 *
 * Get the current playback/recording position and stream length. If any of two arguments are null the queries will be skipped.
 *
 * Returns: TRUE if any of the two queries was ok.
 */
gboolean
dicto_get_position(Dicto *d, gdouble *length, gdouble *position)
{
DictoPrivate *p=GET_PRIVATE(d);
GstFormat fmt=GST_FORMAT_TIME;
GstElement *pipe;
gint64 pos, len;
gboolean r=FALSE;

g_return_val_if_fail(d, FALSE);

if (p->record.active)
	pipe=p->record.pipeline;
else
	pipe=p->play.pipeline;

if (length!=NULL && gst_element_query_duration(pipe, &fmt, &len)) {
	*length=GST_TIME_TO_SECS(len);
	r=TRUE;
}

if (position!=NULL && gst_element_query_position(pipe, &fmt, &pos)) {
	*position=GST_TIME_TO_SECS(pos);
	r=TRUE;
}

return r;
}

/**
 * dicto_new_full:
 * @basedir: Directory to use for file lists and new recordings
 * @f: Audio format to use
 * @sink: #GstElement audio sink element
 * @source: #GstElement audio source element
 *
 * Creates a new #Dicto object ready for use.
 *
 * Returns: The new Dicto object if pipelines where successfully built, NULL on failure.
 *
 */
Dicto *
dicto_new_full(gchar *basedir, DictoFormat f, gchar *sink, gchar *source)
{
Dicto *d;
DictoPrivate *p;

d=g_object_new(DICTO_TYPE,
	"basedir", basedir,
	"prefix" ,"an-",
	"suffix", "",
	"format", f,
	"source", source ? source : AUDIO_SRC,
	"sink", sink ? sink : AUDIO_SINK,
	NULL);

p=GET_PRIVATE(d);

if (!dicto_create_play_pipeline(d)) {
	g_object_unref(d);
	return NULL;
}
if (!dicto_create_record_pipeline(d)) {
	g_object_unref(d);
	return NULL;
}

p->pos_int=POS_INT_DEFAULT;

dicto_notes_load_list(d, FALSE);

return d;
}

/**
 * dicto_new:
 * @basedir: Directory to use for file lists and new recordings
 *
 * Creates a new #Dicto object ready for use with default format and audio sink/source, determined at ./configure time.
 *
 * Default for desktop gtk build uses #alsasink and #alsasrc.
 * Default for hildon 2.2 build uses #pulsesink and #pulsesrc.
 * Default for hildon 2.0 build uses #dsppcmsink and #dsppcmsrc.
 *
 * Returns: See dicto_new_full()
 */
Dicto *
dicto_new(gchar *basedir)
{
return dicto_new_full(basedir, FORMAT_DEFAULT, NULL, NULL);
}
