#!/usr/bin/env python
# -*- coding: UTF-8 -*-

"""
	Simple Function graph Plotter
	© Thomas Führinger, Sam Tygier 2005-2007
	http://lybniz2.sourceforge.net/

	Version 1.3.2
	Requires PyGtk 2.6
	Released under the terms of the revised BSD license
	Modified: 2007-12-14
"""

from __future__ import division
import gtk, pango
import sys
import math
from math import *
import hildon, os
import osso

app_version = '1.3.2'

fhsize = gtk.HILDON_SIZE_FINGER_HEIGHT

try:
	import gnome
	props = {gnome.PARAM_APP_DATADIR : '/usr/share'}
	prog = gnome.program_init("lybniz", str(app_version), properties=props)
except:
	print "Gnome not found"

import gettext
gettext.install('lybniz')

# profiling
enable_profiling = False
if enable_profiling:
	from time import time

app_win = None
actions = gtk.ActionGroup("General")
graph = None
connect_points = True

x_res = 2

x_max = "5.0"
x_min = "-5.0"
x_scale = "1.0"

y_max = "3.0"
y_min = "-3.0"
y_scale = "1.0"

y1 = "sin(x)"
y2 = ""
y3 = ""

icon_file = "/usr/share/lybniz/images/lybniz.png"

# some extra maths functions
def fac(x):
	if type(x) != int or x < 0:
		raise ValueError
	if x==0:
		return 1
	for n in range(2,x):
		x = x*n
	return x

def sinc(x):
	if x == 0:
		return 1
	return sin(x)/x


lybniz_app = osso.Context('maemo_lybniz', '1.3.2', False)


# create a safe namespace for the eval()s in the graph drawing code
def sub_dict(somedict, somekeys, default=None):
	return dict([ (k, somedict.get(k, default)) for k in somekeys ])
# a list of the functions from math that we want.
safe_list = ['math','acos', 'asin', 'atan', 'atan2', 'ceil', 'cos', 'cosh', 'degrees', 'e', 'exp', 'fabs', 'floor', 'fmod', 'frexp', 'hypot', 'ldexp', 'log', 'log10', 'modf', 'pi', 'pow', 'radians', 'sin', 'sinh', 'sqrt', 'tan', 'tanh','fac','sinc']
safe_dict = sub_dict(locals(), safe_list)

#add any needed builtins back in.
safe_dict['abs'] = abs

def marks(min_val,max_val,minor=1):
	"yield positions of scale marks between min and max. For making minor marks, set minor to the number of minors you want between majors"
	try:
		min_val = float(min_val)
		max_val = float(max_val)
	except:
		print "needs 2 numbers"
		raise ValueError

	if(min_val >= max_val):
		print "min bigger or equal to max"
		raise ValueError

	a = 0.2 # tweakable control for when to switch scales
	          # big a value results in more marks

	a = a + log10(minor)

	width = max_val - min_val
	log10_range = log10(width)

	interval = 10 ** int(floor(log10_range - a))
	lower_mark = min_val - fmod(min_val,interval)

	if lower_mark < min_val:
		lower_mark += interval

	a_mark = lower_mark
	while a_mark <= max_val:
		if abs(a_mark) < interval / 2:
			a_mark = 0
		yield a_mark
		a_mark += interval


class GraphClass:
	def __init__(self):
		# Create backing pixmap of the appropriate size
		def configure_event(widget, event):
			x, y, w, h = widget.get_allocation()
			self.pix_map = gtk.gdk.Pixmap(widget.window, w, h)

			# make colors
			self.gc = dict()
			for name, color in (('black',(0,0,0)),('red',(32000,0,0)),('blue',(0,0,32000)),('green',(0,32000,0))):
				self.gc[name] =self.pix_map.new_gc()
				self.gc[name].set_rgb_fg_color(gtk.gdk.Color(red=color[0],green=color[1],blue=color[2]))
			self.layout = pango.Layout(widget.create_pango_context())
			self.canvas_width = w
			self.canvas_height = h
			self.x_max = eval(x_max,{"__builtins__":{}},safe_dict)
			self.x_min = eval(x_min,{"__builtins__":{}},safe_dict)
			self.x_scale = eval(x_scale,{"__builtins__":{}},safe_dict)
			self.y_max = eval(y_max,{"__builtins__":{}},safe_dict)
			self.y_min = eval(y_min,{"__builtins__":{}},safe_dict)
			self.y_scale = eval(y_scale,{"__builtins__":{}},safe_dict)
			self.plot()
			return True

		# Redraw the screen from the backing pixmap
		def expose_event(widget, event):
			x, y, w, h = event.area
			widget.window.draw_drawable(widget.get_style().fg_gc[gtk.STATE_NORMAL], self.pix_map, x, y, x, y, w, h)
			return False

		# Start marking selection
		def button_press_event(widget, event):
			global x_sel, y_sel

			if event.button == 1:
				self.selection[0][0], self.selection[0][1] = int(event.x), int(event.y)
				self.selection[1][0], self.selection[1][1] = None, None

		# End of selection
		def button_release_event(widget, event):

			if event.button == 1 and event.x != self.selection[0][0] and event.y != self.selection[0][1]:
				xmi, ymi = min(self.graph_x(self.selection[0][0]), self.graph_x(event.x)), min(self.graph_y(self.selection[0][1]), self.graph_y(event.y))
				xma, yma = max(self.graph_x(self.selection[0][0]), self.graph_x(event.x)), max(self.graph_y(self.selection[0][1]), self.graph_y(event.y))
				self.x_min, self.y_min, self.x_max, self.y_max = xmi, ymi, xma, yma
				parameter_entries_repopulate()
				graph.plot()
				self.selection[1][0] = None
				self.selection[0][0] = None

		# Draw rectangle during mouse movement
		def motion_notify_event(widget, event):

			if event.is_hint:
				x, y, state = event.window.get_pointer()
			else:
				x = event.x
				y = event.y
				state = event.state

			if state & gtk.gdk.BUTTON1_MASK and self.selection[0][0] is not None:
				gc = self.drawing_area.get_style().black_gc
				gc.set_function(gtk.gdk.INVERT)
				if self.selection[1][0] is not None:
					x0 = min(self.selection[1][0], self.selection[0][0])
					y0 = min(self.selection[1][1], self.selection[0][1])
					w = abs(self.selection[1][0] - self.selection[0][0])
					h = abs(self.selection[1][1] - self.selection[0][1])
					self.pix_map.draw_rectangle(gc, False, x0, y0, w, h)
				x0 = min(self.selection[0][0], int(x))
				y0 = min(self.selection[0][1], int(y))
				w = abs(int(x) - self.selection[0][0])
				h = abs(int(y) - self.selection[0][1])
				self.pix_map.draw_rectangle(gc, False, x0, y0, w, h)
				self.selection[1][0], self.selection[1][1] = int(x), int(y)
				self.draw_drawable()

		self.prev_y = [None, None, None]

		# Marked area point[0, 1][x, y]
		self.selection = [[None, None], [None, None]]

		self.drawing_area = gtk.DrawingArea()
		self.drawing_area.connect("expose_event", expose_event)
		self.drawing_area.connect("configure_event", configure_event)
		self.drawing_area.connect("button_press_event", button_press_event)
		self.drawing_area.connect("button_release_event", button_release_event)
		self.drawing_area.connect("motion_notify_event", motion_notify_event)
		self.drawing_area.set_events(gtk.gdk.EXPOSURE_MASK | gtk.gdk.LEAVE_NOTIFY_MASK | gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.BUTTON_RELEASE_MASK | gtk.gdk.POINTER_MOTION_MASK |gtk.gdk.POINTER_MOTION_HINT_MASK)
		self.scale_style = "dec"

	def draw_drawable(self):
		x, y, w, h = self.drawing_area.get_allocation()
		self.drawing_area.window.draw_drawable(self.drawing_area.get_style().fg_gc[gtk.STATE_NORMAL], self.pix_map, 0, 0, 0, 0, w, h)

	def plot(self):
		self.pix_map.draw_rectangle(self.drawing_area.get_style().white_gc, True, 0, 0, self.canvas_width, self.canvas_height)

		if (self.scale_style == "cust"):

			#draw cross
			self.pix_map.draw_lines(self.gc['black'], [(int(round(self.canvas_x(0))),0),(int(round(self.canvas_x(0))),self.canvas_height)])
			self.pix_map.draw_lines(self.gc['black'], [(0,int(round(self.canvas_y(0)))),(self.canvas_width,int(round(self.canvas_y(0))))])
			# old style axis marks
			iv = self.x_scale * self.canvas_width / (self.x_max - self.x_min) # pixel interval between marks
			os = self.canvas_x(0) % iv # pixel offset of first mark
			# loop over each mark.
			for i in xrange(int(self.canvas_width / iv + 1)):
				#multiples of iv, cause adding of any error in iv, so keep iv as float
				# use round(), to get to closest pixel, int() to prevent warning
				self.pix_map.draw_lines(self.gc['black'], [(int(round(os + i * iv)), int(round(self.canvas_y(0) - 5))), (int(round(os + i * iv)), int(round(self.canvas_y(0) + 5)))])

			# and the y-axis
			iv = self.y_scale * self.canvas_height / (self.y_max - self.y_min)
			os = self.canvas_y(0) % iv
			for i in xrange(int(self.canvas_height / iv + 1)):
				self.pix_map.draw_lines(self.gc['black'], [(int(round(self.canvas_x(0) - 5)), int(round(i * iv + os))), (int(round(self.canvas_x(0) + 5)), int(round(i * iv + os)))])

		else:
			#new style
			factor = 1
			if (self.scale_style == "rad"): factor = pi

			# where to put the numbers
			numbers_x_pos = -10
			numbers_y_pos = 10

			# where to center the axis
			center_x_pix = int(round(self.canvas_x(0)))
			center_y_pix = int(round(self.canvas_y(0)))
			if (center_x_pix < 5): center_x_pix = 5
			if (center_x_pix < 20):numbers_x_pos = 10
			if (center_y_pix < 5): center_y_pix = 5
			if (center_x_pix > self.canvas_width - 5): center_x_pix = self.canvas_width - 5
			if (center_y_pix > self.canvas_height -5): center_y_pix = self.canvas_height - 5;
			if (center_y_pix > self.canvas_height -20): numbers_y_pos = - 10

			# draw cross
			self.pix_map.draw_lines(self.gc['black'], [(center_x_pix,0),(center_x_pix,self.canvas_height)])
			self.pix_map.draw_lines(self.gc['black'], [(0,center_y_pix),(self.canvas_width,center_y_pix)])

			for i in marks(self.x_min / factor, self.x_max / factor):
				label = '%g' % i
				if (self.scale_style == "rad"): label += '\xCF\x80'
				i = i * factor

				self.pix_map.draw_lines(self.gc['black'], [(int(round(self.canvas_x(i))), center_y_pix - 5), (int(round(self.canvas_x(i))), center_y_pix + 5)])

				self.layout.set_text(label)
				extents = self.layout.get_pixel_extents()[1]
				if (numbers_y_pos < 0): adjust = extents[3]
				else: adjust = 0
				self.pix_map.draw_layout(self.gc['black'],int(round(self.canvas_x(i))), center_y_pix + numbers_y_pos - adjust,self.layout)

			for i in marks(self.y_min,self.y_max):
				label = '%g' % i

				self.pix_map.draw_lines(self.gc['black'], [(center_x_pix - 5, int(round(self.canvas_y(i)))), (center_x_pix + 5, int(round(self.canvas_y(i))))])

				self.layout.set_text(label)
				extents = self.layout.get_pixel_extents()[1]
				if (numbers_x_pos < 0): adjust = extents[2]
				else: adjust = 0
				self.pix_map.draw_layout(self.gc['black'],center_x_pix +numbers_x_pos - adjust,int(round(self.canvas_y(i))),self.layout)

			# minor marks
			for i in marks(self.x_min / factor, self.x_max / factor, minor=10):
				i = i * factor
				self.pix_map.draw_lines(self.gc['black'], [(int(round(self.canvas_x(i))), center_y_pix - 2), (int(round(self.canvas_x(i))), center_y_pix +2)])

			for i in marks(self.y_min, self.y_max, minor=10):
				label = '%g' % i
				self.pix_map.draw_lines(self.gc['black'], [(center_x_pix - 2, int(round(self.canvas_y(i)))), (center_x_pix +2, int(round(self.canvas_y(i))))])

		plots = []
		# precompile the functions
		try:
			compiled_y1 = compile(y1.replace("^","**"),"",'eval')
			plots.append((compiled_y1,0,self.gc['blue']))
		except:
			compiled_y1 = None
		try:
			compiled_y2 = compile(y2.replace("^","**"),"",'eval')
			plots.append((compiled_y2,1,self.gc['red']))
		except:
			compiled_y2 = None
		try:
			compiled_y3 = compile(y3.replace("^","**"),"",'eval')
			plots.append((compiled_y3,2,self.gc['green']))
		except:
			compiled_y3 = None

		self.prev_y = [None, None, None]

		if enable_profiling:
			start_graph = time()

		if len(plots) != 0:
			for i in xrange(0,self.canvas_width,x_res):
				x = self.graph_x(i + 1)
				for e in plots:
					safe_dict['x']=x
					try:
						y = eval(e[0],{"__builtins__":{}},safe_dict)
						y_c = int(round(self.canvas_y(y)))

						if y_c < 0 or y_c > self.canvas_height:
							raise ValueError

						if connect_points and self.prev_y[e[1]] is not None:
							self.pix_map.draw_lines(e[2], [(i, self.prev_y[e[1]]), (i + x_res, y_c)])
						else:
							self.pix_map.draw_points(e[2], [(i + x_res, y_c)])
						self.prev_y[e[1]] = y_c
					except:
						#print "Error at %d: %s" % (x, sys.exc_value)
						self.prev_y[e[1]] = None

		if enable_profiling:
			print "time to draw graph:", (time() - start_graph) * 1000, "ms"

		self.draw_drawable()

	def canvas_x(self, x):
		"Calculate position on canvas to point on graph"
		return (x - self.x_min) * self.canvas_width / (self.x_max - self.x_min)

	def canvas_y(self, y):
		return (self.y_max - y) * self.canvas_height / (self.y_max - self.y_min)

	def canvas_point(self, x, y):
		return (self.canvas_x(x), self.canvas_y(y))

	def graph_x(self, x):
		"Calculate position on graph from point on canvas"
		return x  * (self.x_max - self.x_min) / self.canvas_width + self.x_min

	def graph_y(self, y):
		return self.y_max - (y * (self.y_max - self.y_min) / self.canvas_height)


def style_selector():
	selector = hildon.hildon_touch_selector_new_text()
	data = ["Decimal", "Radians"]
	for i in range(len(data)):
		selector.append_text(data[i])

	return selector

def on_set_scale_style(button):
	dlg = hildon.PickerDialog(app_win)
	selector = style_selector()
	selector.connect("changed", set_style, dlg, button)
	dlg.set_selector(selector)

	dlg.show_all()
	dlg.run()

def set_style(selector, column, dlg, button):
	value = selector.get_current_text()
	dlg.destroy()
	if value == 'Decimal':
		scale_dec(button)
	elif value == 'Radians':
		scale_rad(button)
	button.set_value(value)

def menu_create():

	menu = hildon.AppMenu()

	button = gtk.Button(_("Save"))
	button.connect("clicked", save)
	menu.append(button)

	#button = gtk.Button(_("Scale Style"))
	button = hildon.Button(fhsize, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
	button.set_value("Decimal")
	button.set_title(_("Scale Style"))
	button.connect("clicked", on_set_scale_style)
	menu.append(button)

	button = hildon.CheckButton(fhsize)
	button.set_label(_("Connect Points"))
	button.set_active(True)
	button.connect("toggled", toggle_connect)
	menu.append(button)

	button = gtk.Button(_("Help"))
	button.connect("clicked", show_help)
	menu.append(button)

	button = gtk.Button(_("About"))
	button.connect("clicked", show_about_dialog)
	menu.append(button)

	menu.show_all()
	app_win.set_app_menu(menu)

def menu_fs_func(widget):
	if app_win.menu_fs.get_active() == False:
		app_win.unfullscreen()
	else:
		app_win.fullscreen()

def options_toolbar(widget):
	if app_win.menu_toolbar.get_active() == False:
		app_win.tool_bar.destroy()
		app_win.show_all()
	else:
		toolbar_create(widget)

def toolbar_create(widget):
	app_win.tool_bar = gtk.Toolbar()

	tool1 = gtk.ToolButton(gtk.STOCK_JUMP_TO)
	tool1.connect("clicked", fun_dialog)
	tool1.set_expand(True)
	app_win.tool_bar.insert(tool1, -1)

	tool2 = gtk.ToolButton(gtk.STOCK_EXECUTE)
	tool2.connect("clicked", evaluate)
	tool2.set_expand(True)
	app_win.tool_bar.insert(tool2, -1)

	app_win.tool_bar.insert(gtk.SeparatorToolItem(), -1)

	tool3 = gtk.ToolButton(gtk.STOCK_ZOOM_IN)
	tool3.connect("clicked", zoom_in)
	tool3.set_expand(True)
	app_win.tool_bar.insert(tool3, -1)

	tool4 = gtk.ToolButton(gtk.STOCK_ZOOM_OUT)
	tool4.connect("clicked", zoom_out)
	tool4.set_expand(True)
	app_win.tool_bar.insert(tool4, -1)

	tool5 = gtk.ToolButton(gtk.STOCK_ZOOM_100)
	tool5.connect("clicked", zoom_reset)
	tool5.set_expand(True)
	app_win.tool_bar.insert(tool5, -1)

	app_win.add_toolbar(app_win.tool_bar)
	app_win.show_all()

def plot(widget, event=None):
	global x_max, x_min, x_scale, y_max, y_min, y_scale, y1, y2, y3

	x_max = app_win.x_max_entry.get_text()
	x_min = app_win.x_min_entry.get_text()
	#x_scale = app_win.x_scale_entry.get_text()

	y_max = app_win.y_max_entry.get_text()
	y_min = app_win.y_min_entry.get_text()
	#y_scale = app_win.y_scale_entry.get_text()

	graph.x_max = eval(x_max,{"__builtins__":{}},safe_dict)
	graph.x_min = eval(x_min,{"__builtins__":{}},safe_dict)
	#graph.x_scale = eval(x_scale,{"__builtins__":{}},safe_dict)

	graph.y_max = eval(y_max,{"__builtins__":{}},safe_dict)
	graph.y_min = eval(y_min,{"__builtins__":{}},safe_dict)
	#graph.y_scale = eval(y_scale,{"__builtins__":{}},safe_dict)

	y1 = app_win.y1_entry.get_text()
	y2 = app_win.y2_entry.get_text()
	y3 = app_win.y3_entry.get_text()

	graph.plot()


def evaluate(widget, event=None):
	"Evaluate a given x for the three functions"

	def entry_changed(self):
		for e in ((y1, dlg_win.y1_entry), (y2, dlg_win.y2_entry), (y3, dlg_win.y3_entry)):
			try:
				x = float(dlg_win.x_entry.get_text())
				safe_dict['x']=x
				e[1].set_text(str(eval(e[0].replace("^","**"),{"__builtins__":{}},safe_dict)))
			except:
				if len(e[0]) > 0:
					e[1].set_text("Error: %s" % sys.exc_value)
				else:
					e[1].set_text("")

	def close(self):
		dlg_win.destroy()


	dlg_win = gtk.Dialog(title='Evaluate', parent=None, flags=gtk.DIALOG_MODAL | gtk.WIN_POS_CENTER)
	#dlg_win.set_default_size(560, 200)

	dlg_win.x_entry = hildon.Entry(fhsize)
	#dlg_win.x_entry.set_size_request(200, 40)
	dlg_win.x_entry.connect("changed", entry_changed)
	dlg_win.y1_entry = hildon.Entry(fhsize)
	#dlg_win.y1_entry.set_size_request(200, 40)
	dlg_win.y1_entry.set_sensitive(False)
	dlg_win.y2_entry = hildon.Entry(fhsize)
	#dlg_win.y2_entry.set_size_request(200, 40)
	dlg_win.y2_entry.set_sensitive(False)
	dlg_win.y3_entry = hildon.Entry(fhsize)
	#dlg_win.y3_entry.set_size_request(200, 40)
	dlg_win.y3_entry.set_sensitive(False)

	table = gtk.Table(2, 5)
	label = gtk.Label("x = ")
	label.set_alignment(0, .5)
	table.attach(label, 0, 1, 0, 1, xpadding=5, ypadding=5, xoptions=gtk.FILL)
	table.attach(dlg_win.x_entry, 1, 2, 0, 1)
	label = gtk.Label("y1 = ")
	label.set_alignment(0, .5)
	label.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse("blue"))
	table.attach(label, 0, 1, 1, 2, xpadding=5, ypadding=5, xoptions=gtk.FILL)
	table.attach(dlg_win.y1_entry, 1, 2, 1, 2)
	label = gtk.Label("y2 = ")
	label.set_alignment(0, .5)
	label.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse("red"))
	table.attach(label, 0, 1, 2, 3, xpadding=5, ypadding=5, xoptions=gtk.FILL)
	table.attach(dlg_win.y2_entry, 1, 2, 2, 3)
	label = gtk.Label("y3 = ")
	label.set_alignment(0, .5)
	label.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse("DarkGreen"))
	table.attach(label, 0, 1, 3, 4, xpadding=5, ypadding=5, xoptions=gtk.FILL)
	table.attach(dlg_win.y3_entry, 1, 2, 3, 4)

	table.set_border_width(24)

	#hbuttonbox1 = gtk.HButtonBox()
	#hbuttonbox1.show()
	#hbuttonbox1.set_spacing(2)
	#hbuttonbox1.set_border_width(2)
	#hbuttonbox1.set_layout(gtk.BUTTONBOX_END)

	#button1 = gtk.Button(label="Close", stock=gtk.STOCK_CLOSE)
	#button1.show()
	#button1.connect("clicked", close)
	#hbuttonbox1.pack_start(button1)

	#dlg_win.action_area.pack_start(hbuttonbox1, False, False, 8)

	dlg_win.vbox.pack_start(table, True, True, 8)
	dlg_win.show_all()


def zoom_in(widget, event=None):
	"Narrow the plotted section by half"
	center_x = (graph.x_min + graph.x_max) / 2
	center_y = (graph.y_min + graph.y_max) / 2
	range_x = (graph.x_max - graph.x_min)
	range_y = (graph.y_max - graph.y_min)

	graph.x_min = center_x - (range_x / 4)
	graph.x_max = center_x + (range_x / 4)
	graph.y_min = center_y - (range_y / 4)
	graph.y_max = center_y +(range_y / 4)

	parameter_entries_repopulate()
	graph.plot()


def zoom_out(widget, event=None):
	"Double the plotted section"
	center_x = (graph.x_min + graph.x_max) / 2
	center_y = (graph.y_min + graph.y_max) / 2
	range_x = (graph.x_max - graph.x_min)
	range_y = (graph.y_max - graph.y_min)

	graph.x_min = center_x - (range_x)
	graph.x_max = center_x + (range_x)
	graph.y_min = center_y - (range_y)
	graph.y_max = center_y +(range_y)

	parameter_entries_repopulate()
	graph.plot()


def zoom_reset(widget, event=None):
	"Set the range back to the user's input"

	graph.x_min = eval(x_min,{"__builtins__":{}},safe_dict)
	graph.y_min = eval(y_min,{"__builtins__":{}},safe_dict)
	graph.x_max = eval(x_max,{"__builtins__":{}},safe_dict)
	graph.y_max = eval(y_max,{"__builtins__":{}},safe_dict)
	parameter_entries_populate()
	graph.plot()


def scale_dec(widget, event=None):
	graph.scale_style = "dec"
	plot(None)


def scale_rad(widget, event=None):
	graph.scale_style = "rad"
	plot(None)


def scale_cust(widget, event=None):
	graph.scale_style = "cust"
	plot(None)


def toggle_connect(widget, event=None):
	"Toggle between a graph that connects points with lines and one that does not"

	global connect_points
	connect_points = not connect_points
	graph.plot()


def save(widget, event=None):
	import os
	"Save graph as .png"
	HOME = os.path.expanduser("~")
	m = hildon.FileSystemModel()
	file_dialog = hildon.FileChooserDialog(app_win, gtk.FILE_CHOOSER_ACTION_SAVE, m)
	file_dialog.set_default_response(gtk.RESPONSE_OK)
	file_dialog.set_current_folder(HOME + "/MyDocs/.images")
	file_dialog.set_current_name("graph.png")

#	filter = gtk.FileFilter()
#	filter.set_name("All files")
#	filter.add_pattern("*")
#	file_dialog.add_filter(filter)

#	filter = gtk.FileFilter()
#	filter.set_name("PNG")
#	filter.add_pattern("*.png")
#	file_dialog.add_filter(filter)

#	filter = gtk.FileFilter()
#	filter.add_mime_type("image/png")
#	filter.add_pattern("*.png")
#	file_dialog.add_filter(filter)
#	file_dialog.set_filename("FunctionGraph.png")

	response = file_dialog.run()
	if response == gtk.RESPONSE_OK:
		x, y, w, h = graph.drawing_area.get_allocation()
		pix_buffer = gtk.gdk.Pixbuf(gtk.gdk.COLORSPACE_RGB, False, 8, w, h)
		pix_buffer.get_from_drawable(graph.pix_map, graph.pix_map.get_colormap(), 0, 0, 0, 0, w, h)
		pix_buffer.save(file_dialog.get_filename(), "png")
	file_dialog.destroy()


def quit_dlg(widget, event=None):
	gtk.main_quit()


def show_help(widget):
	import webbrowser
	webbrowser.open("file:///usr/share/lybniz/doc/lybniz.html")
	#os.system("dbus-send --print-reply --dest=com.nokia.osso_browser /com/nokia/osso_browser/service com.nokia.osso_browser.load_url string:'file:///usr/share/lybniz/doc/lybniz.html'")


def show_about_dialog(widget):
	about_dialog = gtk.AboutDialog()
	about_dialog.set_name("Lybniz")
	about_dialog.set_version(str(app_version))
	#about_dialog.set_copyright(u"© 2005 by Thomas Führinger")
	about_dialog.set_authors([u"Thomas Führinger","Sam Tygier", "", u"Maemo Port: Daniel Martín Yerga"])
	about_dialog.set_comments(_("Function graph Plotter"))
	about_dialog.set_license("Revised BSD")
	about_dialog.set_website("http://lybniz2.sourceforge.net/")
	try:
		lybniz_icon = gtk.gdk.pixbuf_new_from_file(icon_file)
		about_dialog.set_logo(lybniz_icon)
	except:
		print "icon not found at", icon_file
	about_dialog.connect ("response", lambda d, r: d.destroy())
	about_dialog.run()


def parameter_entries_populate():
	# set text in entries for parameters with user's input

	app_win.y1_entry.set_text(y1)
	app_win.y2_entry.set_text(y2)
	app_win.y3_entry.set_text(y3)
	app_win.x_min_entry.set_text(x_min)
	app_win.x_max_entry.set_text(x_max)
	app_win.y_min_entry.set_text(y_min)
	app_win.y_max_entry.set_text(y_max)


def parameter_entries_repopulate():
	# set text in entries for parameters

	app_win.y1_entry.set_text(y1)
	app_win.y2_entry.set_text(y2)
	app_win.y3_entry.set_text(y3)
	app_win.x_min_entry.set_text(str(graph.x_min))
	app_win.x_max_entry.set_text(str(graph.x_max))
	app_win.y_min_entry.set_text(str(graph.y_min))
	app_win.y_max_entry.set_text(str(graph.y_max))


def key_press_plot(widget, event):
	if event.keyval == 65293:
		plot(None)
		return True
	else:
		return False

## Functions for fullscreen
def on_window_state_change(widget, event, *args):
	if event.new_window_state & gtk.gdk.WINDOW_STATE_FULLSCREEN:
		global app_win_in_fullscreen
		app_win_in_fullscreen = True
	else:
		app_win_in_fullscreen = False

def on_key_press(widget, event, *args):
	if event.keyval == gtk.keysyms.F6:
		if app_win_in_fullscreen:
			app_win.unfullscreen()
			app_win.menu_fs.set_active(False)
		else:
			app_win.fullscreen()
			app_win.menu_fs.set_active(True)
	elif event.keyval == gtk.keysyms.F7:
		zoom_in(widget)
	elif event.keyval == gtk.keysyms.F8:
		zoom_out(widget)
	elif event.keyval == gtk.keysyms.Escape:
		zoom_reset(widget)
	elif event.keyval == gtk.keysyms.Return:
		fun_dialog(widget)

def fun_dialog(widget):
		def on_cerrar_clicked(widget):
			dlg.destroy()

		def on_aceptar_clicked(widget):
			plot(None)
			dlg.destroy()

		dlg = gtk.Dialog(title='Define Functions', parent=None, flags=gtk.DIALOG_MODAL | gtk.WIN_POS_CENTER)


		hbox = gtk.HBox()
		label = gtk.Label("y1 = ")
		label.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse("blue"))
		app_win.y1_entry = hildon.Entry(fhsize)
		hbox.pack_start(label, False, False, 0)
		hbox.pack_start(app_win.y1_entry, True, True, 0)
		dlg.vbox.pack_start(hbox, True, True, 0)

		hbox = gtk.HBox()
		label = gtk.Label("y2 = ")
		label.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse("red"))
		app_win.y2_entry = hildon.Entry(fhsize)
		hbox.pack_start(label, False, False, 0)
		hbox.pack_start(app_win.y2_entry, True, True, 0)
		dlg.vbox.pack_start(hbox, True, True, 0)

		hbox = gtk.HBox()
		label = gtk.Label("y3 = ")
		label.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse("DarkGreen"))
		app_win.y3_entry = hildon.Entry(fhsize)
		hbox.pack_start(label, False, False, 0)
		hbox.pack_start(app_win.y3_entry, True, True, 0)
		dlg.vbox.pack_start(hbox, True, True, 0)

		hbox = gtk.HBox()
		label = gtk.Label(_("X min"))
		app_win.x_min_entry = hildon.Entry(fhsize)
		hbox.pack_start(label, False, False, 0)
		hbox.pack_start(app_win.x_min_entry, True, True, 0)

		label = gtk.Label(_("X max"))
		app_win.x_max_entry = hildon.Entry(fhsize)
		hbox.pack_start(label, False, False, 0)
		hbox.pack_start(app_win.x_max_entry, True, True, 0)

		label = gtk.Label(_("Y min"))
		app_win.y_min_entry = hildon.Entry(fhsize)
		hbox.pack_start(label, False, False, 0)
		hbox.pack_start(app_win.y_min_entry, True, True, 0)

		label = gtk.Label(_("Y max"))
		app_win.y_max_entry = hildon.Entry(fhsize)
		hbox.pack_start(label, False, False, 0)
		hbox.pack_start(app_win.y_max_entry, True, True, 0)
		dlg.vbox.pack_start(hbox, True, True, 0)

		parameter_entries_populate()

		app_win.y1_entry.connect("key-press-event", key_press_plot)
		app_win.y2_entry.connect("key-press-event", key_press_plot)
		app_win.y3_entry.connect("key-press-event", key_press_plot)
		app_win.x_min_entry.connect("key-press-event", key_press_plot)
		app_win.y_min_entry.connect("key-press-event", key_press_plot)
		app_win.x_max_entry.connect("key-press-event", key_press_plot)
		app_win.y_max_entry.connect("key-press-event", key_press_plot)

		button1 = hildon.Button(fhsize, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
		button1.set_title("Plot")
		button1.connect("clicked", on_aceptar_clicked)

		dlg.vbox.pack_start(button1, True, True, 2)

		dlg.show_all()

def main():
	global app_win, graph, app_win_in_fullscreen

	app_win = hildon.Window()
	app_win.set_default_size(800, 480)
	gtk.set_application_name("Lybniz")
	app_win.connect("delete-event", quit_dlg)

	app_win.accel_group = gtk.AccelGroup()
	app_win.add_accel_group(app_win.accel_group)

	app_win.connect("key-press-event", on_key_press)
	app_win.connect("window-state-event", on_window_state_change)
	app_win_in_fullscreen = False

	app_win.v_box = gtk.VBox(False, 1)
	app_win.v_box.set_border_width(1)
	app_win.add(app_win.v_box)

	menu_create()
	toolbar_create(app_win)

	graph = GraphClass()
	app_win.v_box.pack_start(graph.drawing_area, True, True, 0)

	app_win.show_all()

	fun_dialog(None)

	gtk.main()


# Start it all
if __name__ == '__main__': main()
