#include "luxwidget.h"


#include <QPainter>
#include <QPaintEvent>
#include <QPen>
#include <QBrush>
#include <QFont>

#include <QTimer>
#include <QtDebug>


#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <QtGui/QX11Info>
#include <QtCore/QByteArray>
#include <QtCore/QCoreApplication>

#include <QtGui/QMessageBox>




LuxWidget::LuxWidget()
    : QWidget(0, Qt::FramelessWindowHint),
      dirty(true)
{
    QCoreApplication::instance()->setApplicationName("Luxus LUXmeter");

    // Get required atoms
    Atom winTypeAtom = XInternAtom(QX11Info::display(), "_NET_WM_WINDOW_TYPE", false);
    Atom homeAppletAto = XInternAtom(QX11Info::display(), "_HILDON_WM_WINDOW_TYPE_HOME_APPLET", false);
    Atom appletIDAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_ID", false);
    Atom utf8Atom = XInternAtom(QX11Info::display(), "UTF8_STRING", false); 
    Atom appletSettingAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_SETTINGS", false);
   
    // Set correct window type
    XChangeProperty(QX11Info::display(), winId(), winTypeAtom, XA_ATOM, 32,
		    PropModeReplace, (unsigned char *) &homeAppletAto, 1);

    // Use application name to fill AppletID
    QByteArray id (QCoreApplication::instance()->applicationName().remove(' ').toUtf8());
    XChangeProperty(QX11Info::display(), winId(), appletIDAtom, utf8Atom, 8, 
    	            PropModeReplace, (unsigned char *)id.constData(), id.length());

    // Add setting button. This button is shown when hildon-desktop is in edit mode. 
    int settings = 0;
    XChangeProperty(QX11Info::display(), winId(), appletSettingAtom, XA_CARDINAL, 32,
		    PropModeReplace, (unsigned char*)&settings, 1);



    // window update timer
    timerId = startTimer(400);

    // default metering unit
    currentUnit = LuxusUnits_Lux;
  
}


void LuxWidget::paintEvent(QPaintEvent *e)
{
    QPainter p(this);
    p.setRenderHint(QPainter::Antialiasing);
    p.setClipRect(e->rect());

    //make sure you clean your widget with a transparent
    //  color before doing any rendering
    //  note the usage of a composition mode Source
    //  it's important!
    p.save();
    p.setCompositionMode(QPainter::CompositionMode_Source);
    p.setOpacity(0.25);
    p.fillRect(rect(), Qt::lightGray);
    p.restore();
    

    if (dirty) {
        cache.fill(Qt::transparent);
        QPainter p(&cache);
        p.setRenderHint(QPainter::Antialiasing);

        // draw outer frame
        p.setPen(QPen(Qt::darkGray, 3, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin ));
        p.setOpacity(0.8);
        p.drawRect(0,0, this->width()-1, this->height()-1 );


        // draw title

        QString textTitle("Luxus light meter");
        QFont fontTitle("Sans", 12, QFont::SemiCondensed);
        QFontMetrics fmTitle(fontTitle);
        int textWidthTitle = fmTitle.width(textTitle);
        int textHeightTitle = fmTitle.height();
        p.setFont(fontTitle);
        p.setOpacity(0.94);
        p.translate(width()/2, 0);
        p.setPen(QPen(QBrush("#ffffff"), 1));
        p.drawText(-textWidthTitle/2, textHeightTitle, textTitle);
        p.resetTransform();
        // paint meter value

        QString unitText;
        QString valueText;

        int lux = getLux();

        if(currentUnit==LuxusUnits_Lux) {
            unitText = "LUX";
            valueText = QString::number(lux);
        }
        if(currentUnit==LuxusUnits_EV_100) {
            unitText = "EV";
            QPair<int,int> ev = Lux2EV(lux);
            valueText = QString::number(ev.first);
            if(ev.second > 0) {
                valueText = valueText + " " + QString::number(ev.second) + "/3";
            }
        }


        QString textValue(valueText);



        // white part of text
        p.setPen(QPen(QBrush("#ffffff"), 1));


        QFont fontValue("Sans", 24, QFont::Bold);
        QFontMetrics fmValue(fontValue);
        int textWidthValue = fmValue.width(textValue);

        p.setFont(fontValue);

        p.setOpacity(1);

        p.translate(QPoint(width()/2, height()/2));
        p.drawText(-textWidthValue/2, 10, textValue);


        // blue part of text
        p.setPen(QPen(QBrush("#0000ff"), 1));
        p.setOpacity(1);
        p.drawText(-textWidthValue/2 +1, 11, textValue);


        // end of paint meter value


        // paint meter unit


        QFont fontUnit("Courier", 12, QFont::Bold);
        QFontMetrics fmUnit(fontUnit);
        int textWidthUnit = fmUnit.width(unitText);
        int textHeightUnit = fmUnit.height();
        p.setFont(fontUnit);
        p.setOpacity(0.94);
        p.resetTransform();

        p.setPen(QPen(QBrush("#ffffff"), 1));
        p.drawText(width()-textWidthUnit-3, height()-textHeightUnit, unitText);
        p.resetTransform();

        // end of paint meter unit



        p.end();
        dirty = false;
    }
    p.drawPixmap(0, 0, cache);
        
}

//Filter show setting requests
bool QWidget::x11Event ( XEvent * event )
{
  static Atom appletShowSettingAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_SHOW_SETTINGS", false);

  if (event->xclient.message_type == appletShowSettingAtom ) { 
      QMessageBox::information(0, "About", "Luxus light meter widget for the N900 (C) 2010, Toni Nikkanen <toni@tuug.fi>.", QMessageBox::Ok, QMessageBox::Ok);
      return true;
  }
  return false;
}



void LuxWidget::resizeEvent(QResizeEvent *e)
{
    if (e->size() != cache.size()) {
        cache = QPixmap(e->size());
        dirty = true;
        int w = e->size().width();
        int h = e->size().height();
	Q_UNUSED(w);
	Q_UNUSED(h);
    }
}


void LuxWidget::timerEvent(QTimerEvent *event)
{
    dirty = true;
    repaint();
}

void LuxWidget::mouseReleaseEvent(QMouseEvent *evt)
{
    if (currentUnit == LuxusUnits_Lux) {
        currentUnit = LuxusUnits_EV_100;
    }
    else if (currentUnit == LuxusUnits_EV_100) {
        currentUnit = LuxusUnits_Lux;
    }

    dirty = true;
    repaint();
}

