#include "luxwidget.h"

#include <QPainter>
#include <QPaintEvent>
#include <QPen>
#include <QBrush>
#include <QFont>

#include <QTimer>
#include <QtDebug>
#include <QFile>


#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <QtGui/QX11Info>
#include <QtCore/QByteArray>
#include <QtCore/QCoreApplication>

#include <QtGui/QMessageBox>

#include <math.h>



LuxWidget::LuxWidget()
    : QWidget(0, Qt::FramelessWindowHint),
      dirty(true)
{
    QCoreApplication::instance()->setApplicationName("Luxus LUXmeter");

    // Get required atoms
    Atom winTypeAtom = XInternAtom(QX11Info::display(), "_NET_WM_WINDOW_TYPE", false);
    Atom homeAppletAto = XInternAtom(QX11Info::display(), "_HILDON_WM_WINDOW_TYPE_HOME_APPLET", false);
    Atom appletIDAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_ID", false);
    Atom utf8Atom = XInternAtom(QX11Info::display(), "UTF8_STRING", false); 
    Atom appletSettingAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_SETTINGS", false);
   
    // Set correct window type
    XChangeProperty(QX11Info::display(), winId(), winTypeAtom, XA_ATOM, 32,
		    PropModeReplace, (unsigned char *) &homeAppletAto, 1);

    // Use application name to fill AppletID
    QByteArray id (QCoreApplication::instance()->applicationName().remove(' ').toUtf8());
    XChangeProperty(QX11Info::display(), winId(), appletIDAtom, utf8Atom, 8, 
    	            PropModeReplace, (unsigned char *)id.constData(), id.length());

    // Add setting button. This button is shown when hildon-desktop is in edit mode. 
    int settings = 0;
    XChangeProperty(QX11Info::display(), winId(), appletSettingAtom, XA_CARDINAL, 32,
		    PropModeReplace, (unsigned char*)&settings, 1);



    // window update timer
    timerId = startTimer(400);

    // default metering unit
    currentUnit = LuxusUnits_Lux;
  
}


void LuxWidget::paintEvent(QPaintEvent *e)
{
    QPainter p(this);
    p.setRenderHint(QPainter::Antialiasing);
    p.setClipRect(e->rect());

    //make sure you clean your widget with a transparent
    //  color before doing any rendering
    //  note the usage of a composition mode Source
    //  it's important!
    p.save();
    p.setCompositionMode(QPainter::CompositionMode_Source);
    p.setOpacity(0.25);
    p.fillRect(rect(), Qt::lightGray);
    p.restore();
    
    if (dirty) {
        cache.fill(Qt::transparent);
        QPainter p(&cache);
        p.setRenderHint(QPainter::Antialiasing);


        // paint text here!


        QString unitText;
        QString valueText;

        int lux = getLux();

        if(currentUnit==LuxusUnits_Lux) {
            unitText = "LUX";
            valueText = QString::number(lux);
        }
        if(currentUnit==LuxusUnits_EV_100) {
            unitText = "EV";
            QPair<int,int> ev = Lux2EV(lux);
            valueText = QString::number(ev.first);
            if(ev.second > 0) {
                valueText = valueText + " " + QString::number(ev.second) + "/3";
            }
        }


        QString text(valueText + " " + unitText);
        qDebug() << text;

        // white text
        p.setPen(QPen(QBrush("#ffffff"), 1));


        QFont font("Sans", 24, QFont::Bold);
        QFontMetrics fm(font);
        int textWidth = fm.width(text);

        p.setFont(font);

        p.setOpacity(1);

         int h = height();
         int w = width();

         p.translate(QPoint(w/2, h/2));
         p.drawText(-textWidth/2, 0, text);


         // blue text
         p.setPen(QPen(QBrush("#0000ff"), 1));
         p.setOpacity(1);
         p.drawText(-textWidth/2 +1, 1, text);


         // end of paint text




        p.end();
        dirty = false;
    }
    p.drawPixmap(0, 0, cache);
        
}

//Filter show setting requests
bool QWidget::x11Event ( XEvent * event )
{
  static Atom appletShowSettingAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_SHOW_SETTINGS", false);

  if (event->xclient.message_type == appletShowSettingAtom ) { 
      QMessageBox::information(0, "About", "Luxus light meter widget for the N900 (C) 2010, Toni Nikkanen <toni@tuug.fi>.", QMessageBox::Ok, QMessageBox::Ok);
      return true;
  }
  return false;
}



void LuxWidget::resizeEvent(QResizeEvent *e)
{
    if (e->size() != cache.size()) {
        cache = QPixmap(e->size());
        dirty = true;
        int w = e->size().width();
        int h = e->size().height();
	Q_UNUSED(w);
	Q_UNUSED(h);
    }
}


void LuxWidget::timerEvent(QTimerEvent *event)
{
    qDebug() << "timerEvent entered";
    dirty = true;
    repaint();
}

void LuxWidget::mouseReleaseEvent(QMouseEvent *evt)
{
    if (currentUnit == LuxusUnits_Lux) {
        currentUnit = LuxusUnits_EV_100;
    }
    else if (currentUnit == LuxusUnits_EV_100) {
        currentUnit = LuxusUnits_Lux;
    }

    qDebug() << "mouseReleaseEvent entered";
    dirty = true;
    repaint();
}




// the following functions should be split into their own utility class some time

int LuxWidget::getLux()
{
    // scratchbox detected, supply fake value
    if(detectSB()) {
        static int fakeLux = 0;

        int retval = fakeLux;
        fakeLux++;
        if(fakeLux > 65000) {
            fakeLux = 0;
        }
        qDebug () << "getLux returns " << retval;
        return retval;
    }
    // assume real HW, read LUX file
    else {
        QFile sysFile("/sys/class/i2c-adapter/i2c-2/2-0029/lux");
        if (!sysFile.open(QIODevice::ReadOnly | QIODevice::Text)) {
            return -1;
        }
        QTextStream in(&sysFile);
        QString lux = in.readLine();
        qDebug() << "data is " << lux;

        return lux.toInt(0, 10);

    } // real lux file

}


bool LuxWidget::detectSB()
{
    static bool sbDetectedAlready = false;
    static bool isSB = false;

    if(sbDetectedAlready) {
        return isSB;
    }


    sbDetectedAlready = true;
    QFile sb("/scratchbox");

    if(sb.exists()) {
        qDebug() << "/scratchbox exists, using fake values";
        isSB = true;
        return true;
    }
    else {
        qDebug() << "/scratchbox does not exist, using real values";
        isSB = false;
        return false;
    }


}

qreal LuxWidget::log2(qreal n)
{
    return log(n) / log(2);
}

// return value is integral part, and number of one thirds of a stop
// EV is according to ISO 100 speed.
QPair<int,int> LuxWidget::Lux2EV(int lux)
{

    qreal ev = log2((qreal)lux/2.5);

    int q = (int) ev;
    qreal rem_tmp = ev-q;
    int r = 0;
    if (rem_tmp >= 0.00 && rem_tmp < 0.16666666666) {
        r = 0;
    }
    else if (rem_tmp >= 0.16666666666 && rem_tmp < 0.5) {
        r = 1;
    }
    else if (rem_tmp >= 0.5 && rem_tmp < 0.8333333333333) {
        r = 2;
    }
    else if (rem_tmp >= 0.833333333333 && rem_tmp < 1) {
        r = 0;
        q++;  // round up to next integral number
    }


    return QPair<int,int>(q,r);

}
