#include <QtGui/QApplication>
#include <QtGui/QFont>
#include <QtGui/QGridLayout>
#include <QtGui/QHBoxLayout>
#include <QtGui/QLCDNumber>
#include <QtGui/QLabel>
#include <QtGui/QPushButton>
#include <QtGui/QShortcut>
#include <QtGui/QVBoxLayout>
#include <QtGui/QLineEdit>
#include <QtGui/QWidget>
#include <QtGui/QScrollArea>
#include <QtGui/QMenu>
#include <QtGui/QPixmap>

#include "lookup.h"

#include <arpa/inet.h>
#include <arpa/nameser.h>
#include <validator/resolver.h>
#include <validator/validator.h>

#include "QDNSItemModel.h"

int
calculate_time_diff(const struct timeval *then, const struct timeval *now)
{
    struct timeval  tmp, diff;
    memcpy(&tmp, now, sizeof(struct timeval));
    tmp.tv_sec--;
    tmp.tv_usec += 1000000L;
    diff.tv_sec = tmp.tv_sec - then->tv_sec;
    diff.tv_usec = tmp.tv_usec - then->tv_usec;
    if (diff.tv_usec > 1000000L) {
        diff.tv_usec -= 1000000L;
        diff.tv_sec++;
    }
    // return in msec
    return diff.tv_usec/1000 + diff.tv_sec * 1000;
}

void
Lookup::setQueryType(int type)
{
    m_queryType = type;
    dolookup();
}

void
Lookup::setTypeText(const QString &label)
{
    m_queryButton->setText(label);
}

Lookup::Lookup(QWidget *parent)
    : QWidget(parent), found(false), m_queryType(ns_t_a)
{
    //labels = new QLabel[fields];

    //
    // Input widget bar
    //
    QHBoxLayout *hlayout = new QHBoxLayout();

    lookupline = new QLineEdit("www.dnssec-tools.org");
    hlayout->addWidget(lookupline);

    m_queryButton = new QPushButton("A", this);
    hlayout->addWidget(m_queryButton);

    // Icons
    m_validated = QPixmap(":/images/validated.png");
    m_trusted   = QPixmap(":/images/trusted.png");
    m_bad       = QPixmap(":/images/bad.png");
    m_unknown   = QPixmap(":/images/unknown.png");

    m_resultsIcon = new QLabel();
    m_resultsIcon->setPixmap(m_unknown);
    hlayout->addWidget(m_resultsIcon);

    //
    // create the QUERY TYPE menu
    //
    QMenu *querymenu = new QMenu(m_queryButton);
    m_queryButton->setMenu(querymenu);

    m_signalMapper = new QSignalMapper(this);

    QMap<int, QString> valuemap, dnssecmap, othermap;
    
    valuemap[1] = "A";
    valuemap[2] = "NS";
    valuemap[5] = "CNAME";
    valuemap[6] = "SOA";
    valuemap[12] = "PTR";
    valuemap[15] = "MX";
    valuemap[16] = "TXT";
    valuemap[28] = "AAAA";
    valuemap[33] = "SRV";
    valuemap[255] = "ANY";

    dnssecmap[43]    = "DS";
    dnssecmap[46]    = "RRSIG";
    dnssecmap[47]    = "NSEC";
    dnssecmap[48]    = "DNSKEY";
    dnssecmap[50]    = "NSEC3";
    dnssecmap[32769] = "DLV";

    othermap[3] = "MD";
    othermap[4] = "MF";
    othermap[7] = "MB";
    othermap[8] = "MG";
    othermap[9] = "MR";
    othermap[10] = "NULL";
    othermap[11] = "WKS";
    othermap[13] = "HINFO";
    othermap[14] = "MINFO";
    othermap[17] = "RP";
    othermap[18] = "AFSDB";
    othermap[19] = "X25";
    othermap[20] = "ISDN";
    othermap[21] = "RT";
    othermap[22] = "NSAP";
    othermap[23] = "NSAP_PTR";
    othermap[24] = "SIG";
    othermap[25] = "KEY";
    othermap[26] = "PX";
    othermap[27] = "GPOS";
    othermap[29] = "LOC";
    othermap[30] = "NXT";
    othermap[31] = "EID";
    othermap[32] = "NIMLOC";
    othermap[34] = "ATMA";
    othermap[35] = "NAPTR";
    othermap[36] = "KX";
    othermap[37] = "CERT";
    othermap[38] = "A6";
    othermap[39] = "DNAME";
    othermap[40] = "SINK";
    othermap[41] = "OPT";
    othermap[250] = "TSIG";
    othermap[251] = "IXFR";
    othermap[252] = "AXFR";
    othermap[253] = "MAILB";
    othermap[254] = "MAILA";


    QAction *action;

    QMenu *submenu = querymenu->addMenu("Common");
    for(QMap<int, QString>::iterator iter = valuemap.begin();
        iter != valuemap.end();
        iter++) {
        
        action = submenu->addAction(iter.value());
        connect(action, SIGNAL(triggered()), m_signalMapper, SLOT(map()));
        m_signalMapper->setMapping(action, iter.key());
        m_signalMapper->setMapping(action, iter.value());
    }

    submenu = querymenu->addMenu("DNSSEC");
    for(QMap<int, QString>::iterator iter = dnssecmap.begin();
        iter != dnssecmap.end();
        iter++) {
        
        action = submenu->addAction(iter.value());
        connect(action, SIGNAL(triggered()), m_signalMapper, SLOT(map()));
        m_signalMapper->setMapping(action, iter.key());
        m_signalMapper->setMapping(action, iter.value());
    }

    submenu = querymenu->addMenu("Others");
    for(QMap<int, QString>::iterator iter = othermap.begin();
        iter != othermap.end();
        iter++) {
        
        action = submenu->addAction(iter.value());
        connect(action, SIGNAL(triggered()), m_signalMapper, SLOT(map()));
        m_signalMapper->setMapping(action, iter.key());
        m_signalMapper->setMapping(action, iter.value());
    }

    connect(m_signalMapper, SIGNAL(mapped(int)), this, SLOT(setQueryType(int)));
    connect(m_signalMapper, SIGNAL(mapped(const QString &)),
            this, SLOT(setTypeText(const QString &)));

    gobutton = new QPushButton("Go");
    hlayout->addWidget(gobutton);
    connect(gobutton, SIGNAL(clicked()), this, SLOT(dolookup()));
    connect(lookupline, SIGNAL(textEdited(QString)), this, SLOT(entryTextChanged(QString)));
    connect(lookupline, SIGNAL(returnPressed()), this, SLOT(dolookup()));

    //
    // Create the vertical answer sheet
    //
    vlayout = new QVBoxLayout();
    vlayout->addLayout(hlayout);

    m_answerView = new QTreeView(this);
    m_answers = new QDNSItemModel(this);
    m_answerView->setModel(m_answers);

    vlayout->addWidget(m_answerView);

    setLayout(vlayout);
    resize(QSize(800,400));
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
}

QSize Lookup::sizeHint()
{
    return QSize(640,480);
}

Lookup::~Lookup()
{
}

void
Lookup::dolookup()
{
    val_status_t val_status;
    struct addrinfo *ai = NULL, *aitop = NULL;
    int ret;
    struct timeval start, stop;
    u_char buf[4096];
    char printbuf[4096];

    m_answers->clear();

    // perform the lookup
    gettimeofday(&start, NULL);
    ret = val_res_query(NULL, lookupline->text().toUtf8(), ns_c_in,
                        m_queryType, buf, sizeof(buf), &val_status);
    gettimeofday(&stop, NULL);

    // do something with the results
    if (ret <= 0) {
        m_answers->appendRow(new QStandardItem(p_val_status(val_status)));
        
        if (!val_istrusted(val_status)) {
            // untrusted error for host
        }
 	if (!val_istrusted(val_status)) {
            // untrusted for ip address
        }
    } else {

        ns_msg          handle;
        int             id, qdcount, ancount, nscount, arcount;
        ns_rr           rr;
        int             rrnum = 0;
        int             n;
        QString         text;

        if (ns_initparse(buf, ret, &handle) < 0) {
            // Error
            return;
        }

        id = ns_msg_id(handle);
        qdcount = ns_msg_count(handle, ns_s_qd);
        ancount = ns_msg_count(handle, ns_s_an);
        nscount = ns_msg_count(handle, ns_s_ns);
        arcount = ns_msg_count(handle, ns_s_ar);

//         do_section(&handle, ns_s_qd, RES_PRF_QUES, file);
//         do_section(&handle, ns_s_an, RES_PRF_ANS, file);
//         do_section(&handle, ns_s_ns, RES_PRF_AUTH, file);
//         do_section(&handle, ns_s_ar, RES_PRF_ADD, file);

        QMap<ns_sect, QStandardItem *> sections;
        sections[ns_s_qd] = new QStandardItem(QString("Question: %1").arg(qdcount));
        sections[ns_s_an] = new QStandardItem(QString("Answers: %1").arg(ancount));
        sections[ns_s_ns] = new QStandardItem(QString("Authority: %1").arg(nscount));
        sections[ns_s_ar] = new QStandardItem(QString("Additional: %1").arg(arcount));

        QStandardItem *results = new QStandardItem("Results");
        results->appendRow(sections[ns_s_qd]);
        results->appendRow(sections[ns_s_an]);
        results->appendRow(sections[ns_s_ns]);
        results->appendRow(sections[ns_s_ar]);
        m_answers->appendRow(results);

        for(QMap<ns_sect, QStandardItem *>::iterator iter = sections.begin();
            iter != sections.end(); iter++) {
            rrnum = 0;
            while(1) {
                if (ns_parserr(&handle, iter.key(), rrnum, &rr)) {
                    break;
                }
                n = ns_sprintrr(&handle, &rr, NULL, NULL,
                                printbuf, sizeof(printbuf));
                if (n < 0) {
                    // error
                    return;
                }
                iter.value()->appendRow(new QStandardItem(printbuf));
                rrnum++;
            }
        }

        QStandardItem *security = new QStandardItem("Security");
        m_answers->appendRow(security);
        
        if (val_isvalidated(val_status)) {
            m_securityStatus = 
                new QStandardItem("Validated");
            m_answers->setSecurityStatus(m_securityStatus,
                                         QDNSItemModel::validated);
            m_resultsIcon->setPixmap(m_validated);
#ifndef BROKENBACKGROUND
            m_answerView->setStyleSheet("QTreeView { background-color: #96ff96; }");
#endif
        } else if (val_istrusted(val_status)) {
            m_securityStatus = 
                new QStandardItem("Trusted");
            m_answers->setSecurityStatus(m_securityStatus,
                                         QDNSItemModel::trusted);
            m_resultsIcon->setPixmap(m_trusted);
#ifndef BROKENBACKGROUND
            m_answerView->setStyleSheet("QTreeView { background-color: #ffff96; }");
#endif
        } else {
            m_securityStatus = new QStandardItem("Bogus");
            m_answers->setSecurityStatus(m_securityStatus,
                                         QDNSItemModel::bad);
            m_resultsIcon->setPixmap(m_bad);
#ifndef BROKENBACKGROUND
            m_answerView->setStyleSheet("QTreeView { background-color: #ff9696; }");
#endif
        }

        security->appendRow(m_securityStatus);
        security->appendRow(new QStandardItem(p_val_status(val_status)));

        m_answerView->setExpanded(results->index(), true);
        m_answerView->setExpanded(security->index(), true);
        m_answerView->setExpanded(sections[ns_s_an]->index(), true);

    }

    m_answers->appendRow(new QStandardItem(QString("Time: %1 msec").arg(calculate_time_diff(&start, &stop))));


    m_answerView->setHeaderHidden(true);
    m_answerView->setRootIsDecorated(false);
    
    vlayout->invalidate();

    freeaddrinfo(aitop);
}

void Lookup::entryTextChanged(const QString &newtext) {
    Q_UNUSED(newtext);
    m_answers->clear();
    m_answers->setSecurityStatus(m_securityStatus,
                                 QDNSItemModel::unknown);
#ifndef BROKENBACKGROUND
    m_answerView->setStyleSheet("QTreeView { background-color: #ffffff; }");
#endif
    m_resultsIcon->setPixmap(m_unknown);
}
