#include "player.h"
#include <QDebug>
#include <stdlib.h>

QStringList Player::RanksKai = QStringList() << "Novice" << "Intuite" << "Doan" << "Acolyte"
                                             << "Initiate" << "Aspirant" << "Guardian" << "Warmarn"
                                             << "Savant" << "Master";

QStringList Player::RanksMagnakai = QStringList() << "Kai Master" << "Kai Master Senior" << "Kai Master Superior"
                                                  << "Primate" << "Tutelary" << "Principalin" << "Mentora"
                                                  << "Scion-kai" << "Archmaster" << "Kai Grand Master";

QStringList Player::RanksGrandMaster = QStringList() << "Kai Grand Master Senior" << "Kai Grand Master Superior"
                                                     << "Kai Grand Sentinel" << "Kai Grand Defender"
                                                     << "Kai Grand Guardian" << "Sun Knight" << "Sun Lord"
                                                     << "Sun Thane" << "Grand Thane" << "Grand Crown" << "Sun Prince"
                                                     << "Kai Supreme Master";

Player::Player(QObject *parent) :
    QObject(parent)
{
    TemporaryCombatSkillDelta = 0;
    BaseCombatSkill = 0;
    BaseEndurance = 0;
    Endurance = 0;
    GoldCrowns = 0;
    KaiDisciplines.clear();    
    m_Backpack = true;
    m_CanHunt = true;

    DeathsCount = 0;
}

Player::Player(const Player& player) :
        QObject(player.parent())
{
    TemporaryCombatSkillDelta = player.TemporaryCombatSkillDelta;
    BaseCombatSkill = player.BaseCombatSkill;
    BaseEndurance = player.BaseEndurance;
    Endurance = player.Endurance;
    GoldCrowns = player.GoldCrowns;
    m_Backpack = player.m_Backpack;
    m_CanHunt = player.m_CanHunt;

    DeathsCount = player.DeathsCount;
    Properties = QHash<QString, QString>(player.Properties);
}

QList<Weapon*> Player::Weapons(bool includeSpecial)
{
    QList<Weapon*> res;

    QList<Item*>::const_iterator iterator;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        if (i->GetIsWeapon() && ((!includeSpecial && !i->GetIsSpecial()) || includeSpecial))
            res.append((Weapon*)i);
    }
    return res;
}

QList<Item*> Player::BackpackItems()
{
    QList<Item*> res;
    QList<Item*>::const_iterator iterator;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        if (i->GetIsBackpackItem())
            res.append(i);
    }
    return res;
}

QList<Item*> Player::SpecialItems()
{
    QList<Item*> res;
    QList<Item*>::const_iterator iterator;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        if (i->GetIsSpecial())
            res.append(i);
    }
    return res;
}

QString Player::GetRankDescription(int serie, int kaiDisciplinesCount)
{
    switch (serie){
    case Book::Kai:
        return RanksKai.at(kaiDisciplinesCount - 1);
        break;
    case Book::Magnakai:
        return RanksMagnakai.at(kaiDisciplinesCount - 1);
        break;
    case Book::GrandMaster:
        return RanksGrandMaster.at(kaiDisciplinesCount - 1);
        break;
    case Book::NewOrder:
        return RanksGrandMaster.at(kaiDisciplinesCount - 4);
        break;
    }

    return "";
}

int Player::GetRankDisciplines(int serie, QString rank)
{
    int iRank = 0;
    switch (serie){
    case Book::Kai:
        iRank = RanksKai.indexOf(rank);
        break;
    case Book::Magnakai:
        iRank = RanksMagnakai.indexOf(rank);
        break;
    case Book::GrandMaster:
        iRank = RanksGrandMaster.indexOf(rank);
        break;
    case Book::NewOrder:
        iRank = RanksGrandMaster.indexOf(rank);
        break;
    }

    return iRank + 1;
}

bool Player::HasRank(QString rank)
{
    int iRank = GetRankDisciplines(GetSerie(), rank);
    return GetDisciplineCount() >= iRank;
}

int Player::MealsCount()
{
    int result = 0;
    QList<Item*>::const_iterator iterator;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        if (i->GetIsMeal())
            result += 1;
    }
    return result;
}

bool Player::IsDead()
{
    return Endurance <= 0;
}

int Player::GetCombatSkill()
{
    return GetCombatSkill(NULL);
}

int Player::GetCombatSkill(Enemy* enemy)
{
    return GetCombatSkill(enemy, NULL);
}

QStringList Player::GetCombatSkillDescription(Enemy* enemy)
{
    QStringList res;
    GetCombatSkill(enemy, &res);
    return res;
}

int Player::GetCombatSkill(Enemy* enemy, QStringList* description)
{
    if (description){
        description->append(QString("Base|%1").arg(BaseCombatSkill));
        if (TemporaryCombatSkillDelta > 0)
            description->append(QString("Temporary bonus|+%1").arg(TemporaryCombatSkillDelta));
    }

    int result = BaseCombatSkill + TemporaryCombatSkillDelta;

    QList<Item*>::const_iterator iterator;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        if (i->GetIsEquipped()){
            result += i->GetCombatskillDelta();
            if (description && i->GetCombatskillDelta())
                description->append(QString("%1|+%2").arg(i->Name)
                                                          .arg(i->GetCombatskillDelta()));

        }
    }

    foreach (KaiDiscipline* d, KaiDisciplines){
        if (d->GetIsActive()){
            // Count CombatSkillDelta only versus enemies:
            if (enemy != NULL && !enemy->DisciplineImmunities.contains(d->Name)){
                result += d->CombatSkillDelta;
                if (description && d->CombatSkillDelta > 0)
                    description->append(QString("%1|+%2").arg(d->Name).arg(d->CombatSkillDelta));
            }

            // Weaponskill
            Weaponskill* ws = qobject_cast<Weaponskill*>(d);
            if (ws){
                KaiDiscipline* td = GetDiscipline("Weaponmastery");
                Weaponmastery* wm = td != NULL ? qobject_cast<Weaponmastery*>(td) : NULL;
                bool done = false;
                foreach(Item* i, Items){
                    if (i->GetIsWeapon()){
                        Weapon* w = (Weapon*)i;
                        if (w->GetIsEquipped() && w->Classes().contains(ws->Weapon) &&
                            (!wm || !wm->Weapons.contains(ws->Weapon))){
                            result += 2;
                            done = true;
                            if (description)
                                description->append(QString("%1|+2").arg(ws->Description()));
                            break;
                        }
                    }
                    if (done)
                        break;
                }
            }

            // Weaponmastery
            Weaponmastery* wm = qobject_cast<Weaponmastery*>(d);
            if (wm){
                bool done = false;
                foreach(Item* i, Items){
                    if (i->GetIsWeapon()){
                        Weapon* w = (Weapon*)i;
                        if (w->GetIsEquipped()){
                            foreach (int wc, w->Classes()){
                                if (wm->Weapons.contains(wc)){
                                    int delta = 3;
                                    if (GetDisciplineCount() >= 8)
                                        delta = 4;
                                    result += delta;
                                    done = true;
                                    if (description)
                                        description->append(QString("%1 in %2|+%3").arg(wm->Description())
                                                                                    .arg(Weapon::ClassDescription(wc))
                                                                                    .arg(delta));
                                    break;
                                }
                            }

                        }
                    }
                    if (done)
                        break;
                }
            }
        }
    }

    foreach (LoreCircle* lc, LoreCircles::Circles){
        if (lc->Check(this)){
            result += lc->CombatSkillBonus;
            if (description && lc->CombatSkillBonus > 0)
                description->append(QString("Lore-circle of %1|+%2").arg(lc->Name()).arg(lc->CombatSkillBonus));
        }
    }

    if (GetEquippedWeapon() == NULL){
        int delta = 4;
        if (GetSerie() == Book::Magnakai){
            if (GetDisciplineCount() >= 5)
                delta = 2;
            else if (GetDisciplineCount() >= 8)
                delta = 1;
        }

        result -= delta;
        if (description)
            description->append(QString("No weapon equipped|-%1").arg(delta));
    }

    return result;
}

int Player::GetBaseEndurance(QStringList* description)
{
    if (description)
        description->append(QString("Base|%1").arg(BaseEndurance));

    int result = BaseEndurance;
    QList<Item*>::const_iterator iterator;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        if (i->GetIsEquipped()){
            result += i->GetEnduranceDelta();
            if (description && i->GetEnduranceDelta())
                description->append(QString("%1|+%2").arg(i->Name)
                                                          .arg(i->GetEnduranceDelta()));

        }
    }

    foreach (LoreCircle* lc, LoreCircles::Circles){
        if (lc->Check(this)){
            result += lc->EnduranceBonus;
            if (description && lc->EnduranceBonus > 0)
                description->append(QString("Lore-circle of %1|+%2").arg(lc->Name()).arg(lc->EnduranceBonus));
        }
    }
    return result;
}

QStringList Player::GetBaseEnduranceDescription()
{
    QStringList res;
    GetBaseEndurance(&res);
    return res;
}

int Player::GetBaseEndurance()
{
    return GetBaseEndurance(NULL);
}

void Player::RemoveDiscipline(QString name)
{
    KaiDiscipline* remove = NULL;
    foreach (KaiDiscipline* d, KaiDisciplines){
        if (QString::compare(d->Name, name, Qt::CaseInsensitive) == 0){
            remove = d;
        }
    }

    if (remove)
        KaiDisciplines.removeOne(remove);
}

void Player::RemoveDisciplines()
{
    // Reset the discipline's serie to avoid counting them with GetDisciplineCount
    foreach (KaiDiscipline* d, KaiDisciplines)
        d->Serie = 0;
}

int Player::GetDisciplineCount()
{
    int res = 0;
    foreach (KaiDiscipline* d, KaiDisciplines){
        if (d->Serie != 0)
            res++;
    }
    return res;
}

KaiDiscipline* Player::GetDiscipline(QString name)
{
    foreach (KaiDiscipline* d, KaiDisciplines){
        if (QString::compare(d->Name, name, Qt::CaseInsensitive) == 0)
            return d;
    }
    return NULL;
}

bool Player::HasDiscipline(QString name)
{
    foreach (KaiDiscipline* d, KaiDisciplines){
        if (QString::compare(d->Name, name, Qt::CaseInsensitive) == 0)
            return true;
    }
    return false;
}

bool Player::IsDisciplineActive(QString name)
{
    KaiDiscipline* d = GetDiscipline(name);
    if (d != NULL)
        return d->GetIsActive();
    return false;
}

bool Player::HasWeaponskillIn(QString className)
{
    foreach (KaiDiscipline* d, KaiDisciplines){
        if (QString::compare(d->Name, "Weaponskill", Qt::CaseInsensitive) == 0){
            Weaponskill* ws = qobject_cast<Weaponskill*>(d);
            if (ws)
                return ws->Weapon == Weapon::GetClass(className);

        }
    }
    return false;
}

bool Player::HasWeaponskillBonus()
{
    foreach (KaiDiscipline* d, KaiDisciplines){
        if (QString::compare(d->Name, "Weaponskill", Qt::CaseInsensitive) == 0){
            Weaponskill* ws = qobject_cast<Weaponskill*>(d);
            foreach(Item* i, Items){
                if (i->GetIsWeapon()){
                    Weapon* w = (Weapon*)i;
                    if (w->GetIsEquipped() && w->Classes().contains(ws->Weapon)){
                        return true;
                    }
                }
            }
            break;
        }
    }
    return false;
}

bool Player::HasLoreCircle(QString name)
{
    foreach (LoreCircle* lc, LoreCircles::Circles){
        if (QString::compare(lc->Name(), name, Qt::CaseInsensitive) == 0)
            return lc->Check(this);
    }

    return false;
}

bool Player::HasWeaponmasteryIn(QString className)
{
    foreach (KaiDiscipline* d, KaiDisciplines){
        if (QString::compare(d->Name, "Weaponmastery", Qt::CaseInsensitive) == 0){
            Weaponmastery* ws = qobject_cast<Weaponmastery*>(d);
            if (ws)
                return ws->Weapons.contains(Weapon::GetClass(className));

        }
    }
    return false;
}

int Player::GetEndurance()
{
    return Endurance;
}

void Player::SetEndurance(int endurance)
{
    if (endurance < 0)
        endurance = 0;

    if (endurance <= GetBaseEndurance())
        Endurance = endurance;
    else
        Endurance = GetBaseEndurance();
}

void Player::ForceEndurance(int endurance)
{
    Endurance = endurance;
}

void Player::CheckEndurance()
{
    if (Endurance > GetBaseEndurance())
        Endurance = GetBaseEndurance();
    else if (Endurance < 0)
        Endurance = 0;
}

int Player::GetBaseCombatSkill()
{
    return BaseCombatSkill;
}

void Player::SetBaseCombatSkill(int baseCombatSkill)
{
    BaseCombatSkill = baseCombatSkill;
}

void Player::SetTempCombatSkillDelta(int tempCombastSkill)
{
    TemporaryCombatSkillDelta = tempCombastSkill;
}

int Player::GetGoldCrowns()
{
    return GoldCrowns;
}

void Player::SetGoldCrowns(int crowns)
{
    if (crowns >= 0){
        if (crowns > 50)
            crowns = 50;
        GoldCrowns = crowns;
    }else
        GoldCrowns = 0;
}

int Player::GetMaxArrows()
{
    int res = 0;
    QList<Item*>::const_iterator iterator;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        if (i->GetIsSpecial() && i->GetIsQuiver())
            res += 6;
    }
    return res;
}

int Player::GetArrows()
{
    int res = 0;
    QList<Item*>::const_iterator iterator;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        if (i->GetIsSpecial() && i->GetIsQuiver())
            res += i->GetArrows();
    }
    return res;
}

void Player::SetArrows(int value)
{
    int quivers = 0;
    int arrows = 0;
    QList<Item*>::const_iterator iterator;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        if (i->GetIsSpecial() && i->GetIsQuiver()){
            quivers++;
            arrows += i->GetArrows();
        }
    }

    if (value == arrows)
        return;

    if (value > quivers * 6)
        value = quivers * 6;
    else if (value < 0)
        value = 0;

    int delta = value - arrows;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        if (i->GetIsSpecial() && i->GetIsQuiver()){
            int absDelta = abs(delta);
            if (delta < 0){
                if (absDelta < i->GetArrows()){
                    i->SetArrows(i->GetArrows() + delta);
                    delta = 0;
                }else{
                    delta += i->GetArrows();
                    i->SetArrows(0);
                }
            }else{
                if (i->GetArrows() + absDelta <= 6){
                    i->SetArrows(i->GetArrows() + delta);
                    delta = 0;
                }else{
                    delta -= (6 - i->GetArrows());
                    i->SetArrows(6);
                }
            }
        }
    }
}

bool Player::HasItem(QString name)
{
    foreach(Item* i, Items){
        if (QString::compare(i->Name, name, Qt::CaseInsensitive) == 0)
            return true;
    }
    return false;
}

bool Player::HasSpecialItem(QString name)
{
    foreach(Item* i, Items){
        if (QString::compare(i->Name, name, Qt::CaseInsensitive) == 0 && i->GetIsSpecial())
            return true;
    }
    return false;
}

Item* Player::AddItem(QString name, bool special)
{
    if (!name.isEmpty() && (HasBackpack() || special)){
        Item* i = new Item(this);
        i->Name = name;
        i->SetIsSpecial(special);
        Items.append(i);
        return i;
    }
    return NULL;
}

Item* Player::AddMeal(QString name)
{
    if (!name.isEmpty() && HasBackpack()){
        Item* i = new Item(this);
        i->Name = name;
        i->SetIsMeal(true);
        Items.append(i);
        return i;
    }
    return NULL;
}

Item* Player::AddHealingItem(QString name, int enduranceRestore)
{
    if (!name.isEmpty() && enduranceRestore > 0 && HasBackpack()){
        Item* i = new Item(this);
        i->Name = name;
        i->SetEnduranceRestore(enduranceRestore);
        Items.append(i);
        return i;
    }
    return NULL;
}

bool Player::RemoveItem(QString name)
{
    if (!name.isEmpty()){
        Item* found = NULL;
        foreach(Item* i, Items){
            if (QString::compare(i->Name, name) == 0){
                found = i;
                break;
            }
        }
        if (found){
            Items.removeOne(found);
            return true;
        }
    }
    return false;
}

bool Player::RemoveItemByUUID(QString UUID)
{
    Item* i = GetItemByUUID(UUID);
    if (i){
        Items.removeOne(i);
        return true;
    }
    return false;
}

QStringList Player::GetHiddenItems()
{
    QStringList res;
    foreach (Item* i, HiddenItems)
        res.append(i->Name);
    return res;
}

bool Player::HideItem(QString name)
{
    if (!name.isEmpty()){
        Item* found = NULL;
        foreach(Item* i, Items){
            if (QString::compare(i->Name, name) == 0){
                found = i;
                break;
            }
        }
        if (found){
            found->Unequip();
            Items.removeOne(found);
            HiddenItems.append(found);
            return true;
        }
    }
    return false;
}

bool Player::RestoreHiddenItem(QString name)
{
    if (!name.isEmpty()){
        Item* found = NULL;
        foreach(Item* i, HiddenItems){
            if (QString::compare(i->Name, name) == 0){
                found = i;
                break;
            }
        }
        if (found){
            HiddenItems.removeOne(found);
            Items.append(found);
            return true;
        }
    }
    return false;
}

Item* Player::GetItem(QString name)
{
    if (!name.isEmpty()){
        foreach(Item* i, Items){
            if (QString::compare(i->Name, name) == 0)
                return i;
        }
    }
    return NULL;
}

Item* Player::GetItemByUUID(QString UUID)
{
    if (!UUID.isEmpty()){
        foreach(Item* i, Items){
            if (QString::compare(i->GetUUID(), UUID) == 0)
                return i;
        }
    }
    return NULL;
}

int Player::SpecialItemsCount()
{
    int res = 0;
    foreach(Item* i, Items){
        if (i->GetIsSpecial())
            res++;
    }
    return res;
}

int Player::ItemsCount()
{
    int res = 0;
    foreach(Item* i, Items){
        if (i->GetIsBackpackItem())
            res++;
    }
    return res;
}

void Player::SetHasBackpack(bool value)
{
    m_Backpack = value;
    if (!m_Backpack){
        QList<Item*>::iterator iterator = Items.begin();
        while (iterator != Items.end()) {
            Item* i = *iterator;
            if (i->GetIsBackpackItem())
                iterator = Items.erase(iterator);
            else
                iterator++;
        }
    }
}

bool Player::IsBackpackFull()
{
    return GetBackpackFreeSpace() <= 0;
}

int Player::GetBackpackFreeSpace()
{
    if (!HasBackpack())
        return 0;

    int size = 0;
    foreach(Item* i, Items){
        if (i->GetIsBackpackItem())
            size += i->GetSize();
    }
    return 8 - size;
}

int Player::WeaponsCount()
{
    return Weapons().count();
}

QStringList Player::GetWeapons()
{
    QStringList res;
    foreach (Weapon* w, Weapons()){
        res.append(w->Name);
    }
    return res;
}

QStringList Player::GetAllWeapons()
{
    QStringList res;
    foreach (Weapon* w, Weapons(true)){
        res.append(w->Name);
    }
    return res;
}

Weapon* Player::GetEquippedWeapon()
{
    QList<Item*>::const_iterator iterator;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        if (i->GetIsWeapon() && i->GetIsEquipped())
            return (Weapon*)i;
    }
    return NULL;
}

QList<Armor*> Player::GetEquippedArmors()
{
    QList<Armor*> res;
    QList<Item*>::const_iterator iterator;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        if (i->GetIsArmor() && i->GetIsEquipped())
            res.append((Armor*)i);
    }
    return res;
}

Armor* Player::GetEquippedShield()
{
    QList<Item*>::const_iterator iterator;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        if (i->GetIsArmor() && i->GetArmorClass() == Armor::Shield && i->GetIsEquipped())
            return (Armor*)i;
    }
    return NULL;
}

QStringList Player::GetAllItems()
{
    QStringList res;
    QList<Item*>::const_iterator iterator;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        res.append(i->Name);
    }
    return res;
}

QStringList Player::GetSpecialItems()
{
    QStringList res;
    QList<Item*>::const_iterator iterator;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        if (i->GetIsSpecial())
            res.append(i->Name);
    }
    return res;
}

QStringList Player::GetItems()
{
    QStringList res;
    QList<Item*>::const_iterator iterator;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        if (i->GetIsBackpackItem())
            res.append(i->Name);
    }
    return res;
}

QStringList Player::GetMeals()
{
    QStringList res;
    QList<Item*>::const_iterator iterator;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        if (i->GetIsMeal())
            res.append(i->Name);
    }
    return res;
}

bool Player::HasWeapon(QString name)
{
    foreach(Weapon* w, Weapons()){
        if (QString::compare(w->Name, name, Qt::CaseInsensitive) == 0)
            return true;
    }
    return false;
}

bool Player::HasWeaponOfClass(QString wClass)
{
    int wc = Weapon::GetClass(wClass);
    if (wc != Weapon::None){
        foreach(Weapon* w, Weapons()){
            if (w->Classes().contains(wc))
                return true;
        }
    }
    return false;
}

bool Player::CanUseBow()
{
    return GetHands() == 2 && HasWeaponOfClass("Bow") && GetArrows() > 0;
}

Weapon* Player::GetWeapon(QString name)
{
    foreach(Weapon* w, Weapons(true)){
        if (QString::compare(w->Name, name, Qt::CaseInsensitive) == 0)
            return w;
    }
    return NULL;
}

bool Player::IsEquipped(QString name)
{
    foreach(Item* w, Items){
        if (QString::compare(w->Name, name, Qt::CaseInsensitive) == 0 && w->GetIsEquipped())
            return true;
    }

    return false;
}

void Player::Equip(QString name)
{
    foreach(Item* w, Items){
        if ((w->GetIsWeapon() || w->GetIsArmor()) && QString::compare(w->Name, name, Qt::CaseInsensitive) == 0)
            w->Equip();
    }
}

bool Player::GetCanUseShield()
{
    if (HasProperty("CanUseShield"))
        return GetBoolProperty("CanUseShield");
    return true;
}

void Player::SetCanUseShield(bool value)
{
    SetBoolProperty("CanUseShield", value);
    QList<Armor*> equipped = GetEquippedArmors();
    foreach (Armor* a, equipped){
        if (a->GetArmorClass() == Armor::Shield)
            a->Unequip();
    }
}

int Player::GetHands()
{
    if (HasProperty("Hands"))
        return GetProperty("Hands").toInt();
    return 2;
}

void Player::SetHands(int value)
{
    SetProperty("Hands", QString::number(value));
    if (GetFreeHands() < 0){
        QList<Armor*> equipped = GetEquippedArmors();
        foreach (Armor* a, equipped){
            if (a->GetArmorClass() == Armor::Shield)
                a->Unequip();
        }

        if (GetFreeHands() < 0){
            Weapon* w = GetEquippedWeapon();
            if (w != NULL)
                w->Unequip();
        }
    }
}

int Player::GetFreeHands()
{
    int res = GetHands();
    foreach(Item* i, Items){
        if (i->GetIsEquipped()){
            Weapon* w = qobject_cast<Weapon*>(i);
            if (w)
                res -= w->GetHands();
            else if (i->GetIsArmor() && i->GetArmorClass() == Armor::Shield)
                res -= 1;
            else
                res -= i->GetHands();
        }
    }
    return res;
}

Weapon* Player::AddWeapon(QString name, QString classname)
{
    if (!name.isEmpty() && !classname.isEmpty() && WeaponsCount() < 2){
        Weapon* w = new Weapon(this);
        w->Name = name;
        w->AddClass(Weapon::GetClass(classname));
        Items.append(w);
        return w;
    }
    return NULL;
}

Weapon* Player::AddSpecialWeapon(QString name, QString classname)
{
    if (!name.isEmpty() && !classname.isEmpty()){
        Weapon* w = new Weapon(this);
        w->Name = name;
        w->AddClass(Weapon::GetClass(classname));
        w->SetIsSpecial(true);
        Items.append(w);
        return w;
    }
    return NULL;
}

void Player::RemoveWeapon(QString name)
{
    if (!name.isEmpty()){
        Weapon* found = NULL;
        foreach(Weapon* w, Weapons(true)){
            if (QString::compare(w->Name, name) == 0){
                found = w;
                break;
            }
        }
        if (found)
            Items.removeOne(found);
    }
}

QString Player::GetProperty(QString name)
{
    if (Properties.contains(name))
        return Properties[name];
    return "";
}

int Player::GetIntProperty(QString name)
{
    QString p = GetProperty(name);
    if (!p.isEmpty())
        return p.toInt();
    return 0;
}

void Player::SetProperty(QString name, QString value)
{
    Properties[name] = value;
}

void Player::RemoveProperty(QString name)
{
    if (Properties.contains(name))
        Properties.remove(name);
}

void Player::ClearProperties()
{
    Properties.clear();
}

bool Player::HasQuiver()
{
    QList<Item*>::const_iterator iterator;
    for (iterator = Items.constBegin(); iterator != Items.constEnd(); ++iterator){
        Item* i = *iterator;
        if (i->GetIsSpecial() && i->GetIsQuiver())
            return true;
    }
    return false;
}

int Player::GetThrowBonus()
{
    int bonus = 0;
    // When using a Bow or thrown Weapon and instructed to pick a number from the
    // Random Number Table, add 2 to the number picked if you are a Mentora
    // with the Magnakai Discipline of Weaponmastery.
    if (HasDiscipline("Weaponmastery") && GetDisciplineCount() >= 7)
        bonus += 2;

    return bonus;
}

int Player::GetBowBonus(bool includeDisciplines)
{
    int bonus = 0;
    if (includeDisciplines){
        if (HasWeaponmasteryIn("Bow"))
            bonus += 3;
        // When using a Bow or thrown Weapon and instructed to pick a number from the
        // Random Number Table, add 2 to the number picked if you are a Mentora
        // with the Magnakai Discipline of Weaponmastery.
        if (HasDiscipline("Weaponmastery") && GetDisciplineCount() >= 7)
            bonus += 2;
    }

    foreach (Weapon* w, Weapons()){
        if (w->Classes().contains(Weapon::Bow))
            bonus += w->GetRandomBonus();
    }

    return bonus;
}

int Player::GetSerie()
{
    int max = Book::Unknown;
    foreach (KaiDiscipline* d, KaiDisciplines){
        if (d->Serie > max)
            max = d->Serie;
    }
    return max;
}

void Player::AppendNotesLine(QString value)
{
    QString notes = GetNotes();
    if (notes.length())
        notes.append("\n");
    notes.append(value);
    SetNotes(value);
}

bool Player::CanEquipArmor(Armor* armor, QString* error)
{
    if (armor->GetIsEquipped()){
        return true;
    }else{
        if (armor->GetClass() == Armor::Shield && !GetCanUseShield()){
            if (error)
                *error = "You cannot equip a shield now.";
            return false;
        }

        // Che armors of the same class:
        QList<Armor*> equipped = GetEquippedArmors();
        foreach (Armor* a, equipped){
            if (a->GetArmorClass() == armor->GetClass()){
                if (error)
                    *error = "You cannot equip this armor now.";
                return false;
            }
        }

        // Check free hands (only for shields)
        if (armor->GetClass() == Armor::Shield){
            if (GetFreeHands() >= 1){
                return true;
            }else{
                if (error)
                    *error = "You cannot equip a shield now.";
                return false;
            }
        }else
            return true;
    }
    return true;
}

bool Player::CanTakeItem(Item* item, QString* error,
                         int deltaWeapons, int deltaItems, int deltaSpecial, int deltaArrows)
{
    if (item->GetIsWeapon() && !item->GetIsSpecial()){
        //Take weapon:
        if (Weapons().count() + deltaWeapons + 1 > 2){
            if (error)
                *error = "You cannot carry more than two weapons.";
            return false;
        }else
            return true;
    }

    if (item->GetIsSpecial()){
        //Take special item:
        if (SpecialItemsCount() + item->GetQuantity() + deltaSpecial > 12){
            if (error)
                *error = "You cannot carry more special items.";
            return false;
        }else
            return true;
    }else{
        //Take arrows:
        if (item->GetIsArrow()){
            if (!HasQuiver()){
                if (error)
                    *error = "You don't have a quiver and cannot carry arrows.";
                return false;
            }
            if (GetArrows() + deltaArrows + item->GetQuantity() > GetMaxArrows()){
                if (error)
                    *error = "You cannot carry more arrows.";
                return false;
            }
            return true;
        }else{
            //Take normal item:
            if (!HasBackpack() && item->GetIsBackpack()){
                return true;
            }else if (!HasBackpack()){
                if (error)
                    *error = "You don't have the backpack and cannot carry any item.";
                return false;
            }else if (HasBackpack() && item->GetIsBackpack()){
                if (error)
                    *error = "You have the backpack and cannot carry another one.";
                return false;
            }else if (GetBackpackFreeSpace() - deltaItems - item->GetTotalSize() < 0){
                if (error)
                    *error = "You cannot carry more backpack items.";
                return false;
            }else
                return true;
        }
    }
    return false;
}

void Player::Dump()
{
    qDebug() << "Base properties:";
    qDebug() << "  Endurance        :" << GetEndurance();
    qDebug() << "  Base Combat Skill:" << GetBaseCombatSkill();
    qDebug() << "  Combat Skill     :" << GetCombatSkill();
    qDebug() << "Properties:";
    foreach(QString key, Properties.keys()){
        qDebug() << "  " << key << ":" << Properties[key];
    }
    qDebug() << "Items:";
    foreach(Item* i, Items){
        qDebug() << "  " << i->GetUUID() << i->GetName() << "Special:" << i->GetIsSpecial()
                 << "Size:" << i->GetSize();
    }
}
