#include "bookwindow.h"
#include <QMessageBox>
#include <QWebFrame>
#include "ui_bookwindow.h"
#include "fightwindow.h"
#include "endwindow.h"
#include "statuswindow.h"
#include "chooseitemwindow.h"
#include "ruleswindow.h"
#include "aboutwindow.h"
#include "mapwindow.h"
#include "bonusdisciplinewindow.h"
#include "pickitemswindow.h"
#include "selectgoldcrownswindow.h"
#include "shopwindow.h"
#include "minigames/cartwheelwindow.h"
#include "minigames/portholeswindow.h"
#include "minigames/betwindow.h"
#include "bookinfowindow.h"
#include "inputnumberwindow.h"
#include "core/books.h"
#include "commonstatic.h"
#include <QTimer>
#ifdef Q_WS_MAEMO_5
#include "qwebviewselectionsuppressor.h"
#include <QAbstractKineticScroller>
#endif

////////////////////////////////////////////////////////////////////////////////
// METATYPES
////////////////////////////////////////////////////////////////////////////////
Q_DECLARE_METATYPE(Player)
QDataStream &operator<<(QDataStream &out, const Player &player)
{
    out << player.BaseCombatSkill << player.BaseEndurance
        << player.Endurance << player.GoldCrowns << player.HasBackpack() << player.DeathsCount << player.TemporaryCombatSkillDelta
        << player.Properties << player.CanHunt();
    return out;
}

QDataStream &operator>>(QDataStream &in, Player &player)
{
    bool backpack = true;
    bool canHunt  = true;
    in >> player.BaseCombatSkill >> player.BaseEndurance
       >> player.Endurance >> player.GoldCrowns >> backpack >> player.DeathsCount >> player.TemporaryCombatSkillDelta
       >> player.Properties >> canHunt;

    player.SetHasBackpack(backpack);
    player.SetCanHunt(canHunt);

    return in;
}

Q_DECLARE_METATYPE(Choice*)
QScriptValue ChoiceToScriptValue(QScriptEngine *engine, Choice* const &in)
{ return engine->newQObject(in); }

void ChoiceFromScriptValue(const QScriptValue &object, Choice* &out)
{ out = qobject_cast<Choice*>(object.toQObject()); }


Q_DECLARE_METATYPE(Item)
Q_DECLARE_METATYPE(Item*)
QScriptValue ItemToScriptValue(QScriptEngine *engine, Item* const &in)
{ return engine->newQObject(in); }

void ItemFromScriptValue(const QScriptValue &object, Item* &out)
{ out = qobject_cast<Item*>(object.toQObject()); }

QDataStream &operator<<(QDataStream &out, const Item &item)
{
    out << item.Name << item.Description << item.Properties;
    return out;
}

QDataStream &operator>>(QDataStream &in, Item &item)
{
    in >> item.Name >> item.Description >> item.Properties;
    return in;
}

Q_DECLARE_METATYPE(Weapon)
Q_DECLARE_METATYPE(Weapon*)
QScriptValue WeaponToScriptValue(QScriptEngine *engine, Weapon* const &in)
{ return engine->newQObject(in); }

void WeaponFromScriptValue(const QScriptValue &object, Weapon* &out)
{ out = qobject_cast<Weapon*>(object.toQObject()); }

QDataStream &operator<<(QDataStream &out, const Weapon &weapon)
{
    out << weapon.Name << weapon.Description
        << weapon.Properties;
    return out;
}

QDataStream &operator>>(QDataStream &in, Weapon &weapon)
{
    in >> weapon.Name >> weapon.Description
       >> weapon.Properties;
    return in;
}

Q_DECLARE_METATYPE(Armor)
Q_DECLARE_METATYPE(Armor*)
Q_DECLARE_METATYPE(QList<Armor*>)
QScriptValue ArmorToScriptValue(QScriptEngine *engine, Armor* const &in)
{ return engine->newQObject(in); }

void ArmorFromScriptValue(const QScriptValue &object, Armor* &out)
{ out = qobject_cast<Armor*>(object.toQObject()); }

QDataStream &operator<<(QDataStream &out, const Armor &armor)
{
    out << armor.Name << armor.Description
        << armor.Properties;
    return out;
}

QDataStream &operator>>(QDataStream &in, Armor &armor)
{
    in >> armor.Name >> armor.Description
       >> armor.Properties;
    return in;
}

Q_DECLARE_METATYPE(KaiDiscipline)
Q_DECLARE_METATYPE(KaiDiscipline*)
QScriptValue KaiDisciplineToScriptValue(QScriptEngine *engine, KaiDiscipline* const &in)
{ return engine->newQObject(in); }

void KaiDisciplineFromScriptValue(const QScriptValue &object, KaiDiscipline* &out)
{ out = qobject_cast<KaiDiscipline*>(object.toQObject()); }


QDataStream &operator<<(QDataStream &out, const KaiDiscipline &d)
{
    out << d.Name << d.CombatSkillDelta << d.Serie << d.HelpText << d.GetIsActive();
    return out;
}

QDataStream &operator>>(QDataStream &in, KaiDiscipline &d)
{
    bool active = false;
    in >> d.Name >> d.CombatSkillDelta >> d.Serie >> d.HelpText >> active;
    d.SetIsActive(active);
    return in;
}

Q_DECLARE_METATYPE(Weaponskill)
QDataStream &operator<<(QDataStream &out, const Weaponskill &d)
{
    out << d.Name << d.CombatSkillDelta
        << d.Weapon << d.Serie << d.HelpText;
    return out;
}

QDataStream &operator>>(QDataStream &in, Weaponskill &d)
{
    in >> d.Name >> d.CombatSkillDelta
       >> d.Weapon;
    return in;
}

////////////////////////////////////////////////////////////////////////////////
// END METATYPES
////////////////////////////////////////////////////////////////////////////////



BookWindow::BookWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::BookWindow)
{
    ui->setupUi(this);

#ifndef Q_WS_MAEMO_5
#if defined(Q_OS_WIN) || defined(Q_OS_WINCE) || defined(Q_OS_LINUX) || defined (Q_OS_MACX)
    CommonStatic::CenterWindow(this);
#endif 
#endif

    m_Settings = CommonStatic::GeneralSettings(this);
    QString font = m_Settings->value("font", "").toString();
    QStringList elem = font.split(";", QString::SkipEmptyParts);
    if (elem.count() == 3)
        m_Font = QFont(elem.at(0), elem.at(1).toInt(), elem.at(2) == "true");

    qRegisterMetaTypeStreamOperators<Player>("Player");
    qRegisterMetaType<Player>("Player");
    qRegisterMetaTypeStreamOperators<Item>("Item");
    qRegisterMetaType<Item>("Item");
    qRegisterMetaTypeStreamOperators<Weapon>("Weapon");
    qRegisterMetaType<Weapon>("Weapon");
    qRegisterMetaTypeStreamOperators<Armor>("Armor");
    qRegisterMetaType<Armor>("Armor");
    qRegisterMetaTypeStreamOperators<KaiDiscipline>("KaiDiscipline");
    qRegisterMetaType<KaiDiscipline>("KaiDiscipline");
    qRegisterMetaTypeStreamOperators<Weaponskill>("Weaponskill");
    qRegisterMetaType<Weaponskill>("Weaponskill");

    setAttribute(Qt::WA_DeleteOnClose);
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5AutoOrientation);
    setAttribute(Qt::WA_Maemo5StackedWindow);

    ui->scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    QWebViewSelectionSuppressor* suppressor = new QWebViewSelectionSuppressor(ui->webView);
    suppressor->enable();
    ui->webView->property("kineticScroller").value<QAbstractKineticScroller *>()->setEnabled(false);

    ui->lblCSkill->setText("C.Skill:");
    ui->lblEnd->setText("End.:");
    ui->line->setVisible(false);
#endif

#ifdef Q_OS_SYMBIAN
    ui->label_3->setVisible(false);
    ui->lblCrowns->setVisible(false);
    ui->lblCSkill->setText("C.Skill:");
    ui->lblEnd->setText("End.:");
    ui->line->setVisible(false);
#endif

#ifdef QT_NO_DEBUG
    ui->spinSection->setVisible(false);
    ui->pbnGo->setVisible(false);
#endif

    ui->webView->setContextMenuPolicy(Qt::NoContextMenu);

    connect(ui->webView, SIGNAL(loadFinished(bool)), this, SLOT(LoadFinishedSlot(bool)));
    connect(ui->pbnItems, SIGNAL(clicked()), SLOT(PickItemsSlot()));
    connect(ui->pbnRandom, SIGNAL(clicked()), SLOT(GetRandomSlot()));
    connect(ui->pbnFight, SIGNAL(clicked()), SLOT(FightSlot()));
    connect(ui->pbnGo, SIGNAL(clicked()), SLOT(GoSlot()));
    connect(ui->pbnEnd, SIGNAL(clicked()), SLOT(EndSlot()));

    m_ChoiceButtons.append(ui->pbnChoice_1);
    m_ChoiceButtons.append(ui->pbnChoice_2);
    m_ChoiceButtons.append(ui->pbnChoice_3);
    m_ChoiceButtons.append(ui->pbnChoice_4);
    m_ChoiceButtons.append(ui->pbnChoice_5);
    m_ChoiceButtons.append(ui->pbnChoice_6);
    m_ChoiceButtons.append(ui->pbnChoice_7);
    foreach (QPushButton* pbn, m_ChoiceButtons)
        connect(pbn, SIGNAL(clicked()), SLOT(ChoiceSlot()));

    ui->pbnEnd->setVisible(false);
    ui->pbnFight->setVisible(false);
    ui->pbnItems->setVisible(false);
    ui->pbnRandom->setVisible(false);
    ui->pbnShop->setVisible(false);
    SetVisibleChoiceButtons(0);

    connect(ui->pbnShop, SIGNAL(clicked()), SLOT(ShopSlot()));
    connect(ui->pbnInventory, SIGNAL(clicked()), SLOT(StatusSlot()));

    connect(ui->actionRules, SIGNAL(triggered()), this, SLOT(RulesSlot()));
    connect(ui->actionAbout, SIGNAL(triggered()), this, SLOT(AboutSlot()));
    connect(ui->actionMap, SIGNAL(triggered()), this, SLOT(MapSlot()));
    connect(ui->actionRestart_book, SIGNAL(triggered()), this, SLOT(BookRestartSlot()));
    connect(ui->actionBookInfo, SIGNAL(triggered()), this, SLOT(BookInfoSlot()));

    m_RandomPicker = new RandomNumberPicker(this, ui->lblDice);
    connect(m_RandomPicker, SIGNAL(Random(int)), SLOT(RandomSlot(int)));

    m_Section = NULL;
    m_Book = NULL;
    m_Player = NULL;

    m_Loaded = false;

    // Register script types:
    qScriptRegisterMetaType(&m_Engine, ChoiceToScriptValue, ChoiceFromScriptValue);
    qScriptRegisterMetaType(&m_Engine, ItemToScriptValue, ItemFromScriptValue);
    qScriptRegisterMetaType(&m_Engine, WeaponToScriptValue, WeaponFromScriptValue);
    qScriptRegisterMetaType(&m_Engine, ArmorToScriptValue, ArmorFromScriptValue);
    qScriptRegisterMetaType(&m_Engine, KaiDisciplineToScriptValue, KaiDisciplineFromScriptValue);

    qScriptRegisterSequenceMetaType< QList<Armor*> >(&m_Engine);
}

void BookWindow::Init()
{
    QSettings settings;
    if (QFile::exists(settings.fileName())){
        if (!Load()){
            QMessageBox msgBox(this);
            msgBox.setWindowTitle("Error");

            msgBox.setIcon(QMessageBox::Warning);
            msgBox.setText("Error loading last saved position.");
            msgBox.exec();
            return;
        }
    }

    QScriptValue objectValue = m_Engine.newQObject(m_Player);
    m_Engine.globalObject().setProperty("Player", objectValue);
    //ShowSection(m_Section != NULL ? m_Section->Number : 0);
}

BookWindow::~BookWindow()
{
    delete ui;
}

void BookWindow::ShowSection(int section)
{
    if (m_Book->Sections.contains(section)){
        bool newSection = false;
        if (m_Loaded || !m_Section || m_Section->Number != section)
            newSection = true;

        m_Loaded = false;
        m_Section = m_Book->Sections[section];

        ui->spinSection->setValue(m_Section->Number + 1);
        QScriptValue objectValue = m_Engine.newQObject(m_Section);
        m_Engine.globalObject().setProperty("Section", objectValue);

        // Autosave:
        if (newSection){
            bool saved = false;

            ui->lblMessage->clear();
            ui->lblMessage->setVisible(false);

            QString tempSaveFile = GetTempSaveFilename();
            if (m_Section->Number == 0)
                Save(GetBookSaveFilename(m_Book->Number), m_Player, m_Book->Number, m_Section->Number);

            if (m_Section->AutoSave && m_Section->GetEnemiesCount() == 0 && !m_Player->IsDead()){
                Save(tempSaveFile, m_Player, m_Book->Number, m_Section->Number);
                saved = true;
            }

            m_Fought = false;
            m_Ate = 0;

            QTimer::singleShot(0, this, SLOT(ScrollToTopSlot()));
            ExecuteSystemScript("EnterScript");

            if (!m_Section->EnterScript.isEmpty())
                ExecuteScript("EnterScript", m_Section->EnterScript);

            if (saved && m_Section->AutoSave && m_Section->GetEnemiesCount() == 0 && !m_Player->IsDead()){
                QFile::remove(QSettings().fileName());
                QFile::rename(tempSaveFile, QSettings().fileName());
            }
        }

        if (section > 0)
            setWindowTitle(QString("%1 - %2").arg(m_Book->Title).arg(section));
        else
            setWindowTitle(m_Book->Title);

        QUrl baseUrl = QUrl::fromLocalFile(QString("%1/Data/dummy.htm").arg(m_Book->BaseDirectory));
#ifdef Q_WS_MAEMO_5
        QString css = "main_n900.css";
#else
        QString css = "main.css";
#endif

        QString html = QString("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\" \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\"> \
                        <html xmlns=\"http://www.w3.org/1999/xhtml\"> \
                        <head> \
                        <link type=\"text/css\" href=\"%1\" rel=\"stylesheet\" /> \
                        ").arg(css);


        if (m_Settings->value("overridecss", false).toBool()){
            QString customStyle = QString("<style type=\"text/css\"> \
                                          html, body { \
                                           background-color: %1; \
                                           color: %2; \
                                           font-family: %3, Souvenir, Georgia, \"Times New Roman\", Times, serif; \
                                           font-size: %4px; \
                                          } \
                                          </style>").arg(m_Settings->value("background", "#ffffe6").toString())
                                                    .arg(m_Settings->value("color", "#003300").toString())
                                                    .arg(m_Font.family())
                                                    .arg(m_Font.pixelSize());
            html.append(customStyle);
        }
        html.append("</head><body>");
        if (section > 0)
            html.append(QString("<h3 style=\"text-align:center;\">- %1 -</h3>").arg(section));
        html.append(m_Section->Text);
        html.append("</body></html>");

        ui->webView->setFixedHeight(30);
        ui->webView->setHtml(html, baseUrl);

        ShowSectionChoices(section);
        if (!m_Section->ShowScript.isEmpty())
            ExecuteScript("ShowScript", m_Section->ShowScript);
    }else{
        QMessageBox msgBox(this);
        msgBox.setWindowTitle(QString("Invalid section"));

        msgBox.setIcon(QMessageBox::Warning);
        msgBox.setText(QString("Invalid section number:\n%1").arg(section));
        msgBox.exec();
    }
    UpdateToolbar();
}

void BookWindow::ShowSectionChoices(int section)
{
    if (m_Book->Sections.contains(section)){
        m_Section = m_Book->Sections[section];

        ui->pbnInventory->setEnabled(true);
        bool enemies = m_Section->GetEnemiesCount() > 0;
        if (enemies){
            ui->pbnFight->setVisible(true);
            ui->pbnFight->setEnabled(true);
            ui->pbnItems->setVisible(false);
            ui->pbnRandom->setVisible(false);
            SetVisibleChoiceButtons(0);            
            foreach (Choice* c, m_Section->Choices){
                if (c->ShowWithEnemies){
                    QPushButton* pbn = m_ChoiceButtons.at(m_Section->Choices.indexOf(c));
                    pbn->setVisible(true);
                    pbn->setEnabled(c->Enabled);
                }
            }

            ui->pbnShop->setVisible(false);

            if (m_Section->Choices.count() > 0)
                SetChoicesText();
        }else{
            if (m_Section->ShopItems.count())
                ui->pbnShop->setVisible(true);
            else
                ui->pbnShop->setVisible(false);

            ui->pbnFight->setVisible(false);
            if (m_Section->Items.count() > 0 || m_Section->GoldCrowns > 0)
                ui->pbnItems->setVisible(true);
            else
                ui->pbnItems->setVisible(false);

            if (m_Section->HasRandomChoices()){
                ui->pbnRandom->setVisible(true);
                SetVisibleChoiceButtons(0);
            }else{
                ui->pbnRandom->setVisible(false);
                SetVisibleChoiceButtons(0);
            }

            int choicesCount = m_Section->Choices.count();
            SetVisibleChoiceButtons(choicesCount);
            SetChoicesText();
        }

        if (m_Player->IsDead())
            EndMission();
        else
            ui->pbnEnd->setVisible(false);
    }
}

void BookWindow::SetChoicesText()
{
    if (!m_Section)
        return;
    int choicesCount = m_Section->Choices.count();
    for (int i=0; i<choicesCount; i++)
        SetChoiceText(m_ChoiceButtons.at(i), m_Section->Choices.at(i)->Text);
}

void BookWindow::SetChoiceText(QPushButton* button, QString text)
{
#ifdef Q_WS_MAEMO_5
    int maxWidth = width() - 48;
#else
    int maxWidth = width() - 32;
#endif
    QFontMetrics fm = button->fontMetrics();
    QStringList parts;
    QStringList words = text.split(" ");

    QString line;
    foreach (QString w, words){
        if (fm.width(line) + fm.width(" ") + fm.width(w) > maxWidth){
            parts.append(line.trimmed());
            line.clear();
        }
        line.append(" ");
        line.append(w);
    }
    if (!line.isEmpty())
        parts.append(line);

    button->setText(parts.join("\n"));
    button->setMinimumHeight(fm.height() * (parts.count() + 1));
}

void BookWindow::ExitSection()
{
    if (m_Ate < m_Section->MustEat){
        if (m_Section->CanHunt &&
            (m_Player->HasDiscipline("Hunting") || m_Player->HasDiscipline("Huntmastery")) &&
            m_Player->CanHunt()){
            //nothing to do
        }else if (m_Player->MealsCount() < m_Section->MustEat){
            CommonStatic::Message(this, "Info", QString("You don't have enough meals and lose %1 ENDURANCE points.").arg(m_Section->DidntEatEnduranceLoss));
            m_Player->Endurance -= m_Section->DidntEatEnduranceLoss;
        }else{
            if (CommonStatic::ConfirmMessage(this, "Eat", "Do you want to eat now?") == QMessageBox::Yes){
                for (int i=0; i<m_Section->MustEat; i++){
                    ChooseItemWindow* dlg = new ChooseItemWindow(this, m_Player, ChooseItemWindow::ChooseMeal);
                    dlg->exec();
                    Item* m = dlg->SelectedItem();
                    if (m == NULL){
                        CommonStatic::Message(this, "Info", QString("You lose %1 ENDURANCE points.").arg(m_Section->DidntEatEnduranceLoss));
                        m_Player->Endurance -= m_Section->DidntEatEnduranceLoss;
                        break;
                    }else{
                        m_Player->Items.removeOne(m);
                    }
                }
            }else{
                CommonStatic::Message(this, "Info", QString("You lose %1 ENDURANCE points.").arg(m_Section->DidntEatEnduranceLoss));
                m_Player->Endurance -= m_Section->DidntEatEnduranceLoss;
            }
        }
        m_Section->MustEat = 0;
    }

    if (!m_Section->ExitScript.isEmpty()){
        ExecuteScript("ExitScript", m_Section->ExitScript);
    }

    EndTempCombatSkillEffect();

    if (m_Player->HasDiscipline("Healing") && !m_Fought)
        m_Player->SetEndurance(m_Player->GetEndurance() + 1);

    if(m_Player->HasDiscipline("Curing") && !m_Fought)
        m_Player->SetEndurance(m_Player->GetEndurance() + 1);

}

void BookWindow::EndTempCombatSkillEffect()
{
    if (m_Player->TemporaryCombatSkillDelta > 0){
        m_Player->TemporaryCombatSkillDelta = 0;
        // Run the "end effect" script if present:
        QString name = m_Player->GetTemporaryCombatSkillItem();
        if (!name.isEmpty()){
            QScriptValue f = m_Functions.property(QString("endeffect_%1").arg(name.replace("'", "").replace(" ", "")));
            if (f.isValid()){
                QScriptValue objectValue = m_Engine.newQObject(this);
                m_Engine.globalObject().setProperty("Window", objectValue);
                f.call();
            }
            if (m_Engine.hasUncaughtException()){
                QStringList backTrace = m_Engine.uncaughtExceptionBacktrace();
                QMessageBox msgBox(this);
                msgBox.setWindowTitle(QString("Error in %1").arg(name));

                msgBox.setIcon(QMessageBox::Warning);
                msgBox.setText(QString("%1\n\nBacktrace:\n%2").arg(m_Engine.uncaughtException().toString()).arg(backTrace.join("\n")));
                msgBox.exec();
            }
        }
        m_Player->SetTemporaryCombatSkillItem("");
    }
}

void BookWindow::DoChoice(int number)
{
    Choice* choice = m_Section->Choices.at(number);
    int section = m_Section->Number;

    if (!choice->Script.isEmpty()){
        if (!ExecuteChoiceScript(choice) && choice->Target == 0)
            choice->Abort();
        if (m_Player->IsDead())
            choice->Abort();
    }

    if (!choice->IsAborted()){
        ExitSection();        
        ShowSection(choice->Target);
    }else{
        if (section == m_Section->Number)
            ShowSection(m_Section->Number);
    }
}

void BookWindow::ChoiceSlot()
{
    QPushButton* pbn = (QPushButton*)sender();
    DoChoice(m_ChoiceButtons.indexOf(pbn));
}

void BookWindow::FightSlot()
{
#ifndef Q_WS_MAEMO_5
#if defined(Q_OS_WIN) || defined(Q_OS_WINCE) || defined(Q_OS_LINUX) || defined (Q_OS_MACX)
    hide();
#endif
#endif

    bool evaded = false;
    foreach(Enemy* e, m_Section->Enemies){
        FightWindow* dlg = new FightWindow(this, m_Player, e, e->CanBeEvaded, &m_Engine);
#ifndef Q_WS_MAEMO_5
#if defined(Q_OS_WIN) || defined(Q_OS_WINCE) || defined(Q_OS_LINUX) || defined (Q_OS_MACX)
        dlg->setParent(NULL);
#endif
#endif
        dlg->exec();
        evaded = dlg->HasEvaded();
        delete dlg;
        if (m_Player->IsDead() || evaded)
            break;        
    }

    if (evaded){
        foreach(Enemy* e, m_Section->Enemies)
            e->Endurance = 0;
    }
#ifndef Q_WS_MAEMO_5
    show();
#endif
    ShowSection(m_Section->Number);
    m_Fought = true;

    EndTempCombatSkillEffect();
}

void BookWindow::GetRandomSlot()
{
    ui->pbnRandom->setEnabled(false);
    m_RandomPicker->GetRandomNumber(this);
}

void BookWindow::RandomSlot(int number)
{
    m_DiceValue = number;
    ui->pbnRandom->setEnabled(true);
    EvaluateRandomChoice(number);
}

void BookWindow::EvaluateRandomChoice(int number)
{
    foreach(Choice* c, m_Section->Choices){
        if (c->MinValue <= number && c->MaxValue >= number){
            if (!c->Script.isEmpty()){
                ExecuteChoiceScript(c);            
                if (m_Player->IsDead())
                    c->Abort();
            }
            if (!c->IsAborted()){
                ExitSection();
                ShowSection(c->Target);
            }else{
                ShowSection(m_Section->Number);
            }
            break;
        }
    }
}

void BookWindow::UpdateToolbar()
{
    ui->lblCombatSkill->setText(QString::number(m_Player->GetCombatSkill()));
    ui->lblEndurance->setText(QString::number(m_Player->Endurance));
    ui->lblCrowns->setText(QString::number(m_Player->GoldCrowns));
}

bool BookWindow::ExecuteChoiceScript(Choice* choice)
{
    QScriptValue objectValue = m_Engine.newQObject(choice);
    m_Engine.globalObject().setProperty("Choice", objectValue);

    choice->Accept();
    return ExecuteScript("ChoiceScript", choice->Script);
}

bool BookWindow::ExecuteSystemScript(QString name)
{
    QScriptValue objectValue = m_Engine.newQObject(this);
    m_Engine.globalObject().setProperty("Window", objectValue);

    QScriptValue f = m_Functions.property(QString("sys_%1").arg(name));
    if (f.isValid()){
        f.call();
    }
    if (m_Engine.hasUncaughtException()){
        QStringList backTrace = m_Engine.uncaughtExceptionBacktrace();
        QMessageBox msgBox(this);
        msgBox.setWindowTitle(QString("Error in %1").arg(name));

        msgBox.setIcon(QMessageBox::Warning);
        msgBox.setText(QString("%1\n\nBacktrace:\n%2").arg(m_Engine.uncaughtException().toString()).arg(backTrace.join("\n")));
        msgBox.exec();
        return false;
    }
    return true;
}

bool BookWindow::ExecuteScript(QString name, QString script)
{
    QScriptValue objectValue = m_Engine.newQObject(this);
    m_Engine.globalObject().setProperty("Window", objectValue);

    m_Engine.evaluate(script);
    if (m_Engine.hasUncaughtException()){
        QStringList backTrace = m_Engine.uncaughtExceptionBacktrace();
        QMessageBox msgBox(this);
        msgBox.setWindowTitle(QString("Error in %1").arg(name));

        msgBox.setIcon(QMessageBox::Warning);
        msgBox.setText(QString("%1\n\nBacktrace:\n%2").arg(m_Engine.uncaughtException().toString()).arg(backTrace.join("\n")));
        msgBox.exec();
        return false;
    }
    return true;
}

void BookWindow::LoadBookFunctions()
{
    if (!m_Book->FunctionsScript.isEmpty()){
        m_Functions = m_Engine.evaluate(m_Book->FunctionsScript);
        if (m_Engine.hasUncaughtException()){
            QStringList backTrace = m_Engine.uncaughtExceptionBacktrace();
            QMessageBox msgBox(this);
            msgBox.setWindowTitle("Error in functions");

            msgBox.setIcon(QMessageBox::Warning);
            msgBox.setText(QString("%1\n\nBacktrace:\n%2").arg(m_Engine.uncaughtException().toString()).arg(backTrace.join("\n")));
            msgBox.exec();
        }
    }
}

void BookWindow::GoSlot()
{
    if (!m_Book->Sections.contains(ui->spinSection->value())) {
        QMessageBox msgBox(this);
        msgBox.setWindowTitle("Section not found");

        msgBox.setIcon(QMessageBox::Warning);
        msgBox.setText(QString("The section %1 does not exist").arg(ui->spinSection->value()));
        msgBox.exec();
    }else{
        ExitSection();
        ShowSection(ui->spinSection->value());
    }
}

void BookWindow::Print(QString message)
{
    ui->lblMessage->setText(message);
    if (!message.isEmpty())
        ui->lblMessage->setVisible(true);
}

void BookWindow::EndMission()
{
    ui->pbnInventory->setEnabled(false);
    ui->pbnEnd->setVisible(true);
    SetVisibleChoiceButtons(0);
    ui->pbnFight->setVisible(false);
    ui->pbnRandom->setVisible(false);
    ui->pbnShop->setVisible(false);
}

void BookWindow::EndSlot()
{
    EndWindow* dlg = new EndWindow(this);
    if (dlg->exec() == QDialog::Accepted){
        int result = dlg->GetResult();
        switch (result)
        {
        case EndWindow::Quit:
            QApplication::exit(0);
            break;
        case EndWindow::Load:
            Init();
            ShowSection(m_Section != NULL ? m_Section->Number : 0);
            break;
        case EndWindow::Restart:{
                Load(GetBookSaveFilename(m_Book->Number));
                QScriptValue objectValue = m_Engine.newQObject(m_Player);
                m_Engine.globalObject().setProperty("Player", objectValue);
                ShowSection(m_Section->Number);
            }
            break;
        case EndWindow::MainMenu:
            close();
            break;
        }
    }

    delete dlg;
}

void BookWindow::StatusSlot()
{
#ifndef Q_WS_MAEMO_5
    hide();
#endif
    StatusWindow* dlg = new StatusWindow(this, m_Player, m_Section, &m_Engine, &m_Functions);
    dlg->exec();

    foreach (Item* i, dlg->UsedItems()){
        if (i->GetIsMeal()){
            m_Ate++;
            break;
        }
    }

    delete dlg;
#ifndef Q_WS_MAEMO_5    
    show();
#endif
    ShowSection(m_Section->Number);
}

void BookWindow::PickItemsSlot()
{
#ifndef Q_WS_MAEMO_5
    hide();
#endif
    StatusWindow* dlg = new StatusWindow(this, m_Player, m_Section, &m_Engine, &m_Functions);
    dlg->exec();
    delete dlg;
#ifndef Q_WS_MAEMO_5
    show();
#endif
    ShowSection(m_Section->Number);
}

void BookWindow::MessageBox(QString title, QString message)
{
    CommonStatic::Message(this, title, message);
}

void BookWindow::LoadFinishedSlot(bool)
{
    QWebFrame* frame = ui->webView->page()->currentFrame();
    if (frame != NULL){
        ui->webView->setMinimumHeight(frame->contentsSize().height() + 64);
        ui->webView->setMaximumHeight(16777215);
    }
}

void BookWindow::resizeEvent(QResizeEvent*)
{
    ui->webView->setFixedHeight(30);
    LoadFinishedSlot(true);
    SetChoicesText();
}

void BookWindow::show()
{
    QMainWindow::show();
    if (m_Section)
        QTimer::singleShot(250, this, SLOT(ShowSectionSlot()));
}

void BookWindow::ShowSectionSlot()
{
    ShowSection(m_Section->Number);
}

QString BookWindow::GetTempSaveFilename()
{
    return QString("%1/%2").arg(QDir::tempPath()).arg("temp.save");
}

void BookWindow::Save(QString filename, Player* player, int book, int section)
{
    if (filename.isEmpty())
        filename = QSettings().fileName();

    QSettings settings(filename, QSettings::IniFormat, this);
    settings.clear();

    settings.setValue("book", book);
    settings.setValue("section", section);
    settings.setValue("player", qVariantFromValue(*player));

    settings.beginWriteArray("items");
    int index = 0;
    foreach(Item* i, player->Items){
        settings.setArrayIndex(index);
        if (!i->GetIsWeapon())
            settings.setValue("item", qVariantFromValue(*i));
        else
            settings.setValue("weapon", qVariantFromValue(*(Weapon*)i));
        index++;
    }
    settings.endArray();

    settings.beginWriteArray("hiddenitems");
    index = 0;
    foreach(Item* i, player->HiddenItems){
        settings.setArrayIndex(index);
        if (!i->GetIsWeapon())
            settings.setValue("item", qVariantFromValue(*i));
        else
            settings.setValue("weapon", qVariantFromValue(*(Weapon*)i));
        index++;
    }
    settings.endArray();


    settings.beginWriteArray("kaidisciplines");
    index = 0;
    foreach(KaiDiscipline* d, player->KaiDisciplines){
        settings.setArrayIndex(index);
        settings.setValue("kaidiscipline", d->ToString());
        index++;
    }
    settings.endArray();
    settings.sync();
}

bool BookWindow::Load(QString filename)
{
    if (filename.isEmpty())
        filename = QSettings().fileName();

    QSettings settings(filename, QSettings::IniFormat, this);

    m_Section = NULL;
    int bookNumber = settings.value("book").toInt();
    if (m_Book != NULL){
        delete m_Book;
        m_Book = NULL;
    }
    m_Book = new Book(this);
    if (!m_Book->Load(Books::GetBookFilename(bookNumber))){
        QMessageBox msgBox(this);
        msgBox.setWindowTitle(QString("Error loading book"));

        msgBox.setIcon(QMessageBox::Warning);
        msgBox.setText(QString("Error loading book:\n%1").arg(Books::GetBookFilename(bookNumber)));
        msgBox.exec();

        delete m_Book;
        m_Book = NULL;
        return false;
    }else{
        // Check book minimum interpreter version:
        VersionClass interpreter;
        interpreter.FromString(QCoreApplication::applicationVersion());
        if (m_Book->MinInterpreterVersion > interpreter){
            QMessageBox msgBox(this);
            msgBox.setIcon(QMessageBox::Warning);
            msgBox.setText(QString("This book needs version %1 (you have %2)")
                           .arg(m_Book->MinInterpreterVersion.ToString())
                           .arg(interpreter.ToString()));
            msgBox.exec();

            delete m_Book;
            m_Book = NULL;
            return false;
        }
        LoadBookFunctions();
        int sectionNumber = settings.value("section").toInt();
        m_Section = m_Book->Sections[sectionNumber];
    }

    if (m_Player != NULL){
        delete m_Player;
        m_Player = NULL;
    }

    QVariant var = settings.value("player");
    if (var.isValid()){
        m_Player = new Player(var.value<Player>());
    }

    int index = settings.beginReadArray("items");
    for (int i=0; i<index; i++){
        settings.setArrayIndex(i);
        var = settings.value("item");
        if (var.isValid()){
            Item* i = new Item(var.value<Item>());
            if (i->GetUUID().isEmpty())
                i->SetUUID(QUuid::createUuid().toString());
            // For compatibility with previously saved positions, check if the item has
            // the CanBeEquipped property
            if ((i->GetIsArmor() || i->GetIsWeapon()) && !i->HasProperty("CanBeEquipped"))
                i->SetCanBeEquipped(true);

            // Fix armor class:
            if (i->GetIsArmor()){
                if (!QString::compare(i->GetName(), "Chainmail Waistcoat", Qt::CaseInsensitive))
                    i->SetArmorClass(Armor::Body);
                else if (!QString::compare(i->GetName(), "Padded Leather Waistcoat", Qt::CaseInsensitive))
                    i->SetArmorClass(Armor::Body);
                else if (!QString::compare(i->GetName(), "Silver Helm", Qt::CaseInsensitive))
                    i->SetArmorClass(Armor::Head);
                else if (!QString::compare(i->GetName(), "Shield", Qt::CaseInsensitive)){
                    i->SetArmorClass(Armor::Shield);
                    i->SetHands(1);
                }
            }
            m_Player->Items.append(i);
        }else{
            var = settings.value("weapon");
            if (var.isValid()){
                Weapon* w = new Weapon(var.value<Weapon>());
                if (w->GetUUID().isEmpty())
                    w->SetUUID(QUuid::createUuid().toString());
                // For compatibility with previously saved positions, check if the weapon has
                // the CanBeEquipped property
                if (!w->HasProperty("CanBeEquipped"))
                    w->SetCanBeEquipped(true);

                // Fix weapon class:
                if (w->Classes().count() == 0 || w->Classes().contains(Weapon::None)){
                    if (!QString::compare(w->GetName(), "Dagger", Qt::CaseInsensitive))
                        w->Classes().append(Weapon::Dagger);
                    else if (!QString::compare(w->GetName(), "Spear", Qt::CaseInsensitive))
                        w->Classes().append(Weapon::Spear);
                    else if (!QString::compare(w->GetName(), "Mace", Qt::CaseInsensitive))
                        w->Classes().append(Weapon::Mace);
                    else if (!QString::compare(w->GetName(), "Warhammer", Qt::CaseInsensitive))
                        w->Classes().append(Weapon::WarHammer);
                    else if (!QString::compare(w->GetName(), "Axe", Qt::CaseInsensitive))
                        w->Classes().append(Weapon::Axe);
                    else if (!QString::compare(w->GetName(), "Sword", Qt::CaseInsensitive))
                        w->Classes().append(Weapon::Sword);
                    else if (!QString::compare(w->GetName(), "Quarterstaff", Qt::CaseInsensitive))
                        w->Classes().append(Weapon::Quarterstaff);
                    else if (!QString::compare(w->GetName(), "Broadsword", Qt::CaseInsensitive))
                        w->Classes().append(Weapon::Broadsword);
                }

                // Fix Sommerswerd (in book 2 the CombatskillDelta was wrong):
                if (w->GetName() == "Sommerswerd"){
                    w->SetDescription("");
                    w->SetIsSpecial(true);

                    if (w->GetCombatskillDelta() == 10){
                        w->Classes().clear();
                        w->AddClass(Weapon::Sword);
                        w->AddClass(Weapon::Broadsword);
                        w->AddClass(Weapon::ShortSword);
                        w->SetCombatskillDelta(8);
                    }
                }
                m_Player->Items.append(w);
            }
        }
    }
    settings.endArray();

#ifndef QT_NO_DEBUG
        m_Player->Dump();
#endif

    index = settings.beginReadArray("hiddenitems");
    for (int i=0; i<index; i++){
        settings.setArrayIndex(i);
        var = settings.value("item");
        if (var.isValid()){
            Item* i = new Item(var.value<Item>());
            if (i->GetUUID().isEmpty())
                i->SetUUID(QUuid::createUuid().toString());
            m_Player->HiddenItems.append(i);
        }else{
            var = settings.value("weapon");
            if (var.isValid()){
                Weapon* w = new Weapon(var.value<Weapon>());
                if (w->GetUUID().isEmpty())
                    w->SetUUID(QUuid::createUuid().toString());
                m_Player->HiddenItems.append(w);
            }
        }
    }
    settings.endArray();

    index = settings.beginReadArray("kaidisciplines");
    for (int i=0; i<index; i++){
        settings.setArrayIndex(i);
        QString discipline = settings.value("kaidiscipline").toString();
        QStringList ele = discipline.split("|");
        KaiDiscipline* d = NULL;
        // Kai:
        if (ele.at(0) == "Camouflage")
            d = new Camouflage(m_Player);
        else if (ele.at(0) == "Hunting")
            d = new Hunting(m_Player);
        else if (ele.at(0) == "Sixth Sense")
            d = new SixthSense(m_Player);
        else if (ele.at(0) == "Tracking")
            d = new Tracking(m_Player);
        else if (ele.at(0) == "Healing")
            d = new Healing(m_Player);
        else if (ele.at(0) == "Weaponskill")
            d = new Weaponskill(m_Player);
        else if (ele.at(0) == "Mindshield")
            d = new Mindshield(m_Player);
        else if (ele.at(0) == "Mindblast")
            d = new Mindblast(m_Player);
        else if (ele.at(0) == "Animal Kinship")
            d = new AnimalKinship(m_Player);
        else if (ele.at(0) == "Mind Over Matter")
            d = new MindOverMatter(m_Player);
        //Magnakai:
        else if (ele.at(0) == "Weaponmastery")
            d = new Weaponmastery(m_Player);
        else if (ele.at(0) == "Animal Control")
            d = new AnimalControl(m_Player);
        else if (ele.at(0) == "Curing")
            d = new Curing(m_Player);
        else if (ele.at(0) == "Invisibility")
            d = new Invisibility(m_Player);
        else if (ele.at(0) == "Huntmastery")
            d = new Huntmastery(m_Player);
        else if (ele.at(0) == "Psi-surge")
            d = new PsiSurge(m_Player);
        else if (ele.at(0) == "Psi-screen")
            d = new PsiScreen(m_Player);
        else if (ele.at(0) == "Nexus")
            d = new Nexus(m_Player);
        else if (ele.at(0) == "Divination")
            d = new Divination(m_Player);

        if (d){
            d->FromString(discipline);
            m_Player->KaiDisciplines.append(d);
        }
    }
    settings.endArray();

    //////////////////////////////////////////////////////////////////////////////
    // Fixes for bugs and anomalous situations:
    //////////////////////////////////////////////////////////////////////////////
    // Fix for discipline bug in book 6 (the Kai disciplines were removed):
    if (m_Book->GetSerie() == Book::Magnakai){
        if(!m_Player->HasDiscipline("Camouflage")){
            Camouflage* d = new Camouflage(m_Player);
            m_Player->KaiDisciplines.append(d);
        }
        if(!m_Player->HasDiscipline("Hunting")){
            Hunting* d = new Hunting(m_Player);
            m_Player->KaiDisciplines.append(d);
        }
        if(!m_Player->HasDiscipline("Sixth Sense")){
            SixthSense* d = new SixthSense(m_Player);
            m_Player->KaiDisciplines.append(d);
        }
        if(!m_Player->HasDiscipline("Tracking")){
            Tracking* d = new Tracking(m_Player);
            m_Player->KaiDisciplines.append(d);
        }
        if(!m_Player->HasDiscipline("Healing")){
            Healing* d = new Healing(m_Player);
            m_Player->KaiDisciplines.append(d);
        }
        if(!m_Player->HasDiscipline("Weaponskill")){
            Weaponskill* d = new Weaponskill(m_Player);
            d->Weapon = Weapon::Sword; // Default to sword
            m_Player->KaiDisciplines.append(d);
        }
        if(!m_Player->HasDiscipline("Mindshield")){
            Mindshield* d = new Mindshield(m_Player);
            m_Player->KaiDisciplines.append(d);
        }
        if(!m_Player->HasDiscipline("Mindblast")){
            Mindblast* d = new Mindblast(m_Player);
            m_Player->KaiDisciplines.append(d);
        }
        if(!m_Player->HasDiscipline("Animal Kinship")){
            AnimalKinship* d = new AnimalKinship(m_Player);
            m_Player->KaiDisciplines.append(d);
        }
        if(!m_Player->HasDiscipline("Mind Over Matter")){
            MindOverMatter* d = new MindOverMatter(m_Player);
            m_Player->KaiDisciplines.append(d);
        }
    }
    // Fix for discipline bug in version 0.1.15:
    if (m_Player->HasDiscipline("Psi-surge") && !m_Player->HasDiscipline("Mindblast")){
        Mindblast* m = new Mindblast(m_Player);
        m->Serie = 0; // To avoid counting it in GetDisciplineCount
        m->SetIsActive(false);
        m_Player->KaiDisciplines.append(m);
    }

    // Change Psi-surge when reached Archmaster (9 Disciplines)
    if (m_Player->HasDiscipline("Psi-surge") && m_Player->GetDisciplineCount() >= 9){
        KaiDiscipline* d = m_Player->GetDiscipline("Psi-surge");
        d->CombatSkillDelta = 6;
        d->HelpText = "+6 COMBAT SKILL points but -1 ENDURANCE points per round";
    }

    // Reset the serie of disciplines from other series (to avoid counting it in GetDisciplineCount):
    foreach (KaiDiscipline* d, m_Player->KaiDisciplines){
        if (d->Serie != m_Book->GetSerie())
            d->Serie = 0;
    }

    // Check player hands and armor:
    int hands = m_Player->GetHands();
    foreach (Item* i, m_Player->Items){
        if (i->GetIsEquipped()){
            if (i->GetIsArmor()){
                i->Unequip();
                if (m_Player->CanEquipArmor((Armor*)i, NULL)){
                    i->Equip();
                    hands -= i->GetHands();
                }
            }else if(i->GetIsWeapon()){
                hands -= i->GetHands();
            }

            if(hands < 0 && i->GetHands() > 0){
                i->Unequip();
                hands += i->GetHands();
            }
        }
    }

    m_Loaded = true;
    return true;
}

QString BookWindow::GetBookSaveFilename(int number)
{
    QString filename = QSettings().fileName();
    filename = filename.replace(".ini", QString("_%1.ini").arg(number), Qt::CaseInsensitive);
    return filename;
}

QString BookWindow::ChooseMeal(QString title)
{
    ChooseItemWindow* dlg = new ChooseItemWindow(this, m_Player, ChooseItemWindow::ChooseMeal);
    dlg->setWindowTitle(title);
    dlg->SetMessage(title);
    dlg->exec();
    return dlg->SelectedItem() ? dlg->SelectedItem()->Name : "";
}

QString BookWindow::ChooseWeapon(QString title)
{
    ChooseItemWindow* dlg = new ChooseItemWindow(this, m_Player, ChooseItemWindow::ChooseWeapon);
    dlg->setWindowTitle(title);
    dlg->SetMessage(title);
    dlg->exec();
    return dlg->SelectedItem() ? dlg->SelectedItem()->Name : "";
}

QString BookWindow::ChooseItem(QString title)
{
    ChooseItemWindow* dlg = new ChooseItemWindow(this, m_Player, ChooseItemWindow::ChooseItem);
    dlg->setWindowTitle(title);
    dlg->SetMessage(title);
    dlg->exec();
    return dlg->SelectedItem() ? dlg->SelectedItem()->Name : "";
}

QString BookWindow::ChooseItemExceptMeals(QString title)
{
    ChooseItemWindow* dlg = new ChooseItemWindow(this, m_Player, ChooseItemWindow::ChooseItemExceptMeals);
    dlg->setWindowTitle(title);
    dlg->SetMessage(title);
    dlg->exec();
    return dlg->SelectedItem() ? dlg->SelectedItem()->Name : "";
}

QString BookWindow::ChooseSpecialItem(QString title)
{
    ChooseItemWindow* dlg = new ChooseItemWindow(this, m_Player, ChooseItemWindow::ChooseSpecialItem);
    dlg->setWindowTitle(title);
    dlg->SetMessage(title);
    dlg->exec();
    return dlg->SelectedItem() ? dlg->SelectedItem()->Name : "";
}

bool BookWindow::LoadNextBook()
{
    int bookNumber = m_Book->Number + 1;
    Book* book = new Book(this);
    if (!book->Load(Books::GetBookFilename(bookNumber))){
        QMessageBox msgBox(this);
        msgBox.setWindowTitle(QString("Error loading book"));

        msgBox.setIcon(QMessageBox::Warning);
        msgBox.setText(QString("Error loading book:\n%1").arg(Books::GetBookFilename(bookNumber)));
        msgBox.exec();

        delete book;
        book = NULL;
        return false;
    }
    m_Book = book;
    LoadBookFunctions();
    return true;
}

void BookWindow::RulesSlot()
{
    RulesWindow* dlg = new RulesWindow(this, m_Book->BaseDirectory);
    dlg->show();
}

void BookWindow::MapSlot()
{
    QString filename = QString("%1/Data/map.jpg").arg(m_Book->BaseDirectory);
    MapWindow* dlg = new MapWindow(this, filename);
    dlg->show();
}

void BookWindow::AboutSlot()
{
    AboutWindow* frm = new AboutWindow(this);
    frm->show();
    frm->Init();
    frm->exec();

    delete frm;
}

void BookWindow::AddBonusDiscipline()
{
    BonusDisciplineWindow* dlg = new BonusDisciplineWindow(this, m_Book, m_Player, 1);
    while (dlg->exec() != QDialog::Accepted	){}
    delete dlg;
}

void BookWindow::AddBonusDisciplines(int number)
{
    BonusDisciplineWindow* dlg = new BonusDisciplineWindow(this, m_Book, m_Player, number);
    while (dlg->exec() != QDialog::Accepted	){}
    delete dlg;
}

void BookWindow::ClearPickItems()
{
    m_PickGoldCrowns = 0;
    m_PickItems.clear();
}


void BookWindow::AddPickGoldCrowns(int goldCrowns)
{
    m_PickGoldCrowns = goldCrowns;
}

void BookWindow::AddPickRandomGoldCrowns(int bonus)
{
    AddPickGoldCrowns(-bonus - 1);
}

Item* BookWindow::AddPickItem(QString name, bool special)
{
    if (!name.isEmpty()){
        Item* i = new Item(m_Player);
        i->Name = name;
        i->SetIsSpecial(special);
        m_PickItems.append(i);
        return i;
    }
    return NULL;
}

Item* BookWindow::AddMealPickItem(QString name)
{
    if (!name.isEmpty()){
        Item* i = new Item(m_Player);
        i->Name = name;
        i->SetIsMeal(true);
        m_PickItems.append(i);
        return i;
    }
    return NULL;
}

Weapon* BookWindow::AddWeaponPickItem(QString name, QString className)
{
    if (!name.isEmpty() && !className.isEmpty()){
        Weapon* w = new Weapon(m_Player);
        w->Name = name;
        w->AddClass(Weapon::GetClass(className));
        m_PickItems.append(w);
        return w;
    }
    return NULL;
}

Item* BookWindow::AddHealingPickItem(QString name, int enduranceRestore)
{
    if (!name.isEmpty() && enduranceRestore > 0){
        Item* i = new Item(m_Player);
        i->Name = name;
        i->SetEnduranceRestore(enduranceRestore);
        i->Description = QString("Restores %1 ENDURANCE POINTS").arg(enduranceRestore);
        m_PickItems.append(i);
        return i;
    }
    return NULL;
}

int BookWindow::PickItems(int max)
{
    PickItemsWindow* dlg = new PickItemsWindow(this, m_Player, m_Book->GetSerie(), m_PickGoldCrowns, &m_PickItems, max);
    while (dlg->exec() != QDialog::Accepted	){}
    int picked = dlg->GetPickedItemsCount();
    delete dlg;
    return picked;
}

int BookWindow::SelectGoldCrowns(int min)
{
    SelectGoldCrownsWindow* dlg = new SelectGoldCrownsWindow(this, m_Player, min);
    while (dlg->exec() != QDialog::Accepted	){}
    int res = dlg->GetSelectedValue();
    delete dlg;
    return res;
}

void BookWindow::PlayCartwheel()
{
    CartwheelWindow* dlg = new CartwheelWindow(this, m_Player);
    dlg->exec();
    delete dlg;
}

void BookWindow::PlayPortholes()
{
    PortholesWindow* dlg = new PortholesWindow(this, m_Player);
    dlg->exec();
    delete dlg;
}

int BookWindow::PlayBetGame()
{
    BetWindow* dlg = new BetWindow(this, m_Player);
    dlg->exec();
    int rounds = dlg->GetRounds();
    delete dlg;

    return rounds;
}

int BookWindow::GetRandomNumber()
{
    RandomNumberPicker* rnd = new RandomNumberPicker(this, ui->lblDice);

    int res = 0;
    rnd->GetRandomNumber(this, &res);

    delete rnd;
    return res;
}

int BookWindow::GetRandomNumberWithBonus(int bonus)
{
    RandomNumberPicker* rnd = new RandomNumberPicker(this, ui->lblDice, true, 0, 9, bonus);

    int res = 0;
    rnd->GetRandomNumber(this, &res);

    delete rnd;
    return res;
}

void BookWindow::DisableChoiceButton(int choice)
{
    m_ChoiceButtons.at(choice)->setEnabled(false);
}

void BookWindow::EnableChoiceButton(int choice)
{
    m_ChoiceButtons.at(choice)->setVisible(true);
    m_ChoiceButtons.at(choice)->setEnabled(true);
}

void BookWindow::DisableChoiceButtons()
{
    for (int i=0; i<m_Section->Choices.count(); i++){
        m_ChoiceButtons.at(i)->setEnabled(false);
    }
}

void BookWindow::EnableChoiceButtons()
{
    for (int i=0; i<m_Section->Choices.count(); i++){
        m_ChoiceButtons.at(i)->setEnabled(m_Section->Choices.at(i)->Enabled);
    }
}

void BookWindow::ShopSlot()
{
#ifndef Q_WS_MAEMO_5
    hide();
#endif
    ShopWindow* dlg = new ShopWindow(this, m_Player, m_Section);
    dlg->exec();
#ifndef Q_WS_MAEMO_5
    show();
#endif
    delete dlg;
}

void BookWindow::BookInfoSlot()
{
    BookInfoWindow* dlg = new BookInfoWindow(this, m_Book);
    dlg->exec();
    delete dlg;
}

void BookWindow::BookRestartSlot()
{
    if (CommonStatic::ConfirmMessage(this, "Confirm",
                                     QString("Do you want to restart '%1'?").arg(m_Book->Title)) == QMessageBox::Yes){
        Load(GetBookSaveFilename(m_Book->Number));
        QScriptValue objectValue = m_Engine.newQObject(m_Player);
        m_Engine.globalObject().setProperty("Player", objectValue);
        ShowSection(m_Section->Number);
    }
}

void BookWindow::DisableFightButton()
{
    ui->pbnFight->setEnabled(false);
}

void BookWindow::EnableFightButton()
{
    ui->pbnFight->setEnabled(true);
}

void BookWindow::SetVisibleChoiceButtons(int number)
{
    foreach (QPushButton* pbn, m_ChoiceButtons){
        pbn->setVisible(false);
        pbn->setEnabled(false);
    }

    if (number > 0){
        int index = 0;
        foreach (QPushButton* pbn, m_ChoiceButtons){
            pbn->setVisible(true);
            if (!m_Section->HasRandomChoices() && m_Section->Choices.count() > index)
                pbn->setEnabled(m_Section->Choices.at(index)->Enabled);
            if (++index >= number)
                break;
        }
    }
}

int BookWindow::InputNumber(QString title, int min, int max)
{
    InputNumberWindow* dlg = new InputNumberWindow(this, title, min, max);
    dlg->exec();
    return dlg->GetValue();
}

void BookWindow::ScrollToTopSlot()
{
    ui->scrollArea->ensureVisible(0, 0, 0, 0);
}
