/*
 * loldongs
 *
 * Copyright (C) 2010 Collabora Ltd. <http://www.collabora.co.uk/>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <string.h>

#include <conic.h>

#include <libmcclient/mc-account-manager.h>
#include <libmcclient/mc-profile.h>
#include <libmcclient/mc-account.h>

#include <libosso-abook/osso-abook.h>

#include <telepathy-glib/dbus.h>
#include <telepathy-glib/proxy.h>
#include <telepathy-glib/account.h>
#include <telepathy-glib/interfaces.h>
#include <telepathy-glib/util.h>

#define ACCOUNT_PATH_BASE "/org/freedesktop/Telepathy/Account/"

enum
{
  COL_ACCOUNT = 0,
  COL_ACCOUNT_ICON,
  NUM_COLS
};

static struct
{
  const gchar *text;
  guint value;
  gboolean manual;
  gboolean whitelist;
} account_types[] = {
  { "Manually", 0, TRUE, FALSE },
  { "Only on some connections", 1, FALSE, TRUE },
  { "Except on some connections", 2, FALSE, FALSE },
  { NULL }
};

typedef struct
{
  TpDBusDaemon *dbus;
  gchar *filename;
  GKeyFile *key;
  GtkListStore *store;

  /* on an account */
  GtkWidget *connection_window_vbox;
  gchar *connection_window_object_path;
} LoldongsData;

static void
window_destroy_cb (GtkWindow *window,
    LoldongsData *data)
{
  gchar *key_data;

  key_data = g_key_file_to_data (data->key, NULL, NULL);
  g_file_set_contents (data->filename, key_data, strlen (key_data), NULL);
  g_free (key_data);

  if (data->connection_window_object_path != NULL)
    g_free (data->connection_window_object_path);

  g_key_file_free (data->key);
  g_free (data->filename);
  g_object_unref (data->dbus);
  g_slice_free (LoldongsData, data);

  gtk_main_quit ();
}

static void
connection_selector_changed_cb (HildonTouchSelector *widget,
    gint column,
    gpointer user_data)
{
  LoldongsData *data = user_data;
  GList *selected, *l;
  GtkTreeModel *model;
  GPtrArray *array;

  selected = hildon_touch_selector_get_selected_rows (widget, column);
  model = hildon_touch_selector_get_model (widget, column);

  array = g_ptr_array_sized_new (g_list_length (selected));

  for (l = selected; l != NULL; l = l->next)
    {
      GtkTreeIter iter;
      gchar *name;

      gtk_tree_model_get_iter (model, &iter, (GtkTreePath *) l->data);
      gtk_tree_model_get (model, &iter, 0, &name, -1);

      g_ptr_array_add (array, name);
    }

  g_key_file_set_string_list (data->key, data->connection_window_object_path,
      "connections", (const gchar * const *) array->pdata, array->len);

  g_ptr_array_free (array, TRUE);

  g_list_free (selected);
}

static GtkWidget *
create_connection_view (LoldongsData *data)
{
  GtkWidget *selector;
  GSList *connections, *l;
  ConIcConnection *conn;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar **conns;

  selector = hildon_touch_selector_new_text ();
  hildon_touch_selector_set_column_selection_mode (
      HILDON_TOUCH_SELECTOR (selector),
      HILDON_TOUCH_SELECTOR_SELECTION_MODE_MULTIPLE);

  conn = con_ic_connection_new ();
  connections = con_ic_connection_get_all_iaps (conn);

  for (l = connections; l != NULL; l = l->next)
    {
      ConIcIap *ap = CON_IC_IAP (l->data);

      hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector),
          con_ic_iap_get_name (ap));

      g_object_unref (ap);
    }

  g_object_unref (conn);

  if (g_slist_length (connections) == 0)
    return NULL;

  g_slist_free (connections);

  g_signal_connect (selector, "changed",
      G_CALLBACK (connection_selector_changed_cb), data);

  conns = g_key_file_get_string_list (data->key,
      data->connection_window_object_path, "connections", NULL, NULL);

  if (conns == NULL)
    return selector;

  model = hildon_touch_selector_get_model (HILDON_TOUCH_SELECTOR (selector), 0);

  gtk_tree_model_get_iter_first (model, &iter);

  do
    {
      gchar *c;

      gtk_tree_model_get (model, &iter, 0, &c, -1);

      if (tp_strv_contains ((const gchar * const *) conns, c))
        {
          hildon_touch_selector_select_iter (HILDON_TOUCH_SELECTOR (selector), 0,
              &iter, FALSE);
        }

      g_free (c);
    }
  while (gtk_tree_model_iter_next (model, &iter));

  g_strfreev (conns);

  return selector;
}

static void
account_type_selector_changed_cb (HildonTouchSelector *widget,
    gint column,
    gpointer user_data)
{
  LoldongsData *data = user_data;
  GList *children, *l;
  GtkWidget *vbox = data->connection_window_vbox;
  gint old_type, row;

  row = hildon_touch_selector_get_active (widget, column);

  children = gtk_container_get_children (
      GTK_CONTAINER (data->connection_window_vbox));

  /* remove old children */
  for (l = children; l != NULL; l = l->next)
    {
      /* don't get rid of the top button */
      if (!HILDON_IS_BUTTON (l->data))
        gtk_container_remove (GTK_CONTAINER (vbox), GTK_WIDGET (l->data));
    }

  old_type = g_key_file_get_integer (data->key,
      data->connection_window_object_path, "account-type", NULL);

  g_key_file_set_integer (data->key, data->connection_window_object_path,
      "account-type", row);

  /* if the type has changed, then remove the previous saved connections */
  if (old_type != row)
    {
      g_key_file_remove_key (data->key,
          data->connection_window_object_path, "connections", NULL);
    }

  /* add new child */
  if (account_types[row].manual)
    {
      GtkWidget *label;

      label = gtk_label_new (
          "Manually means you decide when an account goes online\n"
          "by toggling its Enabled setting. Only on some connections\n"
          "allows you to choose which connections this account\n"
          "should be enabled on. Except on some connections allows\n"
          "you to choose which connections this account should be\n"
          "disabled on.");
      gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
      gtk_widget_show (label);
    }
  else
    {
      GtkWidget *view;

      view = create_connection_view (data);

      if (view == NULL)
        view = gtk_label_new ("(no connections)");

      gtk_box_pack_start (GTK_BOX (vbox), view, TRUE, TRUE, 5);
      gtk_widget_show (view);
    }
}

static GtkWidget *
create_account_type_selector ()
{
  GtkWidget *selector = NULL;
  gint i;

  selector = hildon_touch_selector_new_text ();

  for (i = 0; account_types[i].text != NULL ; i++)
    {
      hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector),
          account_types[i].text);
  }

  hildon_touch_selector_set_column_selection_mode (
      HILDON_TOUCH_SELECTOR (selector),
      HILDON_TOUCH_SELECTOR_SELECTION_MODE_SINGLE);

  return selector;
}

static void
show_account_details (LoldongsData *data,
    McAccount *account)
{
  GtkWidget *window;
  GtkWidget *vbox, *button;
  GtkWidget *selector;
  gint val = 0;

  if (data->connection_window_object_path != NULL)
    g_free (data->connection_window_object_path);

  data->connection_window_object_path = g_strdup (
      tp_proxy_get_object_path (account));

  window = hildon_stackable_window_new ();
  gtk_container_set_border_width (GTK_CONTAINER (window), 5);
  gtk_window_set_title (GTK_WINDOW (window),
      osso_abook_mc_account_get_bound_name (account));

  vbox = gtk_vbox_new (FALSE, 0);
  data->connection_window_vbox = vbox;

  button = hildon_picker_button_new (HILDON_SIZE_FINGER_HEIGHT,
      HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);
  hildon_button_set_title (HILDON_BUTTON (button), "Enable this account");

  selector = create_account_type_selector ();
  g_signal_connect (selector, "changed",
      G_CALLBACK (account_type_selector_changed_cb), data);

  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
      HILDON_TOUCH_SELECTOR (selector));

  val = g_key_file_get_integer (data->key,
      tp_proxy_get_object_path (account), "account-type", NULL);
  hildon_picker_button_set_active (HILDON_PICKER_BUTTON (button), val);

  gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
  gtk_widget_show (button);

  gtk_container_add (GTK_CONTAINER (window), vbox);
  gtk_widget_show (vbox);

  gtk_box_reorder_child (GTK_BOX (vbox), button, 0);

  gtk_widget_show (window);
}

static void
account_tree_view_row_activated_cb (GtkTreeView *view,
    GtkTreePath *path,
    GtkTreeViewColumn *column,
    LoldongsData *data)
{
  GtkTreeIter iter;
  McAccount *account;

  if (!gtk_tree_model_get_iter (GTK_TREE_MODEL (data->store), &iter, path))
    return;

  gtk_tree_model_get (GTK_TREE_MODEL (data->store), &iter,
      COL_ACCOUNT, &account,
      -1);

  show_account_details (data, account);

  g_object_unref (account);
}

static gchar *
account_selector_account_type (LoldongsData *data,
    McAccount *account)
{
  GError *error = NULL;
  gint val;

  val = g_key_file_get_integer (data->key, tp_proxy_get_object_path (account),
      "account-type", &error);

  if (error != NULL)
    {
      g_error_free (error);
      return g_strdup (account_types[0].text);
    }

  return g_strdup (account_types[val].text);
}

static void
account_data_func (GtkTreeViewColumn *column,
    GtkCellRenderer *renderer,
    GtkTreeModel *model,
    GtkTreeIter *iter,
    gpointer user_data)
{
  LoldongsData *data = user_data;
  McAccount *account;
  McProfile *profile;
  gchar *str, *type;
  gchar *text_color;
  GtkStyle *style;
  GdkColor color;

  gtk_tree_model_get (model, iter,
      COL_ACCOUNT, &account,
      -1);

  profile = mc_profile_lookup (mc_account_compat_get_profile (account));

  style = gtk_rc_get_style_by_paths (gtk_settings_get_default (),
      NULL, NULL, GTK_TYPE_LABEL);

  gtk_style_lookup_color (style, "SecondaryTextColor", &color);

  text_color = g_strdup_printf ("#%02x%02x%02x",
      color.red / 256, color.green / 256, color.blue / 256);

  type = account_selector_account_type (data, account);

  str = g_markup_printf_escaped (
      "%s (%s)\n<span size=\"x-small\" foreground=\"%s\">%s</span>",
      osso_abook_strip_sip_prefix (
          osso_abook_mc_account_get_bound_name (account)),
      mc_profile_get_display_name (profile), text_color, type);

  g_object_set (renderer, "markup", str, NULL);

  g_free (type);
  g_free (text_color);
  g_free (str);

  g_object_unref (profile);
  g_object_unref (account);
}

static void
account_compat_ready_cb (McAccount *account,
    const GError *error,
    gpointer user_data)
{
  LoldongsData *data = user_data;
  GtkTreeIter iter;
  McProfile *profile;
  const gchar *tmp;
  gchar *icon_name;
  GdkPixbuf *pixbuf;

  profile = mc_profile_lookup (mc_account_compat_get_profile (account));

  tmp = mc_profile_get_icon_name (profile);

  if (tmp == NULL)
    icon_name = g_strdup_printf ("im-%s", mc_profile_get_unique_name (profile));
  else
    icon_name = g_strdup (tmp);

  pixbuf = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (),
      icon_name, 40, 0, NULL);

  g_free (icon_name);

  gtk_list_store_append (data->store, &iter);
  gtk_list_store_set (data->store, &iter,
      COL_ACCOUNT, account,
      COL_ACCOUNT_ICON, pixbuf,
      -1);

  g_object_unref (pixbuf);
  g_object_unref (profile);
}

static void
account_ready_cb (McAccount *account,
    const GError *error,
    gpointer user_data)
{
  mc_account_compat_call_when_ready (account, account_compat_ready_cb,
      user_data);
}

static void
account_manager_ready_cb (McAccountManager *manager,
    const GError *error,
    gpointer user_data)
{
  LoldongsData *data = user_data;
  const gchar * const *accounts;

  accounts = mc_account_manager_get_valid_accounts (manager);

  for (; *accounts != NULL; accounts++)
    {
      McAccount *account;

      if (!tp_strdiff (*accounts, ACCOUNT_PATH_BASE "ring/tel/ring"))
        continue;

      account = mc_account_new (data->dbus, *accounts);
      mc_account_call_when_ready (account, account_ready_cb, data);
      g_object_unref (account);
    }
}

static GtkWidget *
create_account_tree_view (LoldongsData *data)
{
  GtkWidget *view;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
  McAccountManager *manager;

  data->store = gtk_list_store_new (NUM_COLS, MC_TYPE_ACCOUNT, GDK_TYPE_PIXBUF);

  manager = mc_account_manager_new (data->dbus);
  mc_account_manager_call_when_ready (manager, account_manager_ready_cb, data);
  g_object_unref (manager);

  view = hildon_gtk_tree_view_new (HILDON_UI_MODE_NORMAL);
  g_signal_connect (view, "row-activated",
      G_CALLBACK (account_tree_view_row_activated_cb), data);

  /* Account icon column */
  renderer = gtk_cell_renderer_pixbuf_new ();
  g_object_set (renderer,
      "stock-size", HILDON_ICON_SIZE_FINGER,
      "width", HILDON_ICON_PIXEL_SIZE_FINGER + HILDON_MARGIN_DOUBLE,
      NULL);

  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
      -1, "Account icon", renderer, "pixbuf", COL_ACCOUNT_ICON, NULL);

  /* Text column */
  renderer = gtk_cell_renderer_text_new ();

  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_append_column (GTK_TREE_VIEW (view), column);

  gtk_tree_view_column_set_cell_data_func (column, renderer,
      account_data_func, data, NULL);

  gtk_tree_view_set_model (GTK_TREE_VIEW (view), GTK_TREE_MODEL (data->store));

  return view;
}

int
main (gint argc,
    gchar **argv)
{
  HildonProgram *program;
  GtkWidget *window, *account_selector, *pannable_area;
  LoldongsData *data;
  GError *error = NULL;

  hildon_gtk_init (&argc, &argv);
  program = hildon_program_get_instance ();

  data = g_slice_new0 (LoldongsData);
  data->connection_window_object_path = NULL;

  data->dbus = tp_dbus_daemon_dup (&error);
  if (data->dbus == NULL)
    {
      g_critical ("Failed to dup dbus daemon: %s",
          error ? error->message : "no message");
      g_slice_free (LoldongsData, data);
      return 1;
    }

  data->filename = g_build_filename (g_get_home_dir (),
      ".loldongs", NULL);
  data->key = g_key_file_new ();

  if (!g_key_file_load_from_file (data->key, data->filename,
          G_KEY_FILE_NONE, &error))
    g_warning ("Failed to open %s", data->filename);

  window = hildon_stackable_window_new ();
  gtk_window_set_title (GTK_WINDOW (window), "Automatic Connections");
  gtk_container_set_border_width (GTK_CONTAINER (window), 5);
  g_signal_connect (window, "destroy", G_CALLBACK (window_destroy_cb), data);
  gtk_widget_show (window);

  account_selector = create_account_tree_view (data);

  pannable_area = hildon_pannable_area_new ();
  gtk_container_add (GTK_CONTAINER (pannable_area), account_selector);
  gtk_widget_show (account_selector);

  gtk_container_add (GTK_CONTAINER (window), pannable_area);
  gtk_widget_show (pannable_area);

  gtk_main ();

  return 0;
}

