/*
 * This file is part of the Liquid project.
 * It provides drawing and all tools of editor. Also it realizes
 * working with graphic files(open, edit, save).
 *
 * Copyright (C) 2009 Kirpichonock K.N. <kirpiche@cs.karelia.ru>
 * Copyright (C) 2009 Volkov A.A. <volkov@cs.karelia.ru>
 * Copyright (C) 2009 Dmitriev V.V. <vdmitrie@cs.karelia.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

#include "core/painter.h"
#include "ui_mainwindow.h"
#include "filters/filters.h"
#include <QDebug>
#include "error_handler/errors.h"

#define MCE_DISPLAY_ON_STRING "on"
#define MCE_DISPLAY_OFF_STRING "off"
#define BG_MODE_ON "bg_on"
#define BG_MODE_OFF "bg_off"

#define dT 10
#define CONTEXT_MENU_DELAY 500

#define WIDTH 721
#define HEIGHT 423
#define FS_HEIGHT 480

#define SMALL 30
#define MEDIUM 50
#define LARGE 75
#define HUGE_SIZE 100

// w - 721, h - 423
// full w - 721, h - 479

// It is used for getting brush boundaries
static inline QRect circle_bounds(const QPointF &center, qreal radius, qreal compensation) {
    return QRect(qRound(center.x() - radius - compensation),
                 qRound(center.y() - radius - compensation),
                 qRound((radius + compensation) * 2),
                 qRound((radius + compensation) * 2));
}

// Constructor
Painter::Painter(QWidget *parent, ErrorHandler *errorHandler) :
    QWidget(parent),
    m_color(Qt::black),
    m_brushColor(Qt::black),
    m_size(MEDIUM),
    currentSize(MEDIUM)
{
    setBackgroundRole(QPalette::Base);

    eh = errorHandler;
    acc = new Accelerometer(this, errorHandler);
    filter = new Filters();
    mce = new MCE();

    isPainting = false;
    isModified = false;
    isIce = false;
    activated = false;

    currentEffect = 0;

    touchTimer = new QTimer;
    touchTimer->setSingleShot(true);
    activeAcc = true;
    bg_mode = false;

    //Start position for brush is center of workspace
    m_pos = QPointF(int(WIDTH/2 - m_size/2), int(HEIGHT/2 - m_size/2));
    oldPos = m_pos;
    lastX = 0;
    lastY = 0;

    workspaceHeight = HEIGHT;

    theImage = new QImage(WIDTH, FS_HEIGHT, QImage::Format_RGB16);
    theImage->fill(qRgb(255, 255, 255));

    drawPearl();
    createActions();

    connect(touchTimer, SIGNAL(timeout()), SLOT(showContextMenu()));
    connect(acc, SIGNAL(deviceOrientationChanged(QVector3D)), SLOT(setPosition(QVector3D)));
    connect(acc, SIGNAL(accShaking()), SLOT(clearWorkspace()));
    connect(acc, SIGNAL(accRolling()), SLOT(rollHandler()));
    connect(acc, SIGNAL(accModeDisabled()), SLOT(changeModeByShaking()));
    connect(mce, SIGNAL(sigDisplayStateChanged(QString)), SLOT(changeState(QString)));
}

void Painter::changeState(const QString &state) {
    if (!state.isEmpty()) {
        if ((state == MCE_DISPLAY_ON_STRING && !bg_mode) || (state == BG_MODE_OFF))
            acc->disableAccelerometer(false);   // Display on
        else if ((state == MCE_DISPLAY_OFF_STRING && !bg_mode) || (state == BG_MODE_ON)) {
            // Disable getting data from accelerometer when display is off
            // and reset current velocuty of Brush
            acc->disableAccelerometer(true);    // Display off
        }
    }
}

// Creates new empty image
void Painter::createImage() {
    QImage image(WIDTH, FS_HEIGHT, QImage::Format_RGB16);
    image.fill(qRgb(255, 255, 255)); // white color for background
    isModified = false;
    setImage(image);
}

// Loads local graphic file to workspace
bool Painter::openImage(const QString &fileName) {
    QImage image;
    if (!image.load(fileName))
        return false;
    image = image.scaled(QSize(WIDTH, FS_HEIGHT), Qt::KeepAspectRatio, Qt::SmoothTransformation);
    isModified = false;
    setImage(image);
    return true;
}

// Saves current image
bool Painter::saveImage(const QString &fileName, const char *fileFormat) {
    isModified = false;
    return theImage->save(fileName, fileFormat);
}

// Sets current image
void Painter::setImage(const QImage &image) {
    *theImage = image.convertToFormat(QImage::Format_RGB32);
    update();
    updateGeometry();
}

// Handler for mousePressed event
void Painter::mousePressEvent(QMouseEvent *e) {   
    if (!activeAcc) {
        if(e->button() == Qt::LeftButton) {
            touchTimer->start(CONTEXT_MENU_DELAY);

            if(!isIce) {
                if(!((fabs(e->x() - lastX) < m_size/5) && (fabs(e->y() - lastY) < m_size/5))) {
                    m_pos = e->pos();
                    lastX = e->pos().x();
                    lastY = e->pos().y();
                }

                QPainter painter(theImage);
                painter.setRenderHint(QPainter::Antialiasing, true);

                QPen pen(m_color, m_size, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin);
                painter.setPen(pen);
                painter.drawPoint(e->x(), e->y());

                isModified = true;
                checkBoundaries();
            } else {
                if(!((abs(e->x() - lastX) < m_size/5) && (abs(e->y() - lastY) < m_size/5)))
                    m_pos = e->pos();                

                isModified = true;                
                *theImage = filter->blurCircle(e->x(),e->y(),(int)m_size/2, theImage);
            }

            if(currentEffect == BLUR_EFFECT) {
                *theImage = filter->blur(theImage);               
            }
        } else {
            return;
        }
    } else {
        if (!((e->x() > 650) && (e->y() > 350))) {
            if(e->button() == Qt::LeftButton) {
                if ((fabs(m_pos.x() - e->x())< m_size + 10) &&  (fabs(m_pos.y() - e->y()) < m_size + 10))
                    touchTimer->start(CONTEXT_MENU_DELAY);
            } else {
                // If user touchs in Finger Zone, then context menu won't show
                return;
            }
        } else {
            // If user touchs in Finger Zone painting is active
            isPainting = true;
        }
    }
    update();
}

// Handler for contextMenu event
void Painter::showContextMenu() {
    QContextMenuEvent *event = new QContextMenuEvent(QContextMenuEvent::Other,
                                                     QPoint(m_pos.x(), m_pos.y()));
    Painter::contextMenuEvent(event);
}

// Handler for mouseRelease event
void Painter::mouseReleaseEvent(QMouseEvent *e) {
    if(!((e->x() > 650) && (e->y() > 350)))
        if(touchTimer->isActive())
            touchTimer->stop();
    if (activeAcc)
            isPainting = false;
}

// Draws a texture for brush
void Painter::drawPearl() {    
    qreal rad = m_size/2;
    QRect bounds = circle_bounds(QPointF(), rad, 0);

    QPainter painter;

    m_pearl_pixmap = QPixmap(bounds.size());
    m_pearl_pixmap.fill(Qt::transparent);

    if(!painter.begin(&m_pearl_pixmap))
        eh->returnError(PAINTER_PIXMAP_ERROR);

    QRadialGradient gradient(rad, rad, rad, 3 * rad / 5, 3 * rad / 5);
    gradient.setColorAt(0.0, QColor(255, 255, 255, 191));
    gradient.setColorAt(0.2, QColor(241, 241, 241, 191));
    gradient.setColorAt(0.9, m_color.toRgb());
    gradient.setColorAt(0.95, QColor(0, 0, 0, 127));
    gradient.setColorAt(1, QColor(0, 0, 0, 0));
    // painter.setRenderHint(QPainter::Antialiasing);
    painter.setBrush(gradient);
    painter.setPen(Qt::NoPen);
    painter.drawEllipse(0, 0, bounds.width(), bounds.height());
    painter.end();
}

// Handler for mouse movement event
void Painter::mouseMoveEvent(QMouseEvent *e) {
    if (!activeAcc) {
        if ((e->buttons() & Qt::LeftButton)) {
            // need testing !!!
            if(!((fabs(e->x() - lastX) < 5) && fabs(e->y() - lastY) < 5))
                touchTimer->stop();
            if(!isIce) {
                QPainter painter(theImage);
                painter.setRenderHint(QPainter::Antialiasing, true);

                QPen pen(m_color, m_size, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin);
                painter.setPen(pen);
                painter.drawLine(lastX, lastY, e->x(), e->y());

                isModified = true;
                m_pos = e->pos();

                checkBoundaries();
            }
            else {
                m_pos = e->pos();                
                if (currentEffect == NO_EFFECT)
                    *theImage = filter->blurCircle(e->x(),e->y(),(int)m_size/2, theImage);
                isModified = true;
            }
            update();
        }
        lastX = e->x();
        lastY = e->y();
    }
}

// Cleares workspace
void Painter::clearWorkspace() {
    createImage();
 }

// Applies blur filter to image if user rolls device
void Painter::rollHandler() {
    *theImage = filter->blur(theImage);
    update();
}

// Overloading paintEvent of widget
void Painter::paintEvent(QPaintEvent *e) {
    QPainter painter(this);
    painter.drawImage(0, 0 , *theImage);
    painter.drawPixmap(m_pos - QPointF(m_size/2, m_size/2), m_pearl_pixmap);
}

// Sets current color for brush, also change tool to Brush
void Painter::setColor(QColor color) {
    m_brushColor = m_color = color;
    setTool(BRUSH);
    isIce = false;
    update();
}

// Sets current tool
void Painter::setTool(int tool) {
    currentEffect = NO_EFFECT;
    switch(tool) {
        case BRUSH :
            m_color = m_brushColor;
            isIce = false;
            drawPearl();
            break;
        case DROP :
            m_color = Qt::white;
            drawPearl();
            isIce = false;
            break;
        case ICE:
            m_color = Qt::white;
            drawPearl();
            isIce = true;
            break;
        default:
            break;
    }
    update();
}

// Sets thickness of different tools (brush, drop, ice)
void Painter::setThickness(float size) {
    m_size = size;
    currentSize = size;
    drawPearl();
    update();
}

// Sets new position of Brush (accelerometer mode)
void Painter::setPosition(QVector3D accVector3) {
    oldPos = m_pos;
    m_pos.setX(oldPos.x() + accVector3.x());
    m_pos.setY(oldPos.y() + accVector3.y());

    // Brush size depends from current velocity
    vz = accVector3.z()/10;
    size = fabs(log10(vz)/log10(0.25));

    // Upper boundary for brush size
    if(size > 1)
        size = 1;
    else
        // Lower boundary for brush size
        if(size < 0.2)
            size = 0.2;

    m_size = currentSize*size;

    // if brush is out of boundaries, then set it velocity to zero
    if (!checkBoundaries())
        acc->resetVelocity();

    // If user touchs Finger Zone, then painting is active
    if(isPainting) {
        isModified = true;
        if(isIce)
            *theImage = filter->blurCircle(m_pos.x(),m_pos.y(),(int)m_size/2, theImage);
        else {
            QPainter painter(theImage);
            painter.setRenderHint(QPainter::Antialiasing, true);
            QPen pen(m_color, m_size, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin);
            painter.setPen(pen);
            painter.drawLine(oldPos, m_pos);
        }
    }
    drawPearl();
    update();
}

// Sets current effect
void Painter::setEffect(int effect){
    currentEffect = effect;
}

// Controls a movement of drop.
bool  Painter::checkBoundaries() {
    bool res = true;
    if(m_pos.x() >= WIDTH){
        m_pos.setX(WIDTH);
        res = false;
    }
    if(m_pos.x() <= 0){
        m_pos.setX(0);
        res = false;
     }
    if(m_pos.y() >= workspaceHeight){
        m_pos.setY(workspaceHeight);
        res = false;
    }
    if(m_pos.y() <= 0){
        m_pos.setY(0);
        res = false;
     }
    return res;
}

// Activates or disables accelerometer mode
bool Painter::setAccMode(bool turnOn) {
    if(turnOn) {
        activeAcc = true;
        acc->setActivated(true);
    } else {
        activeAcc = false;        
        acc->setActivated(false);
        m_size = currentSize;
        drawPearl();
        update();
    }    
    return true;
}

// Creates actions for size menu
void Painter::createActions() {
    smallBrush = new QAction(tr("Small"),this);
    smallBrush->setCheckable(true);
    connect(smallBrush, SIGNAL(triggered()), SLOT(setSmallSize()));
    sizeActions.append(smallBrush);

    mediumBrush = new QAction(tr("Medium"),this);
    mediumBrush->setCheckable(true);
    mediumBrush->setChecked(true);
    connect(mediumBrush, SIGNAL(triggered()), SLOT(setMediumSize()));
    sizeActions.append(mediumBrush);

    largeBrush = new QAction(tr("Large"),this);
    largeBrush->setCheckable(true);
    connect(largeBrush, SIGNAL(triggered()), SLOT(setLargeSize()));
    sizeActions.append(largeBrush);

    hugeBrush = new QAction(tr("Huge"),this);
    hugeBrush->setCheckable(true);
    connect(hugeBrush, SIGNAL(triggered()), SLOT(setHugeSize()));
    sizeActions.append(hugeBrush);
}

// Overloading of contextMenuEvent
void Painter::contextMenuEvent(QContextMenuEvent *event) {
    if(event->reason() == QContextMenuEvent::Other) {
        QMenu menu(this);
        menu.addActions(sizeActions);
        menu.exec(event->globalPos());
    }
}

// Sets current size of Brush to Small
void Painter::setSmallSize() {
    checkBrushSize(smallBrush);
    m_size = SMALL;
    setThickness(SMALL);
}

// Sets current size of Brush to Medium
void Painter::setMediumSize() {
    checkBrushSize(mediumBrush);
    m_size = MEDIUM;
    setThickness(MEDIUM);
}

// Sets current size of Brush to Large
void Painter::setLargeSize() {
    checkBrushSize(largeBrush);
    m_size = LARGE;
    setThickness(LARGE);
}

// Sets current size of Brush to Huge
void Painter::setHugeSize() {
    checkBrushSize(hugeBrush);
    m_size = HUGE_SIZE;
    setThickness(HUGE_SIZE);
}

// Checks current brush size in context menu
void Painter::checkBrushSize(QAction *action) {
    foreach(QAction *entry, sizeActions) {
        if(entry == action)
            entry->setChecked(true);
        else
            entry->setChecked(false);
    }
}

// Disables accelerometer, if user shakes device by Z axis
void Painter::changeModeByShaking() {
    if(activeAcc) {
        emit disableAccelerometer();
    } else {
        emit activateAccelerometer();
    }
}

// Sets height of workspace to full/normal
void Painter::setFullHeight(bool full) {
    if(full)
        workspaceHeight = FS_HEIGHT;
    else
        workspaceHeight = HEIGHT;
}
