/*
 * This file is part of the Liquid project.
 * It is header file of class Accelerometer, which is interface
 * beetween accelerometer sensor (devOrientation class) and
 * Core Module (Painter class).
 *
 * Copyright (C) 2009 Kirpichonock K.N. <kirpiche@cs.karelia.ru>
 * Copyright (C) 2009 Volkov A.A. <volkov@cs.karelia.ru>
 * Copyright (C) 2009 Dmitriev V.V. <vdmitrie@cs.karelia.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
//! Interface between Accelerometer and Core Module.
/*!
  This class provides an access to Accelerometer functions from
  Core Modules.
*/

#ifndef ACCELEROMETER_H
#define ACCELEROMETER_H

#include "accelerometer/devorientation.h"
#include "error_handler/errorhandler.h"
#include <QString>
#include <QVector3D>
#include <QVector2D>
#include "math.h"
#include <QFile>
#include <QTextStream>
#include <QDebug>

class Accelerometer : public QWidget
{
    Q_OBJECT

public:
    ErrorHandler *eh;
    //! A constructor.
    Accelerometer(QWidget *parent = 0, ErrorHandler *errorHandler = 0);
    //! A destructor.
    ~Accelerometer();
    //! An instance of devOrientation class
    devOrientation *orientation;

    bool disableAccelerometer(bool); /*!< Disables accelerometer. */ 
    bool activated;

    qreal pitch; /*!< A value of pitch angle. */
    qreal roll; /*!< A value of roll angle. */

    float vx; /*!< A value of velocity on X axis. */
    float vy; /*!< A value of velocity on Y axis. */

    QVector2D prevAcc;

    QTimer *blurTimer;

    float offsetX; /*!< A value of offset from last position on X axis. */
    float offsetY; /*!< A value of offset from last position on Y axis. */
    void setActivated(bool);

signals:
    //! A signal.
    /*!
      It generates when data from accelerometer is processed.
      MainWindow class catches this signal.
      \param ax - a value of acceleration on the X axis.
      \param ay - a value of acceleration on the Y axis.
      \param az - a value of acceleration on the Z axis.
    */
    void deviceOrientationChanged(QVector3D);
    //! A signal.
    /*!
      It generates when device is shaking.
      It's uses for clearing workspace.
    */
    void accShaking();
    //! A signal.
    /*!
      It generates when device is rolling.
      It's uses for appling blur filter.
    */
    void accRolling();
    //! A signal.
    /*!
      It's uses for disabling accelerometer mode.
    */
    void accModeDisabled();

public slots:
    //! Slot, which handles a signal devOrientation::dataProcessed().
    /*!
      It converts a string values of acceleration to float and
      emits a signal deviceOrientationChanged().
      \param x - a string with value of accelertion on the X axis.
      \param y - a string with value of accelertion on the Y axis.
      \param z - a string with value of accelertion on the Z axis.
    */
    void setOrientation(QVector3D);
    //! Resets velocity of brush to zero
    void resetVelocity();    
    void applyBlur();
};

#endif // ACCELEROMETER_H
