/*
 * This file is part of the Liquid project.
 * It is interface beetween accelerometer sensor (devOrientation class)
 * and Core Module (Painter class).
 *
 * Copyright (C) 2009 Kirpichonock K.N. <kirpiche@cs.karelia.ru>
 * Copyright (C) 2009 Volkov A.A. <volkov@cs.karelia.ru>
 * Copyright (C) 2009 Dmitriev V.V. <vdmitrie@cs.karelia.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

#include "accelerometer/accelerometer.h"
#include "mainwindow.h"
#include <QDebug>
#include <QLabel>

#define SHAKING_BARRIER 17
#define kAccFilter 0.1
#define EPS 0.000000000001
#define INTERVAL 20

// Constructor
Accelerometer::Accelerometer(QWidget *parent, ErrorHandler *errorHandler)
    : QWidget(parent)
{
    eh = errorHandler;
    orientation = new devOrientation();
    vx = 0.0f;
    vy = 0.0f;
    offsetX = 0.0f;
    offsetY = 0.0f ;
    activated = true;

    prevAcc = QVector2D(0, 0);

    connect(orientation,SIGNAL(dataProcessed(QVector3D)),
            this,SLOT(setOrientation(QVector3D)));
}

Accelerometer::~Accelerometer()
{
}

void Accelerometer::applyBlur() {
    emit accRolling();
}

// Processes values of accelertion and counts offset of Brush
// Also it handles shaking of tablet
void Accelerometer::setOrientation(QVector3D accVector3) {
    // Checks device shaking (clearing workspace)
    if(fabs(accVector3.y()) > SHAKING_BARRIER)
        emit accShaking();

    // Checks device rolling (Blur filter)
    if(fabs(accVector3.x()) > SHAKING_BARRIER)
        emit accRolling();

    // Checks device shaking on Z axis (disable Accelerometer)
    if(fabs(accVector3.z()) > SHAKING_BARRIER)
        emit accModeDisabled();

    if(activated) {
        pitch = (atan2(accVector3.x(), -accVector3.y()));
        roll = (atan2(accVector3.y(), -accVector3.x()));

        accVector3.setX(G*sin(pitch));
        accVector3.setY(G*sin(roll));

        // high-pass filter
        accVector3.setX(accVector3.x()*kAccFilter + (1.0 - kAccFilter)*prevAcc.x());
        accVector3.setY(accVector3.y()*kAccFilter + (1.0 - kAccFilter)*prevAcc.y());

        prevAcc = QVector2D(accVector3.x(), accVector3.y());

        // Counting offset of Brush
        offsetX = vx + (accVector3.x())*dT*dT/2;
        offsetY = vy + (accVector3.y())*dT*dT/2;

        vx = vx + accVector3.x()*dT;
        vy = vy + accVector3.y()*dT;

        if(fabs(vx) < EPS)
            vx = 0;

        if(fabs(vy) < EPS)
            vy = 0;

        accVector3.setX(offsetX);
        accVector3.setY(offsetY);

        accVector3.setZ(sqrt(vx*vx + vy*vy));

        emit deviceOrientationChanged(accVector3);
    } else {
        resetVelocity();
    }
}

void Accelerometer::setActivated(bool activate) {
   activated = activate;
}

// Disables accelerometer
bool Accelerometer::disableAccelerometer(bool disable) {
    if(disable && !orientation->disabled) {
        orientation->killTimer(orientation->timerID);
        resetVelocity();
        orientation->disabled = true;
    } else {
        if (orientation->disabled) {
            orientation->timerID = orientation->startTimer(INTERVAL);
            orientation->disabled = false;
        }
    }

    return true;
}

// Resets current velocity to zero
void Accelerometer::resetVelocity() {
    pitch = roll = 0;
    vx = 0.0f;
    vy = 0.0f;
    offsetX = 0.0f;
    offsetY = 0.0f;
}
